/**********************************************************************************************
    Copyright (C) 2006, 2007 Oliver Eichler oliver.eichler@gmx.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA

  Garmin and MapSource are registered trademarks or trademarks of Garmin Ltd.
  or one of its subsidiaries.

**********************************************************************************************/
#include "CToolViewTrack.h"
#include "CGarminTrack.h"
#include "CCentralResources.h"
#include "icons.h"

#include <QtGui>

CToolViewTrack::CToolViewTrack(QWidget * parent, CGarminDBTrack* db)
    : QWidget(parent)
    , originator(false)
{
    setupUi(this);
    setObjectName("Tracks");
    connect(listTracks,SIGNAL(itemClicked(QTreeWidgetItem*,int) ),this,SLOT(slotItemClicked(QTreeWidgetItem*)));
    connect(listTracks,SIGNAL(itemDoubleClicked(QTreeWidgetItem*,int) ),this,SLOT(slotItemDoubleClicked(QTreeWidgetItem*)));

    connect(db,SIGNAL(sigTrackListChanged()),this,SLOT(slotUpdateTackList()));

    connect(listTracks,SIGNAL(customContextMenuRequested(const QPoint&)),this,SLOT(slotContextMenu(const QPoint&)));

    contextMenu = new QMenu(this);
    contextMenu->addAction(QPixmap(iconTrash16x16),tr("Delete"),this,SLOT(slotDelete()));

}

CToolViewTrack::~CToolViewTrack()
{
}

void CToolViewTrack::slotUpdateTackList()
{
    if(originator) return;
    listTracks->clear();
    QPixmap icon(15,15);

    QMap<QString,CGarminTrack*>& tracks = gpResources->trackdb().getTracks();
    QMap<QString,CGarminTrack*>::iterator track = tracks.begin();

    while(track != tracks.end()){
        QTreeWidgetItem * entry = new QTreeWidgetItem(listTracks);
        icon.fill((*track)->getColor());
        entry->setText(0,(*track)->getName());
        entry->setData(0,Qt::UserRole,track.key());
        entry->setIcon(0,icon);
        entry->setText(1,track.key().section('|',0,0));
        if((*track)->isHighlighted()) entry->setSelected(true);
        ++track;
    }

    listTracks->header()->setResizeMode(0,QHeaderView::Interactive);
    listTracks->resizeColumnToContents(0);
}


void CToolViewTrack::slotItemDoubleClicked(QTreeWidgetItem * item)
{
    QString key = item->data(0,Qt::UserRole).toString();
    CGarminTrack* track = gpResources->trackdb().getTracks()[key];

    QRectF area;

    QVector<CGarminTrack::TrkPt_t>::const_iterator pt1 = track->begin();
    QVector<CGarminTrack::TrkPt_t>::const_iterator pt2 = track->begin();
    area.setTopLeft(QPointF(pt1->lon,pt1->lat));
    area.setBottomRight(QPointF(pt1->lon,pt1->lat));
    ++pt1;
    while(pt1 != track->end()){
        QRectF r;
        r.setTopLeft(QPointF(pt1->lon,pt1->lat));
        r.setBottomRight(QPointF(pt2->lon,pt2->lat));

        area |= r;
        ++pt1; ++pt2;
    }

    gpResources->canvas().zoom(area);
}

void CToolViewTrack::slotItemClicked(QTreeWidgetItem * item)
{
    originator = true;
    gpResources->trackdb().highlightTrack(item->data(0,Qt::UserRole).toString());
    originator = false;
}

void CToolViewTrack::keyPressEvent(QKeyEvent * e)
{
    if(e->key() == Qt::Key_Delete){
        QStringList keys;
        QTreeWidgetItem * item;
        const QList<QTreeWidgetItem*>& items = listTracks->selectedItems();
        foreach(item,items){
            keys << item->data(0,Qt::UserRole).toString();
            delete item;
        }
        originator = true;
        gpResources->trackdb().removeTracks(keys);
        originator = false;
    }
    else{
        QWidget::keyPressEvent(e);
    }
}

void CToolViewTrack::getSelectedTracks(QStringList& keys)
{
    keys.clear();

    QTreeWidgetItem * item;
    QList<QTreeWidgetItem*> items = listTracks->selectedItems();
    foreach(item,items){
        keys << item->data(0,Qt::UserRole).toString();
    }
}

void CToolViewTrack::slotContextMenu(const QPoint& pos)
{
    if(!listTracks->selectedItems().isEmpty()){
        QPoint p = listTracks->mapToGlobal(pos);
        p.setY(p.y() + listTracks->header()->height());
        contextMenu->exec(p);
    }
}

void CToolViewTrack::slotDelete()
{
    QStringList keys;
    QTreeWidgetItem * item;
    const QList<QTreeWidgetItem*>& items = listTracks->selectedItems();
    foreach(item,items){
        keys << item->data(0,Qt::UserRole).toString();
        delete item;
    }
    originator = true;
    gpResources->trackdb().removeTracks(keys);
    originator = false;
}
