/**********************************************************************************************
    Copyright (C) 2006, 2007 Oliver Eichler oliver.eichler@gmx.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA

**********************************************************************************************/
#include "CCentralResources.h"
#include "CMainWindow.h"
#include "CGarminDBMap.h"
#include "CGarminDBWpt.h"
#include "CGarminDBTrack.h"
#include "CGarminDBRoute.h"
#include "CGarminLiveLog.h"
#include "CDlgDevice.h"


#include <IDevice.h>

#include <QtGui>

#define FT2M(x) (int)((double)x * 0.3048 + 0.5)
#define XSTR(x) STR(x)
#define STR(x) #x

#define TIME_OFFSET (631018800 + 12*3600)

CCentralResources * gpResources = 0;

CCentralResources::CCentralResources(CMainWindow& parent)
    : QObject(&parent)
    , main(parent)
    , m_useHttpProxy(false)
    , m_httpProxyPort(8080)
    , m_eBrowser(eFirefox)
    , cmdFirefox("firefox \"%s\" &")
    , cmdKonqueror("kfmclient exec \"%s\"")
    , time_offset(TIME_OFFSET)
    , uploadMap(true)
    , uploadWpt(true)
    , uploadRte(true)
    , downloadMap(true)
    , downloadWpt(true)
    , downloadTrk(true)
{
    gpResources = this;

    try{
        m_pLiveLog  = new CGarminLiveLog(main.tabWidget);
        m_pMapDB    = new CGarminDBMap(main.tabWidget);
        m_pWptDB    = new CGarminDBWpt(main.tabWidget);
        m_pTrackDB  = new CGarminDBTrack(main.tabWidget, main.rightSplitter);
        m_pRouteDB  = new CGarminDBRoute(main.tabWidget);
        m_pCanvas   = new CCanvas(&main);
    }
    catch(const QString& msg){
        qDebug() << msg;
    }
    // reset and set sticky element. That's a good idea?
    m_pMapDB->clear();
    m_pWptDB->clear();
    m_pTrackDB->clear();

    main.rightSplitter->insertWidget(0,m_pCanvas);

    m_pFSM = new CFunctionStateMachine(main);
    main.leftSplitter->addWidget(m_pFSM);

    QSettings cfg;
    QString family  = cfg.value("environment/mapfont/family","Arial").toString();
    int size        = cfg.value("environment/mapfont/size",8).toInt();
    m_mapfont = QFont(family,size);

    m_doMetric        = cfg.value("environment/doMetric",true).toBool();
    m_offsetUTC       = cfg.value("environment/UTCOffset",100).toUInt();
    if(m_offsetUTC == 100){
        QMessageBox::information(0,tr("Time offset ...")
                                ,tr("QLandkarte assumes that your hardware clock is "
                                    "set to UTC time. You must setup the correct "
                                    "time offset in 'Setup->Config'")
                                ,QMessageBox::Ok,QMessageBox::NoButton);
    }
    else{
        setUTCOffset(m_offsetUTC);
    }

    m_useHttpProxy    = cfg.value("network/useProxy",m_useHttpProxy).toBool();
    m_httpProxy       = cfg.value("network/proxy/url",m_httpProxy).toString();
    m_httpProxyPort   = cfg.value("network/proxy/port",m_httpProxyPort).toUInt();

    m_eBrowser        = (bowser_e)cfg.value("network/browser",m_eBrowser).toInt();
    cmdOther          = cfg.value("network/browser/other","my command \"%s\"").toString();

    m_DeviceName      = cfg.value("device/name",m_DeviceName).toString();

    uploadMap         = cfg.value("device/upload/maps",true).toBool();
    uploadWpt         = cfg.value("device/upload/waypoints",true).toBool();
    uploadRte         = cfg.value("device/upload/routes",true).toBool();
    downloadMap       = cfg.value("device/download/maps",true).toBool();
    downloadWpt       = cfg.value("device/download/waypoints",true).toBool();
    downloadTrk       = cfg.value("device/download/tracks",true).toBool();

    serialPort        = cfg.value("device/port","/dev/ttyS0").toString();

    emit sigProxyChanged();

    altPluginPath.clear();
    QStringList env = QProcess::systemEnvironment();
    env = env.filter(QRegExp("QLANDKARTE_LIBDIR=.*"));
    if(!env.isEmpty()){
        altPluginPath = env[0].section('=',1,1);
        qDebug() << "Use alt. plugin path:" << altPluginPath;
    }
}

CCentralResources::~CCentralResources()
{
    QSettings cfg;
    cfg.setValue("environment/mapfont/family",m_mapfont.family());
    cfg.setValue("environment/mapfont/size",m_mapfont.pointSize());
    cfg.setValue("environment/doMetric",m_doMetric);
    cfg.setValue("environment/UTCOffset",m_offsetUTC);

    cfg.setValue("network/useProxy",m_useHttpProxy);
    cfg.setValue("network/proxy/url",m_httpProxy);
    cfg.setValue("network/proxy/port",m_httpProxyPort);

    cfg.setValue("network/browser",m_eBrowser);
    cfg.setValue("network/browser/other",cmdOther);

    cfg.setValue("device/name",m_DeviceName);
    cfg.setValue("device/upload/maps",uploadMap);
    cfg.setValue("device/upload/waypoints",uploadWpt);
    cfg.setValue("device/upload/routes",uploadRte);
    cfg.setValue("device/download/maps",downloadMap);
    cfg.setValue("device/download/waypoints",downloadWpt);
    cfg.setValue("device/download/tracks",downloadTrk);

    cfg.setValue("device/port",serialPort);

    gpResources = 0;
}

const QString CCentralResources::distance2str(double d)
{
    QString str;
    if(doMetric()){
        if(d > 1000){
            str = QString("%1km").arg(d/1000,0,'f',2);
        }
        else{
            str = QString("%1m").arg(d,0,'f',0);
        }
    }
    else{
        str = QString("%1ml").arg(d * 0.6213699E-3,0,'f',2);
    }
    return str;
}

void CCentralResources::upload()
{
    if(uploadMap){
        m_pMapDB->uploadSelectedTiles(false);
    }

    if(uploadWpt){
        m_pWptDB->uploadWaypoints();
    }

    if(uploadRte){
        m_pRouteDB->uploadRoutes();
    }

}

void CCentralResources::download()
{
    if(downloadMap){
        m_pMapDB->downloadMapInfo();
    }

    if(downloadWpt){
        m_pWptDB->downloadWaypoints();
    }

    if(downloadTrk){
        m_pTrackDB->downloadTracks();
    }
}


void CCentralResources::setUTCOffset(int offset)
{
    time_offset = TIME_OFFSET + offset * 3600;
}

void CCentralResources::openLink(const QString& link)
{
    QString cmd;
    if(m_eBrowser == eFirefox){
        cmd.sprintf(cmdFirefox.toAscii(),link.toAscii().data());
    }
    else if(m_eBrowser == eKonqueror){
        cmd.sprintf(cmdKonqueror.toAscii(),link.toAscii().data());
    }
    else if(m_eBrowser == eOther){
        cmd.sprintf(cmdOther.toAscii(),link.toAscii().data());
    }
    else{
        return;
    }

    system(cmd.toAscii());

}

bool CCentralResources::getHttpProxy(QString& url, quint16& port)
{
    url  = m_httpProxy;
    port = m_httpProxyPort;
    return m_useHttpProxy;
}

void CCentralResources::setCurrentToolBoxWidget(QWidget * w)
{
    main.tabWidget->setCurrentWidget(w);
}

void CCentralResources::convertHeight(QString& str)
{
    bool ok;
    quint32 val = str.toUInt(&ok,10);
    if(ok){
        if(m_doMetric){
            str = QString("%1m").arg(FT2M(val));
        }
        else{
            str = QString("%1ft").arg(val);
        }
    }
}

Garmin::IDevice * CCentralResources::device()
{
    Garmin::IDevice * dev = 0;
    Garmin::IDevice * (*func)(const char*);

    while(1){

        if(m_DeviceName.isEmpty()){
            CDlgDevice dlg(0);
            if(dlg.exec() == QDialog::Rejected){
                return 0;
            }
        }
#ifdef WIN32
        QString path        = altPluginPath.isEmpty() ? XSTR(QL_LIBDIR) : altPluginPath;
        QString libname;
        if ( m_DeviceName == "NMEATcp" )
            libname     = QString("%1/NMEATcp.dll").arg(path);
        else
            libname     = QString("%1/GPSMap60CSx.dll").arg(path);
        QString funcname    = QString("init%1").arg(m_DeviceName);
//        QMessageBox::information(0,"path=",path);
//        QMessageBox::information(0,"libname=",libname);
//        QMessageBox::information(0,"funcname=",funcname);
#else        
        QString path        = altPluginPath.isEmpty() ? XSTR(QL_LIBDIR) : altPluginPath;
        QString libname     = QString("%1/lib%2.so").arg(path).arg(m_DeviceName);
        QString funcname    = QString("init%1").arg(m_DeviceName);
#endif

        func = (Garmin::IDevice * (*)(const char*))QLibrary::resolve(libname,funcname.toAscii());
        if(func == 0){
            QMessageBox::warning(0,tr("Error ..."),tr("Failed to load driver."),QMessageBox::Ok,QMessageBox::NoButton);
            m_DeviceName.clear();
            continue;
        }
        dev = func(INTERFACE_VERSION);
        if(dev == 0){
            QMessageBox::warning(0,tr("Error ..."),tr("Driver version mismatch."),QMessageBox::Ok,QMessageBox::NoButton);
            m_DeviceName.clear();
            continue;
        }
        break;
    }
    dev->setPort(serialPort.toLatin1());
    return dev;
}


