/**********************************************************************************************
    Copyright (C) 2007 Oliver Eichler oliver.eichler@gmx.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA

**********************************************************************************************/
#include "CGarminRoute.h"
#include "IProjection.h"
#include "GarminIcons.h"

#include <QtCore>

const XY CGarminRoute::null = {0,0};

CGarminRoute::CGarminRoute(const QString& name, QObject * parent)
: QObject(parent)
, name(name)
, distance(0.0)
, smbl(8198)

{
    getWptIconById(smbl, icon);
}


CGarminRoute::~CGarminRoute()
{

}


void CGarminRoute::setSmbl(const uint16_t symbolid)
{
    smbl = symbolid;
    getWptIconById(smbl, icon);
    emit sigRouteChanged();
}

const Garmin::Route_t& CGarminRoute::toGarminRoute(int id)
{
    Garmin::Route_t::ident = name.toAscii().data();

    XY pt; unsigned cnt = 0;
    Garmin::Route_t::route.clear();
    foreach(pt,routeDegree) {
        Garmin::RtePt_t rtept;
        rtept.lon               = pt.u;
        rtept.lat               = pt.v;
        rtept.Wpt_t::ident      = QString("%1.%2").arg(id).arg(++cnt,3,10,QChar('0')).toAscii().data();
        rtept.Wpt_t::comment    = name.toAscii().data();
        rtept.Wpt_t::smbl       = smbl;

//         getWptIconByName("Waypoint",rtept.Wpt_t::smbl);

        Garmin::Route_t::route.push_back(rtept);
    }

    return *this;
}


const XY& CGarminRoute::getPosition(int idx)
{
    if(idx >= routeDegree.size() || idx >= routeMeter.size() || idx < 0) return null;
    return routeDegree[idx];
}


void CGarminRoute::addPosition(const double lon, const double lat)
{
    XY pt;
    pt.u = lon;
    pt.v = lat;
    routeDegree << pt;

    pt.u *= DEG_TO_RAD;
    pt.v *= DEG_TO_RAD;
    pt = pj_fwd(pt,*gpProj);
    routeMeter << pt;

    calcDistance();

    emit sigRouteChanged();
}


void CGarminRoute::delPosition(const int idx)
{
    if(idx >= routeDegree.size() || idx >= routeMeter.size() || idx < 0) return;
    routeMeter.removeAt(idx);
    routeDegree.removeAt(idx);

    calcDistance();

    emit sigRouteChanged();
}


void CGarminRoute::movePosition(const int idx, const double lon, const double lat)
{
    if(idx >= routeDegree.size() || idx >= routeMeter.size() || idx < 0) return;
    XY pt;
    pt.u = lon;
    pt.v = lat;
    routeDegree[idx] = pt;
    pt.u *= DEG_TO_RAD;
    pt.v *= DEG_TO_RAD;
    pt = pj_fwd(pt,*gpProj);
    routeMeter[idx] = pt;

    calcDistance();

    emit sigRouteChanged();
}


void CGarminRoute::calcDistance()
{
    distance = 0.0;
    if(routeDegree.size() < 2) return;

    XY pt1,pt2;
    double a1,a2;

    QList<XY>::const_iterator p1 = routeDegree.begin();
    QList<XY>::const_iterator p2 = routeDegree.begin() + 1;
    while(p2 != routeDegree.end()) {
        pt1.u = p1->u * DEG_TO_RAD; pt1.v = p1->v * DEG_TO_RAD;
        pt2.u = p2->u * DEG_TO_RAD; pt2.v = p2->v * DEG_TO_RAD;
        distance += gpProj->distance(pt1,pt2,a1,a2);
        ++p2; ++p1;
    }

}
