/*
    libfame - Fast Assembly MPEG Encoder Library
    Copyright (C) 2000-2001 Vivien Chappelier

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "fame.h"
#include "fame_profile.h"
#include "fame_profile_mpeg.h"
#include "fame_encoder.h"
#include "fame_decoder.h"
#include "fame_motion.h"
#include "fame_syntax.h"
#include "fame_shape.h"

static void profile_mpeg_init(fame_profile_t *profile,
			      fame_context_t *context,
			      fame_parameters_t *params,
			      unsigned char *buffer,
			      unsigned int size);
static int profile_mpeg_encode(fame_profile_t *profile,
			       fame_yuv_t *yuv,
			       unsigned char *shape);
static int profile_mpeg_close(fame_profile_t *profile);

FAME_CONSTRUCTOR(fame_profile_mpeg_t)
{
  FAME_OBJECT(this)->name = "MPEG profile";
  FAME_PROFILE(this)->init = profile_mpeg_init;
  FAME_PROFILE(this)->encode = profile_mpeg_encode;
  FAME_PROFILE(this)->close = profile_mpeg_close;
  this->encoder_flags = 0;
  this->decoder_flags = 0;
  this->motion_flags = 0;
  this->syntax_flags = 0;
  this->shape_flags = 0;
  this->rate_flags = 0;
  return(this);
}

/*  profile_mpeg_init                                                        */
/*                                                                           */
/*  Description:                                                             */
/*    Initialize the profile.                                                */
/*                                                                           */
/*  Arguments:                                                               */
/*    fame_profile_t *profile: the profile to initialize                     */
/*    fame_parameters_t *params: the parameters for initialization           */
/*    unsigned char *buffer: the buffer to output data                       */
/*    unsigned int size: the size of the output buffer                       */
/*                                                                           */
/*  Return value:                                                            */
/*    None.                                                                  */

static void profile_mpeg_init(fame_profile_t *profile,
			      fame_context_t *context,
			      fame_parameters_t *params,
			      unsigned char *buffer,
			      unsigned int size)
{
  fame_profile_mpeg_t *profile_mpeg = FAME_PROFILE_MPEG(profile);

  profile_mpeg->width = params->width;
  profile_mpeg->height = params->height;
  profile_mpeg->coding = strdup(params->coding);
  profile_mpeg->quant_scale = 1 + (30*(100-params->quality)+50)/100;
  profile_mpeg->bitrate = params->bitrate;
  profile_mpeg->slices_per_frame = params->slices_per_frame;
  profile_mpeg->frames_per_gop = strlen(profile_mpeg->coding);
  profile_mpeg->frames_per_sequence = params->frames_per_sequence;
  profile_mpeg->mbs_per_slice = ((((params->height + 15) >> 4) + 
			     profile_mpeg->slices_per_frame - 1) / 
			    profile_mpeg->slices_per_frame);
  profile_mpeg->slice_number = 0;
  profile_mpeg->frame_number = 0;
  profile_mpeg->gop_number = 0;
  profile_mpeg->fps_num = params->frame_rate_num;
  profile_mpeg->fps_den = params->frame_rate_den;
  profile_mpeg->alpha_th = 255*(100 - params->shape_quality)/100;
  profile_mpeg->search_range = params->search_range;
  profile_mpeg->intra_y_scale = 0;
  profile_mpeg->intra_c_scale = 0;
  profile_mpeg->intra_matrix = NULL;
  profile_mpeg->inter_matrix = NULL;
  profile_mpeg->verbose = params->verbose;

  /* Get the components */
  profile_mpeg->decoder = 
    (fame_decoder_t *) fame_get_object(context, "decoder");
  profile_mpeg->encoder = 
    (fame_encoder_t *) fame_get_object(context, "encoder");
  profile_mpeg->motion =
    (fame_motion_t *) fame_get_object(context, "motion");
  profile_mpeg->syntax = 
    (fame_syntax_t *) fame_get_object(context, "syntax");
  profile_mpeg->shape = 
    (fame_shape_t *) fame_get_object(context, "shape");
  profile_mpeg->rate = 
    (fame_rate_t *) fame_get_object(context, "rate");

  /* VBR/CBR coding */
  if(profile_mpeg->bitrate == 0)
    profile_mpeg->rate = NULL; /* don't need rate control */

  /* Initialize buffer */
  memset(buffer, 0, size);
  profile_mpeg->buffer = buffer;
  profile_mpeg->size = size;
  profile_mpeg->dirty = 0;

  /* Allocate reference frame ring */
  profile_mpeg->ref[0][0] = 
    (fame_yuv_t *) malloc(sizeof(fame_yuv_t));
  profile_mpeg->ref[0][0]->y = 
    (unsigned char *) malloc(profile_mpeg->width*profile_mpeg->height*12/8+
			     (profile_mpeg->width>>1)+1);/*for interpolation*/
  profile_mpeg->ref[0][0]->u = 
    profile_mpeg->ref[0][0]->y + profile_mpeg->width*profile_mpeg->height;
  profile_mpeg->ref[0][0]->v = 
    profile_mpeg->ref[0][0]->u + profile_mpeg->width*profile_mpeg->height/4;
  profile_mpeg->ref[0][1] = 
    (fame_yuv_t *) malloc(sizeof(fame_yuv_t));
  profile_mpeg->ref[0][1]->y = 
    (unsigned char *) malloc(profile_mpeg->width*profile_mpeg->height*12/8);
  profile_mpeg->ref[0][1]->u = 
    profile_mpeg->ref[0][1]->y + profile_mpeg->width*profile_mpeg->height;
  profile_mpeg->ref[0][1]->v =
    profile_mpeg->ref[0][1]->u + profile_mpeg->width*profile_mpeg->height/4;
  profile_mpeg->ref[0][2] =
    (fame_yuv_t *) malloc(sizeof(fame_yuv_t));
  profile_mpeg->ref[0][2]->y =
    (unsigned char *) malloc(profile_mpeg->width*profile_mpeg->height*12/8);
  profile_mpeg->ref[0][2]->u =
    profile_mpeg->ref[0][2]->y + profile_mpeg->width*profile_mpeg->height;
  profile_mpeg->ref[0][2]->v =
    profile_mpeg->ref[0][2]->u + profile_mpeg->width*profile_mpeg->height/4;
  profile_mpeg->ref[0][3] = 
    (fame_yuv_t *) malloc(sizeof(fame_yuv_t));
  profile_mpeg->ref[0][3]->y =
    (unsigned char *) malloc(profile_mpeg->width*profile_mpeg->height*12/8);
  profile_mpeg->ref[0][3]->u =
    profile_mpeg->ref[0][3]->y + profile_mpeg->width*profile_mpeg->height;
  profile_mpeg->ref[0][3]->v =
    profile_mpeg->ref[0][3]->u + profile_mpeg->width*profile_mpeg->height/4;
  profile_mpeg->ref[1][0] =
    (fame_yuv_t *) malloc(sizeof(fame_yuv_t));
  profile_mpeg->ref[1][0]->y =
    (unsigned char *) malloc(profile_mpeg->width*profile_mpeg->height*12/8+
			     (profile_mpeg->width>>1)+1);/*for interpolation*/
  profile_mpeg->ref[1][0]->u =
    profile_mpeg->ref[1][0]->y + profile_mpeg->width*profile_mpeg->height;
  profile_mpeg->ref[1][0]->v =
    profile_mpeg->ref[1][0]->u + profile_mpeg->width*profile_mpeg->height/4;
  profile_mpeg->ref[1][1] =
    (fame_yuv_t *) malloc(sizeof(fame_yuv_t));
  profile_mpeg->ref[1][1]->y =
    (unsigned char *) malloc(profile_mpeg->width*profile_mpeg->height*12/8);
  profile_mpeg->ref[1][1]->u =
    profile_mpeg->ref[1][1]->y + profile_mpeg->width*profile_mpeg->height;
  profile_mpeg->ref[1][1]->v =
    profile_mpeg->ref[1][1]->u + profile_mpeg->width*profile_mpeg->height/4;
  profile_mpeg->ref[1][2] =
    (fame_yuv_t *) malloc(sizeof(fame_yuv_t));
  profile_mpeg->ref[1][2]->y =
    (unsigned char *) malloc(profile_mpeg->width*profile_mpeg->height*12/8);
  profile_mpeg->ref[1][2]->u =
    profile_mpeg->ref[1][2]->y + profile_mpeg->width*profile_mpeg->height;
  profile_mpeg->ref[1][2]->v =
    profile_mpeg->ref[1][2]->u + profile_mpeg->width*profile_mpeg->height/4;
  profile_mpeg->ref[1][3] =
    (fame_yuv_t *) malloc(sizeof(fame_yuv_t));
  profile_mpeg->ref[1][3]->y =
    (unsigned char *) malloc(profile_mpeg->width*profile_mpeg->height*12/8);
  profile_mpeg->ref[1][3]->u =
    profile_mpeg->ref[1][3]->y + profile_mpeg->width*profile_mpeg->height;
  profile_mpeg->ref[1][3]->v =
    profile_mpeg->ref[1][3]->u + profile_mpeg->width*profile_mpeg->height/4;
  profile_mpeg->past = profile_mpeg->current = profile_mpeg->future = 0;

  /* Allocate reconstructed shape */
  if(profile_mpeg->shape)
    profile_mpeg->ref_shape = (unsigned char *) malloc(profile_mpeg->width*profile_mpeg->height);
  else
    profile_mpeg->ref_shape = NULL;

  /* Initialize motion estimation */
  if(profile_mpeg->motion && profile_mpeg->motion->init)
    profile_mpeg->motion->init(profile_mpeg->motion, 
				(profile_mpeg->width >> 4),
				(profile_mpeg->height >> 4),
				FAME_PROFILE_MPEG(profile)->motion_flags);
  
  /* Initialize the syntax */
  if(profile_mpeg->syntax && profile_mpeg->syntax->init)
    profile_mpeg->syntax->init(profile_mpeg->syntax,
				(profile_mpeg->width >> 4),
				(profile_mpeg->height >> 4),
				profile_mpeg->search_range,
				&profile_mpeg->intra_matrix,
				&profile_mpeg->inter_matrix,
				&profile_mpeg->mismatch,
				FAME_PROFILE_MPEG(profile)->syntax_flags);
  
  /* Initialize the encoder */
  if(profile_mpeg->encoder && profile_mpeg->encoder->init)
    profile_mpeg->encoder->init(profile_mpeg->encoder,
				 profile_mpeg->width,
				 profile_mpeg->height,
				 profile_mpeg->intra_matrix,
				 profile_mpeg->inter_matrix,
				 profile_mpeg->mismatch);

  /* Initialize the decoder */
  if(profile_mpeg->decoder && profile_mpeg->decoder->init)
    profile_mpeg->decoder->init(profile_mpeg->decoder,
				 profile_mpeg->width,
				 profile_mpeg->height,
				 profile_mpeg->intra_matrix,
				 profile_mpeg->inter_matrix,
				 profile_mpeg->mismatch);

  /* Initialize the shape coder */
  if(profile_mpeg->shape && profile_mpeg->shape->init)
    profile_mpeg->shape->init(profile_mpeg->shape,
			       (profile_mpeg->width >> 4),
			       (profile_mpeg->height >> 4),
			       FAME_PROFILE_MPEG(profile)->shape_flags);

  /* Initialize rate control */
  if(profile_mpeg->rate && profile_mpeg->rate->init)
    profile_mpeg->rate->init(profile_mpeg->rate,
			     (profile_mpeg->width >> 4),
			     (profile_mpeg->height >> 4),
			     profile_mpeg->bitrate/
			     profile_mpeg->fps_num*profile_mpeg->fps_den,
			     profile_mpeg->coding,
			     FAME_PROFILE_MPEG(profile)->rate_flags);
}

/*  profile_mpeg_encode                                                      */
/*                                                                           */
/*  Description:                                                             */
/*    Encode a single frame.                                                 */
/*                                                                           */
/*  Arguments:                                                               */
/*    fame_profile_t * profile: the profile handle returned by fame_open     */
/*    fame_yuv_t * yuv: the input frame in raw YUV format (YV12 planar)      */
/*    unsigned char * mask: the input mask (0 = transparent, 255 = opaque)   */
/*                                                                           */
/*  Return value:                                                            */
/*    int : the number of bytes written to buffer                            */

static int profile_mpeg_encode(fame_profile_t *profile,
			       fame_yuv_t *yuv,
			       unsigned char *shape)
{
  fame_profile_mpeg_t *profile_mpeg = FAME_PROFILE_MPEG(profile);
  int x, y;
  /* the 4 Y and 2 C blocks in a macroblock */
  short * blocks[6]; 
  /* the binary alpha block */
  unsigned char *bab;
  fame_bab_t bab_type;
  fame_motion_coding_t motion_coding;
  char coding, next;
  fame_motion_vector_t forward[6];
  fame_box_t bounding_box;
  unsigned char pattern;
  int intra, inter;

  /* Initialize syntax buffer */
  if(profile_mpeg->syntax && profile_mpeg->syntax->use)
    profile_mpeg->syntax->use(profile_mpeg->syntax, profile_mpeg->buffer, profile_mpeg->dirty);

  /* Generate sequence */
  if(profile_mpeg->frame_number % profile_mpeg->frames_per_sequence == 0)
    if(profile_mpeg->syntax && profile_mpeg->syntax->start_sequence)
      profile_mpeg->syntax->start_sequence(profile_mpeg->syntax,
					   profile_mpeg->width,
					   profile_mpeg->height,
					   profile_mpeg->fps_num,
					   profile_mpeg->fps_den,
					   profile_mpeg->size*
					   profile_mpeg->frames_per_gop,
					   profile_mpeg->bitrate);

  /* Generate group of pictures */
  if(profile_mpeg->frame_number % profile_mpeg->frames_per_gop == 0)
    if(profile_mpeg->syntax && profile_mpeg->syntax->start_GOP)
      profile_mpeg->syntax->start_GOP(profile_mpeg->syntax, profile_mpeg->frame_number);

  /* TODO: find bounding box */
  bounding_box.x = 0;
  bounding_box.y = 0;
  bounding_box.w = profile_mpeg->width;
  bounding_box.h = profile_mpeg->height;

  /* Generate picture */
  coding = profile_mpeg->coding[profile_mpeg->frame_number % strlen(profile_mpeg->coding)];
  next = profile_mpeg->coding[(profile_mpeg->frame_number + 1) % strlen(profile_mpeg->coding)];
  if(profile_mpeg->syntax && profile_mpeg->syntax->start_picture)
    profile_mpeg->syntax->start_picture(profile_mpeg->syntax,
					 coding,
					 profile_mpeg->frame_number%profile_mpeg->frames_per_gop,
					 &bounding_box);

  /* Enter the encoder */
  if(profile_mpeg->encoder && profile_mpeg->encoder->enter)
    profile_mpeg->encoder->enter(profile_mpeg->encoder,
				  profile_mpeg->ref[profile_mpeg->past],
				  profile_mpeg->ref[profile_mpeg->current],
				  profile_mpeg->ref[profile_mpeg->future],
				  yuv,
				  profile_mpeg->ref_shape);
  
  if(profile_mpeg->decoder && profile_mpeg->decoder->enter)
    profile_mpeg->decoder->enter(profile_mpeg->decoder,
				  profile_mpeg->ref[profile_mpeg->past],
				  profile_mpeg->ref[profile_mpeg->current],
				  profile_mpeg->ref[profile_mpeg->future],
				  yuv,
				  profile_mpeg->ref_shape);

  if(profile_mpeg->shape && profile_mpeg->shape->enter)
    profile_mpeg->shape->enter(profile_mpeg->shape,
				shape,
				profile_mpeg->ref_shape,
				profile_mpeg->alpha_th);
  
  if(profile_mpeg->motion && profile_mpeg->motion->enter)
    profile_mpeg->motion->enter(profile_mpeg->motion,
				 profile_mpeg->ref[profile_mpeg->future],
				 yuv,
				 profile_mpeg->ref_shape,
				 profile_mpeg->search_range);

  if(profile_mpeg->rate && profile_mpeg->rate->enter)
    profile_mpeg->rate->enter(profile_mpeg->rate,
			      profile_mpeg->ref[profile_mpeg->future],
			      yuv,
			      profile_mpeg->ref_shape,
			      coding);			      

  /* estimate quantiser scale for frame */
  if(profile_mpeg->rate && profile_mpeg->rate->global_estimation)
    profile_mpeg->quant_scale =
      profile_mpeg->rate->global_estimation(profile_mpeg->rate);

  bab_type = bab_all_coded;
  motion_coding = motion_intra;
  pattern = 0x0f; /* all blocks coded */
  
  intra = inter = 0;
  for (y = 0; y < (profile_mpeg->height >> 4); y++) 
  {
    /* Generate slice */
    if(y % profile_mpeg->mbs_per_slice == 0) {
      if(profile_mpeg->syntax && profile_mpeg->syntax->start_slice)
	profile_mpeg->syntax->start_slice(profile_mpeg->syntax,
					   y,
					   profile_mpeg->mbs_per_slice*
					   (profile_mpeg->width>>4),
					   profile_mpeg->quant_scale,
					   &profile_mpeg->intra_y_scale,
					   &profile_mpeg->intra_c_scale);
      if(profile_mpeg->encoder && profile_mpeg->encoder->set_quantisation)
	profile_mpeg->encoder->set_quantisation(profile_mpeg->encoder,
						 profile_mpeg->quant_scale,
						 profile_mpeg->intra_y_scale,
						 profile_mpeg->intra_c_scale);
      if(profile_mpeg->decoder && profile_mpeg->decoder->set_quantisation)
	profile_mpeg->decoder->set_quantisation(profile_mpeg->decoder,
						 profile_mpeg->quant_scale,
						 profile_mpeg->intra_y_scale,
						 profile_mpeg->intra_c_scale);
    }

    for (x = 0; x < (profile_mpeg->width >> 4); x++)
    {
      if(profile_mpeg->shape && profile_mpeg->shape->encode_intra_shape)
	bab_type = profile_mpeg->shape->encode_intra_shape(profile_mpeg->shape,
							    x, y, &bab, &pattern);

      /* compensate motion */
      if(coding == 'P') {
	if(profile_mpeg->syntax && profile_mpeg->syntax->predict_vector)
	  profile_mpeg->syntax->predict_vector(profile_mpeg->syntax,
						x, y, 0, forward);
	if(profile_mpeg->motion && profile_mpeg->motion->estimation)
	  motion_coding = profile_mpeg->motion->estimation(profile_mpeg->motion, x, y, forward);

	/* U and V vectors */
	if(profile_mpeg->syntax && profile_mpeg->syntax->compute_chrominance_vectors)
	  profile_mpeg->syntax->compute_chrominance_vectors(profile_mpeg->syntax, forward);
      } else
	motion_coding = motion_intra;

      if(motion_coding == motion_intra) {
	intra++;
	/* Code intra macroblock */
	if(profile_mpeg->encoder && profile_mpeg->encoder->encode_intra_mb)
	  profile_mpeg->encoder->encode_intra_mb(profile_mpeg->encoder,
						  x, y, blocks,
						  bab_type);

	if(next != 'I')
	  if(profile_mpeg->decoder && profile_mpeg->decoder->reconstruct_intra_mb)
	    profile_mpeg->decoder->reconstruct_intra_mb(profile_mpeg->decoder, x, y, 
							 blocks, bab_type);
	
	/* Write macroblock */
	if(profile_mpeg->syntax && profile_mpeg->syntax->write_intra_mb)
	  profile_mpeg->syntax->write_intra_mb(profile_mpeg->syntax,
						x, y, blocks,
						bab, bab_type, pattern);
      } else {
	inter++;
	/* TODO: check for coded in syntax macroblock depending on error   */
	/* of motion estimation, for inter blocks only.                    */
	
	/* Code inter macroblock */
	if(profile_mpeg->encoder && profile_mpeg->encoder->encode_inter_mb)
	  profile_mpeg->encoder->encode_inter_mb(profile_mpeg->encoder,
						  x, y, blocks,
						  forward, NULL, motion_coding,
						  bab_type);

	if(next != 'I')
	  if(profile_mpeg->decoder && profile_mpeg->decoder->reconstruct_inter_mb)
	    profile_mpeg->decoder->reconstruct_inter_mb(profile_mpeg->decoder, 
							 x, y, blocks,
							 forward, NULL, motion_coding,
							 bab_type);
	/* Write macroblock */
	if(profile_mpeg->syntax && profile_mpeg->syntax->write_inter_mb)
	  profile_mpeg->syntax->write_inter_mb(profile_mpeg->syntax, x, y, blocks,
						bab, bab_type, pattern,
						forward, NULL, motion_coding);
      }
    }
  }

  /* Pad and interpolate for half-pel estimation/compensation */
  if(next != 'I')
    if(profile_mpeg->decoder && profile_mpeg->decoder->pad)
      profile_mpeg->decoder->pad(profile_mpeg->decoder, &bounding_box);

  if(next != 'I' && (profile_mpeg->motion->flags & FAME_MOTION_SUBPEL_SEARCH))
    if(profile_mpeg->decoder && profile_mpeg->decoder->interpolate)
      profile_mpeg->decoder->interpolate(profile_mpeg->decoder, 0);

#ifdef DEBUG_WRITE_RECONSTRUCTED_FRAMES
  /* Write reconstructed frame to standard output */
  fwrite(profile_mpeg->ref[profile_mpeg->current][0]->y,
	 profile_mpeg->width*profile_mpeg->height,
	 1, stdout);
  fwrite(profile_mpeg->ref[profile_mpeg->current][0]->u,
	 profile_mpeg->width*profile_mpeg->height/4,
	 1, stdout);
  fwrite(profile_mpeg->ref[profile_mpeg->current][0]->v,
	 profile_mpeg->width*profile_mpeg->height/4,
	 1, stdout);
#endif

  /* Leave the encoder */
  if(profile_mpeg->encoder && profile_mpeg->encoder->leave)
    profile_mpeg->encoder->leave(profile_mpeg->encoder);

  /* Leave the decoder */
  if(profile_mpeg->decoder && profile_mpeg->decoder->leave)
    profile_mpeg->decoder->leave(profile_mpeg->decoder);

  /* Leave the motion */
  if(profile_mpeg->motion && profile_mpeg->motion->leave)
    profile_mpeg->motion->leave(profile_mpeg->motion);

  /* Rotate reference ring */
  switch(coding) {
    case 'I':
    case 'P':
      profile_mpeg->past = profile_mpeg->future;
      profile_mpeg->future = profile_mpeg->current;
      profile_mpeg->current = !profile_mpeg->current;
    break;
    case 'B':
    break;
    default:
    break;
  }

  /* Increment frame number */
  profile_mpeg->frame_number++;

  /* End of picture */
  if(profile_mpeg->syntax && profile_mpeg->syntax->end_picture)
    profile_mpeg->syntax->end_picture(profile_mpeg->syntax);

  /* Return the number of bytes encoded */
  if(profile_mpeg->syntax && profile_mpeg->syntax->flush)
    profile_mpeg->dirty = profile_mpeg->syntax->flush(profile_mpeg->syntax);
  else
    profile_mpeg->dirty = 0;

  if(profile_mpeg->rate && profile_mpeg->rate->leave)
    profile_mpeg->rate->leave(profile_mpeg->rate, profile_mpeg->dirty * 8);

  /* Show picture info */
  if(profile_mpeg->verbose)
    FAME_INFO("inter/intra %3d%% %dkbits/s quality %d%% %c frame #%d\033[K\r",
	      100*inter/(intra+inter), 
	      (profile_mpeg->dirty * profile_mpeg->fps_num * 8) / 
	      (profile_mpeg->fps_den * 1000),
	      (31 - profile_mpeg->quant_scale) * 100 / 30,
	      coding, profile_mpeg->frame_number);
  return(profile_mpeg->dirty);
}

/*  profile_mpeg_close                                                */
/*                                                                           */
/*  Description:                                                             */
/*    Flush remaining encoded data and cleanup everything.                   */
/*                                                                           */
/*  Arguments:                                                               */
/*    fame_profile_t * profile: the profile handle returned by fame_open     */
/*                                                                           */
/*  Return value:                                                            */
/*    int : the number of bytes written to buffer                            */

static int profile_mpeg_close(fame_profile_t *profile)
{
  fame_profile_mpeg_t *profile_mpeg = FAME_PROFILE_MPEG(profile);
  int bytes_written;

  /* Initialize syntax buffer */
  if(profile_mpeg->syntax && profile_mpeg->syntax->use)
    profile_mpeg->syntax->use(profile_mpeg->syntax, profile_mpeg->buffer, profile_mpeg->dirty);

  /* Generate the sequence ender code */
  if(profile_mpeg->syntax && profile_mpeg->syntax->end_sequence)
    profile_mpeg->syntax->end_sequence(profile_mpeg->syntax);

  /* Flush the syntax buffer */
  if(profile_mpeg->syntax && profile_mpeg->syntax->flush)
    bytes_written = profile_mpeg->syntax->flush(profile_mpeg->syntax);
  else
    bytes_written = 0;

  /* Release the decoder */
  if(profile_mpeg->decoder && profile_mpeg->decoder->close)
    profile_mpeg->decoder->close(profile_mpeg->decoder);

  /* Release the encoder */
  if(profile_mpeg->encoder && profile_mpeg->encoder->close)
    profile_mpeg->encoder->close(profile_mpeg->encoder);

  /* Release the syntax */
  if(profile_mpeg->syntax && profile_mpeg->syntax->close)
    profile_mpeg->syntax->close(profile_mpeg->syntax);

  /* Release the motion estimation  */
  if(profile_mpeg->motion && profile_mpeg->motion->close)
    profile_mpeg->motion->close(profile_mpeg->motion);

  /* Release the shape coder  */
  if(profile_mpeg->shape && profile_mpeg->shape->close)
    profile_mpeg->shape->close(profile_mpeg->shape);

  /* Release the rate controller */
  if(profile_mpeg->rate && profile_mpeg->rate->close)
    profile_mpeg->rate->close(profile_mpeg->rate);

  /* Free reference shape */
  if(profile_mpeg->ref_shape)
    free(profile_mpeg->ref_shape);

  /* Free reference ring */
  free(profile_mpeg->ref[0][0]->y);
  free(profile_mpeg->ref[0][0]);
  free(profile_mpeg->ref[0][1]->y);
  free(profile_mpeg->ref[0][1]);
  free(profile_mpeg->ref[0][2]->y);
  free(profile_mpeg->ref[0][2]);
  free(profile_mpeg->ref[0][3]->y);
  free(profile_mpeg->ref[0][3]);
  free(profile_mpeg->ref[1][0]->y);
  free(profile_mpeg->ref[1][0]);
  free(profile_mpeg->ref[1][1]->y);
  free(profile_mpeg->ref[1][1]);
  free(profile_mpeg->ref[1][2]->y);
  free(profile_mpeg->ref[1][2]);
  free(profile_mpeg->ref[1][3]->y);
  free(profile_mpeg->ref[1][3]);

  /* Print newline for picture codes */
  if(profile_mpeg->verbose)
    FAME_INFO("\n");

  /* Return the number of bytes written to buffer */
  return(bytes_written);
}
