/*
    libfame - Fast Assembly MPEG Encoder Library
    Copyright (C) 2000-2001 Vivien Chappelier

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/**************************** mpeg decoder ***********************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "fame.h"
#include "fame_decoder.h"
#include "fame_decoder_mpeg.h"
#include "table_scale.h"
#if defined(HAS_MMX)
#define arch_enter_state()
#define arch_leave_state() asm("emms")
#include "transpose_mmx.h"
#include "idct_mmx.h"
#include "dequantise_mmx.h"
#include "reconstruct_mmx.h"
#include "pad_int.h" /* TODO */
#include "half_mmx.h"
#else
#define arch_enter_state() 
#define arch_leave_state() 
#include "idct_float.h"
#include "dequantise_float.h"
#include "reconstruct_float.h"
#include "pad_int.h"
#include "half_int.h"
#endif

static void mpeg_init(fame_decoder_t *decoder,
		      int width,
		      int height,
		      unsigned char *intra_quantisation_table,
		      unsigned char *inter_quantisation_table,
		      fame_mismatch_t mismatch_type);
static void mpeg_enter(fame_decoder_t *decoder,
			fame_yuv_t **past_ref,
			fame_yuv_t **new_ref,
			fame_yuv_t **future_ref,
			fame_yuv_t *yuv,
			unsigned char *shape);
static void mpeg_set_quantisation(fame_decoder_t *decoder,
				  unsigned char quant_scale,
				  unsigned char intra_y_scale,
				  unsigned char intra_c_scale);
static void mpeg_reconstruct_intra_mb(fame_decoder_t *decoder,
				       short x,
				       short y,
				       short *blocks[6],
				       fame_bab_t bab_type);
static void mpeg_reconstruct_inter_mb(fame_decoder_t *decoder,
				       short x,
				       short y,
				       short *blocks[6],
				       fame_motion_vector_t *forward,
				       fame_motion_vector_t *backward,
				       fame_motion_coding_t motion_coding,
				       fame_bab_t bab_type);
static void mpeg_pad(fame_decoder_t *decoder,
		     fame_box_t *box);
static void mpeg_interpolate(fame_decoder_t *decoder, int rounding);
static void mpeg_leave(fame_decoder_t *decoder);
static void mpeg_close(fame_decoder_t *decoder);

FAME_CONSTRUCTOR(fame_decoder_mpeg_t)
{
  FAME_OBJECT(this)->name = "MPEG decoder";
  FAME_DECODER(this)->init = mpeg_init;
  FAME_DECODER(this)->enter = mpeg_enter;
  FAME_DECODER(this)->set_quantisation = mpeg_set_quantisation;
  FAME_DECODER(this)->reconstruct_intra_mb = mpeg_reconstruct_intra_mb;
  FAME_DECODER(this)->reconstruct_inter_mb = mpeg_reconstruct_inter_mb;
  FAME_DECODER(this)->pad = mpeg_pad;
  FAME_DECODER(this)->interpolate = mpeg_interpolate;
  FAME_DECODER(this)->leave = mpeg_leave;
  FAME_DECODER(this)->close = mpeg_close;
  return(this);
}

/*  mpeg_init                                                                */
/*                                                                           */
/*  Description:                                                             */
/*    Initialize the decoder.                                                */
/*                                                                           */
/*  Arguments:                                                               */
/*    fame_decoder_t *decoder: the decoder to initialize                     */
/*    int width: width of the frame                                          */
/*    int height: height of the frame                                        */
/*    unsigned char *intra_quantisation_table: quantisation matrix for intra */
/*    unsigned char *inter_quantisation_table: quantisation matrix for inter */
/*    fame_mismatch_t mismatch_type: type of mismatch control                */
/*                                                                           */
/*  Return value:                                                            */
/*    None.                                                                  */

static void mpeg_init(fame_decoder_t *decoder,
		      int width,
		      int height,
		      unsigned char *intra_quantisation_table,
		      unsigned char *inter_quantisation_table,
		      fame_mismatch_t mismatch_type)
{
  fame_decoder_mpeg_t *decoder_mpeg = FAME_DECODER_MPEG(decoder);

  /* set width and height */
  decoder_mpeg->width = width;
  decoder_mpeg->height = height;

  /* allocate padded shape buffer */
  decoder_mpeg->padded = (unsigned char *) malloc(decoder_mpeg->width*
						   decoder_mpeg->height);
  decoder_mpeg->intra_quantisation_table = intra_quantisation_table;
  decoder_mpeg->inter_quantisation_table = inter_quantisation_table;
  decoder_mpeg->mismatch = mismatch_type;
}

/*  mpeg_enter                                                               */
/*                                                                           */
/*  Description:                                                             */
/*    Start encoding a new picture.                                          */
/*                                                                           */
/*  Arguments:                                                               */
/*    fame_decoder_t *decoder: the decoder                                   */
/*    fame_yuv_t **past_ref: past reference images                           */
/*    fame_yuv_t **new_ref: new reconstructed reference images               */
/*    fame_yuv_t **future_ref: future reference images                       */
/*    fame_yuv_t *yuv: source image                                          */
/*    unsigned char *shape: shape binary mask                                */
/*                                                                           */
/*  Return value:                                                            */
/*    None.                                                                  */
  
static void mpeg_enter(fame_decoder_t *decoder,
			fame_yuv_t **past_ref,
			fame_yuv_t **new_ref,
			fame_yuv_t **future_ref,
			fame_yuv_t *yuv,
			unsigned char *shape)
{
  fame_decoder_mpeg_t *decoder_mpeg = FAME_DECODER_MPEG(decoder);

  /* Make pointers on the input frame and reference frame */
  decoder_mpeg->input = yuv;
  decoder_mpeg->past_ref = past_ref;
  decoder_mpeg->new_ref = new_ref;
  decoder_mpeg->future_ref = future_ref;
  decoder_mpeg->shape = shape;
  arch_enter_state();
}

/*  mpeg_set_quantisation                                                    */
/*                                                                           */
/*  Description:                                                             */
/*    Updates the matrices used for quantisation.                            */
/*                                                                           */
/*  Arguments:                                                               */
/*    fame_decoder_t *decoder: the decoder                                   */
/*    unsigned char quant_scale: quantisation scale                          */
/*    unsigned char intra_y_scale: intra DC scaler for Y component           */
/*    unsigned char intra_c_scale: intra DC scaler for U and V components    */
/*  Return value:                                                            */
/*    None.                                                                  */

static void mpeg_set_quantisation(fame_decoder_t *decoder,
				  unsigned char quant_scale,
				  unsigned char intra_y_scale,
				  unsigned char intra_c_scale)
{
  fame_decoder_mpeg_t *decoder_mpeg = FAME_DECODER_MPEG(decoder);
  unsigned char *iqtable = decoder_mpeg->intra_quantisation_table;
  unsigned char *niqtable = decoder_mpeg->inter_quantisation_table;
  dct_t *yiqmatrix = decoder_mpeg->yiqmatrix;
  dct_t *ciqmatrix = decoder_mpeg->ciqmatrix;
  dct_t *niqmatrix = decoder_mpeg->niqmatrix;
  dct_t *yidqmatrix = decoder_mpeg->yidqmatrix;
  dct_t *cidqmatrix = decoder_mpeg->cidqmatrix;
  dct_t *nidqmatrix = decoder_mpeg->nidqmatrix;
  dct_t *psmatrix = decoder_mpeg->psmatrix;
  int i;

  /* compute the intra quantisation and dequantisation DC scaler */
#ifdef HAS_MMX
  asm("emms");
  yiqmatrix[0] = (short) ((double)(1UL<<16)*postscale[0]/intra_y_scale); /*16*/
  yidqmatrix[0] =  (short) (intra_y_scale << 3);
  ciqmatrix[0] = (short) ((double)(1UL<<16)*postscale[0]/intra_c_scale); /*16*/
  cidqmatrix[0] =  (short) (intra_c_scale << 3);
#else
  yiqmatrix[0] = postscale[0] / intra_y_scale;
  yidqmatrix[0] =  intra_y_scale;
  ciqmatrix[0] = postscale[0] / intra_c_scale;
  cidqmatrix[0] =  intra_c_scale;
#endif

  /* compute the intra quantisation and dequantisation matrix */
  for(i = 1; i < 64; i++)
  {
#ifdef HAS_MMX
    yiqmatrix[i] = ciqmatrix[i] =
      (short) ((double)(1UL<<19)*postscale[i] / (quant_scale*iqtable[i]));
    yidqmatrix[i] = cidqmatrix[i] = (short) quant_scale*iqtable[i];
#else
    yiqmatrix[i] = ciqmatrix[i] =
      8.0 * postscale[i] / (quant_scale * iqtable[i]);
    yidqmatrix[i] = cidqmatrix[i] = quant_scale*iqtable[i];
#endif
  }

  /* compute the inter quantisation and dequantisation matrix */
  for(i = 0; i < 64; i++)
  {
#ifdef HAS_MMX
    niqmatrix[i] = (short) ((double)(1UL<<19)*postscale[i]/
			   (quant_scale*niqtable[i]));
    nidqmatrix[i] = (short) quant_scale*niqtable[i];
    psmatrix[i] = (short) ((double)(1UL << 16) * prescale[i]);
#else
    niqmatrix[i] = 8.0 * postscale[i] / (quant_scale * niqtable[i]);
    nidqmatrix[i] = quant_scale*niqtable[i];
    psmatrix[i] = prescale[i];
#endif
  }		     
}  

/*  mpeg_reconstruct_intra_mb                                                */
/*                                                                           */
/*  Description:                                                             */
/*    Reconstruct an intra macroblock for further motion estimation.         */
/*                                                                           */
/*  Arguments:                                                               */
/*    fame_decoder_t *decoder: the decoder                                   */
/*    bitbuffer_t *bb: a bit buffer to write the resulting encoded data to.  */
/*    short x: the x location of the macroblock in macroblock units          */
/*    short y: the y location of the macroblock in macroblock units          */
/*    short *blocks[6]:  the DCT coded blocks                                */
/*    fame_bab_t bab_type: binary alpha block type                           */
/*                                                                           */
/*  Return value:                                                            */
/*    None.                                                                  */

static void mpeg_reconstruct_intra_mb(fame_decoder_t *decoder,
				       short x,
				       short y,
				       short *blocks[6],
				       fame_bab_t bab_type)
{
  fame_decoder_mpeg_t *decoder_mpeg = FAME_DECODER_MPEG(decoder);
  unsigned long offset[6];
  int i, pitch;

  pitch = decoder_mpeg->width;

  /* Make offsets to blocks */
  offset[0] = (y << 4) * pitch + (x << 4);         /* Y(0,0) */
  offset[1] = offset[0] + 8;                       /* Y(0,1) */
  offset[2] = offset[0] + (pitch << 3);            /* Y(1,0) */
  offset[3] = offset[2] + 8;                       /* Y(1,1) */
  offset[4] = (y << 3) * (pitch >> 1) + (x << 3);  /* Cb     */
  offset[5] = (y << 3) * (pitch >> 1) + (x << 3);  /* Cr     */

  /* Reconstruct blocks */
  for(i = 0; i < 4; i++) { /* Y */
    if(decoder_mpeg->mismatch == fame_mismatch_local) {
      dequantise_intra_local(blocks[i],
			     decoder_mpeg->tmpblock,
			     decoder_mpeg->yidqmatrix,
			     decoder_mpeg->psmatrix);
    } else {
      dequantise_intra_global(blocks[i],
			      decoder_mpeg->tmpblock,
			      decoder_mpeg->yidqmatrix,
			      decoder_mpeg->psmatrix);
    }
    idct(decoder_mpeg->tmpblock);
    reconstruct_Y(decoder_mpeg->new_ref[0]->y + offset[i],
		  decoder_mpeg->tmpblock,
		  pitch);
  }
  /* U */
  if(decoder_mpeg->mismatch == fame_mismatch_local) {
    dequantise_intra_local(blocks[4],
			   decoder_mpeg->tmpblock,
			   decoder_mpeg->cidqmatrix,
			   decoder_mpeg->psmatrix);
  } else {
    dequantise_intra_global(blocks[4],
			   decoder_mpeg->tmpblock,
			   decoder_mpeg->cidqmatrix,
			   decoder_mpeg->psmatrix);
  }
  idct(decoder_mpeg->tmpblock);
  reconstruct_C(decoder_mpeg->new_ref[0]->u + offset[4],
		decoder_mpeg->tmpblock,
		pitch >> 1);
  /* V */
  if(decoder_mpeg->mismatch == fame_mismatch_local) {
    dequantise_intra_local(blocks[5],
			   decoder_mpeg->tmpblock,
			   decoder_mpeg->cidqmatrix,
			   decoder_mpeg->psmatrix);
  } else {
    dequantise_intra_global(blocks[5],
			    decoder_mpeg->tmpblock,
			    decoder_mpeg->cidqmatrix,
			    decoder_mpeg->psmatrix);
  }
  idct(decoder_mpeg->tmpblock);
  reconstruct_C(decoder_mpeg->new_ref[0]->v + offset[5],
		decoder_mpeg->tmpblock,
		pitch >> 1);
}

/*  mpeg_reconstruct_inter_mb                                               */
/*                                                                           */
/*  Description:                                                             */
/*    Reconstruct an inter macroblock for further motion estimation.         */
/*                                                                           */
/*  Arguments:                                                               */
/*    fame_decoder_t *decoder: the decoder                                   */
/*    bitbuffer_t *bb: a bit buffer to write the resulting encoded data to.  */
/*    short x: the x location of the macroblock in macroblock units          */
/*    short y: the y location of the macroblock in macroblock units          */
/*    short *blocks[6]:  the DCT coded blocks                                */
/*    fame_motion_vector_t *forward: forward motion vectors                  */
/*    fame_motion_vector_t *backward: backward motion vectors                */
/*    fame_bab_t bab_type: binary alpha block type                           */
/*                                                                           */
/*  Return value:                                                            */
/*    None.                                                                  */

static void mpeg_reconstruct_inter_mb(fame_decoder_t *decoder,
				      short x,
				      short y,
				      short *blocks[6],
				      fame_motion_vector_t *forward,
				      fame_motion_vector_t *backward,
				      fame_motion_coding_t motion_coding,
				      fame_bab_t bab_type)
{
  fame_decoder_mpeg_t *decoder_mpeg = FAME_DECODER_MPEG(decoder);
  unsigned long offset[6];
  signed long motion[6];
  int coded[6];
  signed long residual[6];
  int i, j, pitch;

  pitch = decoder_mpeg->width;

  /* Make offsets to blocks */
  offset[0] = (y << 4) * pitch + (x << 4);         /* Y(0,0) */
  offset[1] = offset[0] + 8;                       /* Y(0,1) */
  offset[2] = offset[0] + (pitch << 3);            /* Y(1,0) */
  offset[3] = offset[2] + 8;                       /* Y(1,1) */
  offset[4] = (y << 3) * (pitch >> 1) + (x << 3);  /* Cb     */
  offset[5] = (y << 3) * (pitch >> 1) + (x << 3);  /* Cr     */

  /* Compute motion offsets (motion is half-pixel coded) */
  for(i = 0; i < 4; i++) {
    /* full-pel motion */
    motion[i] = (forward[i].dy >> 1) * pitch + (forward[i].dx >> 1);
    /* half-pel motion */
    residual[i] = ((forward[i].dy & 1) << 1) | (forward[i].dx & 1);
  }
  for(i = 4; i < 6; i++) {
    /* full-pel motion */
    motion[i] = (forward[i].dy >> 1) * (pitch >> 1) + (forward[i].dx >> 1);
    /* half-pel motion */
    residual[i] = ((forward[i].dy & 1) << 1) | (forward[i].dx & 1);
  }

  /* check for not coded blocks */
  for(j = 0; j < 6; j++) {
    coded[j] = 0;
    if(blocks[j] != NULL)
      for(i = 0; i < 64; i++) {
	coded[j] |= blocks[j][i];
      }
  }

  /* Reconstruct blocks */
  for(i = 0; i < 4; i++) { /* Y */
    if(coded[i]) {

      if(decoder_mpeg->mismatch == fame_mismatch_local) {
	dequantise_inter_local(blocks[i],
			       decoder_mpeg->tmpblock,
			       decoder_mpeg->nidqmatrix,
			       decoder_mpeg->psmatrix);
      } else {
	dequantise_inter_global(blocks[i],
				decoder_mpeg->tmpblock,
				decoder_mpeg->nidqmatrix,
				decoder_mpeg->psmatrix);
      }

      idct(decoder_mpeg->tmpblock);

      sum(decoder_mpeg->new_ref[0]->y + offset[i],
	  decoder_mpeg->future_ref[residual[i]]->y + offset[i] + motion[i],
	  &forward[i].error,
	  decoder_mpeg->tmpblock,
	  pitch);
    } else {
      move(decoder_mpeg->new_ref[0]->y + offset[i],
	   decoder_mpeg->future_ref[residual[i]]->y + offset[i] + motion[i],
	   pitch);
      forward[i].error = 0;
    }
  }
  
  /* U */
  if(coded[4]) {
    if(decoder_mpeg->mismatch == fame_mismatch_local) {
      dequantise_inter_local(blocks[4],
			     decoder_mpeg->tmpblock,
			     decoder_mpeg->nidqmatrix,
			     decoder_mpeg->psmatrix);
    } else {
      dequantise_inter_global(blocks[4],
			      decoder_mpeg->tmpblock,
			      decoder_mpeg->nidqmatrix,
			      decoder_mpeg->psmatrix);
    }
    idct(decoder_mpeg->tmpblock);
    sum(decoder_mpeg->new_ref[0]->u + offset[4],
	decoder_mpeg->future_ref[residual[4]]->u + offset[4] + motion[4],
	&forward[4].error,
	decoder_mpeg->tmpblock,
	pitch >> 1);
  } else {
    move(decoder_mpeg->new_ref[0]->u + offset[4],
	 decoder_mpeg->future_ref[residual[4]]->u + offset[4] + motion[4],
	 pitch >> 1);
    forward[4].error = 0;
  }
  
  /* V */
  if(coded[5]) {
    if(decoder_mpeg->mismatch == fame_mismatch_local) {
      dequantise_inter_local(blocks[5],
			     decoder_mpeg->tmpblock,
			     decoder_mpeg->nidqmatrix,
			     decoder_mpeg->psmatrix);
    } else {
      dequantise_inter_global(blocks[5],
			      decoder_mpeg->tmpblock,
			      decoder_mpeg->nidqmatrix,
			      decoder_mpeg->psmatrix);
    }
    idct(decoder_mpeg->tmpblock);
    sum(decoder_mpeg->new_ref[0]->v + offset[5],
	decoder_mpeg->future_ref[residual[5]]->v + offset[5] + motion[5],
	&forward[5].error,
	decoder_mpeg->tmpblock,
	pitch >> 1);
  } else {
    move(decoder_mpeg->new_ref[0]->v + offset[5],
	 decoder_mpeg->future_ref[residual[4]]->v + offset[5] + motion[5],
	 pitch >> 1);
    forward[5].error = 0;
  }
}

/*  mpeg_pad                                                                 */
/*                                                                           */
/*  Description:                                                             */
/*    Perform repetitive padding of arbitrary shape for motion estimation.   */
/*                                                                           */
/*  Arguments:                                                               */
/*    fame_decoder_t *decoder: the decoder                                   */
/*    fame_box_t box: bounding box                                           */
/*                                                                           */
/*  Return value:                                                            */
/*    None.                                                                  */

static void mpeg_pad(fame_decoder_t *decoder,
		     fame_box_t *box)
{
  fame_decoder_mpeg_t *decoder_mpeg = FAME_DECODER_MPEG(decoder);
  if(decoder_mpeg->shape) {
    pad(decoder_mpeg->width,
	decoder_mpeg->height,
	decoder_mpeg->new_ref[0],
	decoder_mpeg->shape,
	decoder_mpeg->padded,
	box);
  }
}


/*  mpeg_interpolate                                                         */
/*                                                                           */
/*  Description:                                                             */
/*    Compute half-pel resolution frames from reference frame.               */
/*                                                                           */
/*  Arguments:                                                               */
/*    fame_decoder_t *decoder: the decoder                                   */
/*                                                                           */
/*  Return value:                                                            */
/*    None.                                                                  */

static void mpeg_interpolate(fame_decoder_t *decoder, int rounding)
{
  fame_decoder_mpeg_t *decoder_mpeg = FAME_DECODER_MPEG(decoder);

  half_interpolate(decoder_mpeg->width,
		   decoder_mpeg->height,
		   decoder_mpeg->new_ref,
		   rounding);
}

/*  mpeg_leave                                                              */
/*                                                                           */
/*  Description:                                                             */
/*    End the encoding of a picture.                                         */
/*                                                                           */
/*  Arguments:                                                               */
/*    fame_decoder_t *decoder: the decoder                                   */
/*                                                                           */
/*  Return value:                                                            */
/*    None.                                                                  */

static void mpeg_leave(fame_decoder_t *decoder)
{
  arch_leave_state();
}

/*  mpeg_close                                                              */
/*                                                                           */
/*  Description:                                                             */
/*    Release the decoder.                                                   */
/*                                                                           */
/*  Arguments:                                                               */
/*    fame_decoder_t *decoder: the decoder                                   */
/*                                                                           */
/*  Return value:                                                            */
/*    None.                                                                  */

static void mpeg_close(fame_decoder_t *decoder)
{
  fame_decoder_mpeg_t *decoder_mpeg = FAME_DECODER_MPEG(decoder);

  /* free shape padding buffer */
  free(decoder_mpeg->padded);
}
