/*(GPL)
------------------------------------------------------------
   Kobo Deluxe - An enhanced SDL port of XKobo
------------------------------------------------------------
 * Copyright (C) 1995, 1996 Akira Higuchi
 * Copyright (C) 2001-2003, 2005-2007 David Olofson
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define	DBG(x)	x

#undef	DEBUG_OUT

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <math.h>

#ifdef DEBUG
#include "audio.h"
extern "C" {
/*For the oscilloscope*/
#include "a_globals.h"
/*For VU and voice info*/
#include "a_struct.h"
}
#endif

#include "kobolog.h"
#include "config.h"
#include "kobo.h"
#include "states.h"
#include "screen.h"
#include "manage.h"
#include "score.h"
#include "gamectl.h"
#include "random.h"
#include "version.h"
#include "options.h"
#include "myship.h"
#include "enemies.h"

#define	MAX_FPS_RESULTS	64

/* Joystick support */
#define DEFAULT_JOY_LR		0	// Joystick axis left-right default
#define DEFAULT_JOY_UD		1	// Joystick axis up-down default
#define DEFAULT_JOY_FIRE	0	// Default fire button on joystick
#define DEFAULT_JOY_START	1


/*----------------------------------------------------------
	Singletons
----------------------------------------------------------*/
KOBO_sound		sound;


/*----------------------------------------------------------
	Globals
----------------------------------------------------------*/
filemapper_t		*fmap = NULL;
prefs_t			*prefs = NULL;
kobo_gfxengine_t	*gengine = NULL;

screen_window_t		*wscreen = NULL;
dashboard_window_t	*wdash = NULL;
bargraph_t		*whealth = NULL;
bargraph_t		*wtemp = NULL;
bargraph_t		*wttemp = NULL;
radar_map_t		*wmap = NULL;
radar_window_t		*wradar = NULL;
window_t		*wmain = NULL;
display_t		*dhigh = NULL;
display_t		*dscore = NULL;
display_t		*dstage = NULL;
display_t		*dships = NULL;
RGN_region		*logo_region = NULL;

int mouse_x = 0;
int mouse_y = 0;
int mouse_left = 0;
int mouse_middle = 0;
int mouse_right = 0;

int exit_game = 0;


static int main_init()
{
	prefs = new prefs_t;
	fmap = new filemapper_t;
	gengine = new kobo_gfxengine_t;
	return 0;
}


static void main_cleanup()
{
	delete gengine;
	gengine = NULL;
	delete fmap;
	fmap = NULL;
	delete prefs;
	prefs = NULL;
}


/*----------------------------------------------------------
	Various functions
----------------------------------------------------------*/

static void setup_dirs(char *xpath)
{
	fmap->exepath(xpath);

	fmap->addpath("DATA", KOBO_DATA_DIR);

	/*
	 * Graphics data
	 */
	/* Current dir; from within the build tree */
	fmap->addpath("GFX", "./data/gfx");
	/* Real data dir */
	fmap->addpath("GFX", "DATA>>gfx");
	/* Current dir */
	fmap->addpath("GFX", "./gfx");

	/*
	 * Sound data
	 */
	/* Current dir; from within the build tree */
	fmap->addpath("SFX", "./data/sfx");
	/* Real data dir */
	fmap->addpath("SFX", "DATA>>sfx");
	/* Current dir */
	fmap->addpath("SFX", "./sfx");

	/*
	 * Score files (user and global)
	 */
	fmap->addpath("SCORES", KOBO_SCORE_DIR);
	/* 'scores' in current dir (For importing scores, perhaps...) */
// (Disabled for now, since filemapper_t can't tell
// when it hits the same dir more than once...)
//	fmap->addpath("SCORES", "./scores");

	/*
	 * Configuration files
	 */
	fmap->addpath("CONFIG", KOBO_CONFIG_DIR);
	/* System local */
	fmap->addpath("CONFIG", SYSCONF_DIR);
	/* In current dir (last resort) */
	fmap->addpath("CONFIG", "./");
}


static void add_dirs(prefs_t *p)
{
	char buf[300];
	if(p->dir[0])
	{
		char *upath = fmap->sys2unix(p->dir);
		snprintf(buf, 300, "%s/sfx", upath);
		fmap->addpath("SFX", buf, 1);
		snprintf(buf, 300, "%s/gfx", upath);
		fmap->addpath("GFX", buf, 1);
		snprintf(buf, 300, "%s/scores", upath);
		fmap->addpath("SCORES", buf, 1);
		snprintf(buf, 300, "%s", upath);
		fmap->addpath("CONFIG", buf, 0);
	}

	if(p->sfxdir[0])
		fmap->addpath("SFX", fmap->sys2unix(p->sfxdir), 1);

	if(p->gfxdir[0])
		fmap->addpath("GFX", fmap->sys2unix(p->gfxdir), 1);

	if(p->scoredir[0])
		fmap->addpath("SCORES", fmap->sys2unix(p->scoredir), 1);
}


#ifdef DEBUG
static void draw_osc(int mode)
{
	int mx = oscframes;
	if(mx > wmain->width())
		mx = wmain->width();
	int yo = wmain->height() - 40;
	wmain->foreground(wmain->map_rgb(0x000099));
	wmain->fillrect(0, yo, wmain->width(), 1);
	wmain->foreground(wmain->map_rgb(0x990000));
	wmain->fillrect(0, yo - 32, wmain->width(), 1);
	wmain->fillrect(0, yo + 31, wmain->width(), 1);

	switch(mode)
	{
	  case 0:
		wmain->foreground(wmain->map_rgb(0x009900));
		for(int s = 0; s < mx; ++s)
			wmain->point(s, ((oscbufl[s]+oscbufr[s]) >> 11) + yo);
		break;
	  case 1:
		wmain->foreground(wmain->map_rgb(0x009900));
		for(int s = 0; s < mx; ++s)
			wmain->point(s, (oscbufl[s] >> 10) + yo);
		wmain->foreground(wmain->map_rgb(0xcc0000));
		for(int s = 0; s < mx; ++s)
			wmain->point(s, (oscbufr[s] >> 10) + yo);
		break;
	  case 2:
		wmain->foreground(wmain->map_rgb(0x009900));
		for(int s = 0; s < mx; ++s)
			wmain->point(s>>1, (oscbufl[s] >> 10) + yo);
		wmain->foreground(wmain->map_rgb(0xcc0000));
		for(int s = 0; s < mx; ++s)
			wmain->point((s>>1) + (mx>>1), (oscbufr[s] >> 10) + yo);
		wmain->fillrect(mx/2, yo-34, 1, 68);
		break;
	}

	wmain->foreground(wmain->map_rgb(0xcc0000));
	wmain->fillrect(0, yo - 32, 6, limiter.attenuation >> 11);
}


static void draw_vu(void)
{
	int xo = (wmain->width() - 5 * AUDIO_MAX_VOICES) / 2;
	int yo = wmain->height() - 50;
	wmain->foreground(wmain->map_rgb(0x000000));
	for(int s = 4; s < 40; s += 4)
		wmain->fillrect(xo, yo+s, 5 * AUDIO_MAX_VOICES-1, 1);
	wmain->foreground(wmain->map_rgb(0x333333));
	wmain->fillrect(xo-1, yo, 5 * AUDIO_MAX_VOICES+1, 1);
	wmain->fillrect(xo-1, yo+40, 5 * AUDIO_MAX_VOICES+1, 4);
	for(int s = 0; s <= AUDIO_MAX_VOICES; ++s)
		wmain->fillrect(xo + s*5 - 1, yo+1, 1, 39);
	for(int s = 0; s < AUDIO_MAX_VOICES; ++s)
	{
		int vu, vu2, vumin;
		if(VS_STOPPED != voicetab[s].state)
		{
			wmain->foreground(wmain->map_rgb(0x009900));
			wmain->fillrect(xo + 1, yo + 41, 2, 2);

			vu = labs((voicetab[s].ic[VIC_LVOL].v>>1) +
					(voicetab[s].ic[VIC_RVOL].v>>1)) >>
					RAMP_BITS;
#ifdef AUDIO_USE_VU
			vu2 = voicetab[s].vu;
			voicetab[s].vu = 0;
#else
			vu2 = 0;
#endif
			vu2 = (vu2>>4) * (vu>>4) >> 8;
			vu2 >>= 11;
			if(vu2 > 40)
				vu2 = 40;
			vu >>= 11;
			if(vu > 40)
				vu = 40;
			vumin = vu < vu2 ? vu : vu2;
		}
		else
			vu = vu2 = vumin = 0;
		wmain->foreground(wmain->map_rgb(0x006600));
		wmain->fillrect(xo, yo + 40 - vu, 4, vu - vu2);
		wmain->foreground(wmain->map_rgb(0xffcc00));
		wmain->fillrect(xo, yo + 40 - vu2, 4, vu2);
		xo += 5;
	}
}
#endif


/*----------------------------------------------------------
	The main object
----------------------------------------------------------*/
class KOBO_main
{
  public:
#ifdef DEBUG
	static int		audio_vismode;
#endif
	static SDL_Joystick	*joystick;
	static int		js_lr;
	static int		js_ud;
	static int		js_fire;
	static int		js_start;

	static FILE		*logfile;
//	static int		l_open;
//	static int		l_format;

	static Uint32		esc_tick;
	static int		esc_count;
	static int		exit_game_fast;

	// Frame rate counter
	static int		fps_count;
	static int		fps_starttime;
	static int		fps_nextresult;
	static int		fps_lastresult;
	static float		*fps_results;
	static display_t	*dfps;

	// Frame rate limiter
	static float		max_fps_filter;
	static int		max_fps_begin;

	static int		xoffs;
	static int		yoffs;

	// Backup in case we screw up we can't get back up
	static prefs_t		safe_prefs;

	static int open();
	static void close();
	static int run();

	static int open_logging(prefs_t *p);
	static void close_logging();
	static void load_config(prefs_t *p);
	static void save_config(prefs_t *p);

	static void build_screen();
	static int init_display(prefs_t *p);
	static void close_display();

	static void show_progress(prefs_t *p);
	static void progress(int percent, const char *msg);
	static int load_graphics(prefs_t *p);
	static int load_sounds(prefs_t *p, int render_all = 0);

	static int init_js(prefs_t *p);
	static void close_js();

	static int escape_hammering();
	static int quit_requested();
	static void brutal_quit();
	static void pause_game();

	static void print_fps_results();
};

#ifdef DEBUG
int		KOBO_main::audio_vismode = 0;
#endif

SDL_Joystick	*KOBO_main::joystick = NULL;
int		KOBO_main::js_lr = DEFAULT_JOY_LR;
int		KOBO_main::js_ud = DEFAULT_JOY_UD;
int		KOBO_main::js_fire = DEFAULT_JOY_FIRE;
int		KOBO_main::js_start = DEFAULT_JOY_START;

FILE		*KOBO_main::logfile = NULL;
//int		KOBO_main::l_open = 0;
//int		KOBO_main::l_format = 0;

Uint32		KOBO_main::esc_tick = 0;
int		KOBO_main::esc_count = 0;
int		KOBO_main::exit_game_fast = 0;

int		KOBO_main::fps_count = 0;
int		KOBO_main::fps_starttime = 0;
int		KOBO_main::fps_nextresult = 0;
int		KOBO_main::fps_lastresult = 0;
float		*KOBO_main::fps_results = NULL;
display_t	*KOBO_main::dfps = NULL;

float		KOBO_main::max_fps_filter = 0.0f;
int		KOBO_main::max_fps_begin = 0;

int		KOBO_main::xoffs = 0;
int		KOBO_main::yoffs = 0;

prefs_t		KOBO_main::safe_prefs;


static KOBO_main km;


void KOBO_main::print_fps_results()
{
	int i, r = fps_nextresult;
	if(fps_lastresult != MAX_FPS_RESULTS-1)
		r = 0;
	for(i = 0; i < fps_lastresult; ++i)
	{
		log_printf(ULOG, "%.1f fps\n", fps_results[r++]);
		if(r >= MAX_FPS_RESULTS)
			r = 0;
	}

	free(fps_results);
	fps_nextresult = 0;
	fps_lastresult = 0;
	fps_results = NULL;
}


int KOBO_main::escape_hammering()
{
	Uint32 nt = SDL_GetTicks();
	if(nt - esc_tick > 500)
		esc_count = 1;
	else
		++esc_count;
	esc_tick = nt;
	return esc_count >= 5;
}


int KOBO_main::quit_requested()
{
	SDL_Event e;
	while(SDL_PollEvent(&e))
	{
		switch(e.type)
		{
		  case SDL_QUIT:
			exit_game_fast = 1;
			break;
		  case SDL_VIDEOEXPOSE:
			gengine->invalidate();
			break;
		  case SDL_KEYUP:
			switch(e.key.keysym.sym)
			{
			  case SDLK_ESCAPE:
				if(escape_hammering())
					exit_game_fast = 1;
				break;
			  default:
				break;
			}
			break;
		}
	}
	if(exit_game_fast)
		return 1;
	return SDL_QuitRequested();
}


void KOBO_main::close_logging()
{
	/* Flush logs to disk, close log files etc. */
	log_close();

	if(logfile)
	{
		fclose(logfile);
		logfile = NULL;
	}
//	l_open = 0;
}


int KOBO_main::open_logging(prefs_t *p)
{
#if 0
	if(l_open)
	{
		//Reopen
		if(p->logformat != l_format)
			l_open = 0;
		if((p->logfile != 0) != (logfile != NULL))
			l_open = 0;
		if(!l_open)
			close_logging();
		l_format = p->logformat;
	}
#endif
	close_logging();

	if(log_open() < 0)
		return -1;

	if(p && p->logfile)
		switch (p->logformat)
		{
		  case 2:
			logfile = fopen("log.html", "wb");
			break;
		  default:
			logfile = fopen("log.txt", "wb");
			break;
		}

	if(logfile)
	{
		log_set_target_stream(0, logfile);
		log_set_target_stream(1, logfile);
	}
	else
	{
		log_set_target_stream(0, stdout);
		log_set_target_stream(1, stderr);
	}

	log_set_target_stream(2, NULL);

	if(p)
		switch(p->logformat)
		{
		  default:
			log_set_target_flags(-1, LOG_TIMESTAMP);
			break;
		  case 1:
			log_set_target_flags(-1, LOG_ANSI | LOG_TIMESTAMP);
			break;
		  case 2:
			log_set_target_flags(-1, LOG_HTML | LOG_TIMESTAMP);
			break;
		}

	/* All levels output to stdout... */
	log_set_level_target(-1, 0);

	/* ...except these, that output to stderr. */
	log_set_level_target(ELOG, 1);
	log_set_level_target(CELOG, 1);

	/* Some fancy colors... */
	log_set_level_attr(ULOG, LOG_YELLOW);
	log_set_level_attr(WLOG, LOG_YELLOW | LOG_BRIGHT);
	log_set_level_attr(ELOG, LOG_RED | LOG_BRIGHT);
	log_set_level_attr(CELOG, LOG_RED | LOG_BRIGHT | LOG_BLINK);
	log_set_level_attr(DLOG, LOG_CYAN);
	log_set_level_attr(D2LOG, LOG_BLUE | LOG_BRIGHT);
	log_set_level_attr(D3LOG, LOG_BLUE);

	/* Disable levels as desired */
	if(p)
		switch(p->logverbosity)
		{
		  case 0:
			log_set_level_target(ELOG, 2);
		  case 1:
			log_set_level_target(WLOG, 2);
		  case 2:
			log_set_level_target(DLOG, 2);
		  case 3:
			log_set_level_target(D2LOG, 2);
		  case 4:
			log_set_level_target(D3LOG, 2);
		  case 5:
			break;
		}

	if(p && p->logfile && !logfile)
		log_printf(ELOG, "Couldn't open log file!\n");

//	l_open = 1;
	return 0;
}


void KOBO_main::build_screen()
{
	gengine->clear(0x000000);

	wdash->place(xoffs, yoffs, SCREEN_WIDTH, SCREEN_HEIGHT);

	whealth->place(xoffs + 4, yoffs + 92, 8, 128);
	whealth->bgimage(B_HEALTH_LID, 0);
	whealth->background(whealth->map_rgb(0x182838));
	whealth->redmax(0);

	wmain->place(xoffs + 8 + MARGIN, yoffs + MARGIN, WSIZE, WSIZE);
	gengine->output(wmain);

	dhigh->place(xoffs + 252, yoffs + 4, 64, 18);
	dhigh->font(B_NORMAL_FONT);
	dhigh->bgimage(B_HIGH_BACK, 0);
	dhigh->caption("HIGHSCORE");
	dhigh->text("000000000");

	dscore->place(dhigh->x(), dhigh->y2() + 4, 64, 18);
	dscore->font(B_NORMAL_FONT);
	dscore->bgimage(B_SCORE_BACK, 0);
	dscore->caption("SCORE");
	dscore->text("000000000");

	wmap->place(0, 0, MAP_SIZEX, MAP_SIZEY);
	wmap->offscreen();

	wradar->place(xoffs + 244,
			yoffs + (SCREEN_HEIGHT - MAP_SIZEY) / 2,
			MAP_SIZEX, MAP_SIZEY);
	wradar->bgimage(B_RADAR_BACK, 0);

	wtemp->place(xoffs + 244, yoffs + 188, 4, 32);
	wtemp->bgimage(B_TEMP_LID, 0);
	wtemp->background(wtemp->map_rgb(0x182838));
	wtemp->redmax(1);

	wttemp->place(xoffs + 248, yoffs + 188, 4, 32);
	wttemp->bgimage(B_TTEMP_LID, 0);
	wttemp->background(wttemp->map_rgb(0x182838));
	wttemp->redmax(1);

	dships->place(xoffs + 264, yoffs + 196, 38, 18);
	dships->font(B_NORMAL_FONT);
	dships->bgimage(B_SHIPS_BACK, 0);
	dships->caption("SHIPS");
	dships->text("000");

	dstage->place(dships->x(), dships->y2() + 4, 38, 18);
	dstage->font(B_NORMAL_FONT);
	dstage->bgimage(B_STAGE_BACK, 0);
	dstage->caption("STAGE");
	dstage->text("000");

	if(prefs->cmd_fps)
	{
		dfps = new display_t;
		dfps->init(gengine);
		dfps->place(0, -9, 32, 18);
		dfps->color(wdash->map_rgb(0, 0, 0));
		dfps->font(B_NORMAL_FONT);
		dfps->caption("FPS");
	}
}


int KOBO_main::init_display(prefs_t *p)
{
	int dw, dh;		// Display size
	int gw, gh;		// Game "window" size
	gengine->title("Kobo Deluxe " VERSION, "kobodl");
	gengine->driver((gfx_drivers_t)p->videodriver);

	dw = p->width;
	dh = p->height;
	if(p->fullscreen)
	{
		// This game assumes 1:1 pixel aspect ratio, or 4:3
		// width:height ratio, so we need to adjust accordingly.
		// Note:
		//	This code assumes 1:1 pixels for all resolutions!
		//	This does not hold true for 1280x1024 or a CRT
		//	for example, but that's an incorrect (although
		//	all too common) setup anyway. (1280x1024 is for
		//	5:4 TFT displays only!)
		if(dw * 3 >= dh * 4)
		{
			// 4:3 or widescreen; Height defines size
			gw = dh * 4 / 3;
			gh = dh;
		}
		else
		{
			// "tallscreen" (probably 5:4 TFT or rotated display)
			// Width defines size
			gw = dw;
			gh = dw * 3 / 4;
		}
	}
	else
	{
		gw = dw;
		gh = dh;
	}

	// Scaling has 16ths granularity, so tiles scale properly!
//	xscale = (int)((gw * 16 + 8) / SCREEN_WIDTH) / 16.f;
//	yscale = (int)((gh * 16 + 8) / SCREEN_HEIGHT) / 16.f;
	gengine->scale((int)((gw * 16 + 8) / SCREEN_WIDTH) / 16.f,
			(int)((gh * 16 + 8) / SCREEN_HEIGHT) / 16.f);

	// Read back and recalculate, in case the engine has some ideas...
//	xscale = gengine->xscale();
//	yscale = gengine->yscale();
	gw = (int)(SCREEN_WIDTH * gengine->xscale() + 0.5f);
	gh = (int)(SCREEN_HEIGHT * gengine->yscale() + 0.5f);

	if(!p->fullscreen)
	{
		//Add thin black border around the game "screen" in windowed mode.
		dw = gw + 8;
		dh = gh + 8;
	}

	xoffs = (int)((dw - gw) / 2 / gengine->xscale());
	yoffs = (int)((dh - gh) / 2 / gengine->yscale());
	gengine->size(dw, dh);

	gengine->mode(0, p->fullscreen);
	gengine->doublebuffer(p->doublebuf);
	gengine->shadow(p->shadow);
	gengine->cursor(0);

	gengine->period(game.speed);
	sound.period(game.speed);
	gengine->timefilter(p->timefilter * 0.01f);
	gengine->interpolation(p->filter);

	gengine->scroll_ratio(LAYER_OVERLAY, 0.0, 0.0);
	gengine->scroll_ratio(LAYER_PLAYER, 1.0, 1.0);
	gengine->scroll_ratio(LAYER_ENEMIES, 1.0, 1.0);
	gengine->scroll_ratio(LAYER_BASES, 1.0, 1.0);
	gengine->scroll_ratio(LAYER_GROUND, 1.0, 1.0);
	gengine->scroll_ratio(LAYER_STARS, 0.5, 0.5);
	gengine->wrap(MAP_SIZEX * CHIP_SIZEX, MAP_SIZEY * CHIP_SIZEY);

	if(gengine->open(ENEMY_MAX) < 0)
		return -1;

	wscreen = new screen_window_t;
	wscreen->init(gengine);
	wscreen->place(0, 0,
			(int)(gengine->width() / gengine->xscale() + 0.5f),
			(int)(gengine->height() / gengine->yscale() + 0.5f));
	wscreen->border((int)(yoffs * gengine->yscale() + 0.5f),
			(int)(xoffs * gengine->xscale() + 0.5f),
			dw - gw - (int)(xoffs * gengine->xscale() + 0.5f),
			dh - gh - (int)(yoffs * gengine->yscale() + 0.5f));

	wdash = new dashboard_window_t;
	wdash->init(gengine);
	whealth = new bargraph_t;
	whealth->init(gengine);
	wmain = new window_t;
	wmain->init(gengine);
	dhigh = new display_t;
	dhigh->init(gengine);
	dscore = new display_t;
	dscore->init(gengine);
	wmap = new radar_map_t;
	wmap->init(gengine);
	wradar = new radar_window_t;
	wradar->init(gengine);
	wtemp = new bargraph_t;
	wtemp->init(gengine);
	wttemp = new bargraph_t;
	wttemp->init(gengine);
	dships = new display_t;
	dships->init(gengine);
	dstage = new display_t;
	dstage->init(gengine);

	build_screen();

	wdash->mode(DASHBOARD_BLACK);
	gengine->flip();

	return 0;
}


void KOBO_main::close_display()
{
	delete dfps;
	dfps = NULL;
	delete dstage;
	dstage = NULL;
	delete dships;
	dships = NULL;
	delete wttemp;
	wttemp = NULL;
	delete wtemp;
	wtemp = NULL;
	delete wradar;
	wradar = NULL;
	delete wmap;
	wmap = NULL;
	delete dscore;
	dscore = NULL;
	delete dhigh;
	dhigh = NULL;
	delete wmain;
	wmain = NULL;
	delete whealth;
	whealth = NULL;
	delete wdash;
	wdash = NULL;
	delete wscreen;
	wscreen = NULL;
}


void KOBO_main::show_progress(prefs_t *p)
{
	const char *fn;
	gengine->noalpha(0);
	fn = fmap->get("GFX>>loading.png");
	if(fn)
		gengine->loadimage(B_LOADING, fn);

	if(p->alpha)
		gengine->noalpha(0);
	else
		gengine->noalpha(NOALPHA_THRESHOLD);
	fn = fmap->get("GFX>>font2b.png");
	if(fn)
		gengine->loadfont(B_NORMAL_FONT, fn);

	wdash->mode(DASHBOARD_LOADING);
	gengine->flip();
}


void KOBO_main::progress(int percent, const char *msg)
{
	wdash->progress(percent, msg);
	gengine->flip();
}


static int progress_cb(int percent, const char *msg)
{
	km.progress(percent, msg);
	if(km.quit_requested())
		return -999;
	return 0;
}


#define	PROGRESS(x, y)				\
	{					\
		if(progress_cb((x), (y)))	\
			return -999;		\
	}
int KOBO_main::load_graphics(prefs_t *p)
{
	const char *fn;
	SDL_Rect r;

	gengine->reset_filters();

	gengine->colorkey(255, 0, 0);
	gengine->clampcolor(0, 0, 0, 0);
	gengine->scalemode((gfx_scalemodes_t) p->scalemode);
	gengine->brightness(0.01f * p->brightness, 0.01f * p->contrast);
	if(p->use_dither)
		gengine->dither(p->dither_type, p->broken_rgba8);
	else
		gengine->dither(-1);

	show_progress(p);

	if(p->use_dither)
		gengine->dither(p->dither_type, p->broken_rgba8);
	else
		gengine->dither(-1);
	if(p->alpha)
		gengine->noalpha(0);
	else
		gengine->noalpha(NOALPHA_THRESHOLD);

	// Just check that the progress func got the font...
	if(!gengine->is_loaded(B_NORMAL_FONT))
		return -8;

	PROGRESS(5, "Loading in-game graphics");
	gengine->scalemode((gfx_scalemodes_t) p->scalemode, 1);
		fn = fmap->get("GFX>>tiles.png");
		if(!fn || gengine->loadtiles(B_TILES, CHIP_SIZEX, CHIP_SIZEY, fn) < 0)
			return -1;
		PROGRESS(7, NULL);
	gengine->scalemode((gfx_scalemodes_t) p->scalemode, 0);
		fn = fmap->get("GFX>>sprites.png");
		if(!fn || gengine->loadtiles(B_SPRITES, 16, 16, fn) < 0)
			return -2;
		gengine->set_hotspot(B_SPRITES, -1, 8, 8);
		PROGRESS(9, NULL);
	gengine->source_scale(2.0f, 2.0f);
		fn = fmap->get("GFX>>bolt.png");
		if(!fn || gengine->loadtiles(B_BOLT, 16, 16, fn) < 0)
			return -2;
		gengine->set_hotspot(B_BOLT, -1, 4, 4);
		PROGRESS(12, NULL);
		fn = fmap->get("GFX>>explo1b.png");
		if(!fn || gengine->loadtiles(B_EXPLO1, 32, 32, fn) < 0)
			return -2;
		gengine->set_hotspot(B_EXPLO1, -1, 8, 8);
		fn = fmap->get("GFX>>explo2b.png");
		if(!fn || gengine->loadtiles(B_EXPLO2, 64, 64, fn) < 0)
			return -2;
		gengine->set_hotspot(B_EXPLO2, -1, 16, 16);
		PROGRESS(15, NULL);
	gengine->source_scale(1.0f, 1.0f);
		fn = fmap->get("GFX>>bullets.png");
		if(!fn || gengine->loadtiles(B_BULLETS, 8, 8, fn) < 0)
			return -3;
		gengine->set_hotspot(B_BULLETS, -1, 4, 4);
		PROGRESS(20, NULL);
		fn = fmap->get("GFX>>bigship.png");
		if(!fn || gengine->loadtiles(B_BIGSHIP, 32, 32, fn) < 0)
			return -4;
		gengine->set_hotspot(B_BIGSHIP, -1, 16, 16);

	PROGRESS(25, "Loading framework graphics");
	gengine->scalemode((gfx_scalemodes_t) p->scalemode, 1);
	gengine->clampcolor(0, 0, 0, 255);
	gengine->source_scale(2.0f, 2.0f);	// This is 640x480!
		fn = fmap->get("GFX>>screen.png");
		if(!fn || gengine->loadimage(B_SCREEN, fn) < 0)
			return -5;
		PROGRESS(30, NULL);
		r.x = (8 + MARGIN);
		r.y = MARGIN;
		r.w = r.h = WSIZE;
		if(gengine->copyrect(B_FRAME, B_SCREEN, 0, &r) < 0)
			return -6;
		PROGRESS(32, NULL);
		r.x = 252;
		r.y = 4;
		r.w = 64;
		r.h = 18;
		if(gengine->copyrect(B_HIGH_BACK, B_SCREEN, 0, &r) < 0)
			return -26;
		PROGRESS(33, NULL);
		r.y += 18 + 4;
		if(gengine->copyrect(B_SCORE_BACK, B_SCREEN, 0, &r) < 0)
			return -27;
		PROGRESS(34, NULL);
		r.x = 244;
		r.y = (SCREEN_HEIGHT - MAP_SIZEY) / 2;
		r.w = MAP_SIZEX;
		r.h = MAP_SIZEY;
		if(gengine->copyrect(B_RADAR_BACK, B_SCREEN, 0, &r) < 0)
			return -28;
		PROGRESS(35, NULL);
		r.x = 264;
		r.y = 196;
		r.w = 38;
		r.h = 18;
		if(gengine->copyrect(B_SHIPS_BACK, B_SCREEN, 0, &r) < 0)
			return -29;
		PROGRESS(36, NULL);
		r.y += 18 + 4;
		if(gengine->copyrect(B_STAGE_BACK, B_SCREEN, 0, &r) < 0)
			return -30;
		PROGRESS(37, NULL);
		r.x = 4;
		r.y = 92;
		r.w = 8;
		r.h = 128;
		if(gengine->copyrect(B_HEALTH_LID, B_SCREEN, 0, &r) < 0)
			return -31;
		PROGRESS(38, NULL);
		r.x = 244;
		r.y = 188;
		r.w = 4;
		r.h = 32;
		if(gengine->copyrect(B_TEMP_LID, B_SCREEN, 0, &r) < 0)
			return -32;
		PROGRESS(39, NULL);
		r.x += 4;
		if(gengine->copyrect(B_TTEMP_LID, B_SCREEN, 0, &r) < 0)
			return -33;
	PROGRESS(40, "Loading logo");
	gengine->scalemode((gfx_scalemodes_t) p->scalemode, 0);
		fn = fmap->get("GFX>>logo.png");
		if(!fn || gengine->loadimage(B_LOGO, fn) < 0)
			return -7;
		fn = fmap->get("GFX>>deluxe.png");
		if(!fn || gengine->loadimage(B_LOGODELUXE, fn) < 0)
			return -8;
	gengine->scalemode(GFX_SCALE_NEAREST, 1);
	gengine->source_scale(4.0f, 4.0f);
		fn = fmap->get("GFX>>logomask.png");
		if(!fn || gengine->loadimage(B_LOGOMASK, fn) < 0)
			return -9;
		s_sprite_t *s = gengine->get_sprite(B_LOGOMASK, 0);
		if(!s || !s->surface)
			return -10;
		RGN_FreeRegion(logo_region);
		logo_region = RGN_ScanMask(s->surface,
				SDL_MapRGB(s->surface->format, 255, 255, 255));
		if(!logo_region)
		{
			log_printf(ELOG, "Could not create logo region!\n");
			return -92;
		}

	PROGRESS(55, "Loading fonts");
	gengine->scalemode((gfx_scalemodes_t) p->scalemode);
	gengine->source_scale(1.0f, 1.0f);
	gengine->clampcolor(0, 0, 0, 0);
		fn = fmap->get("GFX>>font4b.png");
		if(!fn || gengine->loadfont(B_MEDIUM_FONT, fn) < 0)
			return -9;
		PROGRESS(60, NULL);

		fn = fmap->get("GFX>>font3b.png");
		if(!fn || gengine->loadfont(B_BIG_FONT, fn) < 0)
			return -9;
		PROGRESS(70, NULL);

		fn = fmap->get("GFX>>counter.png");
		if(!fn || gengine->loadfont(B_COUNTER_FONT, fn) < 0)
			return -10;

	PROGRESS(80, "Loading special FX graphics");
	gengine->scalemode((gfx_scalemodes_t) p->scalemode, 1);
		fn = fmap->get("GFX>>noise.png");
		if(!fn || gengine->loadtiles(B_NOISE, NOISE_SIZEX, 1, fn) < 0)
			return -1;
		PROGRESS(85, NULL);

		// Alpha + filtered scaling would be very expensive for this one!
	gengine->scalemode(GFX_SCALE_NEAREST, 1);
		fn = fmap->get("GFX>>hitnoise.png");
		if(!fn || gengine->loadtiles(B_HITNOISE, NOISE_SIZEX, 1, fn) < 0)
			return -1;
		PROGRESS(90, NULL);

		fn = fmap->get("GFX>>focusfx.png");
	gengine->scalemode((gfx_scalemodes_t) p->scalemode, 1);
		if(!fn || gengine->loadimage(B_FOCUSFX, fn) < 0)
			return -1;
		PROGRESS(95, NULL);

	gengine->noalpha(1);
	gengine->brightness(1.0f, 1.0f);
	gengine->scalemode(GFX_SCALE_NEAREST, 1);
		fn = fmap->get("GFX>>brushes.png");
		if(!fn || gengine->loadtiles(B_BRUSHES, 16, 16, fn) < 0)
			return -11;
		PROGRESS(100, NULL);
	gengine->scalemode((gfx_scalemodes_t) p->scalemode);
	gengine->brightness(0.01f * p->brightness, 0.01f * p->contrast);
	if(p->alpha)
		gengine->noalpha(0);
	else
		gengine->noalpha(NOALPHA_THRESHOLD);

	return 0;
}
#undef	PROGRESS


int KOBO_main::load_sounds(prefs_t *p, int render_all)
{
	if(!p->use_sound)
		return 0;
	show_progress(p);
	return sound.load(progress_cb, render_all);
}


int KOBO_main::init_js(prefs_t *p)
{
	/* Activate Joystick sub-sys if we are using it */
	if(p->use_joystick)
	{
		if(SDL_InitSubSystem(SDL_INIT_JOYSTICK) < 0)
		{
			log_printf(ELOG, "Error setting up joystick!\n");
			return -1;
		}

		if(SDL_NumJoysticks() > 0)
		{
			SDL_JoystickEventState(SDL_ENABLE);
			joystick = SDL_JoystickOpen(0);
			if(!joystick)
			{
				SDL_QuitSubSystem(SDL_INIT_JOYSTICK);
				return -2;
			}
		}
		else
		{
			log_printf(ELOG, "No joysticks found!\n");
			joystick = NULL;
			return -3;
		}

	}
	return 0;
}


void KOBO_main::close_js()
{
	if(!SDL_WasInit(SDL_INIT_JOYSTICK))
		return;

	if(!joystick)
		return;

	if(SDL_JoystickOpened(0))
		SDL_JoystickClose(joystick);
	joystick = NULL;

	SDL_QuitSubSystem(SDL_INIT_JOYSTICK);
}


void KOBO_main::load_config(prefs_t *p)
{
	FILE *f = fmap->fopen(KOBO_CONFIG_DIR "/" KOBO_CONFIG_FILE, "r");
	if(f)
	{
		p->read(f);
		fclose(f);
	}
}


void KOBO_main::save_config(prefs_t *p)
{
	FILE *f;
#if defined(HAVE_GETEGID) && defined(HAVE_SETGID)
	gid_t oldgid = getegid();
	if(setgid(getgid()) != 0)
	{
		log_printf(ELOG, "Cannot save config! (%s)\n",
				strerror(errno));
		return;
	}
#endif
	f = fmap->fopen(KOBO_CONFIG_DIR "/" KOBO_CONFIG_FILE, "w");
	if(f)
	{
		p->write(f);
		fclose(f);
	}
#if defined(HAVE_GETEGID) && defined(HAVE_SETGID)
	if(setgid(oldgid) != 0)
	{
		log_printf(ELOG, "Cannot restore GID! (%s)\n",
				strerror(errno));
		return;
	}
#endif
}


void KOBO_main::brutal_quit()
{
	if(exit_game_fast)
	{
		log_printf(ULOG, "Second try quitting; using brutal method!\n");
		atexit(SDL_Quit);
		close_logging();
		exit(1);
	}

	exit_game_fast = 1;
	if(gengine)
		gengine->stop();
}


void KOBO_main::pause_game()
{
	if(gsm.current() != &st_pause_game)
		gsm.press(BTN_PAUSE);
}


void kobo_render_highlight(ct_widget_t *wg)
{
	screen.set_highlight(wg->y() + wg->height() / 2 - wmain->y(),
			wg->height());
}


int KOBO_main::open()
{
	if(init_display(prefs) < 0)
		return -1;

	if(load_graphics(prefs) < 0)
		return -2;

	sound.open();

	if(load_sounds(prefs) < 0)
		return -3;

	wdash->nibble();

	ct_engine.render_highlight = kobo_render_highlight;
	wdash->mode(DASHBOARD_GAME);
	wradar->mode(RM_NOISE);
	pubrand.init();
	init_js(prefs);
	gamecontrol.init(prefs->always_fire);
	manage.init();

	gsm.push(&st_intro_1);

	return 0;
}


void KOBO_main::close()
{
	close_js();
	RGN_FreeRegion(logo_region);
	logo_region = NULL;
	close_display();
	if(gengine)
	{
		delete gengine;
		gengine = NULL;
	}
	sound.close();
	SDL_Quit();
}


int KOBO_main::run()
{
	int retry_status = 0;
	int dont_retry = 0;
	while(1)
	{
		if(!retry_status)
		{
			safe_prefs = *prefs;
			gengine->run();

			if(!manage.game_stopped())
				manage.abort();

			if(exit_game_fast)
				break;

			if(exit_game)
			{
				if(wdash)
					wdash->nibble();
				break;
			}
			dont_retry = 0;
		}

		retry_status = 0;
		if(global_status & OS_RESTART_AUDIO)
		{
			if(prefs->use_sound)
			{
				log_printf(ULOG, "--- Restarting audio...\n");
				sound.stop();
				if(load_sounds(prefs) < 0)
					return 5;
				sound.open();
				wdash->nibble();
				log_printf(ULOG, "--- Audio restarted.\n");
				wdash->mode(DASHBOARD_GAME);
				wradar->mode(screen.radar_mode);
				manage.set_bars();
			}
			else
			{
				log_printf(ULOG, "--- Stopping audio...\n");
				sound.close();
				log_printf(ULOG, "--- Audio stopped.\n");
			}
		}
		if((global_status & OS_RELOAD_AUDIO_CACHE) && prefs->cached_sounds)
		{
			log_printf(ULOG, "--- Rendering sounds to disk...\n");
			if(load_sounds(prefs) < 0)
			{
				log_printf(ELOG, "--- Could not render sounds to disk!\n");
				st_error.message("Could not render to disk!",
						"Please, check your installation.");
				gsm.push(&st_error);
			}
			wdash->mode(DASHBOARD_GAME);
			wradar->mode(screen.radar_mode);
			manage.set_bars();
		}
		if(global_status & OS_RESTART_VIDEO)
		{
			log_printf(ULOG, "--- Restarting video...\n");
			wdash->nibble();
			gengine->hide();
			close_display();
			gengine->unload();
			if(init_display(prefs) < 0)
			{
				log_printf(ELOG, "--- Video init failed!\n");
				st_error.message("Video initialization failed!",
						"Try different settings.");
				gsm.push(&st_error);
				if(dont_retry)
					return 6;
				*prefs = safe_prefs;
				dont_retry = 1;
				retry_status |= OS_RESTART_VIDEO;
			}
			else
			{
				gamecontrol.init(prefs->always_fire);
				log_printf(ULOG, "--- Video restarted.\n");
			}
		}
		if(global_status & (OS_RELOAD_GRAPHICS |
					OS_RESTART_VIDEO))
		{
			if(!(global_status & OS_RESTART_VIDEO))
				wdash->nibble();
			gengine->unload();
			RGN_FreeRegion(logo_region);
			logo_region = NULL;
			if(!retry_status)
			{
				log_printf(ULOG, "--- Reloading graphics...\n");
				if(load_graphics(prefs) < 0)
					return 7;
				wdash->nibble();
				wdash->mode(DASHBOARD_GAME);
				wradar->mode(screen.radar_mode);
				manage.set_bars();
				log_printf(ULOG, "--- Graphics reloaded.\n");
			}
		}
#if 0
// This must be severely broken... genigne.close() calls for restarting video,
// reloading graphics etc etc!
		if(global_status & OS_RESTART_ENGINE)
		{
			gengine->close();
			if(!retry_status)
			{
				log_printf(ULOG, "--- Restarting engine...\n");
				gengine->scroll_ratio(LAYER_OVERLAY, 0.0, 0.0);
				gengine->scroll_ratio(LAYER_PLAYER, 1.0, 1.0);
				gengine->scroll_ratio(LAYER_ENEMIES, 1.0, 1.0);
				gengine->scroll_ratio(LAYER_BASES, 1.0, 1.0);
				gengine->scroll_ratio(LAYER_GROUND, 1.0, 1.0);
				gengine->scroll_ratio(LAYER_STARS, 0.5, 0.5);
				gengine->wrap(MAP_SIZEX * CHIP_SIZEX,
						MAP_SIZEY * CHIP_SIZEY);
				if(gengine->open(2048) < 0)
				{
					log_printf(CELOG, "--- Could not restart engine!\n");
					if(dont_retry)
						return 8;
					*prefs = safe_prefs;
					dont_retry = 1;
					retry_status |= OS_RESTART_ENGINE;
				}
				else
				{
					gamecontrol.init(prefs->always_fire);
					log_printf(ULOG, "--- Engine restarted.\n");
				}
			}
		}
#else
		if(global_status & OS_RESTART_ENGINE)
			log_printf(ELOG, "OS_RESTART_ENGINE not implemented!\n");
#endif
		
		if(global_status & OS_RESTART_INPUT)
		{
			close_js();
			init_js(prefs);
			gamecontrol.init(prefs->always_fire);
		}
		if(global_status & OS_RESTART_LOGGER)
			open_logging(prefs);

		global_status = retry_status;
		km.pause_game();
		manage.reenter();
	}
	return 0;
}


/*----------------------------------------------------------
	Kobo Graphics Engine
----------------------------------------------------------*/

kobo_gfxengine_t::kobo_gfxengine_t()
{
}


void kobo_gfxengine_t::frame()
{
	sound.frame();

	if(!gsm.current())
	{
		log_printf(CELOG, "INTERNAL ERROR: No gamestate!\n");
		exit_game = 1;
		stop();
		return;
	}
	if(exit_game || manage.aborted())
	{
		stop();
		return;
	}

	/*
	 * Process input
	 */
	SDL_Event ev;
	while(SDL_PollEvent(&ev))
	{
		int k, ms;
		switch (ev.type)
		{
		  case SDL_KEYDOWN:
			switch(ev.key.keysym.sym)
			{
#ifdef PROFILE_AUDIO
			  case SDLK_F10:
				++km.audio_vismode;
				break;
			  case SDLK_F12:
				audio_print_info();
				break;
#endif
			  case SDLK_DELETE:
				if(prefs->cmd_debug)
				{
					manage.ships = 1;
					myship.hit(1000);
				}
				break;
			  case SDLK_RETURN:
				ms = SDL_GetModState();
				if(ms & (KMOD_CTRL | KMOD_SHIFT | KMOD_META))
					break;
				if(!(ms & KMOD_ALT))
					break;
				km.pause_game();
				prefs->fullscreen = !prefs->fullscreen;
				prefs->changed = 1;
				global_status |= OS_RELOAD_GRAPHICS |
						OS_RESTART_VIDEO;
				stop();
				return;
			  case SDLK_PRINT:
			  case SDLK_SYSREQ:
				gengine->screenshot();
				break;
			  default:
				break;
			}
			k = gamecontrol.map(ev.key.keysym.sym);
			gamecontrol.press(k);
			gsm.press(k, ev.key.keysym.unicode);
			break;
		  case SDL_KEYUP:
			if((ev.key.keysym.sym == SDLK_ESCAPE) && km.escape_hammering())
			{
				km.pause_game();
				prefs->fullscreen = 0;
				prefs->videodriver = (int)GFX_DRIVER_SDL2D;
				prefs->width = 320;
				prefs->height = 240;
				prefs->aspect = 1000;
				prefs->depth = 0;
				prefs->doublebuf = 0;
				prefs->shadow = 1;
				prefs->scalemode = (int)GFX_SCALE_NEAREST;
				prefs->brightness = 100;
				prefs->contrast = 100;
				global_status |= OS_RELOAD_GRAPHICS |
						OS_RESTART_VIDEO;
				stop();
				return;
			}
			k = gamecontrol.map(ev.key.keysym.sym);
			if(k == SDLK_PAUSE)
			{
				gamecontrol.press(BTN_PAUSE);
				gsm.press(BTN_PAUSE);
			}
			else
			{
				gamecontrol.release(k);
				gsm.release(k);
			}
			break;
		  case SDL_VIDEOEXPOSE:
			gengine->invalidate();
			break;
		  case SDL_ACTIVEEVENT:
			// Any type of focus loss should activate pause mode!
			if(!ev.active.gain)
				km.pause_game();
			break;
		  case SDL_QUIT:
			/*gsm.press(BTN_CLOSE);*/
			km.brutal_quit();
			break;
		  case SDL_JOYBUTTONDOWN:
			if(ev.jbutton.button == km.js_fire)
			{
				gamecontrol.press(BTN_FIRE);
				gsm.press(BTN_FIRE);
			}
			else if(ev.jbutton.button == km.js_start)
			{
				gamecontrol.press(BTN_START);
				gsm.press(BTN_START);
			}
			break;
		  case SDL_JOYBUTTONUP:
			if(ev.jbutton.button == km.js_fire)
			{
				gamecontrol.release(BTN_FIRE);
				gsm.release(BTN_FIRE);
			}
			break;
		  case SDL_JOYAXISMOTION:
			// FIXME: We will want to allow these to be
			// redefined, but for now, this works ;-)
			if(ev.jaxis.axis == km.js_lr)
			{
				if(ev.jaxis.value < -3200)
				{
					gamecontrol.press(BTN_LEFT);
					gsm.press(BTN_LEFT);
				}
				else if(ev.jaxis.value > 3200)
				{
					gamecontrol.press(BTN_RIGHT);
					gsm.press(BTN_RIGHT);
				}
				else
				{
					gamecontrol.release(BTN_LEFT);
					gamecontrol.release(BTN_RIGHT);
					gsm.release(BTN_LEFT);
					gsm.release(BTN_RIGHT);
				}

			}
			else if(ev.jaxis.axis == km.js_ud)
			{
				if(ev.jaxis.value < -3200)
				{
					gamecontrol.press(BTN_UP);
					gsm.press(BTN_UP);
				}
				else if(ev.jaxis.value > 3200)
				{
					gamecontrol.press(BTN_DOWN);
					gsm.press(BTN_DOWN);
				}
				else
				{
					gamecontrol.release(BTN_UP);
					gamecontrol.release(BTN_DOWN);
					gsm.release(BTN_UP);
					gsm.release(BTN_DOWN);
				}

			}
		  case SDL_MOUSEMOTION:
			mouse_x = (int)(ev.motion.x / gengine->xscale()) - km.xoffs;
			mouse_y = (int)(ev.motion.y / gengine->yscale()) - km.yoffs;
			if(prefs->use_mouse)
				gamecontrol.mouse_position(
						mouse_x - 8 - MARGIN - WSIZE/2,
						mouse_y - MARGIN - WSIZE/2);
			break;
		  case SDL_MOUSEBUTTONDOWN:
			mouse_x = (int)(ev.motion.x / gengine->xscale()) - km.xoffs;
			mouse_y = (int)(ev.motion.y / gengine->yscale()) - km.yoffs;
			gsm.press(BTN_FIRE);
			if(prefs->use_mouse)
			{
				gamecontrol.mouse_position(
						mouse_x - 8 - MARGIN - WSIZE/2,
						mouse_y - MARGIN - WSIZE/2);
				switch(ev.button.button)
				{
				  case SDL_BUTTON_LEFT:
					mouse_left = 1;
					break;
				  case SDL_BUTTON_MIDDLE:
					mouse_middle = 1;
					break;
				  case SDL_BUTTON_RIGHT:
					mouse_right = 1;
					break;
				}
				gamecontrol.press(BTN_FIRE);
			}
			break;
		  case SDL_MOUSEBUTTONUP:
			mouse_x = (int)(ev.motion.x / gengine->xscale()) - km.xoffs;
			mouse_y = (int)(ev.motion.y / gengine->yscale()) - km.yoffs;
			if(prefs->use_mouse)
			{
				gamecontrol.mouse_position(
						mouse_x - 8 - MARGIN - WSIZE/2,
						mouse_y - MARGIN - WSIZE/2);
				switch(ev.button.button)
				{
				  case SDL_BUTTON_LEFT:
					mouse_left = 0;
					break;
				  case SDL_BUTTON_MIDDLE:
					mouse_middle = 0;
					break;
				  case SDL_BUTTON_RIGHT:
					mouse_right = 0;
					break;
				}
			}
			if(!mouse_left && !mouse_middle && !mouse_right)
			{
				if(prefs->use_mouse)
					gamecontrol.release(BTN_FIRE);
				gsm.release(BTN_FIRE);
			}
			break;
		}
	}
	gamecontrol.process();

	/*
	 * Run the current gamestate for one frame
	 */
	gsm.frame();

#if 0
	{
		static int c = 0;
		++c;
		if(c >= 7)
		{
			gengine->screenshot();
			c = 0;
		}
	}
#endif
}


void kobo_gfxengine_t::pre_render()
{
	sound.run();
	gsm.pre_render();
}


void kobo_gfxengine_t::post_render()
{
	gsm.post_render();

	if(!prefs->cmd_noframe)
		wmain->sprite(0, 0, B_FRAME, 0, 0);

#ifdef DEBUG
	if(prefs->cmd_debug)
	{
		char buf[20];
		snprintf(buf, sizeof(buf), "Obj: %d",
				gengine->objects_in_use());
		wmain->font(B_NORMAL_FONT);
		wmain->string(160, 5, buf);
	}

	switch (km.audio_vismode % 5)
	{
	  case 0:
		break;
	  case 1:
		draw_osc(0);
		break;
	  case 2:
		draw_vu();
		break;
	  case 3:
		draw_osc(1);
		break;
	  case 4:
		draw_osc(2);
		break;
	}
#endif

	// Frame rate counter
	int nt = (int)SDL_GetTicks();
	int tt = nt - km.fps_starttime;
	if((tt > 1000) && km.fps_count)
	{
		float f = km.fps_count * 1000.0 / tt;
		::screen.fps(f);
		km.fps_count = 0;
		km.fps_starttime = nt;
		if(prefs->cmd_fps)
		{
			char buf[20];
			snprintf(buf, sizeof(buf), "%.1f", f);
			km.dfps->text(buf);
			if(!km.fps_results)
				km.fps_results = (float *)
						calloc(MAX_FPS_RESULTS,
						sizeof(float));
			if(km.fps_results)
			{
				km.fps_results[km.fps_nextresult++] = f;
				if(km.fps_nextresult >= MAX_FPS_RESULTS)
					km.fps_nextresult = 0;
				if(km.fps_nextresult > km.fps_lastresult)
					km.fps_lastresult = km.fps_nextresult;
			}
		}
	}
	++km.fps_count;

	// Frame rate limiter
	if(prefs->max_fps)
	{
		if(prefs->max_fps_strict)
		{
			static double nextframe = -1000000.0f;
			while(1)
			{
				double t = (double)SDL_GetTicks();
				if(fabs(nextframe - t) > 1000.0f)
					nextframe = t;
				double d = nextframe - t;
				if(d > 10.0f)
					SDL_Delay(10);
				else if(d > 1.0f)
					SDL_Delay(1);
				else
					break;
			}
			nextframe += 1000.0f / prefs->max_fps;
		}
		else
		{
			int rtime = nt - km.max_fps_begin;
			km.max_fps_begin = nt;
			km.max_fps_filter += (float)(rtime - km.max_fps_filter) * 0.3;
			int delay = (int)(1000.0 / prefs->max_fps -
					km.max_fps_filter + 0.5);
			if((delay > 0) && (delay < 1100))
				SDL_Delay(delay);
			km.max_fps_begin = (int)SDL_GetTicks();
		}
	}
}


/*----------------------------------------------------------
	main() and related stuff
----------------------------------------------------------*/

static void put_usage()
{
	printf("\nKobo Deluxe %s\n", KOBO_VERSION);
	printf("Usage: kobodl [<options>]\n");
	printf("Recognized options:\n");
	int s = -1;
	while(1)
	{
		s = prefs->find_next(s);
		if(s < 0)
			break;
		switch(prefs->type(s))
		{
		  case CFG_BOOL:
			printf("        -[no]%-16.16s[%s]\t%s\n",
					prefs->name(s),
					prefs->get_default_s(s),
					prefs->description(s));
			break;
		  case CFG_INT:
		  case CFG_FLOAT:
			printf("        -%-20.20s[%s]\t%s\n",
					prefs->name(s),
					prefs->get_default_s(s),
					prefs->description(s));
			break;
		  case CFG_STRING:
			printf("        -%-20.20s[\"%s\"]\t%s\n",
					prefs->name(s),
					prefs->get_default_s(s),
					prefs->description(s));
		  default:
			break;
		}
	}
}


extern "C" void emergency_close(void)
{
	km.close();
}


extern "C" RETSIGTYPE breakhandler(int dummy)
{
	/* For platforms that drop the handlers on the first signal... */
	signal(SIGTERM, breakhandler);
	signal(SIGINT, breakhandler);
	km.brutal_quit();
#if (RETSIGTYPE != void)
	return 0;
#endif
}


int main(int argc, char *argv[])
{
	int cmd_exit = 0;
	atexit(emergency_close);
	signal(SIGTERM, breakhandler);
	signal(SIGINT, breakhandler);

	SDL_Init(0);

	if(main_init())
	{
		fprintf(stderr, "INTERNAL ERROR\n");
		return 1;
	};

	km.open_logging(NULL);

	setup_dirs(argv[0]);
	--argc;
	++argv;

	if((argc < 1) || (strcmp("-override", argv[0]) != 0))
		km.load_config(prefs);

	if(prefs->parse(argc, argv) < 0)
	{
		put_usage();
		main_cleanup();
		return 1;
	}

	if(prefs->cmd_noparachute)
	{
		SDL_Quit();
		SDL_Init(SDL_INIT_NOPARACHUTE);
	}

	km.open_logging(prefs);

	for(int a = 0; a < argc; ++a)
		log_printf(DLOG, "argv[%d] = \"%s\"\n", a, argv[a]);

	int k = -1;
	while((k = prefs->find_next(k)) >= 0)
	{
		log_printf(D3LOG, "key %d: \"%s\"\ttype=%d\t\"%s\"\n",
				k,
				prefs->name(k),
				prefs->type(k),
				prefs->description(k)
			);
	}

	add_dirs(prefs);

	if(prefs->cmd_hiscores)
	{
		scorefile.gather_high_scores();
		scorefile.print_high_scores();
		cmd_exit = 1;
	}

	if(prefs->cmd_showcfg)
	{
		printf("Configuration:\n");
		printf("----------------------------------------\n");
		prefs->write(stdout);
		printf("\nPaths:\n");
		printf("----------------------------------------\n");
		fmap->print(stdout, "*");
		printf("----------------------------------------\n");
		cmd_exit = 1;
	}

	if(cmd_exit)
	{
		km.close_logging();
		main_cleanup();
		return 0;
	}

	if(km.open() < 0)
	{
		km.close_logging();
		main_cleanup();
		return 1;
	}

	km.run();

	km.close();

	/* 
	 * Seems like we got all the way here without crashing,
	 * so let's save the current configuration! :-)
	 */
	if(prefs->changed)
	{
		km.save_config(prefs);
		prefs->changed = 0;
	}

	if(prefs->cmd_fps)
		km.print_fps_results();

	km.close_logging();
	main_cleanup();
	return 0;
}
