/////////////////////////////////////////////////////////////////////////////
// Name:        TitlePropDlg.cpp
// Purpose:     DVD title properties dialog
// Author:      Alex Thuering
// Created:     31.01.2004
// RCS-ID:      $Id: TitlePropDlg.cpp,v 1.25 2008/07/14 22:38:55 ntalex Exp $
// Copyright:   (c) Alex Thuering
// Licence:     GPL
/////////////////////////////////////////////////////////////////////////////

#include "TitlePropDlg.h"
#include "TitlesetManager.h"
#include "AVPropDlg.h"
#include "MPEG.h"

enum {
	AV_BT_ID = 7850,
	DO_NOT_TRANSCODE_CB_ID
};

BEGIN_EVENT_TABLE(TitlePropDlg, wxPropDlg)
	EVT_BUTTON(AV_BT_ID, TitlePropDlg::OnAV)
	EVT_CHECKBOX(DO_NOT_TRANSCODE_CB_ID, TitlePropDlg::OnDoNotTranscodeCheck)
END_EVENT_TABLE()

TitlePropDlg::TitlePropDlg(wxWindow* parent, DVD* dvd, Audio* audio, Video* video, Pgc* pgc, Vob* vob):
		wxPropDlg(parent), m_dvd(dvd), m_audio(audio), m_video(video), m_pgc(pgc), m_vob(vob) {
	m_formatCtrlIdx = -1;
	Create();
	SetSize(360,-1);
}

void TitlePropDlg::CreatePropPanel(wxSizer* sizer) {
	if (m_vob->GetSlideshow()) {
		Slideshow* slideshow = m_vob->GetSlideshow();
		wxSizer* grpSizer = BeginGroup(sizer, _("Slideshow"));
		wxFlexGridSizer* grid = new wxFlexGridSizer(2, 2, 4, 4);
		grid->AddGrowableCol(1);
		wxArrayString formats = DVD::GetVideoFormatLabels();
		AddComboProp(grid, _("Format:"), formats[slideshow->GetVideoFormat()-2], formats, wxCB_READONLY);
		AddSpinProp(grid, _("Duration (sec):"), slideshow->GetDuration(), 1, 1000, 100);
		grpSizer->Add(grid, 0, wxEXPAND|wxALL, 6);
	} else {
		wxSizer* grpSizer = BeginGroup(sizer, _("Video object"));
		wxFlexGridSizer* grid = new wxFlexGridSizer(2, 4, 4);
		grid->AddGrowableCol(1);
		int stIdx = 0;
		if (m_vob->GetFilename().length()) {
			AddText(grid, _("File name:"));
			if (m_vob->GetFilename().length())
				AddText(grid, m_vob->GetFilename());
			AddText(grid, wxString(wxT("    ")) + _("Duration:"));
			if (m_vob->GetDuration()>0) {
				int secs = (int) m_vob->GetDuration();
				int mins = secs / 60;
				secs %= 60;
				int hours = mins / 60;
				mins %= 60;
				AddText(grid, wxString::Format(wxT("%02d:%02d:%02d"), hours, mins, secs));
			} else
				AddText(grid, wxT("N/A"));
			m_formatCtrlIdx = GetLastControlIndex() + 1;
			int streamsCount = (int)m_vob->GetStreams().GetCount()
					- m_vob->GetAudioFilenames().GetCount();
			for (; stIdx < streamsCount; stIdx++) {
				Stream* stream = m_vob->GetStreams()[stIdx];
				wxString srcFormat = stream->GetSourceFormat();
				switch (stream->GetType()) {
				case stVIDEO: {
					AddText(grid, wxString(wxT("    ")) + _("Video:"));
					wxBoxSizer* pane = new wxBoxSizer(wxHORIZONTAL);
					AddText(pane, srcFormat);
					pane->Add(4,4);
					wxArrayString formats = DVD::GetVideoFormatLabels(true);
					AddChoiceProp(pane, wxT(""), formats[stream->GetVideoFormat()-1], formats, 100, false);
					GetLastControl()->Enable(!m_vob->GetDoNotTranscode());
					grid->Add(pane);
					break;
				}
				case stAUDIO: {
					AddText(grid, wxString(wxT("    ")) + _("Audio:"));
					wxBoxSizer* pane = new wxBoxSizer(wxHORIZONTAL);
					AddText(pane, srcFormat);
					pane->Add(4,4);
					wxArrayString formats = DVD::GetAudioFormatLabels(true, true);
					AddChoiceProp(pane, wxT(""), formats[stream->GetAudioFormat()], formats, 100, false);
					GetLastControl()->Enable(!m_vob->GetDoNotTranscode());
					grid->Add(pane);
					break;
				}
				case stSUBTITLE:
					AddStaticTextProp(grid, wxString(wxT("    ")) + _("Subtitle:"), srcFormat);
					break;
				default:
					break;
				}
			}
			if (m_vob->GetAudioFilenames().GetCount() == 0) {
				AddText(grid, wxT(""));
				AddCheckProp(grid, _("do not remultiplex/transcode"), m_vob->GetDoNotTranscode(),
						false, DO_NOT_TRANSCODE_CB_ID);
				if (!MPEG::HasNavPacks(m_vob->GetFilename()))
					GetLastControl()->Enable(false);
			}
		}
		for (unsigned int i=0; i<m_vob->GetAudioFilenames().GetCount(); i++) {
			Stream* stream = m_vob->GetStreams()[stIdx++];
			wxString srcFormat = stream->GetSourceFormat();
			grid->Add(1, 1);
			grid->Add(1, 1);
			AddText(grid, _("File name:"));
			AddText(grid, m_vob->GetAudioFilenames()[i]);
			AddText(grid, wxString(wxT("    ")) + _("Audio:"));
			wxBoxSizer* pane = new wxBoxSizer(wxHORIZONTAL);
			AddText(pane, srcFormat);
			pane->Add(4,4);
			wxArrayString formats = DVD::GetAudioFormatLabels(true);
			AddChoiceProp(pane, wxT(""), formats[stream->GetAudioFormat()-1], formats, 100, false);
			grid->Add(pane);
		}
		for (unsigned int i=0; i<m_vob->GetSubtitles().GetCount(); i++) {
			grid->Add(1, 1);
			grid->Add(1, 1);
			AddText(grid, _("Subtitle:"));
			AddText(grid, m_vob->GetSubtitles()[i]->GetFilename());
		}
		grid->Add(2, 2);
		grid->Add(2, 2);
		AddTextProp(grid, _("Chapters:"), m_vob->GetChapters());
		AddSpinProp(grid, _("Pause:"), m_vob->GetPause(), -1, 254, 100);
		grpSizer->Add(grid, 0, wxEXPAND|wxALL, 6);
	}
	sizer->Add(6, 6);

	wxSizer* grpSizer = BeginGroup(sizer, _("Title"));
	wxFlexGridSizer* grid = new wxFlexGridSizer(3, 2, 4, 4);
	grid->AddGrowableCol(1);
	AddFileProp(grid, _("Palette:"), m_pgc->GetPalette(), wxFD_OPEN, wxT("..."), wxT("*.rgb;*.yuv"));
	wxArrayString commands;
	for (int pgci=0; pgci<(int)m_dvd->GetVmgm().Count(); pgci++)
		commands.Add(wxString::Format(_T("call vmgm menu %d;"),pgci+1));
	for (int tsi=0; tsi<(int)m_dvd->GetTitlesets().Count(); tsi++) {
		Titleset* ts = m_dvd->GetTitlesets()[tsi];
		if (ts->GetMenus().Count()>0)
			commands.Add(_T("call menu;"));
		if (m_dvd->IsJumppadsEnabled()) {
			for (int pgci=1; pgci<(int)ts->GetMenus().Count(); pgci++)
				commands.Add(wxString::Format(_T("call menu %d;"),pgci+1));
		}
		for (int pgci=0; pgci<(int)ts->GetTitles().Count(); pgci++)
			commands.Add(wxString::Format(_T("jump title %d;"),pgci+1));
	}
	AddComboProp(grid, _("Pre commands:"), m_pgc->GetPreCommands(), commands);
	AddComboProp(grid, _("Post commands:"), m_pgc->GetPostCommands(), commands);
	grpSizer->Add(grid, 0, wxEXPAND|wxALL, 6);
}

bool TitlePropDlg::SetValues() {
	int n = 0;
	if (m_vob->GetSlideshow()) {
		Slideshow* slideshow = m_vob->GetSlideshow();
		slideshow->SetVideoFormat((VideoFormat) GetInt(n++));
		slideshow->SetDuration(GetInt(n++));
	} else {
		int stIdx = 0;
		if (m_vob->GetFilename().length()) {
			int streamsCount = (int) m_vob->GetStreams().GetCount()
					- m_vob->GetAudioFilenames().GetCount();
			for (; stIdx < streamsCount; stIdx++) {
				Stream* stream = m_vob->GetStreams()[stIdx];
				switch (stream->GetType()) {
				case stVIDEO:
					stream->SetDestinationFormat(GetInt(n++) + 1);
					break;
				case stAUDIO:
					stream->SetDestinationFormat(GetInt(n++));
					break;
				default:
					break;
				}
			}
			if (m_vob->GetAudioFilenames().GetCount() == 0) {
				m_vob->SetDoNotTranscode(GetBool(n++));
			}
		}
		for (unsigned int i=0; i<m_vob->GetAudioFilenames().GetCount(); i++) {
			Stream* stream = m_vob->GetStreams()[stIdx++];
			stream->SetDestinationFormat(GetInt(n++) + 1);
		}
		m_vob->SetChapters(GetString(n++));
		m_vob->SetPause(GetInt(n++));
	}
	m_pgc->SetPalette(GetString(n++));
	m_pgc->SetPreCommands(GetString(n++));
	m_pgc->SetPostCommands(GetString(n++));
	return true;
}

void TitlePropDlg::CreateButtonPane(wxSizer* sizer, bool resetButton) {
	wxBoxSizer* buttonPane = new wxBoxSizer(wxHORIZONTAL);
	wxButton* avBt = new wxButton(this, AV_BT_ID, _("Audio/Video..."));
	buttonPane->Add(avBt, 0, wxALIGN_RIGHT|wxALIGN_CENTER_VERTICAL);
	buttonPane->Add(10, 10, 1, wxEXPAND);
	wxButton* okBt = new wxButton(this, wxID_OK, _("OK"));
	okBt->SetDefault();
	buttonPane->Add(okBt, 0, wxALIGN_RIGHT|wxALIGN_CENTER_VERTICAL);
	buttonPane->Add(8, 0);
	wxButton* cancelBt = new wxButton(this, wxID_CANCEL, _("Cancel"));
	buttonPane->Add(cancelBt, 0, wxALIGN_RIGHT|wxALIGN_CENTER_VERTICAL);
	sizer->Add(buttonPane, 0, wxEXPAND|wxLEFT|wxRIGHT|wxBOTTOM, 10);
}

void TitlePropDlg::OnAV(wxCommandEvent& event) {
	AVPropDlg propDlg(this, m_audio, m_video, false);
	if (propDlg.ShowModal() == wxID_OK) {
		wxCommandEvent evt(EVT_COMMAND_DVD_CHANGED, this->GetId());
		GetEventHandler()->ProcessEvent(evt);
	}
}

void TitlePropDlg::OnDoNotTranscodeCheck(wxCommandEvent& event) {
	for (int i = m_formatCtrlIdx; i < m_formatCtrlIdx + (int)m_vob->GetStreams().GetCount(); i++) {
		wxControlWithItems* ctrl = (wxControlWithItems*) GetControl(i);
		ctrl->Enable(!event.IsChecked());
		ctrl->SetSelection(i == m_formatCtrlIdx ? 0 : 1);
	}
}
