// ColorSliderCtrl.cpp : implementation file
//

#include "stdafx.h"
#include "ContextFree.h"
#include "ColorSliderCtrl.h"
#include <math.h>
#include "CGdiPlusBitmap.h"

using namespace Gdiplus;


// CColorSliderCtrl

Bitmap* CColorSliderCtrl::MarkerBitmap = 0;

IMPLEMENT_DYNAMIC(CColorSliderCtrl, CWnd)

BOOL CColorSliderCtrl::RegisterWindowClass()
{
    WNDCLASS  wndcls;
    HINSTANCE hInst = AfxGetInstanceHandle();

    // Already registered?
    if(!(::GetClassInfo(hInst, C_CCOLORSLIDERCTRL_CLASSNAME, &wndcls))) { 
        wndcls.style            = CS_DBLCLKS | CS_HREDRAW | CS_VREDRAW;
        wndcls.lpfnWndProc      = ::DefWindowProc;
        wndcls.cbClsExtra       = wndcls.cbWndExtra = 0;
        wndcls.hInstance        = hInst;
        wndcls.hIcon            = NULL;
        wndcls.hCursor          = AfxGetApp()->LoadCursor(IDC_ARROW);
        wndcls.hbrBackground    = (HBRUSH) (COLOR_3DFACE + 1);
        wndcls.lpszMenuName     = NULL;
        wndcls.lpszClassName    = C_CCOLORSLIDERCTRL_CLASSNAME;

        if(!AfxRegisterClass(&wndcls)) {
            AfxThrowResourceException();
            return FALSE;
        }
    }

    return TRUE;
}

CColorSliderCtrl::CColorSliderCtrl() 
:   mInitialized(false),
    mMin(0),
    mMax(255),
    mPos(128),
    mFocusPen(0),
    mNoFocusPen(0),
    mChannelBitmap(0),
    mTopColor(255,255,255),
    mBottomColor(0,0,0),
    mBackground(0)
{
    RegisterWindowClass();
}

CColorSliderCtrl::~CColorSliderCtrl()
{
    delete mChannelBitmap;
    delete mBackground;
    delete mFocusPen;
    delete mNoFocusPen;
}


BEGIN_MESSAGE_MAP(CColorSliderCtrl, CWnd)
	ON_WM_LBUTTONDOWN()
	ON_WM_LBUTTONUP()
	ON_WM_MOUSEMOVE()
	ON_WM_PAINT()
    ON_WM_SETFOCUS()
    ON_WM_KILLFOCUS()
    ON_WM_KEYDOWN()
    ON_WM_GETDLGCODE()
END_MESSAGE_MAP()

void CColorSliderCtrl::CalcRect()
{
    mCurrentRect.Y = mHeight - 1 - ((mPos - mMin) * (mHeight - 1)) / (mMax - mMin) + mMinY;
}

bool CColorSliderCtrl::SetPos(int pos)
{
    if (pos < mMin) pos = mMin;
    if (pos > mMax) pos = mMax;
    if (pos == mPos) return false;
    mPos = pos;

    RedrawPoint();
    CalcRect();
    RedrawPoint();
    return true;
}

int CColorSliderCtrl::GetPos() const
{
	return mPos;
}

void CColorSliderCtrl::SetColor(const Color& top, const Color& bottom)
{
    if (top.ToCOLORREF() == mTopColor.ToCOLORREF() &&
        bottom.ToCOLORREF() == mBottomColor.ToCOLORREF()) return;

    mTopColor = top;
    mBottomColor = bottom;
    RedrawChannel();
    CRect cr(mChannelRect.GetLeft(), mChannelRect.GetTop(), 
        mChannelRect.GetRight(), mChannelRect.GetBottom());
    cr.InflateRect(1, 1);
    InvalidateRect(&cr, FALSE);
}

void CColorSliderCtrl::RedrawPoint()
{
    CRect cr(mCurrentRect.GetLeft(), mCurrentRect.GetTop(), 
        mCurrentRect.GetRight(), mCurrentRect.GetBottom());
    cr.InflateRect(1, 1);
    InvalidateRect(&cr, FALSE);
}

void CColorSliderCtrl::RedrawChannel()
{
    Graphics g(mChannelBitmap);
    LinearGradientBrush lgb(Point(), Point(0, mHeight), 
        mTopColor, mBottomColor);
    g.FillRectangle(&lgb, 0, 0, 18, mHeight);
}


// CColorSliderCtrl message handlers

void CColorSliderCtrl::Init()
{
    CRect r;
    GetClientRect(&r);

    mHeight = r.Height() - 11;
    mMinY = r.top + 1;

    mCurrentRect = Rect(r.left + 1, 0, 40, 9);
    CalcRect();

    mChannelRect = Rect(r.left + 12, mMinY + 4, 18, mHeight);
    mChannelBitmap = new Bitmap(18, mHeight, PixelFormat24bppRGB);
    mLeftRect = Rect(r.left, r.top, 12, r.Height());
    mRightRect = Rect(r.left + 30, r.top, 12, r.Height());
    RedrawChannel();
    mFocusRect = Rect(r.left, r.top, 41, r.Height() - 2);

    if (!MarkerBitmap) {
        CGdiPlusBitmapResource bm((LPCTSTR)IDB_BITMAP_B);
        MarkerBitmap = bm.m_pBitmap;
        bm.m_pBitmap = 0;
    }

    DWORD backcolor = ::GetSysColor(COLOR_3DFACE);
    Color back((BYTE)(backcolor & 255), 
        (BYTE)((backcolor >> 8) & 255), 
        (BYTE)((backcolor >> 16) & 255));
    mBackground = new SolidBrush(back);

    mFocusPen = new Pen(Color());
    mFocusPen->SetDashStyle(DashStyleDot);
    mNoFocusPen = new Pen(back);

    mNotify.code = SLIDER_CHANGE;
    mNotify.hwndFrom = GetSafeHwnd();
    mNotify.idFrom = GetDlgCtrlID();
    mInitialized = true;
}

void CColorSliderCtrl::OnPaint()
{
    if (GetUpdateRect(NULL, TRUE)) {
        if (!mInitialized) Init();
        CPaintDC dc(this);
        Graphics g(dc);
        g.DrawImage(mChannelBitmap, mChannelRect);
        g.FillRectangle(mBackground, mLeftRect);
        g.FillRectangle(mBackground, mRightRect);
        g.DrawImage(MarkerBitmap, mCurrentRect);
        bool hasFocus = GetFocus() == this;
        g.DrawRectangle(hasFocus ? mFocusPen : mNoFocusPen, mFocusRect);
    }
}

void CColorSliderCtrl::OnSetFocus(CWnd*)
{
    DrawFocus(mFocusPen);
}

void CColorSliderCtrl::OnKillFocus(CWnd*)
{
    DrawFocus(mNoFocusPen);
}

void CColorSliderCtrl::DrawFocus(Gdiplus::Pen* p)
{
    if (!mInitialized) return;
    Graphics g(GetSafeHwnd());
    g.DrawRectangle(p, mFocusRect);
}

void CColorSliderCtrl::OnLButtonDown(UINT nFlags, CPoint point)
{
    SetFocus();
	SetCapture();
	TrackPoint(point);

	CWnd::OnLButtonDown(nFlags, point);
}

void CColorSliderCtrl::OnLButtonUp(UINT nFlags, CPoint point)
{
	if (GetCapture() == this) {
		ReleaseCapture();
    }
	CWnd::OnLButtonUp(nFlags, point);
}

void CColorSliderCtrl::OnMouseMove(UINT nFlags, CPoint point)
{
	if (GetCapture() == this) {
		TrackPoint(point);
	}
	CWnd::OnMouseMove(nFlags, point);
}

void CColorSliderCtrl::TrackPoint(CPoint point)
{
    int pos = point.y - mMinY;
    if (pos < 0) pos = 0;
    if (pos >= mHeight) pos = mHeight - 1;

    pos = ((mHeight - 1 - pos) * (mMax - mMin)) / (mHeight - 1) + mMin;

    if (SetPos(pos))
        GetParent()->SendNotifyMessage(WM_NOTIFY, mNotify.idFrom, (LPARAM)(&mNotify));
}

void CColorSliderCtrl::OnKeyDown(UINT nChar, UINT nRepCnt, UINT nFlags)
{
    int pos = mPos;
    if (nChar == VK_UP) pos += nRepCnt;
    if (nChar == VK_DOWN) pos -= nRepCnt;
    if (SetPos(pos))
        GetParent()->SendNotifyMessage(WM_NOTIFY, mNotify.idFrom, (LPARAM)(&mNotify));
}

UINT CColorSliderCtrl::OnGetDlgCode()
{
    return DLGC_WANTARROWS;
}


