
#include "stdafx.h"
#include "CRenderView.h"
#include <afxmt.h>
#include "WinCanvas.h"
#include "ChildFrm.h"
#include "tiledCanvas.h"

IMPLEMENT_DYNCREATE( CRenderView, CView )

BEGIN_MESSAGE_MAP( CRenderView, CView )
ON_WM_SIZE()
ON_WM_CONTEXTMENU()
ON_WM_ERASEBKGND()
END_MESSAGE_MAP()

#include <gdiplus.h>

using namespace Gdiplus;

CRenderView::CRenderView()
: mCanvas(0),
mDisplay(true),
mNotRendered(true),
mFrame(0),
mWidth(0),
mHeight(0),
mActualWidth(0),
mActualHeight(0),
mSizeChanged(true),
mBackingStore(0)
{
}

CRenderView::~CRenderView()
{
    delete mBackingStore;
}

void CRenderView::OnContextMenu(CWnd* , CPoint point)
{
    CMenu menu; 
    
    // Try to load context menu
    if (menu.LoadMenu(IDR_RENDER_POPUP)) {
        CMenu* popup = menu.GetSubMenu(0);
        ASSERT(popup);
        
        //show the context menu
        popup->TrackPopupMenu(TPM_LEFTALIGN | TPM_LEFTBUTTON | TPM_RIGHTBUTTON, 
            point.x , point.y , AfxGetMainWnd()); 
    }
}

BOOL CRenderView::OnEraseBkgnd(CDC*)
{
    // CRenderViews have no background to erase
    return -1;
}

void CRenderView::OnSize(UINT type, int width, int height)
{
    // Calculate the size and position of the output window. This is used to psotion the window
    // when drawing and to size a canvas when it is created.
    CView::OnSize(type, width, height);
    mActualWidth = width;
    mActualHeight = height;
    mWidth = width;
    mHeight = height;
    mSizeChanged = true;
    Invalidate(TRUE);
}

void CRenderView::OnDraw(CDC* dc)
{
    CView::OnDraw(dc);
    if (!WinCanvas::GdiPlusInitialized) return;

    Color gray(255, 224, 224, 224);
    Color white(255, 255, 255, 255);
    Pen p(gray, 1.0);
     
    // Create a new backing store if the window size changed, but not if the user
    // is still changing the window size. In that case we wait until the user
    // stops resizing before creating a new backing store.
    bool sizeChanged = mSizeChanged;
    if (mSizeChanged && !theApp.mResizing) {
        delete mBackingStore;
        mBackingStore = new Bitmap(mActualWidth, mActualHeight, PixelFormat32bppARGB);
        Graphics g(mBackingStore);
        g.Clear(mCanvas ? mCanvas->mBackgroundOpaque : white);
        mSizeChanged = false;
    }

    if (theApp.mResizing) {
        // If the user is actively resizing then do a cheap rescale from the backing store
        Graphics screen(dc->GetSafeHdc());
        screen.Clear(mCanvas ? mCanvas->mBackground : white);
        
        SplatBitmap(mBackingStore, mBackingStore->GetWidth(), mBackingStore->GetHeight(),
            screen, mCanvas ? mCanvas->mBackground : white, mActualWidth, mActualHeight, 
            InterpolationModeLowQuality);
    } else {
        if (!mCanvas) {
            Graphics g(mBackingStore);
            SplatBitmap(0, 0, 0, g, white, mWidth, mHeight, 
                InterpolationModeHighQualityBicubic);
        }

        // If the user is not resizing then update the backing store and blit it to the screen
        else if ((sizeChanged || mCanvas->mChanged) && mFrame->mOwnCanvas.Lock(0)) {
            // If we can't lock the canvas then don't draw. Whoever locked the canvas
            // must invalidate me when they unlock the canvas.
            Graphics g(mBackingStore);
            Bitmap* grayBM = mCanvas->MakeBitmap(false);
            
            SplatBitmap(grayBM, mCanvas->mWidth, mCanvas->mHeight,
                g, mCanvas->mBackground, mWidth, mHeight, 
                InterpolationModeHighQualityBicubic);
            
            mCanvas->mChanged = false;
            mFrame->mOwnCanvas.Unlock();
            
            // Free temp bitmap
            delete grayBM;
        }
        
        Graphics screen(dc->GetSafeHdc());
        screen.DrawImage(mBackingStore, 0, 0, mActualWidth, mActualHeight);
	}
}


void CRenderView::SplatBitmap(Bitmap* srcBM, int srcWidth, int srcHeight, 
                              Graphics& destGfx, Color back,
                              int destWidth, int destHeight, 
                              InterpolationMode stretchMode)
{
    Color gray(255, 224, 224, 224);
    Color white(255, 255, 255, 255);
    Pen p(gray, 1.0);
    SolidBrush grayBrush(gray);
    SolidBrush whiteBrush(white);

    if (!mCanvas || !mDisplay || mNotRendered) {
        // If there is no canvas bitmap or we aren't going to display it then
        // draw a nice message bitmap instead.
        destGfx.Clear(white);
        int originX = (destWidth - 202) / 2;
        int originY = (destHeight - 202) / 2;
        destGfx.DrawRectangle(&p, Rect(originX, originY, 202, 202));
        
        FontFamily  family(L"Arial");
        Font f(&family, 12, FontStyleRegular, UnitPixel);
        SolidBrush b(Color(0,0,0));
		if (mNotRendered) {
            destGfx.DrawString(L"Click on 'Render' to start.", -1, &f, 
                PointF(originX + 10.0F, originY + 181.0F), &b);
        } else if (mCanvas && mCanvas->mDone) {
            destGfx.DrawString(L"Rendering complete.", -1, &f, 
                PointF(originX + 10.0F, originY + 165.0F), &b);
            destGfx.DrawString(L"Ready to save.", -1, &f, 
                PointF(originX + 10.0F, originY + 181.0F), &b);
        } else {
            destGfx.DrawString(L"Rendering, please be patient...", -1, &f, 
                PointF(originX + 10.0F, originY + 181.0F), &b);
        }
        
        destGfx.DrawImage(WinCanvas::AppBitmap, originX + 37, originY + 37, 128, 128);
    } else {
        double scale = 1.0;
        
        // check if the bitmap is too big and shrink it to fit
        if (srcWidth > mWidth || srcHeight > mHeight) {
            double widthScale = (double)mWidth / srcWidth;
            double heightScale = (double)mHeight / srcHeight;
            scale = (widthScale < heightScale) ? widthScale : heightScale;
        }
        
        // scale the bitmap if it is too big
        int scaledWidth = (int)(srcWidth * scale);
        int scaledHeight = (int)(srcHeight * scale);
        
        // center the scaled bitmap
        int originX = (destWidth - scaledWidth)/ 2;
        int originY = (destHeight - scaledHeight)/ 2;
        
        // Draw the bitmap scaled
        Rect destRect(originX, originY, scaledWidth, scaledHeight);
        if (back.GetAlpha() < 255) {
            destGfx.Clear(white);
            for (int y = 0; y <= (destHeight >> 3); y++)
                for (int x = 0; x <= (destWidth >> 3); x++)
                    if ((x + y) & 1)
                        destGfx.FillRectangle(&grayBrush, Rect(x * 8, y * 8, 8, 8));

            if (!mFrame->mTiled) {
                SolidBrush b(back);
                destGfx.SetClip(destRect, CombineModeExclude);
                destGfx.FillRectangle(&b, Rect(0, 0, destWidth, destHeight));
                destGfx.ResetClip();
            }
        } else {
            destGfx.Clear(back);
        }

        destGfx.SetInterpolationMode(stretchMode);

        if (mFrame->mTiled && scale == 1.0) {
            tileList points = 
                mFrame->mTiledCanvas->getTesselation(destWidth, destHeight, originX, originY, true);
            
            for (tileList::iterator pt = points.begin(); 
                 pt != points.end(); pt++) {
                destRect = Rect(pt->x, pt->y, srcWidth, srcHeight);

                destGfx.DrawImage(srcBM, destRect, 0, 0, srcWidth, srcHeight,
                    UnitPixel, NULL, NULL, NULL);
            }
        } else {
            destGfx.DrawImage(srcBM, destRect, 0, 0, srcWidth, srcHeight,
                UnitPixel, NULL, NULL, NULL);
        }
        Pen p2(Color(), 1.0);
        p2.SetDashStyle(DashStyleDot);
        destGfx.DrawRectangle(&p2, Rect(originX - 1, originY - 1, scaledWidth + 1, scaledHeight + 1));
    }
}
