/*
 *

 Context Free Design Grammar - version 0.1

 Copyright (C) 2005 Chris Coyne - ccoyne77@gmail.com

 [Send me anything cool you make with it or of it.]

 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License as published
 by the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 02111-1307  USA

 *
 */

#include <iostream>
#include <fstream>
#include <sys/time.h>
#include <signal.h>
#include <unistd.h>
#include <stdlib.h>
#include "cfdg.h"
#include "variation.h"
#include "pngCanvas.h"
#include "SVGCanvas.h"
#include "posixSystem.h"
#include "version.h"
#include "tiledCanvas.h"

using namespace std;


Renderer* runningRenderer = 0;

void
statusTimer(int signal)
{
  if (runningRenderer) runningRenderer->requestUpdate();
}

void
setupTimer(Renderer* renderer)
{
  runningRenderer = renderer;

  struct sigaction doit;
  doit.sa_handler = statusTimer;
  sigemptyset(&doit.sa_mask);
  doit.sa_flags = SA_RESTART;
  sigaction(SIGALRM, &doit, 0);

  itimerval period;
  period.it_interval.tv_sec = 0;
  period.it_interval.tv_usec = 250000;  // 1/4 second
  period.it_value = period.it_interval;
  setitimer(ITIMER_REAL, &period, 0);
}


const char* invokeName = "";

void
usage(bool inError)
{
  ostream& out = inError ? cerr : cout;

  out << APP_NAME << " - " << APP_VERSION << "(v" << APP_BUILD << ")" << endl;
  out << endl;
  out << "Usage: " << endl << APP_NAME
      << " [options] input.cfdg output.png/svg" << endl;
  out << "    or to pipe a cfdg file on standard input:" << endl;
  out << APP_NAME
      << " [options] - output.png/svg" << endl << endl;
  out << "Options: " << endl;
  out << "    -w num    width in pixels(png) or mm(svg) (default 500)" << endl;
  out << "    -h num    height in pixels(png) or mm(svg) (default 500)" << endl;
  out << "    -s num    set both width and height in pixels/mm" << endl;
  out << "    -m num    maximum number of shapes (default none)" << endl;
  out << "    -x float  minimum size of shapes in pixels/mm (default 0.3)" << endl;
  out << "    -b float  border size [-1,2]: -1=-8 pixel border, 0=no border," << endl;
  out << "              1=8 pixel border, 2=variable-sized border" << endl;
  out << "    -v str    set the variation code (default is random)" << endl;
  out << "    -a num    generate num animation frames (PNG only)" << endl;
  out << "    -z        zoom out during animation" << endl;
  out << "    -V        generate SVG (vector) output" << endl;
  out << "    -c        crop image output" << endl;
  out << "    -?        show this message, then exit" << endl;
  out << endl;
  out << "Historical usage: " << invokeName
      << " input.cfdg output.png width height [maxshapes]" << endl;

  exit(inError ? 2 : 0);
}

struct options {
  enum OutputFormat { PNGfile = 0, SVGfile = 1 };
  int   width;
  int   height;
  int   maxShapes;
  float minSize;
  float borderSize;

  int   variation;
  bool  crop;
  int   animationFrames;
  bool  animationZoom;

  const char* input;
  const char* output;
  OutputFormat format;
  
  options()
  : width(500), height(500), maxShapes(0), minSize(0.3),borderSize(2),
    variation(-1), crop(false), animationFrames(0), animationZoom(false),
    input(0), output(0), format(PNGfile)
  { }
};

int
intArg(char arg, const char* str)
{
  char* end;
  long int v = -1;

  v = strtol(str, &end, 10);
  if (end == str || v < 0) {
    if (arg == '-')
      cerr << "Argument must be a positive integer" << endl;
    else
      cerr << "Option -" << arg << " takes a positive integer argument" << endl;
    usage(true);
  }
  return (int)v;
}

float
floatArg(char arg, const char* str)
{
  float v = atof(str);

  if (v < 0) {
    if (arg == '-')
      cerr << "Argument must be a positive float" << endl;
    else
      cerr << "Option -" << arg << " takes a positive float argument" << endl;
    usage(true);
  }
  return v;
}

void
processCommandLine(int argc, char* argv[], options& opt)
{
  extern char *optarg;
  extern int optind, optopt;

  invokeName = argv[0];

  char c;
  while ((c = getopt(argc, argv, ":w:h:s:m:x:b:v:a:cVz")) != -1) {
    switch(c) {
      case 'w':
        opt.width = intArg(c, optarg);
        break;
      case 'h':
        opt.height = intArg(c, optarg);
        break;
      case 's':
        opt.height = opt.width = intArg(c, optarg);
        break;
      case 'm':
        opt.maxShapes = intArg(c, optarg);
        break;
      case 'x':
        opt.minSize = floatArg(c, optarg);
        break;
      case 'b':
        opt.borderSize = atof(optarg);
        if (opt.borderSize < -1 || opt.borderSize > 2) usage(true);
        break;
      case 'v':
        opt.variation = Variation::fromString(optarg);
        break;
      case 'a':
        if (opt.format == options::SVGfile) usage(true);
        opt.animationFrames = intArg(c, optarg);
        break;
      case 'V':
        if (opt.animationFrames) usage(true);
        opt.format = options::SVGfile;
        break;
      case 'c':
        opt.crop = true;
        break;
      case 'z':
        opt.animationZoom = true;
        break;
      case ':':
        cerr << "Option -" << (char)optopt << " requires an argument" << endl;
        usage(true);
        break;
      case '?':
        if (optopt == '?')
          usage(false);
        cerr << "Unrecognized option: -" << optopt << endl;
        usage(true);
        break;
    }
  }

  if (optind < argc) {
    opt.input = argv[optind++];
  }
  if (optind < argc) {
    opt.output = argv[optind++];
  }
  if (optind < argc) {
    opt.width = intArg('-', argv[optind++]);
  }
  if (optind < argc) {
    opt.height = intArg('-', argv[optind++]);
  }
  if (optind < argc) {
    opt.maxShapes = intArg('-', argv[optind++]);
  }
  if (optind < argc) {
    cerr << "Too many arguments" << endl;
    usage(true);
  }

  if (!opt.input) {
    cerr << "Missing input file" << endl;
    usage(true);
  }
  if (!opt.output) {
    cerr << "Missing output file" << endl;
    usage(true);
  }
}


int main (int argc, char* argv[]) {
  options opts;
  int var = Variation::random(6);

  processCommandLine(argc, argv, opts);

  PosixSystem system;
  CFDG* myDesign = CFDG::ParseFile(opts.input, &system);
  if (!myDesign) return 1;

  agg::trans_affine tileTransform;
  bool isTiled = myDesign->isTiled(&tileTransform);
  
  if (opts.variation < 0) opts.variation = var;
  char code[Variation::maxStringLength];
  Variation::toString(opts.variation, code, false);

  bool useRGBA = myDesign->usesColor();
  aggCanvas::PixelFormat pixfmt = aggCanvas::SuggestPixelFormat(myDesign);

  cout << "Generating " << (useRGBA ? "color" : "gray-scale")
       << ((opts.format == options::PNGfile) ? " PNG image" : " SVG vector output") 
       << ", variation " << code << "..." << endl;

  pngCanvas* png = 0;
  SVGCanvas* svg = 0;
  tiledCanvas* tile = 0;
  Canvas* myCanvas;
  
  Renderer* myRenderer = myDesign->renderer(
                                            opts.width, opts.height, opts.minSize,
                                            opts.variation, opts.borderSize);
  myRenderer->setMaxShapes(opts.maxShapes);
  opts.width = myRenderer->m_width;
  opts.height = myRenderer->m_height;
  double scale = myRenderer->run(0, false);
  
  if (opts.format == options::PNGfile) {
	  png = new pngCanvas(opts.output, opts.width, opts.height, pixfmt, opts.crop, opts.animationFrames);
	  myCanvas = (Canvas*)png;
  } else {
	  svg = new SVGCanvas(opts.output, opts.width, opts.height, opts.crop);
	  myCanvas = (Canvas*)svg;
  }
  
  if (isTiled) {
      tile = new tiledCanvas(myCanvas, tileTransform);
      myCanvas = tile;
      tile->scale(scale);
  }

  setupTimer(myRenderer);

  if (opts.animationFrames) {
    myRenderer->animate(myCanvas, opts.animationFrames, opts.animationZoom);
  } else {
    myRenderer->draw(myCanvas);
  }
  cout << endl;

  cout << "DONE!" << endl;
  
  delete png;
  delete svg;
  delete tile;

  return 0;
}


