//
//  GalleryUploader.m
//  Context Free
//
//  Created by Mark Lentczner on 1/12/07.
//  Copyright 2007 __MyCompanyName__. All rights reserved.
//

#import "GalleryUploader.h"
#import "CFDGDocument.h"
#import "GView.h"
#import "VariationFormatter.h"

#include <string>
#include <sstream>

#include "upload.h"

namespace {

	std::string asString(NSString* ns)
	{
		NSData* d = [ns dataUsingEncoding: NSUTF8StringEncoding];
		std::string s((const char*)[d bytes], [d length]);
		return s;
	}
	
	NSString* asNSString(const std::string& s)
	{
		return [NSString stringWithUTF8String: s.c_str()];
	}

}



@implementation GalleryUploader

- (id)initForDocument:(CFDGDocument*)document andView:(GView*)view;
{
	self = [self initWithWindowNibName: @"GalleryUploader"];
	if (!self) return self;

	mDocument = document;
	mView = view;
    mStatus = 0;
		// no need to retain - the document is retaining us!
		
	[self loadWindow];
	[mFormView retain];
	[mProgressView retain];
	[mDoneView retain];
	return self;
}

- (void) dealloc {
	[mDoneView release];
	[mProgressView release];
	[mFormView release];
	[mConnection release];
	[mResponseBody release];
	[super dealloc];
}


- (NSData*)requestBody
{
	Upload upload;
	
	upload.mUserName	= asString([mUserNameField stringValue]);
	upload.mPassword	= asString([mPasswordField stringValue]);
	upload.mTitle		= asString([mTitleField stringValue]);
	upload.mNotes		= asString([mNotesView string]);
	upload.mFileName	= asString([mFileField stringValue]) + ".cfdg";
	upload.mVariation	= [mView variation];
    upload.mTiled       = [mView isTiled];
	upload.mCompression	= (Upload::Compression)
							[[mCompressionMatrix selectedCell] tag];

	NSData* textData	= [mDocument getContent];
	NSData* imageData	= [mView pngImageDataCropped: 
        [[NSUserDefaults standardUserDefaults] boolForKey: @"SaveCropped"]
                                         rectangular: upload.mTiled];
	
	upload.mText		= (const char*)[textData bytes];
	upload.mTextLen		= [textData length];
	upload.mImage		= (const char*)[imageData bytes];
	upload.mImageLen	= [imageData length];
	
	std::ostringstream payloadStream;
	upload.generatePayload(payloadStream);
	std::string payloadString = payloadStream.str();
	
	return [NSData dataWithBytes: payloadString.data()
					length: payloadString.length()];
}

- (void)setView:(NSView*)view
{
	NSRect contentFrame = [mContentView frame];
	
	NSSize oldContentSize = contentFrame.size;
	NSSize newContentSize = [view frame].size;
	NSSize deltaSize;
	deltaSize.width = newContentSize.width - oldContentSize.width;
	deltaSize.height = newContentSize.height - oldContentSize.height;
	
	NSView* container = [mContentView superview];
	[mContentView removeFromSuperview];
	contentFrame.size = newContentSize;
	[view setFrame: contentFrame];
	[container addSubview: view];
	mContentView = view;

	NSWindow* window = [self window];
	NSRect f = [window frame];
	f.origin.x -= deltaSize.width / 2.0;
	f.origin.y -= deltaSize.height;
	f.size.width += deltaSize.width;
	f.size.height += deltaSize.height;
	
	[window setFrame: f display: YES animate: YES];
}

- (void)allDone:(NSString*)message
{
	[mConnection release];		mConnection = nil;

    if (message) mStatus = -1;
    
    [mRetryButton setEnabled:NO];
    switch (mStatus) {
        case -1:
            [mMessage setString: message];
            break;
        case 0:
            [mMessage setString: @"Upload completed without a status code (?!?!?!)."];
            break;
        case 409:
            [mRetryButton setEnabled:YES];
        case 200: {
            NSAttributedString *theParsedHTML;

            // Take the raw HTML data and then initialize an NSMutableAttributed
            // string with HTML code
            theParsedHTML = [[NSAttributedString alloc] initWithHTML:mResponseBody 
                        documentAttributes: nil];
            
            // no parsing or fetching error.. so lets display it
            
            if (theParsedHTML) {
                [[mMessage textStorage] setAttributedString:theParsedHTML];
                [theParsedHTML release];
            }
            break;
        }
        case 404:
            [mMessage setString: @"Upload service is offline."];
            break;
        default:
            [mMessage setString: [NSString localizedStringWithFormat:
                @"Failed with unexpected status code: %d, %@", mStatus, 
                [NSHTTPURLResponse localizedStringForStatusCode: mStatus]]];
            break;
    }
	[mResponseBody release];	mResponseBody = nil;
	[self setView: mDoneView];
}

- (IBAction)retry:(id)sender
{
	[self setView: mFormView];
}

- (IBAction)show:(id)sender
{
	[self setView: mFormView];
	
	if ([[mTitleField stringValue] length] == 0) {
		[mTitleField setStringValue: [mDocument displayName]];
	}

	if ([[mFileField stringValue] length] == 0) {
		NSString* file = [mDocument fileName];
		if (file) {
			file = [[[file stringByDeletingPathExtension] lastPathComponent]
						retain];
		}
		else {
			file = [mDocument displayName];
		}
		[mFileField setStringValue: file];
	}

	[mVariationField setIntValue: [mView variation]];
    
    int bestCompression = [mView canvasColor256] ? 
        Upload::CompressPNG8 : Upload::CompressJPEG;
	[mCompressionMatrix selectCellWithTag: bestCompression];

    [NSApp beginSheet: [self window]
        modalForWindow: [mView window]
        modalDelegate: nil didEndSelector: nil contextInfo: nil];
}


- (IBAction)goToCreateAccount:(id)sender
{
    NSURL* url = [NSURL URLWithString: 
        @"http://www.contextfreeart.org/phpbb/profile.php?mode=register"];
    
    [[NSWorkspace sharedWorkspace] openURL: url];
}

- (IBAction)upload:(id)sender
{
	NSMutableURLRequest* request =
		[NSMutableURLRequest requestWithURL:
				[NSURL URLWithString: @"http://www.contextfreeart.org/gallery/upload.php"]
			cachePolicy: NSURLRequestReloadIgnoringCacheData
			timeoutInterval: 120.0
			];
	[request setHTTPMethod: @"POST"];
	[request setValue: asNSString(Upload::generateContentType())
				forHTTPHeaderField: @"Content-Type"];
	[request setHTTPBody: [self requestBody]];
	
	mResponseBody = [[NSMutableData data] retain];

	mConnection =
		[[NSURLConnection alloc] initWithRequest: request delegate: self];
	if (!mConnection) {
		[self cancel: sender];
	}

	[self setView: mProgressView];
}


-(void)connection:(NSURLConnection *)c didReceiveResponse:(NSURLResponse *)response
{
    [mResponseBody setLength: 0];
    
    if ([response isKindOfClass:[NSHTTPURLResponse class]] ) {
        mStatus = [(NSHTTPURLResponse *)response statusCode];
    }
}

- (void)connection:(NSURLConnection *)c didReceiveData:(NSData *)data
{
    [mResponseBody appendData: data];
}

-(void)connectionDidFinishLoading:(NSURLConnection *)c
{
    [self performSelector:@selector( allDone: )
               withObject:0
               afterDelay:0.0];	
}

- (void)connection:(NSURLConnection *)c didFailWithError:(NSError *)error
{
	[self allDone: [error localizedDescription]];
}




- (IBAction)cancel:(id)sender
{
	if (mConnection) {
		[mConnection cancel];
	}

	[mConnection release];		mConnection = nil;
	[mResponseBody release];	mResponseBody = nil;

	NSWindow* window = [self window];
	[NSApp endSheet: window];
	[window orderOut: sender];
}

@end
