/*
 *  yglue.cpp
 *  CFDA
 *
 *  Created by Mark Lentczner on 4/14/05.
 *  Copyright 2005 __MyCompanyName__. All rights reserved.
 *
 */

#include "yglue.h"

#include <string.h>
#include <stdio.h>

#include "builder.h"
#include "cfdg.h"
#include "cfdgimpl.h"
#include "shape.h"

#ifdef _WIN32
#pragma warning( disable : 4786 )
#endif

extern Builder * builder;

extern "C" {
    extern int yylineno;
    void yyrestart(FILE *new_file);
}

void yyerror(const char *msg) {
    builder->m_CFDG->system()->syntaxError(builder->m_currentPath, yylineno);
    builder->m_CFDG->system()->message(
        "Line %d: Parse error - %s", yylineno, msg);
}


yg_real     yg_BuildReal(const char* token)    
{ 
    builder->addEntropy(token); 
    return atof(token); 
}
yg_string   yg_BuildString(const char* token)  { return strdup(token); }
void        yg_FreeString(yg_string st)        { free(st); }

using namespace std;

// Store parsed contents of include lines
void
yg_IncludeFile(yg_string fname)
{
    string path =
        builder->m_CFDG->system()->relativeFilePath(builder->m_currentPath, fname);
    builder->m_filesToLoad.push(path);
    yg_FreeString(fname);
}


static int
yg_StringToShape(yg_string st)
{
    string name = st;
    yg_FreeString(st);

    return builder->m_CFDG->encodeShapeName(name);
}

// Store parsed contents of startshape lines
void
yg_Initialize(yg_string name)
{
    Shape si;
    si.mShapeType = yg_StringToShape(name);
    
    builder->m_CFDG->setInitialShape(si);
}

// Store parsed contents of background lines
void
yg_Background()
{
    HSBColor solidWhite(0, 0, 1, 1);
    solidWhite.adjustWith(builder->mCurrentModification.m_Color);
    agg::rgba bk;
    solidWhite.getRGBA(bk);
    builder->m_CFDG->setBackgroundColor(bk);

    builder->clearModification();
}

//Store parsed contents of tile lines
const char*
yg_Tile()
{
    const char* errmsg = builder->buildTileTransform();
    builder->clearModification();
    return errmsg;
}

//Store parsed contents of size lines
const char*
yg_SizeLine()
{
    const char* errmsg = builder->buildSizeTransform();
    builder->clearModification();
    return errmsg;
}

void
yg_Rule(yg_string name, yg_real weight)
{
    if (weight <= 0.0) weight = 1.0;

    // Store the current rule and its replacements in the rule list
    builder->m_currRule.mInitialShapeType = yg_StringToShape(name);
    builder->m_currRule.mWeight = weight;
    builder->m_CFDG->addRule(builder->m_currRule);

    // Restore builder current rule to initial state for next rule
    builder->m_currRule = Rule();
}

void
yg_Replacement(yg_string name)
{
    // Store replacement shape in the rules replacement list
    Replacement r;
    builder->addEntropy(name);
    r.mShapeType = yg_StringToShape(name);
    r.mChildChange = builder->mCurrentModification;
    r.mIterationCount = 1;
    builder->m_currRule.addReplacement(r);
    builder->clearModification();
}

void
yg_ReplacementLoop(yg_string name, int count)
{
    // Store replacement shape in the rules replacement list
    Replacement r;
    builder->addEntropy(name);
    r.mShapeType = yg_StringToShape(name);
    r.mIterationChange = builder->mSavedModification;
    r.mChildChange = builder->mCurrentModification;
    r.mIterationCount = count;
    builder->m_currRule.addReplacement(r);
    builder->clearModification();
}

void
yg_ClearTransform()
{
    builder->clearModification();
}

void
yg_SaveTransform()
{
    builder->mSavedModification = builder->mCurrentModification;
    builder->clearModification();
}

// Calls the builder routine to discard the built-up transform that is used
// in the shape [ ... ] syntax and build a new transform use the standard
// transform order for the shape { ... } syntax.
void 
yg_DefTransform() 
{
    builder->buildDefaultTransform();
}

// These routines a) store that the transform was parsed b) multiply the 
// transform into the current transform
void yg_Orientation(yg_real v)          { builder->Orientation(v); }
void yg_Reflection(yg_real v)           { builder->Reflection(v); }
void yg_XLocation(yg_real v)            { builder->XLocation(v); }
void yg_YLocation(yg_real v)            { builder->YLocation(v); }
void yg_ZLocation(yg_real v)            { builder->ZLocation(v); }
void yg_Size(yg_real vx, yg_real vy, yg_real vz)    { builder->Size(vx, vy, vz); }
void yg_Skew(yg_real vx, yg_real vy)    { builder->Skew(vx, vy); }

void yg_Hue(yg_real v, int useTarget)
{
    builder->addEntropy("Hue");
    builder->m_CFDG->addParameter(CFDGImpl::Color);
    if (useTarget) {
        if (v < -1) v = -1;
        if (v >  1) v =  1;
    }
    builder->mCurrentModification.m_Color.h = v;
    if (useTarget) {
        builder->mCurrentModification.m_Color.mUseTarget |= HSBColor::HueTarget;
        builder->addEntropy("target");
    }
}

void yg_Saturation(yg_real v, int useTarget)
{
    builder->addEntropy("Saturation");
    builder->m_CFDG->addParameter(CFDGImpl::Color);
    if (v < -1) v = -1;
    if (v >  1) v =  1;
    builder->mCurrentModification.m_Color.s = v;
    if (useTarget) {
        builder->mCurrentModification.m_Color.mUseTarget |= HSBColor::SaturationTarget;
        builder->addEntropy("target");
    }
}

void yg_Brightness(yg_real v, int useTarget)
{
    builder->addEntropy("Brightness");
    if (v < -1) v = -1;
    if (v >  1) v =  1;
    builder->mCurrentModification.m_Color.b = v;
    if (useTarget) {
        builder->mCurrentModification.m_Color.mUseTarget |= HSBColor::BrightnessTarget;
        builder->addEntropy("target");
    }
}

void yg_Alpha(yg_real v, int useTarget)
{
    builder->addEntropy("Alpha");
    builder->m_CFDG->addParameter(CFDGImpl::Alpha);
    if (v < -1) v = -1;
    if (v >  1) v =  1;
    builder->mCurrentModification.m_Color.a = v;
    if (useTarget) {
        builder->mCurrentModification.m_Color.mUseTarget |= HSBColor::AlphaTarget;
        builder->addEntropy("target");
    }
}

void yg_HueTarget(yg_real v, int )
{
    builder->addEntropy("HueTarget");
    builder->m_CFDG->addParameter(CFDGImpl::Color);
    builder->mCurrentModification.m_ColorTarget.h = v;
}

void yg_SaturationTarget(yg_real v, int )
{
    builder->addEntropy("SaturationTarget");
    builder->m_CFDG->addParameter(CFDGImpl::Color);
    if (v < -1) v = -1;
    if (v >  1) v =  1;
    builder->mCurrentModification.m_ColorTarget.s = v;
}

void yg_BrightnessTarget(yg_real v, int )
{
    builder->addEntropy("BrightnessTarget");
    if (v < -1) v = -1;
    if (v >  1) v =  1;
    builder->mCurrentModification.m_ColorTarget.b = v;
}

void yg_AlphaTarget(yg_real v, int )
{
    builder->addEntropy("AlphaTarget");
    builder->m_CFDG->addParameter(CFDGImpl::Alpha);
    if (v < -1) v = -1;
    if (v >  1) v =  1;
    builder->mCurrentModification.m_ColorTarget.a = v;
}

void
yg_Reset()
{
    yylineno = 1;
    yyrestart(0);
}

void
yg_ResetForNextFile()
{
    yylineno = 1;
}

int
yg_Input(char* buf, int max)
{
    if (!builder->m_input  ||  builder->m_input->eof())
        return 0;
    
    builder->m_input->read(buf, max);
	
	char* r = buf;
	char* w = r;
	char* e = r + builder->m_input->gcount();
    
    // Tack on a line-feed at the end of the cfdg file so that line comments
    // at the end of the file are tokenized correctly even if they have no
    // terminating line feed.
    if ((e - r) < max) *e++ = '\n';
    
    // Convert "\r\n" to " \n" and "\r" to "\n"
	while (r != e) {
		char c = *r++;
		if (c == '\r') {
			*w++ = (*r == '\n') ? ' ' : '\n';
		} else {
            *w++ = c;
		}
	}
	
    return w - buf;
}

