/*
 *  tiledCanvas.cpp
 *  Context Free
 *
 *  Created by John Horigan on 4/11/06.
 *  Copyright 2006 John Horigan. All rights reserved.
 *
 */

#include "tiledCanvas.h"
#include <math.h>
#include "primShape.h"
#include "bounds.h"

void tiledCanvas::start(bool clear, const agg::rgba& bk, int , int )
{
	mTile->start(clear, bk, mTile->mWidth, mTile->mHeight);
}

void tiledCanvas::end() 
{
	mTile->end();
}

void tiledCanvas::circle(RGBA8 c, agg::trans_affine tr)
{
    if (tileTransform(tr)) {
		mTile->circle(c, tr);
    } else {
        for (int i = 0; i < 9; i++)
            mTile->circle(c, tr * mTr[i]);
    }
}

void tiledCanvas::square(RGBA8 c, agg::trans_affine tr)
{
    if (tileTransform(tr)) {
		mTile->square(c, tr);
    } else {
        for (int i = 0; i < 9; i++)
            mTile->square(c, tr * mTr[i]);
    }
}

void tiledCanvas::triangle(RGBA8 c, agg::trans_affine tr)
{
    if (tileTransform(tr)) {
		mTile->triangle(c, tr);
    } else {
        for (int i = 0; i < 9; i++)
            mTile->triangle(c, tr * mTr[i]);
    }
}

static agg::trans_affine_scaling doubleSize(2.0);

bool tiledCanvas::tileTransform(agg::trans_affine& tr)
// Adjust the translation part of the transform so that it falls within the 
// tile parallelogram at the origin. 
//
// Returns whether the shape is close to the edge of the canvas 
// (true=not close, false=close/overlapping).
{
	double dummy;
    mInvert.transform(&(tr.tx), &(tr.ty));  // transform to unit square tesselation
	tr.tx = modf(tr.tx, &dummy);            // translate to unit square at the origin
	tr.ty = modf(tr.ty, &dummy);
    if (tr.tx < 0.0) tr.tx += 1.0;
    if (tr.ty < 0.0) tr.ty += 1.0;
    mOffset.transform(&(tr.tx), &(tr.ty));  // transform back to specified tesselation
	
    Bounds b(doubleSize * tr, primShape::squareType);
    return b.mMin_X > 0 && b.mMax_X < mWidth && b.mMin_Y > 0 && b.mMax_Y < mHeight;
}

tiledCanvas::tiledCanvas(Canvas* tile, const agg::trans_affine& tr) 
:   Canvas(tile->mWidth, tile->mHeight), 
    mTile(tile), 
    mOffset(tr)
{
}

void tiledCanvas::scale(double scaleFactor)
{
    agg::trans_affine_scaling scale(scaleFactor);
    
    // Generate the tiling transform in pixel units
    mOffset *= scale;
    
    // The mInvert transform can transform coordinates from the pixel unit tiling
    // to the unit square tiling.
    mInvert = mOffset;
    mInvert.invert();
    
    int i = 0;
    for (double y = -1.0; y < 1.5; y += 1.0)
        for (double x = -1.0; x < 1.5; x += 1.0) {
            double x1 = x;
            double y1 = y;
            mOffset.transform(&x1, &y1);
            mTr[i++] = agg::trans_affine_translation(x1, y1);
        }
}

tileList tiledCanvas::getTesselation(int w, int h, int x, int y, bool flipY)
{
    // Produce an integer version of mOffset that is centered in the w x h screen
    agg::trans_affine tess(mWidth, floor(mOffset.shy + 0.5), floor(mOffset.shx + 0.5),
        flipY ? -mHeight : mHeight, x, y);
    agg::rect_i screen(0, 0, w - 1, h - 1);
    
    tileList tessPoints;
    tessPoints.push_back(agg::point_i(x, y));   // always include the center tile
    
    // examine rings of tile units around the center unit until you encounter a
    // ring that doesn't have any tile units that intersect the screen. Then stop.
    for (int ring = 1; ; ring++) {
        bool hit = false;
        for (int y = -ring; y <= ring; y++) {
            for (int x = -ring; x <= ring; x++) {
                // These loops enumerate all tile units on and within the ring.
                // Skip tile units that are within (not on) the ring.
                if (abs(x) < ring && abs(y) < ring) continue;
                
                // Find where this tile is on the screen
                double dx = x;
                double dy = y;
                tess.transform(&dx, &dy);
                int px = (int)floor(dx + 0.5);
                int py = (int)floor(dy + 0.5);
                
                // If the tile is visible then record it
                agg::rect_i tile(px, py, px + mWidth - 1, py + mHeight - 1);
                if (tile.overlaps(screen)) {
                    hit = true;
                    tessPoints.push_back(agg::point_i(px, py));
                }
            }
        }
        
        if (!hit) break;
    }
    
    return tessPoints;
}

bool tiledCanvas::isRectangular(agg::point_i* factor)
{
    int shx = (int)floor(mOffset.shx + 0.5);
    int shy = (int)floor(mOffset.shy + 0.5);
    
    if (shx == 0 && shy == 0) {
        if (factor) factor->x = factor->y = 1;
        return true;
    }
    
    if (shx && mWidth % abs(shx) == 0) {
        if (factor) {
            factor->x = 1;
            factor->y = mWidth / abs(shx);
        }
        return true;
    }
    
    if (shx && mWidth % (mWidth - abs(shx)) == 0) {
        if (factor) {
            factor->x = 1;
            factor->y = mWidth / (mWidth - abs(shx));
        }
        return true;
    }
    
    if (shy && mHeight % shy == 0) {
        if (factor) {
            factor->x = mHeight / abs(shy);
            factor->y = 1;
        }
        return true;
    }
    
    if (shy && mHeight % (mHeight - abs(shy)) == 0) {
        if (factor) {
            factor->x = mHeight / (mHeight - abs(shy));
            factor->y = 1;
        }
        return true;
    }
    
    return false;
}
