#ifndef INCLUDE_SHAPE_H
#define INCLUDE_SHAPE_H

#include <vector>
#include <string>
#include <iostream>

#include "agg_trans_affine.h"
#include "agg_color_rgba.h"
#include "HSBColor.h"
#include "cfdg.h"

class Replacement;

// Contains all of the information about a change to a shape 
class Modification {
    public: 
        agg::trans_affine m_transform;
        double m_Z;
        double m_SizeZ;
        
        HSBColor m_Color;
        HSBColor m_ColorTarget;

        unsigned short mRand48Seed[3];

        Modification() : m_Z(0.0), m_SizeZ(1.0)
        { mRand48Seed[0] = mRand48Seed[1] = mRand48Seed[2] = 0; }
        
        double area() const { return fabs(m_transform.determinant()); }
        
        Modification operator*(const Modification& m) const
        {
            Modification n = *this;
            n *= m;
            return n;
        }
        Modification& operator*=(const Modification& m)
        {
            m_transform.premultiply(m.m_transform);
            
            m_Z += m.m_Z * m_SizeZ;
            m_SizeZ *= m.m_SizeZ;
            
            
            m_Color.adjustWith(m.m_Color, m_ColorTarget);
            m_ColorTarget.adjustWith(m.m_ColorTarget, m_ColorTarget);
            
            mRand48Seed[0] ^= m.mRand48Seed[0];
            mRand48Seed[1] ^= m.mRand48Seed[1];
            mRand48Seed[2] ^= m.mRand48Seed[2];
            return *this;
        }
};

// Contains all of the information about a shape that is used during parsing
// and executing a cfdg file
class Shape {
    public: 
        int mShapeType;
        Modification mWorldState;
        double mAreaCache;

        Shape() : mShapeType(-1) { mAreaCache = mWorldState.area(); }

        Shape operator*(const Modification& m) const {
            Shape s = *this;
            s.mWorldState *= m;
            s.mAreaCache = s.mWorldState.area();
            return s;
        }
        
        
        inline Shape& operator*=(const Replacement& r);
        
        Shape& operator*=(const Modification& m) {
            mWorldState *= m;
            mAreaCache = mWorldState.area();
            return *this;
        }

        double area() const { return mAreaCache; }
        bool operator<(const Shape& b) const { return mAreaCache < b.mAreaCache; }

        void write(std::ostream& os) const { os.write((char*)(this), sizeof(Shape)); }
        void read(std::istream& is) { is.read((char *)(this), sizeof(Shape)); }
};

// Contains all of the information about a shape replacement that is used during parsing
// and executing a cfdg file
class Replacement {
    public: 
        int mShapeType;
        Modification mChildChange;
        int mIterationCount;
        Modification mIterationChange;

        Replacement() : mShapeType(-1), mIterationCount(1) {}
};

// Contains the subset of class Shape that is needed for drawing squares,
// circles, and triangles onto a canvas
class FinishedShape {
    public: 
        unsigned char mPrimitiveShapeType;
        float m_MiniTransform[6];
        float m_Z;
        RGBA8 mColor;
		double mCumulativeArea;
		
        FinishedShape() {};
        FinishedShape(const Shape&, double cumulativeArea);

        void GetTransform(agg::trans_affine& tr) const;

        bool operator<(const FinishedShape& b) const { 
            return m_Z == b.m_Z ? mCumulativeArea < b.mCumulativeArea : m_Z < b.m_Z; 
        };
 
        void write(std::ostream& os) const { os.write((char*)(this), sizeof(FinishedShape)); }
        void read(std::istream& is) { is.read((char *)(this), sizeof(FinishedShape)); }
};

inline std::ostream& operator<<(std::ostream& os, const FinishedShape& fs) { fs.write(os); return os; }
inline std::istream& operator>>(std::istream& is, FinishedShape& fs) { fs.read(is); return is; }
inline std::ostream& operator<<(std::ostream& os, const Shape& s) { s.write(os); return os; }
inline std::istream& operator>>(std::istream& is, Shape& s) { s.read(is); return is; }

class Rule {
    public:
        int mInitialShapeType;
        std::vector<Replacement> mReplacements;
        double mWeight;
    public:
        void addReplacement(const Replacement& r);
        bool operator<(const Rule& r) const { 
            return (mInitialShapeType == r.mInitialShapeType) ? 
                (mWeight < r.mWeight) : 
                (mInitialShapeType < r.mInitialShapeType); 
        }
};

inline Shape& Shape::operator*=(const Replacement& r) {
    mShapeType = r.mShapeType;
    mWorldState *= r.mChildChange;
    mAreaCache = mWorldState.area();
    return *this;
}


const double MY_PI =  3.14159265358979323846;

#endif // INCLUDE_SHAPE_H
