/*
 *  renderimpl.cpp
 *  Context Free
 *
 *  Created by Mark Lentczner on 2006-02-18.
 *  Copyright 2006 __MyCompanyName__. All rights reserved.
 *
 */

#include "renderimpl.h"

#include <iterator>
#include <string>
#include <algorithm>

#ifdef WIN32
#include "rand48.h"
#include <float.h>
#define isfinite _finite
#else
#include <math.h>
#endif

#include "shapeSTL.h"
#include "primShape.h"

using namespace std;


//#define DEBUG_SIZES
#ifndef DEBUG_SIZES
const unsigned int MOVE_FINISHED_AT     = 2000000; // when this many, move to file
const unsigned int MOVE_UNFINISHED_AT   =  500000; // when this many, move to files
const unsigned int MAX_MERGE_FILES      =     200; // maximum number of files to merge at once
#else
const unsigned int MOVE_FINISHED_AT     =    1000; // when this many, move to file
const unsigned int MOVE_UNFINISHED_AT   =     200; // when this many, move to files
const unsigned int MAX_MERGE_FILES      =       2; // maximum number of files to merge at once
#endif
const double SHAPE_BORDER = 2.0; // multiplier of shape size when calculating bounding box
const double FIXED_BORDER = 8.0; // fixed extra border, in pixels



RendererImpl::RendererImpl(CFDGImpl* cfdg,
                            int width, int height, float minSize,
                            int variation, double border)
    : m_cfdg(cfdg), m_canvas(0),
    m_maxShapes(500000000),
    mNextFramePoint(-1.0),

    mFinishedFileCount(0),
    mUnfinishedFileCount(0),

    mVariation(variation),

    mScaleArea(0.0), 
    mFixedBorder(1.0), mShapeBorder(1.0),
    mTotalArea(0.0),
    
    m_currScale(0.0), m_minArea(0.3), 
    m_outputSoFar(0)
{
    minSize = (minSize == 0.0F) ? 0.3F : minSize;
    m_minArea = minSize * minSize;

    mFixedBorder = FIXED_BORDER * ((border <= 1) ? border : 1);
    mShapeBorder = SHAPE_BORDER * ((border <= 1) ? 0.5 : (border / 2));
    Bounds::ScaleFactor = agg::trans_affine_scaling(mShapeBorder);

    double tile_x, tile_y;
    m_tiled = m_cfdg->isTiled(0, &tile_x, &tile_y);
    m_sized = m_cfdg->isSized(&tile_x, &tile_y);
    
    m_width = width;
    m_height = height;
    if (m_tiled || m_sized) {
        mFixedBorder = mShapeBorder = 0.0;
        mBounds.mMin_X = -(mBounds.mMax_X = tile_x / 2.0);
        mBounds.mMin_Y = -(mBounds.mMax_Y = tile_y / 2.0);
        mBounds.mValid = true;
        rescaleOutput(m_width, m_height, true);
        mScaleArea = m_currScale * m_currScale;
    }
}

RendererImpl::~RendererImpl()
{
}

void
RendererImpl::setMaxShapes(int n)
{
    m_maxShapes = n ? n : 400000000;
}

void
RendererImpl::resetBounds()
{
    mBounds = Bounds();
}


void
RendererImpl::outputPrep(Canvas* canvas)
{
    m_canvas = canvas;
	
	if (canvas) {
		m_width = canvas->mWidth;
		m_height = canvas->mHeight;
	}
    
    m_pleaseStop = false;
    m_pleaseFinishUp = false;
    m_pleaseSendUpdate = false;
    
    m_stats.inOutput = false;
	m_stats.animating = false;
	
    mNextFramePoint = -1.0;
}


double
RendererImpl::run(Canvas * canvas, bool partialDraw)
{
    outputPrep(canvas);
    if (canvas)
        canvas->scale(m_currScale);
    
    m_stats.shapeCount = m_stats.toDoCount = 0;
    m_unfinishedInFilesCount = 0;

    int reportAt = 250;

    Shape initShape = m_cfdg->getInitialShape();
    initShape.mWorldState.mRand48Seed[0] = 0x330E; 
    initShape.mWorldState.mRand48Seed[1] = (unsigned short)(mVariation >> 16); 
    initShape.mWorldState.mRand48Seed[2] = (unsigned short)(mVariation % 0xFFFF); 
    // Init color to opaque black
    initShape.mWorldState.m_ColorTarget = HSBColor(0, 0, 0, 1);
    initShape.mWorldState.m_Color = HSBColor(0, 0, 0, 1);   // only place where alpha is set to 1
    processShape(initShape);

    for (;;) {
        if (m_pleaseStop) break;
        if (m_pleaseFinishUp) break;
        
        fileIfNecessary();
        
        if (mUnfinishedShapes.empty()) break;
        if ((m_stats.shapeCount + m_stats.toDoCount) > m_maxShapes)
            break;

        // Get the largest unfinished shape
        multiset<Shape>::iterator biggest = mUnfinishedShapes.end();
        Shape s = *(--biggest);
        mUnfinishedShapes.erase(biggest);
        m_stats.toDoCount--;
        
        Rule* rule = m_cfdg->findRule(s.mShapeType, erand48(s.mWorldState.mRand48Seed));
        for (vector<Replacement>::iterator rep = rule->mReplacements.begin(); 
             rep != rule->mReplacements.end(); rep++) {
            // bump random seed
            nrand48(s.mWorldState.mRand48Seed);
            
            Shape si = s;
            nrand48(si.mWorldState.mRand48Seed);
            
            for (int i = rep->mIterationCount; i > 0; i--) {
                Shape child = si;
                child *= *rep;
                double area = child.area();
                if (!isfinite(area)) {
                    m_pleaseStop = true;
                    system()->message("A shape got too big.");
                    break;
                }
                // only add it if it's big enough (or if there are no finished shapes yet)
                if (!mBounds.valid() || (area * mScaleArea >= m_minArea)) {
                    processShape(child);
                }

                if (i > 1) {
                    si *= rep->mIterationChange;
                    nrand48(s.mWorldState.mRand48Seed);
                    nrand48(si.mWorldState.mRand48Seed);
                }
            }
        }


        if (m_pleaseSendUpdate || (m_stats.shapeCount > reportAt)) {
            if (partialDraw)
              outputPartial();
            outputStats();
            reportAt = 2 * m_stats.shapeCount;
        }
    }
    
    if (!m_pleaseStop) {
        stable_sort(mFinishedShapes.begin(), mFinishedShapes.end());
        outputFinal();
    }

    outputStats();
    return m_currScale;
}

void
RendererImpl::draw(Canvas* canvas)
{
    outputPrep(canvas);
    outputFinal();
    outputStats();
}

class OutputBounds : public ShapeOp
{
public:
    OutputBounds(int frames, double frameInc, double shapeBorder);
    void apply(const FinishedShape&);
	
	const Bounds& frameBounds(int frame) { return mFrameBounds[frame]; }
	int           frameCount(int frame) { return mFrameCounts[frame]; }
	
	void finalAccumulate();
		// call after all the frames to compute the bounds at each frame
		
	void backwardFilter(double framesToHalf);
	void smooth(int window);
	
private:
	double			mFrameInc;
	double			mShapeBorder;
    vector<Bounds>	mFrameBounds;
	vector<int>		mFrameCounts;
};

OutputBounds::OutputBounds(int frames, double frameInc, double shapeBorder)
	: mFrameInc(frameInc), mShapeBorder(shapeBorder)
{
	mFrameBounds.resize(frames);
	mFrameCounts.resize(frames, 0);
}

void
OutputBounds::apply(const FinishedShape& s)
{
	int frame = (int)floor(s.mCumulativeArea / mFrameInc);
	if (frame >= (int)mFrameBounds.size()) return;
	
	agg::trans_affine tr;
	s.GetTransform(tr);
	
	mFrameBounds[frame] += Bounds(tr, s.mPrimitiveShapeType);
	mFrameCounts[frame] += 1;
}

void
OutputBounds::finalAccumulate()
{
	vector<Bounds>::iterator prev, curr, end;
	prev = mFrameBounds.begin();
	end = mFrameBounds.end();
	if (prev == end) return;
	
	for (curr = prev + 1; curr != end; prev = curr, ++curr) {
		*curr += *prev;
	}
}

void
OutputBounds::backwardFilter(double framesToHalf)
{
	double alpha = pow(0.5, 1.0 / framesToHalf);
	
	vector<Bounds>::reverse_iterator prev, curr, end;
	prev = mFrameBounds.rbegin();
	end = mFrameBounds.rend();
	if (prev == end) return;

	for (curr = prev + 1; curr != end; prev = curr, ++curr) {
		*curr = curr->interpolate(*prev, alpha);
	}
} 

void
OutputBounds::smooth(int window)
{
	int frames = mFrameBounds.size();
	if (frames == 0) return;
	
	mFrameBounds.resize(frames + window - 1, mFrameBounds.back());
	
	vector<Bounds>::iterator write, read, end;
	read = mFrameBounds.begin();
	
	double factor = 1.0 / window;
	
	Bounds accum;
	for (int i = 0; i < window; ++i)
		accum.gather(*read++, factor);
	
	write = mFrameBounds.begin();
	end = mFrameBounds.end();
	for (;;) {
		Bounds old = *write;
		*write++ = accum;
		accum.gather(old, -factor);
		
		if (read == end) break;
		
		accum.gather(*read++, factor);
	} 
	
	mFrameBounds.resize(frames, Bounds());
}


void
RendererImpl::animate(Canvas* canvas, int frames, bool zoom)
{
    outputPrep(canvas);

	{
		if (zoom) resetBounds();
		
		// start with a blank frame
		
		int curr_width = m_width;
		int curr_height = m_height;
		rescaleOutput(curr_width, curr_height, true);
		
		m_canvas->start(true, m_cfdg->backgroundColor(),
			curr_width, curr_height);
		m_canvas->end();
	}
	
    double frameInc = mTotalArea / frames;
    
	OutputBounds outputBounds(frames, frameInc, mShapeBorder);
	{
        system()->message("Computing zoom");

		forEachShape(true, outputBounds);
		outputBounds.finalAccumulate();
		outputBounds.backwardFilter(10.0);
		//outputBounds.smooth(3);
	}
	
	m_stats.shapeCount = 0;
	m_stats.animating = true;
	
    for (int frameCount = 1; frameCount <= frames; ++frameCount)
    {
        system()->message("Generating frame %d of %d", frameCount, frames);
		
		if (zoom) mBounds = outputBounds.frameBounds(frameCount - 1);
		m_stats.shapeCount += outputBounds.frameCount(frameCount - 1);
        mNextFramePoint = frameInc * frameCount;
        outputFinal();
		outputStats();

        if (m_pleaseStop || m_pleaseFinishUp) break;
    }
	
	m_stats.animating = false;
	outputStats();
	system()->message("Animation of %d frames complete", frames);
}

void
RendererImpl::processShape(const Shape& s)
{
    if (m_cfdg->shapeHasRules(s.mShapeType)) {
        m_stats.toDoCount++;
        mUnfinishedShapes.insert(s);
    } else if (primShape::isPrimShape(s.mShapeType)) {
        m_stats.shapeCount++;
        mTotalArea += s.area();
        FinishedShape fs(s, mTotalArea);
        for (int i = 0; i < 6; i++) {
            if (!isfinite(fs.m_MiniTransform[i])) {
                m_pleaseStop = true;
                system()->message("A shape got too big.");
                return;
            }
        }
        mFinishedShapes.push_back(fs);
        if (!m_tiled && !m_sized) 
            updateBounds(s.mWorldState.m_transform, s.mShapeType);
    } else {
        m_pleaseStop = true;
        system()->message("Shape with no rules encountered: %s.", 
            m_cfdg->decodeShapeName(s.mShapeType).c_str());
    }
}


//-------------------------------------------------------------------------////


void
RendererImpl::updateBounds(const agg::trans_affine& trns, int primShapeType)
{
	mBounds += Bounds(trns, primShapeType);
	
	double minScale
		= mBounds.computeScale(m_width, m_height, mFixedBorder, false);
    mScaleArea = minScale * minScale;
}

//-------------------------------------------------------------------------////


void
RendererImpl::fileIfNecessary()
{
    if (mFinishedShapes.size() > MOVE_FINISHED_AT)
        moveFinishedToFile();

    if (mUnfinishedShapes.size() > MOVE_UNFINISHED_AT) 
        moveUnfinishedToTwoFiles();
    else if (mUnfinishedShapes.empty())
        getUnfinishedFromFile();
}

void
RendererImpl::moveUnfinishedToTwoFiles()
{
    ref_ptr<TempFile> t1 = TempFile::build(system(), "cfdg-temp-unfin-",
                                            "expansion", ++mFinishedFileCount);
    ref_ptr<TempFile> t2 = TempFile::build(system(), "cfdg-temp-unfin-",
                                            "expansion", ++mFinishedFileCount);

    m_unfinishedFiles.push_back(t1);
    m_unfinishedFiles.push_back(t2);

    ostream* f1 = t1->forWrite();
    ostream* f2 = t2->forWrite();
    
    system()->message("Writing %s temp files %d & %d",
                        t1->type().c_str(), t1->number(), t2->number());

    int count = mUnfinishedShapes.size() / 3;

    multiset<Shape>::iterator usi = mUnfinishedShapes.begin();

    for (int i = 0; i < count; ++i, ++usi)
        usi->write(*f1);

    for (int j = 0; j < count; ++j, ++usi)
        usi->write(*f2);

    mUnfinishedShapes.erase(mUnfinishedShapes.begin(), usi);

    delete f1;
    delete f2;
    
    m_unfinishedInFilesCount += 2 * count;
}

void
RendererImpl::getUnfinishedFromFile()
{
    if (m_unfinishedFiles.empty()) return;
    
    ref_ptr<TempFile> t = m_unfinishedFiles.front();
    
    istream* f = t->forRead();

    copy(istream_iterator<Shape>(*f), istream_iterator<Shape>(), 
         inserter<multiset<Shape> >(mUnfinishedShapes, mUnfinishedShapes.end()));

    delete f;

    m_unfinishedFiles.pop_front();
}

//-------------------------------------------------------------------------////

void
RendererImpl::moveFinishedToFile()
{
    ref_ptr<TempFile> t = TempFile::build(system(), "cfdg-temp-fin-",
                                            "shapes", ++mFinishedFileCount);
    m_finishedFiles.push_back(t);
    
    ostream* f = t->forWrite();

    stable_sort(mFinishedShapes.begin(), mFinishedShapes.end());

    copy(mFinishedShapes.begin(), mFinishedShapes.end(), 
         ostream_iterator<FinishedShape>(*f));

    mFinishedShapes.clear();
    
    delete f;
}

//-------------------------------------------------------------------------////

void RendererImpl::rescaleOutput(int& curr_width, int& curr_height, bool final)
{
	agg::trans_affine trans;
    double scale;
	
	scale = mBounds.computeScale(curr_width, curr_height,
		mFixedBorder , true, &trans, m_tiled || m_sized);

    if (final                       // if final output
    || m_currScale == 0.0           // if first time, use this scale
    || (m_currScale * 0.90) > scale)// if grew by more than 10%
    {
        m_currScale = scale;
        m_currTrans = trans;
        m_outputSoFar = 0;
        m_stats.fullOutput = true;
    }
}


void
RendererImpl::forEachShape(bool final, ShapeOp& op)
{
	if (!final) {
		for_each(mFinishedShapes.begin() + m_outputSoFar, mFinishedShapes.end(),
			op.outputFunction());
		m_outputSoFar = mFinishedShapes.size();
	}
	else if (m_finishedFiles.empty()) {
		for_each(mFinishedShapes.begin(), mFinishedShapes.end(),
			op.outputFunction());
	}
	else {
		deque< ref_ptr<TempFile> >::iterator begin, last, end;

		while (m_finishedFiles.size() > MAX_MERGE_FILES) {
			OutputMerge merger(*system());
			
			begin = m_finishedFiles.begin();
			last = begin + (MAX_MERGE_FILES - 1);
			end = last + 1;
							
			for_each(begin, end, merger.tempFileAdder());
				
			ref_ptr<TempFile> t = TempFile::build(system(), "cfdg-temp-mrg-",
													"merge", ++mFinishedFileCount);

			ostream* f = t->forWrite();
			system()->message("Merging temp files %d through %d",
								(*begin)->number(), (*last)->number());
								
			merger.merge(ostream_iterator<FinishedShape>(*f));

			delete f;
			
			m_finishedFiles.erase(begin, end);
			m_finishedFiles.push_back(t);
		}

		OutputMerge merger(*system());
		
		begin = m_finishedFiles.begin();
		end = m_finishedFiles.end();
		
		for_each(begin, end, merger.tempFileAdder());
		
		merger.addShapes(mFinishedShapes.begin(), mFinishedShapes.end());
		merger.merge(op.outputIterator());
	}
}


class OutputDraw : public ShapeOp
{
public:
    OutputDraw(RendererImpl& renderer, bool final);
    
    void apply(const FinishedShape&);

    class Stopped { }; 

private:
    RendererImpl& mRenderer;
    bool mFinal;
        
    int mCircleType;
    int mSquareType;
    int mTriangeType;
    int mBreakpointType;
};

OutputDraw::OutputDraw(RendererImpl& renderer, bool final)
     : mRenderer(renderer), mFinal(final)
{
    static string circle_name = "CIRCLE";
    static string square_name = "SQUARE";
    static string triangle_name = "TRIANGLE";
    
    mCircleType = mRenderer.m_cfdg->encodeShapeName(circle_name);
    mSquareType = mRenderer.m_cfdg->encodeShapeName(square_name);
    mTriangeType = mRenderer.m_cfdg->encodeShapeName(triangle_name);
}

void
OutputDraw::apply(const FinishedShape& s)
{
    if (mRenderer.m_pleaseStop) throw Stopped();
    if (!mFinal  &&  mRenderer.m_pleaseFinishUp) throw Stopped();
    
    if (mRenderer.m_pleaseSendUpdate)
        mRenderer.outputStats();


    if (mRenderer.mNextFramePoint >= 0.0
    &&  mRenderer.mNextFramePoint < s.mCumulativeArea)
        return;


    mRenderer.m_stats.outputDone += 1;

    agg::trans_affine tr;
    s.GetTransform(tr);

    tr *= mRenderer.m_currTrans;

    double a = fabs(tr.determinant()); 
    if (!isfinite(a) || a < mRenderer.m_minArea) return;
    
    if (s.mPrimitiveShapeType == mCircleType) {
        mRenderer.m_canvas->circle(s.mColor, tr);
    }
    else if (s.mPrimitiveShapeType == mSquareType) {
        mRenderer.m_canvas->square(s.mColor, tr);
    }
    else if (s.mPrimitiveShapeType == mTriangeType) {
        mRenderer.m_canvas->triangle(s.mColor, tr);
    }
    else if (s.mPrimitiveShapeType == mBreakpointType) {
        ;
    }
    else {
        mRenderer.system()->message(
            "Non drawable shape with no rules: %s",
            mRenderer.m_cfdg->decodeShapeName(s.mPrimitiveShapeType).c_str());
    }
}


void RendererImpl::output(bool final)
{
    if (!m_canvas)
        return;
        
    if (!final &&  !m_finishedFiles.empty())
        return; // don't do updates once we have temp files
        
    m_stats.inOutput = true;
    m_stats.fullOutput = final;
    m_stats.finalOutput = final;
    m_stats.outputCount = m_stats.shapeCount;

    int curr_width = m_width;
    int curr_height = m_height;
    rescaleOutput(curr_width, curr_height, final);
    
    m_stats.outputDone = m_outputSoFar;
    
    m_canvas->start(m_outputSoFar == 0, m_cfdg->backgroundColor(),
        curr_width, curr_height);

    OutputDraw draw(*this, final);
    try {
		forEachShape(final, draw);
    }
    catch (OutputDraw::Stopped) { }

    m_canvas->end();
    m_stats.inOutput = false;
    m_stats.outputTime = m_canvas->mTime;
}


void
RendererImpl::outputStats()
{
    system()->stats(m_stats);
    m_pleaseSendUpdate = false;
}

void
RendererImpl::requestStop()
{
    m_pleaseStop = true;
}

void
RendererImpl::requestFinishUp()
{
    m_pleaseFinishUp = true;
}

void
RendererImpl::requestUpdate()
{
    m_pleaseSendUpdate = true;
}

