/*
 *  qtCanvas.cpp
 *  Context Free
 *
 *  Created by Mark Lentczner on 2006-02-23.
 *  Copyright 2006 __MyCompanyName__. All rights reserved.
 *
 */

#include "qtCanvas.h"

#ifdef _WIN32

#include <QTML.h>
#include <QuickTimeComponents.h>
#include <GXMath.h>

#define GetPortPixMap(port) (port->portPixMap)

#else
#include <QuickTime/QuickTime.h>
#endif

static void
CheckError(OSErr err, const char* name)
{
    if (err != noErr)
        std::cerr << "error " << std::dec << err << " in " << name << std::endl;
}

static void
CheckMoviesError(const char* name)
{
	CheckError(GetMoviesError(), name);
}

bool qtCanvas::systemInited = false;

bool
qtCanvas::systemInit()
{
    if (!qtCanvas::systemInited) {
        OSErr err = noErr;
#ifdef _WIN32
        err = InitializeQTML(kInitializeQTMLNoSoundFlag);
        CheckError (err, "InitializeQTML error" );
#endif
        qtCanvas::systemInited = err == noErr;
    }
    return qtCanvas::systemInited;
} 

void
qtCanvas::systemExit()
{
    if (qtCanvas::systemInited) {
#ifdef _WIN32
        TerminateQTML();
#endif
        qtCanvas::systemInited = false;
    }
}



class qtCanvas::Impl
{
public:
	Impl(short width, short height);
	~Impl();

	void prepareMovie(CFStringRef);
	
	void addFrame();
	void showSettingsDialog();
	void setPreview(bool isGray);
	
private:
	void setupSCComponent();	void finishSCComponent();
	void setupMovie();			void finishMovie();
	void setupTrack();			void finishTrack();
	void setupImage();			void finishImage();
	void setupCompression();	void finishCompression();

private:
    CFStringRef		mPath;
	double			mFrameRate;
    short             mWidth;
    short             mHeight;

	ComponentInstance
					mSCComponent;
	static Handle	sSCSettings;

    DataHandler		mDataHandler;
	Movie			mMovie;

	Track			mTrack;
	Media			mMedia;

    Rect			mFrame;
    unsigned char*	mImageData;
	unsigned int	mImageStride;
	GWorldPtr		mGWorld;
	
	ImageDescriptionHandle
					mImageDesc;
	
	friend class qtCanvas;
};

Handle qtCanvas::Impl::sSCSettings = nil;


void
qtCanvas::Impl::setupSCComponent()
{
	OSErr err;
	
    err = EnterMovies();
    CheckError (err, "EnterMovies error" );

	mSCComponent = OpenDefaultComponent(
						StandardCompressionType,
						StandardCompressionSubType);
	CheckMoviesError("OpenDefaultComponent");

	if (sSCSettings) {
		SCSetInfo(mSCComponent, scSettingsStateType, &sSCSettings);
	}
	else {
		SCSpatialSettings	spatial;
		SCTemporalSettings	temporal;
		long				preference;
		CodecFlags			codecFlags;
		
		spatial.codecType = kAnimationCodecType;
		spatial.codec = NULL;
		spatial.depth = 32; // reset when the preview is set up
		spatial.spatialQuality = codecNormalQuality;
		
		temporal.temporalQuality = codecNormalQuality;
		temporal.frameRate = FloatToFixed(15.0);
		temporal.keyFrameRate = FixedToInt(temporal.frameRate) * 2;
		
		preference = scListEveryCodec;
		//preference |= scShowBestDepth;
		//preference |= scUseMovableModal;
		
		codecFlags = codecFlagUpdatePreviousComp;
		
		SCSetInfo(mSCComponent, scSpatialSettingsType, &spatial);
		SCSetInfo(mSCComponent, scTemporalSettingsType, &temporal);
		SCSetInfo(mSCComponent, scPreferenceFlagsType, &preference);
		SCSetInfo(mSCComponent, scCodecFlagsType, &codecFlags);
	}
}

void
qtCanvas::Impl::finishSCComponent()
{
	if (sSCSettings)
		DisposeHandle(sSCSettings);
	SCGetInfo(mSCComponent, scSettingsStateType, &sSCSettings);

	CloseComponent(mSCComponent);
	CheckMoviesError("CloseComponent");
}

void
qtCanvas::Impl::setPreview(bool isGray)
{
	SCSpatialSettings	spatial;
	SCGetInfo(mSCComponent, scSpatialSettingsType, &spatial);
	spatial.depth = isGray ? 40 : 32;
	SCSetInfo(mSCComponent, scSpatialSettingsType, &spatial);

	SCSetTestImagePixMap(mSCComponent,
		GetPortPixMap(mGWorld), &mFrame,
		scPreferScaling
		);
    CheckMoviesError("SCSetTestImagePixMap");
}

void
qtCanvas::Impl::showSettingsDialog()
{
	SCRequestSequenceSettings(mSCComponent);
}


void
qtCanvas::Impl::setupMovie()
{
	OSErr err;

	Handle dataRefH = nil;
	OSType dataRefType = 0;
	
	err = QTNewDataReferenceFromFullPathCFString(
			mPath, (UInt32)kQTNativeDefaultPathStyle, 0,
			&dataRefH, &dataRefType);
    CheckError(err, "QTNewDataReferenceFromFullPathCFString");

	CreateMovieStorage(dataRefH, dataRefType,
			'TVOD', smSystemScript,
			createMovieFileDeleteCurFile |
				createMovieFileDontCreateResFile |
				newMovieActive,
			&mDataHandler, &mMovie);
    CheckMoviesError("CreateMovieStorage");
    CheckError((OSErr)((nil != mMovie) ? noErr : -1), "CreateMovieStorage movie");
}

void
qtCanvas::Impl::finishMovie()
{
	if (mMovie) {
		UpdateMovieInStorage(mMovie, mDataHandler);
		CheckMoviesError("UpdateMovieInStorage");
		
		DisposeMovie(mMovie);
	}
	
    if (mDataHandler) {
        CloseMovieStorage(mDataHandler);
        CheckMoviesError("CloseMovieStorage");
    }
}

void
qtCanvas::Impl::setupTrack()
{
    mTrack = NewMovieTrack(mMovie,
				FixRatio(mWidth,1), FixRatio(mHeight,1),
				(short)kNoVolume);
	CheckMoviesError("NewMovieTrack");


	SCTemporalSettings	temporal;
	SCGetInfo(mSCComponent, scTemporalSettingsType, &temporal);

    mMedia = NewTrackMedia(mTrack,
				VideoMediaType, (TimeScale)temporal.frameRate,
				nil, 0);
    CheckMoviesError("NewTrackMedia" );


    BeginMediaEdits(mMedia);
    CheckMoviesError("BeginMediaEdits" );
}

void
qtCanvas::Impl::finishTrack()
{
	if (mMedia) {
		EndMediaEdits(mMedia);
		CheckMoviesError("EndMediaEdits");

		if (mTrack) {
			InsertMediaIntoTrack(mTrack,
						0, 0, GetMediaDuration(mMedia), fixed1);
			CheckMoviesError("InsertMediaIntoTrack" );
		}
	}
}

void
qtCanvas::Impl::setupImage()
{
    MacSetRect(&mFrame, 0, 0, mWidth, mHeight);
    
    mImageStride = mWidth * 4;
    mImageData = new unsigned char[mImageStride * mHeight];

    OSErr err;
	err = QTNewGWorldFromPtr(&mGWorld,
				k32ARGBPixelFormat, &mFrame,
				nil, nil, (GWorldFlags)0,
				mImageData, mImageStride);
    CheckError(err, "QTNewGWorldFromPtr");
}

void
qtCanvas::Impl::finishImage()
{
    if (mGWorld) {
        DisposeGWorld (mGWorld);
    }

    delete[] mImageData;
}

void
qtCanvas::Impl::setupCompression()
{
	SCCompressSequenceBegin(mSCComponent,
		GetPortPixMap(mGWorld), &mFrame,
		&mImageDesc);
	CheckMoviesError("SCCompressSequenceBegin");
}

void
qtCanvas::Impl::finishCompression()
{
	SCCompressSequenceEnd(mSCComponent);
	CheckMoviesError("SCCompressSequenceEnd");
}

void
qtCanvas::Impl::addFrame()
{
	if (!mImageData) return;
	
	Handle	compressedData;
	long	compressedSize;

	short notSyncFlag;
	SCCompressSequenceFrame(mSCComponent,
			GetPortPixMap(mGWorld), &mFrame,
			&compressedData, &compressedSize,
			&notSyncFlag);
    CheckMoviesError("SCCompressSequenceFrame");

    AddMediaSample(mMedia,
			compressedData, 0, compressedSize,
			(TimeValue)fixed1,
			(SampleDescriptionHandle)mImageDesc,
			1, notSyncFlag, nil);
    CheckMoviesError("AddMediaSample");
}


qtCanvas::Impl::Impl(short width, short height)
	: mPath(nil), mFrameRate(12.0),
	mWidth(width), mHeight(height),
	mSCComponent(nil),
	mDataHandler(nil), mMovie(nil),
	mTrack(nil), mMedia(nil),
	mImageData(0), mGWorld(nil),
	mImageDesc(nil)
{
	setupSCComponent();
	setupImage();
}

void
qtCanvas::Impl::prepareMovie(CFStringRef path)
{
    mPath = path;
	CFRetain(mPath);
	
	setupMovie();
	setupTrack();
	setupCompression();
}

qtCanvas::Impl::~Impl()
{
	finishCompression();
	finishTrack();
	finishMovie();
	
	if (mPath) CFRelease(mPath);
	
	finishImage();
	finishSCComponent();
}



qtCanvas::qtCanvas(int width, int height)
    : aggCanvas(QT_Blend), impl(* new Impl((short)width, (short)height))
{
    attach(impl.mImageData, impl.mWidth, impl.mHeight, impl.mImageStride);
}

qtCanvas::~qtCanvas()
{
    delete &impl;
}

void
qtCanvas::end()
{
    aggCanvas::end();
	
	impl.addFrame();
}

void
qtCanvas::prepareMovie(CFStringRef path)
{
	impl.prepareMovie(path);
}
void
qtCanvas::setPreview(void* data, unsigned width, unsigned height,
	int stride, PixelFormat format)
{
	copy(data, width, height, stride, format);
	impl.setPreview(format == Gray8_Blend);
}

void
qtCanvas::showSettingsDialog()
{
	impl.showSettingsDialog();
}

int
qtCanvas::convertToFrames(float seconds)
{
	SCTemporalSettings	temporal;
	
	SCGetInfo(impl.mSCComponent, scTemporalSettingsType, &temporal);
	
	return (int)(FixedToFloat(temporal.frameRate) * seconds);
}

