#ifndef INCLUDE_CFDG_H
#define INCLUDE_CFDG_H

#include <iostream>
#include <string>
#include <time.h>
#include "agg_trans_affine.h"
#include "agg_color_rgba.h"

typedef agg::rgba8 RGBA8;


class System {
    public:
        virtual void message(const char* fmt, ...) = 0;
        virtual void syntaxError(const std::string& filepath, int line) = 0;
        
        virtual std::istream* openFileForRead(const std::string& path) = 0;
        virtual std::istream* tempFileForRead(const std::string& path) = 0;
        virtual std::ostream* tempFileForWrite(std::string& prefixInNameOut) = 0;
            // caller must delete returned streams when done
            
        virtual std::string relativeFilePath(
            const std::string& base, const std::string& rel) = 0;
        
        struct Stats {
            int     shapeCount;     // finished shapes in image
            int     toDoCount;      // unfinished shapes still to expand
            
            bool    inOutput;       // true if we are in the output loop
            bool    fullOutput;     // not an incremental output
            bool    finalOutput;    // the last output
            int     outputCount;    // number to be output
            int     outputDone;     // number output so far
            clock_t outputTime;
			
			bool	animating;		// inside the animation loop
        };

        virtual void stats(const Stats&);
        
        virtual ~System();
};

class Canvas {
    public:
        virtual void start(bool , const agg::rgba& , int , int ) 
        { mTime = clock(); }
        virtual void end()
        { mTime = clock() - mTime; }

        virtual void circle(RGBA8 , agg::trans_affine ) = 0;
        virtual void square(RGBA8 , agg::trans_affine ) = 0;
        virtual void triangle(RGBA8 , agg::trans_affine ) = 0;
        virtual void scale(double ) {}

		Canvas(int width, int height) 
			: mWidth(width), mHeight(height) {}
        virtual ~Canvas();
		
		int mWidth;
		int mHeight;
        clock_t mTime;
};

class Renderer;

class CFDG {
    public:
        static CFDG* ParseFile(const char* fname, System*);
        virtual ~CFDG();

        virtual Renderer* renderer(
                int width, int height, float minSize,
                int variation, double border = 2.0
            ) = 0;
            // caller must delete returned object

        virtual bool usesColor() const = 0;
        virtual bool usesAlpha() const = 0;
        virtual bool isTiled(agg::trans_affine* tr = 0, double* x = 0, double* y = 0) const = 0;
        virtual bool isSized(double* x = 0, double* y = 0) const = 0;
		
		virtual agg::rgba backgroundColor() const = 0;

    protected:
        CFDG() { }
};


class Renderer {
    public:
        virtual ~Renderer();
        
        virtual void setMaxShapes(int n) = 0;        
        virtual void resetBounds() = 0;

        virtual double run(Canvas* canvas, bool partialDraw) = 0;
        virtual void draw(Canvas* canvas) = 0;
        virtual void animate(Canvas* canvas, int frames, bool zoom) = 0;

        virtual void requestStop() = 0;     // stop ASAP
        virtual void requestFinishUp() = 0; // stop expanding, and do final output
        virtual void requestUpdate() = 0;   // call stats() soon
        
        int m_width;
        int m_height;
    
    protected:
        Renderer() { }
};

#endif
