#ifndef __HSBCOLOR_H_
#define __HSBCOLOR_H_

namespace agg {
    struct rgba;
};

#define EQUALITY_THRESHOLD  0.00001

#include <math.h>

struct HSBColor
{
    enum AssignmentTypes { HueTarget = 1, 
                           SaturationTarget = 2, 
                           BrightnessTarget = 4, 
                           AlphaTarget = 8 
                         };
                             
    HSBColor() : h(0), s(0), b(0), a(0), mUseTarget(0) {};
    HSBColor(double hue, double sat, double bri, double alpha) 
        : h(hue), s(sat), b(bri), a(alpha), mUseTarget(0) {};
    HSBColor(const agg::rgba&);

    void adjustWith(const HSBColor& adj, const HSBColor& target = HSBColor(0,0,0,0));
    HSBColor getAdjustmentFrom(const HSBColor& adj) const;

    void getRGBA(agg::rgba& c) const;

	double h, s, b, a;
    int mUseTarget;
    
    static inline double adjust(const double& base, const double& adjustment, 
                                int useTarget = 0, const double& target = 0.0);
    static inline double adjustHue(const double& base, const double& adjustment, 
                                   int useTarget = 0, const double& target = 0.0);
    static inline double delta(const double& to, const double& from);
    static inline double deltaHue(const double& to, const double& from);
};

inline double
HSBColor::adjust(const double& base, const double& adjustment, 
                 int useTarget, const double& target)
{
    if (adjustment == 0.0)
        return base;
    
    if (useTarget) {
            // If we are really close to the target then don't change, even if
            // the adjustment is negative (which way would we go?)
        if (adjustment > 0 && fabs(base - target) < EQUALITY_THRESHOLD)
            return base;
        
            // Otherwise move away from or toward the target
        double edge = base < target ? 0 : 1;
        if (adjustment < 0)
            return base + (  base - edge) * adjustment;
        else 
            return base + (target - base) * adjustment;
    } else {
            // Move toward 0 or toward 1
        if (adjustment < 0)
            return base + base * adjustment;
        else 
            return base + (1 - base) * adjustment;
    }
}

inline double
HSBColor::adjustHue(const double& base, const double& adjustment, 
                    int useTarget, const double& target)
{
    if (adjustment == 0.0)
        return base;
    
    double h;
    if (useTarget & HueTarget) {
        // decrease or increase toward target. If the target hue does not 
        // cooperate by being smaller (or larger) than the current hue then 
        // add/subtract 360 to make it so. This only works if all hues are
        // within the interval [0,360).
        double t = target;
        if (adjustment < 0) {
            if (t > base) t -= 360;
            h = base + (base - t) * adjustment;
        } else {
            if (t < base) t += 360;
            h = base + (t - base) * adjustment;
        }
    } else {
        h = base + adjustment;
    }
    
    // Normalize result to the interval [0,360)
    return h < 0.0 ? fmod(h + 360.0, 360.0) : fmod(h, 360.0);
}

inline double 
HSBColor::delta(const double& to, const double& from)
{
    if (fabs(to - from) < EQUALITY_THRESHOLD) return 0.0;
    return ((to - from) > 0.0) ? 
           ((to - from) / (1.0 - from)) : 
           ((to - from) / from);
}

inline double
HSBColor::deltaHue(const double& to, const double& from)
{ 
    double diff = fmod(to - from, 360.0);
    // Normalize result to the interval (-180,180]
    if (diff <= -180.0) return diff + 360.0;
    if (diff >   180.0) return diff - 360.0;
    return diff;
}



#endif
