/*
  xoris - grabs color from the screen & dumps it to stdout.
  Copyright (c) 2004,2005,2006 Alexander Gromnizki <iwerdon@gmail.com>
  All rights reserved.

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is fur-
  nished to do so, subject to the following conditions:
  
  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FIT-
  NESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
  
  $Id: main.c,v 1.5 2006/04/05 18:50:49 alex Exp $
*/

#include <stdio.h>
#include <stdbool.h>
#include <stdarg.h>
#include <unistd.h>
#include <assert.h>

#include <X11/Xlib.h>
#include <X11/cursorfont.h>

/* global */
char *program_name = "xoris";
const char *program_version = "0.1b";
#define INIT_PROGRAM_NAME program_name=argv[0]
static char *id = "$Id: main.c,v 1.5 2006/04/05 18:50:49 alex Exp $";

/* resources (from command line only) */
char *rgb_file;

typedef enum _XColorSpec {
	XCS_SIMPLE, /* like in rgb.txt */
	XCS_STRING,
	XCS_RGBI, /* not implemented */
	XCS_RGB_4, /* not implemented */
	XCS_RGB_8,  /* not implemented */
	XCS_RGB_12,  /* not implemented */
	XCS_RGB_16,  /* not implemented */
	XCS_RGB_OLD_4,  /* not implemented */
	XCS_RGB_OLD_8, 
	XCS_RGB_OLD_12, /* not implemented */
	XCS_RGB_OLD_16  /* not implemented */
} XColorSpec;

static XColor get_xycolor(Display *dpy, const int x, const int y);
static void print_xcolor(Display *dpy, const int x, const int y,
	const XColorSpec xcs, const bool verbose);
static char* get_def_colname(const XColor *xcol);
static char* get_db_colname(const XColor* xcol, const char* alt_rgb_file);
static void x_exit(Display *dpy, const int ec, char *fmt, ...);
static void usage();

int main(int argc, char *argv[])
{
	Display *dpy;
	Window win_root;
	XEvent event;
	Cursor cursor;
	char *dpy_name = (char*)getenv("DISPLAY");
	
	char ch;
	bool verbose = false;
	bool exit_after_grab = true;
	bool put_newline = true;
	
	INIT_PROGRAM_NAME;
	
	while ((ch = getopt(argc, argv, "d:vENr:h")) != -1)
		switch (ch) {
		case 'd':
			dpy_name = optarg;
			break;
		case 'v':
			 verbose = true;
			break;
		case 'E':
			 exit_after_grab = false;
			break;
		case 'N':
			 put_newline = false;
			break;
		case 'r':
			rgb_file = optarg;
			break;
		case 'h':
			usage();
			exit(0);
		default:
			usage();
			exit(1);
		}
	argc -= optind;
	argv += optind;
	
	dpy = XOpenDisplay(dpy_name);
	if (dpy == NULL) err(1, "unable to open display %s", dpy_name);
	
	win_root = DefaultRootWindow(dpy);
	/* make the target cursor */
	cursor = XCreateFontCursor(dpy, XC_crosshair);
	
	/* grab the pointer using target cursor, letting it room all over */
	if (XGrabPointer(dpy, win_root, False,
			ButtonPressMask|ButtonReleaseMask, GrabModeSync,
			GrabModeAsync, win_root, cursor, CurrentTime) != GrabSuccess)
		x_exit(dpy, 1, "can't grab on %s the mouse", dpy_name);
		
	for (;;) {
		XAllowEvents(dpy, SyncPointer, CurrentTime);
		XWindowEvent(dpy, win_root,	ButtonPressMask|ButtonReleaseMask, &event);

		switch (event.type) {
		case ButtonPress: {
			int x = event.xbutton.x;
			int y = event.xbutton.y;
			
			switch (event.xbutton.button) {
			case Button1:
				/* dump color values*/
				if (event.xbutton.state & ControlMask)
					print_xcolor(dpy, x, y, XCS_SIMPLE, verbose);
				else if (event.xbutton.state & ShiftMask)
					print_xcolor(dpy, x, y, XCS_STRING, verbose);
				else
					print_xcolor(dpy, x, y, XCS_RGB_OLD_8, verbose); 
				break;

 			case Button3:
 				/* exit */
 				x_exit(dpy, 0, NULL);
 				break;
				
			default:
				continue;
			}

			if (put_newline) printf("\n");
			if (exit_after_grab) x_exit(dpy, 0, NULL);
			continue; } /* of ButtonPress */

		default:
			break;
		}

	}
	
	x_exit(dpy, 0, NULL);
}

static XColor get_xycolor(Display *dpy, const int x, const int y)
/*	
	IMHO it works on 24+ bit color display only
	why? google in comp.windows.x
*/
{
	static XColor xcol;
	XImage *xim;
	unsigned long p;
	
	xim = XGetImage(dpy, DefaultRootWindow(dpy),
		x, y, 1, 1, AllPlanes, ZPixmap);
	p = XGetPixel(xim, 0, 0);
	xcol.pixel = p;
	xcol.flags = DoRed | DoGreen | DoBlue;
	XQueryColor(dpy, DefaultColormap(dpy, DefaultScreen(dpy)), &xcol);
	
	XDestroyImage(xim);
	
	return xcol;
}

static void print_xcolor(Display *dpy, const int x, const int y,
	const XColorSpec xcs, const bool verbose)
/*
	x, y - coordinates of display pixel
	verbose - to show coordinates or no
*/
{
	XColor xcol;

	xcol = get_xycolor(dpy, x, y);
	if (verbose) printf("%dx%d ", x, y);
	
	switch (xcs) {
	case XCS_RGB_OLD_8: {
		char r[3], g[3], b[3];

		sprintf(r, "%x", xcol.red >> 8);
		if (strlen(r) < 2) sprintf(r, "0%x", xcol.red >> 8);

		sprintf(g, "%x", xcol.green >> 8);
		if (strlen(g) < 2) sprintf(g, "0%x", xcol.green >> 8);
		
		sprintf(b, "%x", xcol.blue >> 8);
		if (strlen(b) < 2) sprintf(b, "0%x", xcol.blue >> 8);

		printf("#%s%s%s", r, g, b);
		break; }
	case XCS_STRING: {
		char *name;
		bool find = true;
		
	 	if (rgb_file == NULL) {
			name = get_def_colname(&xcol);
			if (name == NULL) find = false;
	 	} else {
			name = get_db_colname(&xcol, rgb_file);
			if (name == NULL) find = false;
		}
		
		if (!find) {
			/* if nothing found - dump XCS_RGB_OLD_8 */
			print_xcolor(dpy, x, y, XCS_RGB_OLD_8, false);
			return;
		}
		else {
			printf("%s", name);
			free(name);
		}
		break; }
	case XCS_SIMPLE:
	default:
		printf("%d %d %d", xcol.red >> 8, xcol.green >> 8, xcol.blue >> 8);
		break;
	}
	
	fflush(stdout);
}

static char* get_def_colname(const XColor *xcol)
/*
	tries to get abstract color name from default X rgb database
	using showrgb(1)
	
	returns NULL if false
	don't forget to free the result!
*/
{
	FILE* rgb;
	char line[BUFSIZ];
	char name[BUFSIZ];
	char *colorname = NULL;
	int red, green, blue;
	
	if ((rgb = popen("showrgb", "r")) == NULL) {
		warn("cant open a process 'showrgb'");
		return NULL;
	}

	while(fgets(line, sizeof(line), rgb)) {
		if (sscanf(line, "%d %d %d %[^\n]\n", &red, &green, &blue, name) == 4)
			if ( ((xcol->red >> 8) == red) &&
				((xcol->green >> 8) == green) &&
				((xcol->blue >> 8) == blue) ) {
				colorname = (char*)malloc(strlen(name)+1);
				assert(colorname);
				strcpy(colorname, name);
				break;
			}
	}

	pclose(rgb);
	
	return colorname;
}

static char* get_db_colname(const XColor* xcol, const char* alt_rgb_file)
/*
	tries to get abstract color name from user .txt file
	
	returns NULL if false
	don't forget to free the result!
*/
{
	char line[BUFSIZ];
	char name[BUFSIZ];
	FILE* rgb;
	int lineno = 0;
	int red, green, blue;
	char *colorname = NULL;

	rgb = fopen(alt_rgb_file, "r");
	if (rgb == NULL) {
		warn("can't open '%s' file", alt_rgb_file);
		return NULL;
	}

	while(fgets(line, sizeof(line), rgb)) {
		lineno++;
		if (sscanf(line, "%d %d %d %[^\n]\n", &red, &green, &blue, name) == 4)
			if (red >= 0 && red <= 0xff &&
				green >= 0 && green <= 0xff &&
				blue >= 0 && blue <= 0xff) {
				if ( ((xcol->red >> 8) == red) &&
					 ((xcol->green >> 8) == green) &&
					 ((xcol->blue >> 8) == blue) ) {
					colorname = (char*)malloc(strlen(name)+1);
					assert(colorname);
					strcpy(colorname, name);
					break;
				}
			}
		else
			warn("value for '%s' is out of range: %s:%d",
				name, alt_rgb_file, lineno);
		else
			if (*line && *line != '!')
				warn("syntax error: %s:%d", rgb_file, lineno);
	}

	fclose(rgb);

	return colorname;
}

static void usage()
{
    printf("xoris %s - grabs color from the screen & dumps it to stdout.\n"
           "Copyright (c) 2004,2005,2006 Alexander Gromnizki <iwerdon@gmail.com>\n"
           "All rights reserved.\n\n"
           "Usage: %s [-d display] [-r filename] [-vENh]\n"
           "    -d host        display to use\n"
           "    -v             show also XY coordinates\n"
           "    -E             don't exit after grab\n"
           "    -N             don't put newline at the end\n"
           "    -r filename    use alternate rgb.txt file\n"
           "    -h             this help\n",
		   program_version, program_name);
}

static void x_exit(Display *dpy, const int ec, char *fmt, ...)
{
	XCloseDisplay(dpy);
	if ((fmt != NULL) && (ec > 0)) {
		va_list argp;
		
		va_start(argp, fmt);
		verr(ec, fmt, argp);
		va_end(argp);
	}
	else exit(0);
}
