//
// DynamicAny.h
//
// $Id: //poco/1.4/Foundation/include/Poco/DynamicAny.h#1 $
//
// Library: Foundation
// Package: Core
// Module:  DynamicAny
//
// Definition of the DynamicAny class.
//
// Copyright (c) 2007, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#ifndef Foundation_DynamicAny_INCLUDED
#define Foundation_DynamicAny_INCLUDED


#include "Poco/Foundation.h"
#include "Poco/DynamicAnyHolder.h"
#include "Poco/Format.h"
#include <typeinfo>


namespace Poco {


class Foundation_API DynamicAny
	/// DynamicAny allows to store data of different types and to convert between these types transparently.
	/// DynamicAny puts forth the best effort to provide intuitive and reasonable conversion semantics and prevent 
	/// unexpected data loss, particularly when performing narrowing or signedness conversions of numeric data types.
	///
	/// An attempt to convert or extract from a non-initialized (empty) DynamicAny variable shall result
	/// in an exception being thrown.
	///
	/// Loss of signedness is not allowed for numeric values. This means that if an attempt is made to convert 
	/// the internal value which is a negative signed integer to an unsigned integer type storage, a RangeException is thrown. 
	/// Overflow is not allowed, so if the internal value is a larger number than the target numeric type size can accomodate, 
	/// a RangeException is thrown.
	///
	/// Precision loss, such as in conversion from floating-point types to integers or from double to float on platforms
	/// where they differ in size (provided internal actual value fits in float min/max range), is allowed.
	/// 
	/// String truncation is allowed -- it is possible to convert between string and character when string length is 
	/// greater than 1. An empty string gets converted to the char '\0', a non-empty string is truncated to the first character. 
	///
	/// Boolean conversion is performed as follows:
	///
	/// A string value "false" (not case sensitive), "0" or "" (empty string) can be converted to a boolean value false,
	/// any other string not being false by the above criteria evaluates to true (e.g: "hi" -> true).
	/// Integer 0 values are false, everything else is true.
	/// Floating point values equal to the minimal FP representation on a given platform are false, everything else is true.
	///
	/// Arithmetic operations with POD types as well as between DynamicAny's are supported, subject to following
	/// limitations:
	/// 
	/// 	- for std::string and const char* values, only '+' and '+=' operations are supported
	/// 
	/// 	- for integral and floating point numeric values, following operations are supported:
	/// 	  '+', '+=', '-', '-=', '*', '*=' , '/' and '/=' 
	///
	/// 	- for integral values, following operations are supported:
	///		  prefix and postfix increment (++) and decement (--)
	/// 
	/// 	- for all other types, InvalidArgumentException is thrown upon attempt of an arithmetic operation
	/// 
	/// A DynamicAny can be created from and converted to a value of any type for which a specialization of 
	/// DynamicAnyHolderImpl is available. For supported types, see DynamicAnyHolder documentation.
{
public:
	DynamicAny();
		/// Creates an empty DynamicAny.

	template <typename T> 
	DynamicAny(const T& val):
		_pHolder(new DynamicAnyHolderImpl<T>(val))
		/// Creates the DynamicAny from the given value.
	{
	}

	DynamicAny(const char* pVal);
		// Convenience constructor for const char* which gets mapped to a std::string internally, i.e. pVal is deep-copied.

	DynamicAny(const DynamicAny& other);
		/// Copy constructor.

	~DynamicAny();
		/// Destroys the DynamicAny.

	void swap(DynamicAny& other);
		/// Swaps the content of the this DynamicAny with the other DynamicAny.

	template <typename T> 
	void convert(T& val) const
		/// Invoke this method to perform a safe conversion.
		///
		/// Example usage:
		///     DynamicAny any("42");
		///     int i;
		///     any.convert(i);
		///
		/// Throws a RangeException if the value does not fit
		/// into the result variable.
		/// Throws a NotImplementedException if conversion is
		/// not available for the given type.
		/// Throws InvalidAccessException if DynamicAny is empty.
	{
		if (!_pHolder)
			throw InvalidAccessException("Can not convert empty value.");

		_pHolder->convert(val);
	}
	
	template <typename T> 
	T convert() const
		/// Invoke this method to perform a safe conversion.
		///
		/// Example usage:
		///     DynamicAny any("42");
		///     int i = any.convert<int>();
		///
		/// Throws a RangeException if the value does not fit
		/// into the result variable.
		/// Throws a NotImplementedException if conversion is
		/// not available for the given type.
		/// Throws InvalidAccessException if DynamicAny is empty.
	{
		if (!_pHolder)
			throw InvalidAccessException("Can not convert empty value.");

		T result;
		_pHolder->convert(result);
		return result;
	}
	
	template <typename T>
	operator T () const
		/// Safe conversion operator for implicit type
		/// conversions. If the requested type T is same as the 
		/// type being held, the operation performed is direct 
		/// extraction, otherwise it is the conversion of the value
		/// from type currently held to the one requested.
		///
		/// Throws a RangeException if the value does not fit
		/// into the result variable.
		/// Throws a NotImplementedException if conversion is
		/// not available for the given type.
		/// Throws InvalidAccessException if DynamicAny is empty.
	{
		if (!_pHolder)
				throw InvalidAccessException("Can not convert empty value.");

		if (typeid(T) == _pHolder->type())
			return extract<T>();
		else
		{
			T result;
			_pHolder->convert(result);
			return result;
		}
	}

	template <typename T>
	const T& extract() const
		/// Returns a const reference to the actual value.
		///
		/// Must be instantiated with the exact type of
		/// the stored value, otherwise a BadCastException
		/// is thrown.
		/// Throws InvalidAccessException if DynamicAny is empty.
	{
		if (_pHolder && _pHolder->type() == typeid(T))
		{
			DynamicAnyHolderImpl<T>* pHolderImpl = static_cast<DynamicAnyHolderImpl<T>*>(_pHolder);
			return pHolderImpl->value();
		}
		else if (!_pHolder)
			throw InvalidAccessException("Can not extract empty value.");
		else
			throw BadCastException(format("Can not convert %s to %s.",
				_pHolder->type().name(),
				typeid(T).name()));
	}

	template <typename T> 
	DynamicAny& operator = (const T& other)
		/// Assignment operator for assigning POD to DynamicAny
	{
		DynamicAny tmp(other);
		swap(tmp);
		return *this;
	}

	bool operator ! () const;
		/// Logical NOT operator.

	DynamicAny& operator = (const DynamicAny& other);
		/// Assignment operator specialization for DynamicAny

	template <typename T>
	const DynamicAny operator + (const T& other) const
		/// Addition operator for adding POD to DynamicAny
	{
		return convert<T>() + other;
	}

	const DynamicAny operator + (const DynamicAny& other) const;
		/// Addition operator specialization for DynamicAny

	const DynamicAny operator + (const char* other) const;
		/// Addition operator specialization for adding const char* to DynamicAny

	DynamicAny& operator ++ ();
		/// Pre-increment operator

	const DynamicAny operator ++ (int);
		/// Post-increment operator

	DynamicAny& operator -- ();
		/// Pre-decrement operator

	const DynamicAny operator -- (int);
		/// Post-decrement operator

	template <typename T> 
	DynamicAny& operator += (const T& other)
		/// Addition assignment operator for addition/assignment of POD to DynamicAny.
	{
		return *this = convert<T>() + other;
	}

	DynamicAny& operator += (const DynamicAny& other);
		/// Addition assignment operator overload for DynamicAny

	DynamicAny& operator += (const char* other);
		/// Addition assignment operator overload for const char*

	template <typename T> 
	const DynamicAny operator - (const T& other) const
		/// Subtraction operator for subtracting POD from DynamicAny
	{
		return convert<T>() - other;
	}

	const DynamicAny operator - (const DynamicAny& other) const;
		/// Subtraction operator overload for DynamicAny

	template <typename T> 
	DynamicAny& operator -= (const T& other)
		/// Subtraction assignment operator
	{
		return *this = convert<T>() - other;
	}

	DynamicAny& operator -= (const DynamicAny& other);
		/// Subtraction assignment operator overload for DynamicAny

	template <typename T> 
	const DynamicAny operator * (const T& other) const
		/// Multiplication operator for multiplying DynamicAny with POD
	{
		return convert<T>() * other;
	}

	const DynamicAny operator * (const DynamicAny& other) const;
		/// Multiplication operator overload for DynamicAny

	template <typename T> 
	DynamicAny& operator *= (const T& other)
		/// Multiplication assignment operator
	{
		return *this = convert<T>() * other;
	}

	DynamicAny& operator *= (const DynamicAny& other);
		/// Multiplication assignment operator overload for DynamicAny

	template <typename T> 
	const DynamicAny operator / (const T& other) const
		/// Division operator for dividing DynamicAny with POD
	{
		return convert<T>() / other;
	}

	const DynamicAny operator / (const DynamicAny& other) const;
		/// Division operator overload for DynamicAny

	template <typename T> 
	DynamicAny& operator /= (const T& other)
		/// Division assignment operator
	{
		return *this = convert<T>() / other;
	}

	DynamicAny& operator /= (const DynamicAny& other);
		/// Division assignment operator specialization for DynamicAny

	template <typename T> 
	bool operator == (const T& other) const
		/// Equality operator
	{
		if (isEmpty()) return false;
		return convert<T>() == other;
	}

	bool operator == (const char* other) const;
		/// Equality operator overload for const char*

	bool operator == (const DynamicAny& other) const;
		/// Equality operator overload for DynamicAny

	template <typename T> 
	bool operator != (const T& other) const
		/// Inequality operator
	{
		if (isEmpty()) return true;
		return convert<T>() != other;
	}

	bool operator != (const DynamicAny& other) const;
		/// Inequality operator overload for DynamicAny

	bool operator != (const char* other) const;
		/// Inequality operator overload for const char*

	template <typename T> 
	bool operator < (const T& other) const
		/// Less than operator
	{
		if (isEmpty()) return false;
		return convert<T>() < other;
	}

	bool operator < (const DynamicAny& other) const;
		/// Less than operator overload for DynamicAny

	template <typename T> 
	bool operator <= (const T& other) const
		/// Less than or equal operator
	{
		if (isEmpty()) return false;
		return convert<T>() <= other;
	}

	bool operator <= (const DynamicAny& other) const;
		/// Less than or equal operator overload for DynamicAny

	template <typename T> 
	bool operator > (const T& other) const
		/// Greater than operator
	{
		if (isEmpty()) return false;
		return convert<T>() > other;
	}

	bool operator > (const DynamicAny& other) const;
		/// Greater than operator overload for DynamicAny

	template <typename T> 
	bool operator >= (const T& other) const
		/// Greater than or equal operator
	{
		if (isEmpty()) return false;
		return convert<T>() >= other;
	}

	bool operator >= (const DynamicAny& other) const;
		/// Greater than or equal operator overload for DynamicAny

	template <typename T>
	bool operator || (const T& other) const
		/// Logical OR operator
	{
		if (isEmpty()) return false;
		return convert<bool>() || other;
	}

	bool operator || (const DynamicAny& other) const;
		/// Logical OR operator operator overload for DynamicAny

	template <typename T>
	bool operator && (const T& other) const
		/// Logical AND operator
	{
		if (isEmpty()) return false;
		return convert<bool>() && other;
	}

	bool operator && (const DynamicAny& other) const;
		/// Logical AND operator operator overload for DynamicAny

	bool isArray() const;
		/// Returns true if DynamicAny represents a vector

	template <typename T>
	DynamicAny& operator [] (T n)
		/// Index operator, only use on DynamicAnys where isArray
		/// returns true! In all other cases InvalidAccessException is thrown.
	{
		return holderImpl<std::vector<DynamicAny>,
			InvalidAccessException>("Not an array.")->operator[](n);
	}

	template <typename T>
	const DynamicAny& operator [] (T n) const
		/// const Index operator, only use on DynamicAnys where isArray
		/// returns true! In all other cases InvalidAccessException is thrown.
	{
		return const_cast<const DynamicAny&>(holderImpl<std::vector<DynamicAny>,
			InvalidAccessException>("Not an array.")->operator[](n));
	}

	const std::type_info& type() const;
		/// Returns the type information of the stored content.

	void empty();
		/// Empties DynamicAny.

	bool isEmpty() const;
		/// Returns true if empty.

	bool isInteger() const;
		/// Returns true if stored value is integer.

	bool isSigned() const;
		/// Returns true if stored value is signed.

	bool isNumeric() const;
		/// Returns true if stored value is numeric.
		/// Returns false for numeric strings (e.g. "123" is string, not number)

	bool isString() const;
		/// Returns true if stored value is std::string.
	
private:
	template <typename T>
	T add(const DynamicAny& other) const
	{
		return convert<T>() + other.convert<T>();
	}

	template <typename T>
	T subtract(const DynamicAny& other) const
	{
		return convert<T>() - other.convert<T>();
	}
	
	template <typename T>
	T multiply(const DynamicAny& other) const
	{
		return convert<T>() * other.convert<T>();
	}

	template <typename T>
	T divide(const DynamicAny& other) const
	{
		return convert<T>() / other.convert<T>();
	}

	template <typename T, typename E>
	DynamicAnyHolderImpl<T>* holderImpl(const std::string errorMessage = "") const
	{
		if (_pHolder && _pHolder->type() == typeid(T))
			return static_cast<DynamicAnyHolderImpl<T>*>(_pHolder);
		else if (!_pHolder)
			throw InvalidAccessException("Can not access empty value.");
		else
			throw E(errorMessage);
	}

	DynamicAnyHolder* _pHolder;
};


///
/// inlines
///


///
/// DynamicAny members
///
inline void DynamicAny::swap(DynamicAny& ptr)
{
	std::swap(_pHolder, ptr._pHolder);
}


inline const std::type_info& DynamicAny::type() const
{
	return _pHolder ? _pHolder->type() : typeid(void);
}


inline const DynamicAny DynamicAny::operator + (const char* other) const
{
	return convert<std::string>() + other;
}


inline DynamicAny& DynamicAny::operator += (const char*other)
{
	return *this = convert<std::string>() + other;
}


inline bool DynamicAny::operator ! () const
{
	return !convert<bool>();
}


inline bool DynamicAny::isEmpty() const
{
	return 0 == _pHolder;
}


inline bool DynamicAny::isArray() const
{
	return _pHolder ? _pHolder->isArray() : false;
}


inline bool DynamicAny::isInteger() const
{
	return _pHolder ? _pHolder->isInteger() : false;
}


inline bool DynamicAny::isSigned() const
{
	return _pHolder ? _pHolder->isSigned() : false;
}


inline bool DynamicAny::isNumeric() const
{
	return _pHolder ? _pHolder->isNumeric() : false;
}


inline bool DynamicAny::isString() const
{
	return _pHolder ? _pHolder->isString() : false;
}


///
/// DynamicAny non-member functions
///
inline const DynamicAny operator + (const char* other, const DynamicAny& da)
	/// Addition operator for adding DynamicAny to const char*
{
	std::string tmp = other;
	return tmp + da.convert<std::string>();
}


inline char operator + (const char& other, const DynamicAny& da)
	/// Addition operator for adding DynamicAny to char
{
	return other + da.convert<char>();
}


inline char operator - (const char& other, const DynamicAny& da)
	/// Subtraction operator for subtracting DynamicAny from char
{
	return other - da.convert<char>();
}


inline char operator * (const char& other, const DynamicAny& da)
	/// Multiplication operator for multiplying DynamicAny with char
{
	return other * da.convert<char>();
}


inline char operator / (const char& other, const DynamicAny& da)
	/// Division operator for dividing DynamicAny with char
{
	return other / da.convert<char>();
}


inline char operator += (char& other, const DynamicAny& da)
	/// Addition assignment operator for adding DynamicAny to char
{
	return other += da.convert<char>();
}


inline char operator -= (char& other, const DynamicAny& da)
	/// Subtraction assignment operator for subtracting DynamicAny from char
{
	return other -= da.convert<char>();
}


inline char operator *= (char& other, const DynamicAny& da)
	/// Multiplication assignment operator for multiplying DynamicAny with char
{
	return other *= da.convert<char>();
}


inline char operator /= (char& other, const DynamicAny& da)
	/// Division assignment operator for dividing DynamicAny with char
{
	return other /= da.convert<char>();
}


inline bool operator == (const char& other, const DynamicAny& da)
	/// Equality operator for comparing DynamicAny with char
{
	if (da.isEmpty()) return false;
	return other == da.convert<char>();
}


inline bool operator != (const char& other, const DynamicAny& da)
	/// Inequality operator for comparing DynamicAny with char
{
	if (da.isEmpty()) return true;
	return other != da.convert<char>();
}


inline bool operator < (const char& other, const DynamicAny& da)
	/// Less than operator for comparing DynamicAny with char
{
	if (da.isEmpty()) return false;
	return other < da.convert<char>();
}


inline bool operator <= (const char& other, const DynamicAny& da)
	/// Less than or equal operator for comparing DynamicAny with char
{
	if (da.isEmpty()) return false;
	return other <= da.convert<char>();
}


inline bool operator > (const char& other, const DynamicAny& da)
	/// Greater than operator for comparing DynamicAny with char
{
	if (da.isEmpty())return false;
	return other > da.convert<char>();
}


inline bool operator >= (const char& other, const DynamicAny& da)
	/// Greater than or equal operator for comparing DynamicAny with char
{
	if (da.isEmpty())return false;
	return other >= da.convert<char>();
}


inline Poco::Int8 operator + (const Poco::Int8& other, const DynamicAny& da)
	/// Addition operator for adding DynamicAny to Poco::Int8
{
	return other + da.convert<Poco::Int8>();
}


inline Poco::Int8 operator - (const Poco::Int8& other, const DynamicAny& da)
	/// Subtraction operator for subtracting DynamicAny from Poco::Int8
{
	return other - da.convert<Poco::Int8>();
}


inline Poco::Int8 operator * (const Poco::Int8& other, const DynamicAny& da)
	/// Multiplication operator for multiplying DynamicAny with Poco::Int8
{
	return other * da.convert<Poco::Int8>();
}


inline Poco::Int8 operator / (const Poco::Int8& other, const DynamicAny& da)
	/// Division operator for dividing DynamicAny with Poco::Int8
{
	return other / da.convert<Poco::Int8>();
}


inline Poco::Int8 operator += (Poco::Int8& other, const DynamicAny& da)
	/// Addition assignment operator for adding DynamicAny to Poco::Int8
{
	return other += da.convert<Poco::Int8>();
}


inline Poco::Int8 operator -= (Poco::Int8& other, const DynamicAny& da)
	/// Subtraction assignment operator for subtracting DynamicAny from Poco::Int8
{
	return other -= da.convert<Poco::Int8>();
}


inline Poco::Int8 operator *= (Poco::Int8& other, const DynamicAny& da)
	/// Multiplication assignment operator for multiplying DynamicAny with Poco::Int8
{
	return other *= da.convert<Poco::Int8>();
}


inline Poco::Int8 operator /= (Poco::Int8& other, const DynamicAny& da)
	/// Division assignment operator for dividing DynamicAny with Poco::Int8
{
	return other /= da.convert<Poco::Int8>();
}


inline bool operator == (const Poco::Int8& other, const DynamicAny& da)
	/// Equality operator for comparing DynamicAny with Poco::Int8
{
	if (da.isEmpty()) return false;
	return other == da.convert<Poco::Int8>();
}


inline bool operator != (const Poco::Int8& other, const DynamicAny& da)
	/// Inequality operator for comparing DynamicAny with Poco::Int8
{
	if (da.isEmpty()) return true;
	return other != da.convert<Poco::Int8>();
}


inline bool operator < (const Poco::Int8& other, const DynamicAny& da)
	/// Less than operator for comparing DynamicAny with Poco::Int8
{
	if (da.isEmpty()) return false;
	return other < da.convert<Poco::Int8>();
}


inline bool operator <= (const Poco::Int8& other, const DynamicAny& da)
	/// Less than or equal operator for comparing DynamicAny with Poco::Int8
{
	if (da.isEmpty()) return false;
	return other <= da.convert<Poco::Int8>();
}


inline bool operator > (const Poco::Int8& other, const DynamicAny& da)
	/// Greater than operator for comparing DynamicAny with Poco::Int8
{
	if (da.isEmpty()) return false;
	return other > da.convert<Poco::Int8>();
}


inline bool operator >= (const Poco::Int8& other, const DynamicAny& da)
	/// Greater than or equal operator for comparing DynamicAny with Poco::Int8
{
	if (da.isEmpty()) return false;
	return other >= da.convert<Poco::Int8>();
}


inline Poco::UInt8 operator + (const Poco::UInt8& other, const DynamicAny& da)
	/// Addition operator for adding DynamicAny to Poco::UInt8
{
	return other + da.convert<Poco::UInt8>();
}


inline Poco::UInt8 operator - (const Poco::UInt8& other, const DynamicAny& da)
	/// Subtraction operator for subtracting DynamicAny from Poco::UInt8
{
	return other - da.convert<Poco::UInt8>();
}


inline Poco::UInt8 operator * (const Poco::UInt8& other, const DynamicAny& da)
	/// Multiplication operator for multiplying DynamicAny with Poco::UInt8
{
	return other * da.convert<Poco::UInt8>();
}


inline Poco::UInt8 operator / (const Poco::UInt8& other, const DynamicAny& da)
	/// Division operator for dividing DynamicAny with Poco::UInt8
{
	return other / da.convert<Poco::UInt8>();
}


inline Poco::UInt8 operator += (Poco::UInt8& other, const DynamicAny& da)
	/// Addition assignment operator for adding DynamicAny to Poco::UInt8
{
	return other += da.convert<Poco::UInt8>();
}


inline Poco::UInt8 operator -= (Poco::UInt8& other, const DynamicAny& da)
	/// Subtraction assignment operator for subtracting DynamicAny from Poco::UInt8
{
	return other -= da.convert<Poco::UInt8>();
}


inline Poco::UInt8 operator *= (Poco::UInt8& other, const DynamicAny& da)
	/// Multiplication assignment operator for multiplying DynamicAny with Poco::UInt8
{
	return other *= da.convert<Poco::UInt8>();
}


inline Poco::UInt8 operator /= (Poco::UInt8& other, const DynamicAny& da)
	/// Division assignment operator for dividing DynamicAny with Poco::UInt8
{
	return other /= da.convert<Poco::UInt8>();
}


inline bool operator == (const Poco::UInt8& other, const DynamicAny& da)
	/// Equality operator for comparing DynamicAny with Poco::UInt8
{
	if (da.isEmpty()) return false;
	return other == da.convert<Poco::UInt8>();
}


inline bool operator != (const Poco::UInt8& other, const DynamicAny& da)
	/// Inequality operator for comparing DynamicAny with Poco::UInt8
{
	if (da.isEmpty()) return true;
	return other != da.convert<Poco::UInt8>();
}


inline bool operator < (const Poco::UInt8& other, const DynamicAny& da)
	/// Less than operator for comparing DynamicAny with Poco::UInt8
{
	if (da.isEmpty()) return false;
	return other < da.convert<Poco::UInt8>();
}


inline bool operator <= (const Poco::UInt8& other, const DynamicAny& da)
	/// Less than or equal operator for comparing DynamicAny with Poco::UInt8
{
	if (da.isEmpty()) return false;
	return other <= da.convert<Poco::UInt8>();
}


inline bool operator > (const Poco::UInt8& other, const DynamicAny& da)
	/// Greater than operator for comparing DynamicAny with Poco::UInt8
{
	if (da.isEmpty()) return false;
	return other > da.convert<Poco::UInt8>();
}


inline bool operator >= (const Poco::UInt8& other, const DynamicAny& da)
	/// Greater than or equal operator for comparing DynamicAny with Poco::UInt8
{
	if (da.isEmpty()) return false;
	return other >= da.convert<Poco::UInt8>();
}


inline Poco::Int16 operator + (const Poco::Int16& other, const DynamicAny& da)
	/// Addition operator for adding DynamicAny to Poco::Int16
{
	return other + da.convert<Poco::Int16>();
}


inline Poco::Int16 operator - (const Poco::Int16& other, const DynamicAny& da)
	/// Subtraction operator for subtracting DynamicAny from Poco::Int16
{
	return other - da.convert<Poco::Int16>();
}


inline Poco::Int16 operator * (const Poco::Int16& other, const DynamicAny& da)
	/// Multiplication operator for multiplying DynamicAny with Poco::Int16
{
	return other * da.convert<Poco::Int16>();
}


inline Poco::Int16 operator / (const Poco::Int16& other, const DynamicAny& da)
	/// Division operator for dividing DynamicAny with Poco::Int16
{
	return other / da.convert<Poco::Int16>();
}


inline Poco::Int16 operator += (Poco::Int16& other, const DynamicAny& da)
	/// Addition assignment operator for adding DynamicAny to Poco::Int16
{
	return other += da.convert<Poco::Int16>();
}


inline Poco::Int16 operator -= (Poco::Int16& other, const DynamicAny& da)
	/// Subtraction assignment operator for subtracting DynamicAny from Poco::Int16
{
	return other -= da.convert<Poco::Int16>();
}


inline Poco::Int16 operator *= (Poco::Int16& other, const DynamicAny& da)
	/// Multiplication assignment operator for multiplying DynamicAny with Poco::Int16
{
	return other *= da.convert<Poco::Int16>();
}


inline Poco::Int16 operator /= (Poco::Int16& other, const DynamicAny& da)
	/// Division assignment operator for dividing DynamicAny with Poco::Int16
{
	return other /= da.convert<Poco::Int16>();
}


inline bool operator == (const Poco::Int16& other, const DynamicAny& da)
	/// Equality operator for comparing DynamicAny with Poco::Int16
{
	if (da.isEmpty()) return false;
	return other == da.convert<Poco::Int16>();
}


inline bool operator != (const Poco::Int16& other, const DynamicAny& da)
	/// Inequality operator for comparing DynamicAny with Poco::Int16
{
	if (da.isEmpty()) return true;
	return other != da.convert<Poco::Int16>();
}


inline bool operator < (const Poco::Int16& other, const DynamicAny& da)
	/// Less than operator for comparing DynamicAny with Poco::Int16
{
	if (da.isEmpty()) return false;
	return other < da.convert<Poco::Int16>();
}


inline bool operator <= (const Poco::Int16& other, const DynamicAny& da)
	/// Less than or equal operator for comparing DynamicAny with Poco::Int16
{
	if (da.isEmpty()) return false;
	return other <= da.convert<Poco::Int16>();
}


inline bool operator > (const Poco::Int16& other, const DynamicAny& da)
	/// Greater than operator for comparing DynamicAny with Poco::Int16
{
	if (da.isEmpty()) return false;
	return other > da.convert<Poco::Int16>();
}


inline bool operator >= (const Poco::Int16& other, const DynamicAny& da)
	/// Greater than or equal operator for comparing DynamicAny with Poco::Int16
{
	if (da.isEmpty()) return false;
	return other >= da.convert<Poco::Int16>();
}


inline Poco::UInt16 operator + (const Poco::UInt16& other, const DynamicAny& da)
	/// Addition operator for adding DynamicAny to Poco::UInt16
{
	return other + da.convert<Poco::UInt16>();
}


inline Poco::UInt16 operator - (const Poco::UInt16& other, const DynamicAny& da)
	/// Subtraction operator for subtracting DynamicAny from Poco::UInt16
{
	return other - da.convert<Poco::UInt16>();
}


inline Poco::UInt16 operator * (const Poco::UInt16& other, const DynamicAny& da)
	/// Multiplication operator for multiplying DynamicAny with Poco::UInt16
{
	return other * da.convert<Poco::UInt16>();
}


inline Poco::UInt16 operator / (const Poco::UInt16& other, const DynamicAny& da)
	/// Division operator for dividing DynamicAny with Poco::UInt16
{
	return other / da.convert<Poco::UInt16>();
}


inline Poco::UInt16 operator += (Poco::UInt16& other, const DynamicAny& da)
	/// Addition assignment operator for adding DynamicAny to Poco::UInt16
{
	return other += da.convert<Poco::UInt16>();
}


inline Poco::UInt16 operator -= (Poco::UInt16& other, const DynamicAny& da)
	/// Subtraction assignment operator for subtracting DynamicAny from Poco::UInt16
{
	return other -= da.convert<Poco::UInt16>();
}


inline Poco::UInt16 operator *= (Poco::UInt16& other, const DynamicAny& da)
	/// Multiplication assignment operator for multiplying DynamicAny with Poco::UInt16
{
	return other *= da.convert<Poco::UInt16>();
}


inline Poco::UInt16 operator /= (Poco::UInt16& other, const DynamicAny& da)
	/// Division assignment operator for dividing DynamicAny with Poco::UInt16
{
	return other /= da.convert<Poco::UInt16>();
}


inline bool operator == (const Poco::UInt16& other, const DynamicAny& da)
	/// Equality operator for comparing DynamicAny with Poco::UInt16
{
	if (da.isEmpty()) return false;
	return other == da.convert<Poco::UInt16>();
}


inline bool operator != (const Poco::UInt16& other, const DynamicAny& da)
	/// Inequality operator for comparing DynamicAny with Poco::UInt16
{
	if (da.isEmpty()) return true;
	return other != da.convert<Poco::UInt16>();
}


inline bool operator < (const Poco::UInt16& other, const DynamicAny& da)
	/// Less than operator for comparing DynamicAny with Poco::UInt16
{
	if (da.isEmpty()) return false;
	return other < da.convert<Poco::UInt16>();
}


inline bool operator <= (const Poco::UInt16& other, const DynamicAny& da)
	/// Less than or equal operator for comparing DynamicAny with Poco::UInt16
{
	if (da.isEmpty()) return false;
	return other <= da.convert<Poco::UInt16>();
}


inline bool operator > (const Poco::UInt16& other, const DynamicAny& da)
	/// Greater than operator for comparing DynamicAny with Poco::UInt16
{
	if (da.isEmpty()) return false;
	return other > da.convert<Poco::UInt16>();
}


inline bool operator >= (const Poco::UInt16& other, const DynamicAny& da)
	/// Greater than or equal operator for comparing DynamicAny with Poco::UInt16
{
	if (da.isEmpty()) return false;
	return other >= da.convert<Poco::UInt16>();
}


inline Poco::Int32 operator + (const Poco::Int32& other, const DynamicAny& da)
	/// Addition operator for adding DynamicAny to Poco::Int32
{
	return other + da.convert<Poco::Int32>();
}


inline Poco::Int32 operator - (const Poco::Int32& other, const DynamicAny& da)
	/// Subtraction operator for subtracting DynamicAny from Poco::Int32
{
	return other - da.convert<Poco::Int32>();
}


inline Poco::Int32 operator * (const Poco::Int32& other, const DynamicAny& da)
	/// Multiplication operator for multiplying DynamicAny with Poco::Int32
{
	return other * da.convert<Poco::Int32>();
}


inline Poco::Int32 operator / (const Poco::Int32& other, const DynamicAny& da)
	/// Division operator for dividing DynamicAny with Poco::Int32
{
	return other / da.convert<Poco::Int32>();
}


inline Poco::Int32 operator += (Poco::Int32& other, const DynamicAny& da)
	/// Addition assignment operator for adding DynamicAny to Poco::Int32
{
	return other += da.convert<Poco::Int32>();
}


inline Poco::Int32 operator -= (Poco::Int32& other, const DynamicAny& da)
	/// Subtraction assignment operator for subtracting DynamicAny from Poco::Int32
{
	return other -= da.convert<Poco::Int32>();
}


inline Poco::Int32 operator *= (Poco::Int32& other, const DynamicAny& da)
	/// Multiplication assignment operator for multiplying DynamicAny with Poco::Int32
{
	return other *= da.convert<Poco::Int32>();
}


inline Poco::Int32 operator /= (Poco::Int32& other, const DynamicAny& da)
	/// Division assignment operator for dividing DynamicAny with Poco::Int32
{
	return other /= da.convert<Poco::Int32>();
}


inline bool operator == (const Poco::Int32& other, const DynamicAny& da)
	/// Equality operator for comparing DynamicAny with Poco::Int32
{
	if (da.isEmpty()) return false;
	return other == da.convert<Poco::Int32>();
}


inline bool operator != (const Poco::Int32& other, const DynamicAny& da)
	/// Inequality operator for comparing DynamicAny with Poco::Int32
{
	if (da.isEmpty()) return true;
	return other != da.convert<Poco::Int32>();
}


inline bool operator < (const Poco::Int32& other, const DynamicAny& da)
	/// Less than operator for comparing DynamicAny with Poco::Int32
{
	if (da.isEmpty()) return false;
	return other < da.convert<Poco::Int32>();
}


inline bool operator <= (const Poco::Int32& other, const DynamicAny& da)
	/// Less than or equal operator for comparing DynamicAny with Poco::Int32
{
	if (da.isEmpty()) return false;
	return other <= da.convert<Poco::Int32>();
}


inline bool operator > (const Poco::Int32& other, const DynamicAny& da)
	/// Greater than operator for comparing DynamicAny with Poco::Int32
{
	if (da.isEmpty()) return false;
	return other > da.convert<Poco::Int32>();
}


inline bool operator >= (const Poco::Int32& other, const DynamicAny& da)
	/// Greater than or equal operator for comparing DynamicAny with Poco::Int32
{
	if (da.isEmpty()) return false;
	return other >= da.convert<Poco::Int32>();
}


inline Poco::UInt32 operator + (const Poco::UInt32& other, const DynamicAny& da)
	/// Addition operator for adding DynamicAny to Poco::UInt32
{
	return other + da.convert<Poco::UInt32>();
}


inline Poco::UInt32 operator - (const Poco::UInt32& other, const DynamicAny& da)
	/// Subtraction operator for subtracting DynamicAny from Poco::UInt32
{
	return other - da.convert<Poco::UInt32>();
}


inline Poco::UInt32 operator * (const Poco::UInt32& other, const DynamicAny& da)
	/// Multiplication operator for multiplying DynamicAny with Poco::UInt32
{
	return other * da.convert<Poco::UInt32>();
}


inline Poco::UInt32 operator / (const Poco::UInt32& other, const DynamicAny& da)
	/// Division operator for dividing DynamicAny with Poco::UInt32
{
	return other / da.convert<Poco::UInt32>();
}


inline Poco::UInt32 operator += (Poco::UInt32& other, const DynamicAny& da)
	/// Addition assignment operator for adding DynamicAny to Poco::UInt32
{
	return other += da.convert<Poco::UInt32>();
}


inline Poco::UInt32 operator -= (Poco::UInt32& other, const DynamicAny& da)
	/// Subtraction assignment operator for subtracting DynamicAny from Poco::UInt32
{
	return other -= da.convert<Poco::UInt32>();
}


inline Poco::UInt32 operator *= (Poco::UInt32& other, const DynamicAny& da)
	/// Multiplication assignment operator for multiplying DynamicAny with Poco::UInt32
{
	return other *= da.convert<Poco::UInt32>();
}


inline Poco::UInt32 operator /= (Poco::UInt32& other, const DynamicAny& da)
	/// Division assignment operator for dividing DynamicAny with Poco::UInt32
{
	return other /= da.convert<Poco::UInt32>();
}


inline bool operator == (const Poco::UInt32& other, const DynamicAny& da)
	/// Equality operator for comparing DynamicAny with Poco::UInt32
{
	if (da.isEmpty()) return false;
	return other == da.convert<Poco::UInt32>();
}


inline bool operator != (const Poco::UInt32& other, const DynamicAny& da)
	/// Inequality operator for comparing DynamicAny with Poco::UInt32
{
	if (da.isEmpty()) return true;
	return other != da.convert<Poco::UInt32>();
}


inline bool operator < (const Poco::UInt32& other, const DynamicAny& da)
	/// Less than operator for comparing DynamicAny with Poco::UInt32
{
	if (da.isEmpty()) return false;
	return other < da.convert<Poco::UInt32>();
}


inline bool operator <= (const Poco::UInt32& other, const DynamicAny& da)
	/// Less than or equal operator for comparing DynamicAny with Poco::UInt32
{
	if (da.isEmpty()) return false;
	return other <= da.convert<Poco::UInt32>();
}


inline bool operator > (const Poco::UInt32& other, const DynamicAny& da)
	/// Greater than operator for comparing DynamicAny with Poco::UInt32
{
	if (da.isEmpty()) return false;
	return other > da.convert<Poco::UInt32>();
}


inline bool operator >= (const Poco::UInt32& other, const DynamicAny& da)
	/// Greater than or equal operator for comparing DynamicAny with Poco::UInt32
{
	if (da.isEmpty()) return false;
	return other >= da.convert<Poco::UInt32>();
}


inline Poco::Int64 operator + (const Poco::Int64& other, const DynamicAny& da)
	/// Addition operator for adding DynamicAny to Poco::Int64
{
	return other + da.convert<Poco::Int64>();
}


inline Poco::Int64 operator - (const Poco::Int64& other, const DynamicAny& da)
	/// Subtraction operator for subtracting DynamicAny from Poco::Int64
{
	return other - da.convert<Poco::Int64>();
}


inline Poco::Int64 operator * (const Poco::Int64& other, const DynamicAny& da)
	/// Multiplication operator for multiplying DynamicAny with Poco::Int64
{
	return other * da.convert<Poco::Int64>();
}


inline Poco::Int64 operator / (const Poco::Int64& other, const DynamicAny& da)
	/// Division operator for dividing DynamicAny with Poco::Int64
{
	return other / da.convert<Poco::Int64>();
}


inline Poco::Int64 operator += (Poco::Int64& other, const DynamicAny& da)
	/// Addition assignment operator for adding DynamicAny to Poco::Int64
{
	return other += da.convert<Poco::Int64>();
}


inline Poco::Int64 operator -= (Poco::Int64& other, const DynamicAny& da)
	/// Subtraction assignment operator for subtracting DynamicAny from Poco::Int64
{
	return other -= da.convert<Poco::Int64>();
}


inline Poco::Int64 operator *= (Poco::Int64& other, const DynamicAny& da)
	/// Multiplication assignment operator for multiplying DynamicAny with Poco::Int64
{
	return other *= da.convert<Poco::Int64>();
}


inline Poco::Int64 operator /= (Poco::Int64& other, const DynamicAny& da)
	/// Division assignment operator for dividing DynamicAny with Poco::Int64
{
	return other /= da.convert<Poco::Int64>();
}


inline bool operator == (const Poco::Int64& other, const DynamicAny& da)
	/// Equality operator for comparing DynamicAny with Poco::Int64
{
	if (da.isEmpty()) return false;
	return other == da.convert<Poco::Int64>();
}


inline bool operator != (const Poco::Int64& other, const DynamicAny& da)
	/// Inequality operator for comparing DynamicAny with Poco::Int64
{
	if (da.isEmpty()) return true;
	return other != da.convert<Poco::Int64>();
}


inline bool operator < (const Poco::Int64& other, const DynamicAny& da)
	/// Less than operator for comparing DynamicAny with Poco::Int64
{
	if (da.isEmpty()) return false;
	return other < da.convert<Poco::Int64>();
}


inline bool operator <= (const Poco::Int64& other, const DynamicAny& da)
	/// Less than or equal operator for comparing DynamicAny with Poco::Int64
{
	if (da.isEmpty()) return false;
	return other <= da.convert<Poco::Int64>();
}


inline bool operator > (const Poco::Int64& other, const DynamicAny& da)
	/// Greater than operator for comparing DynamicAny with Poco::Int64
{
	if (da.isEmpty()) return false;
	return other > da.convert<Poco::Int64>();
}


inline bool operator >= (const Poco::Int64& other, const DynamicAny& da)
	/// Greater than or equal operator for comparing DynamicAny with Poco::Int64
{
	if (da.isEmpty()) return false;
	return other >= da.convert<Poco::Int64>();
}


inline Poco::UInt64 operator + (const Poco::UInt64& other, const DynamicAny& da)
	/// Addition operator for adding DynamicAny to Poco::UInt64
{
	return other + da.convert<Poco::UInt64>();
}


inline Poco::UInt64 operator - (const Poco::UInt64& other, const DynamicAny& da)
	/// Subtraction operator for subtracting DynamicAny from Poco::UInt64
{
	return other - da.convert<Poco::UInt64>();
}


inline Poco::UInt64 operator * (const Poco::UInt64& other, const DynamicAny& da)
	/// Multiplication operator for multiplying DynamicAny with Poco::UInt64
{
	return other * da.convert<Poco::UInt64>();
}


inline Poco::UInt64 operator / (const Poco::UInt64& other, const DynamicAny& da)
	/// Division operator for dividing DynamicAny with Poco::UInt64
{
	return other / da.convert<Poco::UInt64>();
}


inline Poco::UInt64 operator += (Poco::UInt64& other, const DynamicAny& da)
	/// Addition assignment operator for adding DynamicAny to Poco::UInt64
{
	return other += da.convert<Poco::UInt64>();
}


inline Poco::UInt64 operator -= (Poco::UInt64& other, const DynamicAny& da)
	/// Subtraction assignment operator for subtracting DynamicAny from Poco::UInt64
{
	return other -= da.convert<Poco::UInt64>();
}


inline Poco::UInt64 operator *= (Poco::UInt64& other, const DynamicAny& da)
	/// Multiplication assignment operator for multiplying DynamicAny with Poco::UInt64
{
	return other *= da.convert<Poco::UInt64>();
}


inline Poco::UInt64 operator /= (Poco::UInt64& other, const DynamicAny& da)
	/// Division assignment operator for dividing DynamicAny with Poco::UInt64
{
	return other /= da.convert<Poco::UInt64>();
}


inline bool operator == (const Poco::UInt64& other, const DynamicAny& da)
	/// Equality operator for comparing DynamicAny with Poco::UInt64
{
	if (da.isEmpty()) return false;
	return other == da.convert<Poco::UInt64>();
}


inline bool operator != (const Poco::UInt64& other, const DynamicAny& da)
	/// Inequality operator for comparing DynamicAny with Poco::UInt64
{
	if (da.isEmpty()) return true;
	return other != da.convert<Poco::UInt64>();
}


inline bool operator < (const Poco::UInt64& other, const DynamicAny& da)
	/// Less than operator for comparing DynamicAny with Poco::UInt64
{
	if (da.isEmpty()) return false;
	return other < da.convert<Poco::UInt64>();
}


inline bool operator <= (const Poco::UInt64& other, const DynamicAny& da)
	/// Less than or equal operator for comparing DynamicAny with Poco::UInt64
{
	if (da.isEmpty()) return false;
	return other <= da.convert<Poco::UInt64>();
}


inline bool operator > (const Poco::UInt64& other, const DynamicAny& da)
	/// Greater than operator for comparing DynamicAny with Poco::UInt64
{
	if (da.isEmpty()) return false;
	return other > da.convert<Poco::UInt64>();
}


inline bool operator >= (const Poco::UInt64& other, const DynamicAny& da)
	/// Greater than or equal operator for comparing DynamicAny with Poco::UInt64
{
	if (da.isEmpty()) return false;
	return other >= da.convert<Poco::UInt64>();
}


inline float operator + (const float& other, const DynamicAny& da)
	/// Addition operator for adding DynamicAny to float
{
	return other + da.convert<float>();
}


inline float operator - (const float& other, const DynamicAny& da)
	/// Subtraction operator for subtracting DynamicAny from float
{
	return other - da.convert<float>();
}


inline float operator * (const float& other, const DynamicAny& da)
	/// Multiplication operator for multiplying DynamicAny with float
{
	return other * da.convert<float>();
}


inline float operator / (const float& other, const DynamicAny& da)
	/// Division operator for dividing DynamicAny with float
{
	return other / da.convert<float>();
}


inline float operator += (float& other, const DynamicAny& da)
	/// Addition assignment operator for adding DynamicAny to float
{
	return other += da.convert<float>();
}


inline float operator -= (float& other, const DynamicAny& da)
	/// Subtraction assignment operator for subtracting DynamicAny from float
{
	return other -= da.convert<float>();
}


inline float operator *= (float& other, const DynamicAny& da)
	/// Multiplication assignment operator for multiplying DynamicAny with float
{
	return other *= da.convert<float>();
}


inline float operator /= (float& other, const DynamicAny& da)
	/// Division assignment operator for dividing DynamicAny with float
{
	return other /= da.convert<float>();
}


inline bool operator == (const float& other, const DynamicAny& da)
	/// Equality operator for comparing DynamicAny with float
{
	if (da.isEmpty()) return false;
	return other == da.convert<float>();
}


inline bool operator != (const float& other, const DynamicAny& da)
	/// Inequality operator for comparing DynamicAny with float
{
	if (da.isEmpty()) return true;
	return other != da.convert<float>();
}


inline bool operator < (const float& other, const DynamicAny& da)
	/// Less than operator for comparing DynamicAny with float
{
	if (da.isEmpty()) return false;
	return other < da.convert<float>();
}


inline bool operator <= (const float& other, const DynamicAny& da)
	/// Less than or equal operator for comparing DynamicAny with float
{
	if (da.isEmpty()) return false;
	return other <= da.convert<float>();
}


inline bool operator > (const float& other, const DynamicAny& da)
	/// Greater than operator for comparing DynamicAny with float
{
	if (da.isEmpty()) return false;
	return other > da.convert<float>();
}


inline bool operator >= (const float& other, const DynamicAny& da)
	/// Greater than or equal operator for comparing DynamicAny with float
{
	if (da.isEmpty()) return false;
	return other >= da.convert<float>();
}


inline double operator + (const double& other, const DynamicAny& da)
	/// Addition operator for adding DynamicAny to double
{
	return other + da.convert<double>();
}


inline double operator - (const double& other, const DynamicAny& da)
	/// Subtraction operator for subtracting DynamicAny from double
{
	return other - da.convert<double>();
}


inline double operator * (const double& other, const DynamicAny& da)
	/// Multiplication operator for multiplying DynamicAny with double
{
	return other * da.convert<double>();
}


inline double operator / (const double& other, const DynamicAny& da)
	/// Division operator for dividing DynamicAny with double
{
	return other / da.convert<double>();
}


inline double operator += (double& other, const DynamicAny& da)
	/// Addition assignment operator for adding DynamicAny to double
{
	return other += da.convert<double>();
}


inline double operator -= (double& other, const DynamicAny& da)
	/// Subtraction assignment operator for subtracting DynamicAny from double
{
	return other -= da.convert<double>();
}


inline double operator *= (double& other, const DynamicAny& da)
	/// Multiplication assignment operator for multiplying DynamicAny with double
{
	return other *= da.convert<double>();
}


inline double operator /= (double& other, const DynamicAny& da)
	/// Division assignment operator for dividing DynamicAny with double
{
	return other /= da.convert<double>();
}


inline bool operator == (const double& other, const DynamicAny& da)
	/// Equality operator for comparing DynamicAny with double
{
	if (da.isEmpty()) return false;
	return other == da.convert<double>();
}


inline bool operator != (const double& other, const DynamicAny& da)
	/// Inequality operator for comparing DynamicAny with double
{
	if (da.isEmpty()) return true;
	return other != da.convert<double>();
}


inline bool operator < (const double& other, const DynamicAny& da)
	/// Less than operator for comparing DynamicAny with double
{
	if (da.isEmpty()) return false;
	return other < da.convert<double>();
}


inline bool operator <= (const double& other, const DynamicAny& da)
	/// Less than or equal operator for comparing DynamicAny with double
{
	if (da.isEmpty()) return false;
	return other <= da.convert<double>();
}


inline bool operator > (const double& other, const DynamicAny& da)
	/// Greater than operator for comparing DynamicAny with double
{
	if (da.isEmpty()) return false;
	return other > da.convert<double>();
}


inline bool operator >= (const double& other, const DynamicAny& da)
	/// Greater than or equal operator for comparing DynamicAny with double
{
	if (da.isEmpty()) return false;
	return other >= da.convert<double>();
}


inline bool operator == (const bool& other, const DynamicAny& da)
	/// Equality operator for comparing DynamicAny with bool
{
	if (da.isEmpty()) return false;
	return other == da.convert<bool>();
}


inline bool operator != (const bool& other, const DynamicAny& da)
	/// Inequality operator for comparing DynamicAny with bool
{
	if (da.isEmpty()) return true;
	return other != da.convert<bool>();
}


inline bool operator == (const std::string& other, const DynamicAny& da)
	/// Equality operator for comparing DynamicAny with std::string
{
	if (da.isEmpty()) return false;
	return other == da.convert<std::string>();
}


inline bool operator != (const std::string& other, const DynamicAny& da)
	/// Inequality operator for comparing DynamicAny with std::string
{
	if (da.isEmpty()) return true;
	return other != da.convert<std::string>();
}


inline bool operator == (const char* other, const DynamicAny& da)
	/// Equality operator for comparing DynamicAny with const char*
{
	if (da.isEmpty()) return false;
	return da.convert<std::string>() == other;
}


inline bool operator != (const char* other, const DynamicAny& da)
	/// Inequality operator for comparing DynamicAny with const char*
{
	if (da.isEmpty()) return true;
	return da.convert<std::string>() != other;
}


#ifndef POCO_LONG_IS_64_BIT


inline long operator + (const long& other, const DynamicAny& da)
	/// Addition operator for adding DynamicAny to long
{
	return other + da.convert<long>();
}


inline long operator - (const long& other, const DynamicAny& da)
	/// Subtraction operator for subtracting DynamicAny from long
{
	return other - da.convert<long>();
}


inline long operator * (const long& other, const DynamicAny& da)
	/// Multiplication operator for multiplying DynamicAny with long
{
	return other * da.convert<long>();
}


inline long operator / (const long& other, const DynamicAny& da)
	/// Division operator for dividing DynamicAny with long
{
	return other / da.convert<long>();
}


inline long operator += (long& other, const DynamicAny& da)
	/// Addition assignment operator for adding DynamicAny to long
{
	return other += da.convert<long>();
}


inline long operator -= (long& other, const DynamicAny& da)
	/// Subtraction assignment operator for subtracting DynamicAny from long
{
	return other -= da.convert<long>();
}


inline long operator *= (long& other, const DynamicAny& da)
	/// Multiplication assignment operator for multiplying DynamicAny with long
{
	return other *= da.convert<long>();
}


inline long operator /= (long& other, const DynamicAny& da)
	/// Division assignment operator for dividing DynamicAny with long
{
	return other /= da.convert<long>();
}


inline bool operator == (const long& other, const DynamicAny& da)
	/// Equality operator for comparing DynamicAny with long
{
	if (da.isEmpty()) return false;
	return other == da.convert<long>();
}


inline bool operator != (const long& other, const DynamicAny& da)
	/// Inequality operator for comparing DynamicAny with long
{
	if (da.isEmpty()) return true;
	return other != da.convert<long>();
}


inline bool operator < (const long& other, const DynamicAny& da)
	/// Less than operator for comparing DynamicAny with long
{
	if (da.isEmpty()) return false;
	return other < da.convert<long>();
}


inline bool operator <= (const long& other, const DynamicAny& da)
	/// Less than or equal operator for comparing DynamicAny with long
{
	if (da.isEmpty()) return false;
	return other <= da.convert<long>();
}


inline bool operator > (const long& other, const DynamicAny& da)
	/// Greater than operator for comparing DynamicAny with long
{
	if (da.isEmpty()) return false;
	return other > da.convert<long>();
}


inline bool operator >= (const long& other, const DynamicAny& da)
	/// Greater than or equal operator for comparing DynamicAny with long
{
	if (da.isEmpty()) return false;
	return other >= da.convert<long>();
}


#endif // POCO_LONG_IS_64_BIT


} // namespace Poco


#endif // Foundation_DynamicAny_INCLUDED
