/*-
 * Copyright (c) 2010 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Yorick Hardy
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdlib.h>

#include "interrupts.h"
#include "requests.h"
#include "tusb3410.h"

static void do_usb_interrupt();
static void restart_dma1();
static void restart_dma3();

static uint8_t intr_toggle = 0, intr_ready = 0;

static void
do_usb_interrupt(void)
{
	if ((TI3410_IEPCNF_2 & TI3410_EPCNF_TOGLE) != intr_toggle)
		return;
	if (intr_ready == 0)
		return;

	if (intr_toggle == 0) {
		/* if the interrupt LSR has any bits set clear them */
		TI3410_UART_LSR = intrinX[UTICOM_INTR_LSR];
		/* if the interrupt MSR has any bits set clear them */
		TI3410_UART_MSR = intrinX[UTICOM_INTR_MSR];
		/* set transmission length and ack */
		TI3410_IEPBCTX_2 = UTICOM_INTR_LEN;
	} else {
		/* if the interrupt LSR has any bits set clear them */
		TI3410_UART_LSR = intrinY[UTICOM_INTR_LSR];
		/* if the interrupt MSR has any bits set clear them */
		TI3410_UART_MSR = intrinY[UTICOM_INTR_MSR];
		/* set transmission length and ack */
		TI3410_IEPBCTY_2 = UTICOM_INTR_LEN;
	}

	intr_toggle ^= TI3410_EPCNF_TOGLE;
	intr_ready = 0;
}

void
usb_interrupt(void)
{
	/* don't send an interrupt if the interrupt endpoint is disabled */
	if ((TI3410_IEPCNF_2 & TI3410_EPCNF_UBME) == 0)
		return;

	if (intr_toggle == 0) {
		/* setup (and queue) the interrupt information */
		intrinX[UTICOM_INTR_LSR] = TI3410_UART_LSR;
		intrinX[UTICOM_INTR_MSR] = TI3410_UART_MSR;
		intrinX[UTICOM_INTR_STATUS] = 0;
		if ((TI3410_DMACDR3 & TI3410_DMACSR3_TXFT) &&
		    (TI3410_DMACDR3 & TI3410_DMACSR3_OVRUN))
			intrinX[UTICOM_INTR_STATUS]
				|= UTICOM_INTR_STATUS_OVERRUN;
	} else {
		/* setup (and queue) the interrupt information */
		intrinY[UTICOM_INTR_LSR] = TI3410_UART_LSR;
		intrinY[UTICOM_INTR_MSR] = TI3410_UART_MSR;
		intrinY[UTICOM_INTR_STATUS] = 0;
		if ((TI3410_DMACDR3 & TI3410_DMACSR3_TXFT) &&
		    (TI3410_DMACDR3 & TI3410_DMACSR3_OVRUN))
			intrinY[UTICOM_INTR_STATUS]
				|= UTICOM_INTR_STATUS_OVERRUN;
	}

	intr_ready = 1;

	/* interrupt buffer available so send immediately */
	if ((TI3410_IEPBCTX_2 & TI3410_EPBCT_NAK)
	     && (TI3410_IEPBCTY_2 & TI3410_EPBCT_NAK))
		do_usb_interrupt();
}

void
reset_interrupt(void)
{
	intr_ready = 0;
	intr_toggle = 0;
}

static void
restart_dma1(void)
{
	/*
	 * only restart dma1 if the bulk output endpoint is enabled
	 * and if dma1 is not already enabled
	 */
	if ((TI3410_OEPCNF_1 & TI3410_EPCNF_UBME) &&
	    (TI3410_DMACDR1 & TI3410_DMACDR1_EN) == 0) {
		/* clear DMA status */
		TI3410_DMACSR1 = TI3410_DMACSR1;
		TI3410_DMACDR1 |= TI3410_DMACDR1_EN;
	}
}

static void
restart_dma3(void)
{
	/*
	 * only restart dma3 if the bulk input endpoint is enabled
	 * and if dma3 is not already enabled
	 */
	if ((TI3410_IEPCNF_1 & TI3410_EPCNF_UBME) &&
	    (TI3410_DMACDR3 & TI3410_DMACDR3_EN) == 0) {
		/* clear DMA status */
		TI3410_DMACSR3 = TI3410_DMACSR3;
		TI3410_DMACDR3 |= TI3410_DMACDR3_EN;
	}
}

void
external0_isr(void) __interrupt (0)
{
	while (TI3410_VECINT != 0) {
		/* dispatch on vecint */
		switch (TI3410_VECINT) {
		case TI3410_VECINT_IE0:
			/* continue sending setup data */
			do_control_read_data();
			break;
		case TI3410_VECINT_OE0:
			/* ack for the status stage after sending setup data */
			TI3410_OEPBCNT_0 = 0;
			break;
		case TI3410_VECINT_IE1:
			restart_dma3();
			break;
		case TI3410_VECINT_OE1:
			restart_dma1();
			break;
		case TI3410_VECINT_IE2:
			do_usb_interrupt();
			break;
		case TI3410_VECINT_SETUP:
			handle_setup_intr();
			break;
		case TI3410_VECINT_UART_STATUS:
			usb_interrupt();
			break;
		case TI3410_VECINT_UART_MODEM:
			usb_interrupt();
			break;
#if 0
		case TI3410_VECINT_UART_RXF:
		case TI3410_VECINT_UART_TXE:
			if (TI3410_UART_LSR != 0)
				usb_interrupt();
			break;
#endif
		case TI3410_VECINT_DMA3:
			/* buffer overrun: notify and stall */
			if ((TI3410_DMACDR3 & TI3410_DMACSR3_TXFT) &&
			    (TI3410_DMACDR3 & TI3410_DMACSR3_OVRUN)) {
				TI3410_IEPCNF_1 |= TI3410_EPCNF_STALL;
				usb_interrupt();
			}
			if (TI3410_DMACDR3 & TI3410_DMACDR3_XY) {
				if (TI3410_IEPBCTY_1 & TI3410_EPBCT_NAK)
					restart_dma3();
			} else {
				if (TI3410_IEPBCTX_1 & TI3410_EPBCT_NAK)
					restart_dma3();
			}
			break;
#if 0 /* not handled */
		case TI3410_VECINT_OE2:
		case TI3410_VECINT_IE3:
		case TI3410_VECINT_OE3:
		case TI3410_VECINT_RESR:
		case TI3410_VECINT_SUSR:
		case TI3410_VECINT_RSTR:
		case TI3410_VECINT_I2CTXE:
		case TI3410_VECINT_I2CRXF:
		case TI3410_VECINT_DMA1:  /* Restarted at TI3410_VECINT_OE1 */
		case TI3410_VECINT_STPOW: /* checked in handle_setup_intr   */
#endif
		default:
			break;
 		}

		/* check for the next interrupt event */
 		TI3410_VECINT = TI3410_VECINT;
	}
} 

void timer0_isr(void)		__interrupt (1) { } 
void external1_isr(void)	__interrupt (2) { } 
void timer1_isr(void)		__interrupt (3) { } 
void serial_isr(void)		__interrupt (4) { } 
