/*-
 * Copyright (c) 2010 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Yorick Hardy
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdint.h>
#include <stdlib.h>

#include "interrupts.h"
#include "requests.h"
#include "tusb3410.h"

#define MIN(x,y) (((x)<(y))?(x):(y))

/* variables for setup data to be sent to the host */
static volatile uint8_t *read_data = NULL;
static volatile uint8_t read_data_size = 0, end_with_short_pkt = 0;
static volatile	read_packet_size = 0;

/* current configuration, 0 == not configured */
static volatile uint8_t configuration = 0;

/* variables to store setup request information */
static volatile uint8_t bmRequestType, bRequest;
static volatile uint8_t wValueLo, wValueHi;
static volatile uint8_t wIndexLo, wIndexHi;
static volatile uint8_t wLengthLo, wLengthHi;

/* used in a few of the status requests */
static volatile uint8_t status[2];

static __code uint8_t configuration_descriptor[46] = {
	/* Configuration Descriptor */
	0x09,		/* Length: 9 */
	0x02,		/* Type:  configuration descriptor */
	0x2e, 0x00,	/* Total Length: 9+9+7+7+7+7 = 46 */
	0x01,		/* Number of interfaces */
	0x01,		/* Configuration number */
	0x04,		/* Description */
	0x80,		/* Attributes */
	0x32,		/* Max power consumption 100mA */
	/* Interface Descriptor */
		0x09,		/* Length: 9 */
		0x04,		/* Type: Interface descriptor */
		0x00,		/* Interface number */
		0x00,		/* Alternate setting */
		0x03,		/* Number of endpoints */
		0x02,		/* Interface class: communications interface */
		0x01,		/* Interface sub class: direct line */
		0x00,		/* Interface protocol: not specific */
		0x05,		/* Interface description */
#if 0
	/* Endpoint Descriptor */
		0x07,		/* Length: 7 */
		0x05,		/* Type: endpoint descriptor */
		0x00,		/* Endpoint 0 */
		0x00,		/* Attribute: control */
		0x08, 0x00,	/* Maximum packet size: 8 */
		0x00,		/* Interval */
#endif
	/* Endpoint Descriptor */
		0x07,		/* Length: 7 */
		0x05,		/* Type: endpoint descriptor */
		0x81,		/* Endpoint 1: input */
		0x02,		/* Attribute: bulk */
		0x40, 0x00,	/* Maximum packet size: 64 */
		0x05,		/* Interval */
	/* Endpoint Descriptor */
		0x07,		/* Length: 7 */
		0x05,		/* Type: endpoint descriptor */
		0x01,		/* Endpoint 1: output */
		0x02,		/* Attribute: bulk */
		0x40, 0x00,	/* Maximum packet size: 64 */
		0x05,		/* Interval */
	/* Endpoint Descriptor */
		0x07,		/* Length: 7 */
		0x05,		/* Type: endpoint descriptor */
		0x82,		/* Endpoint 2: input */
		0x03,		/* Attribute: interrupt */
		INTR_BUFSIZE, 0x00,	/* Maximum packet size */
		0x00		/* Interval */
};

static __code uint8_t device_descriptor[18] = {
	0x12,		/* Length: 18 */
	0x01,		/* Type: device descriptor */
	0x01, 0x10,	/* USB 1.1 */
	0x02,		/* Class: Communications device */
	0x00,		/* SubClass */
	0x00,		/* Protocol */
	0x08,		/* Maximum packet size for endpoint 0 */
	0x04, 0x51,	/* Vendor ID: Texas Instruments */
	0x34, 0x10,	/* Product ID: TIUSB3410 */
	TI3410_FIRMWARE_VERSION_MAJOR,
	TI3410_FIRMWARE_VERSION_MINOR,
	0x01,		/* Manufacturer */
	0x02,		/* Product */
	0x03,		/* Serial Number */
	0x02		/* Number of configurations */
};

static __code uint8_t str_desc_lang[4] = {
	0x04,		/* Length: 1 language + 2 */
	0x03,		/* Type: String descriptor */
	0x04, 0x09	/* Language ID: English */
};

static __code uint8_t str_desc_mfact[36] = {
	0x24,		/* Length: 34 + 2 */
	0x03,		/* Type: String descriptor */
	'T',0x00, 'e',0x00, 'x',0x00, 'a',0x00, 's',0x00, ' ',0x00,
	'I',0x00, 'n',0x00, 's',0x00, 't',0x00, 'r',0x00, 'u',0x00,
	'm',0x00, 'e',0x00, 'n',0x00, 't',0x00, 's',0x00
};

static __code uint8_t str_desc_prod[20] = {
	0x14,		/* Length: 18 + 2 */
	0x03,		/* Type: String descriptor */
	'T',0x00, 'I',0x00, 'U',0x00, 'S',0x00, 'B',0x00, '3',0x00,
	'4',0x00, '1',0x00, '0',0x00
};

static uint8_t str_desc_serial[34] = {
	0x22,		/* Length: 32 + 2 */
	0x03,		/* Type: String descriptor */
	'0',0x00, '0',0x00, '0',0x00, '0',0x00, '0',0x00, '0',0x00,
	'0',0x00, '0',0x00, '0',0x00, '0',0x00, '0',0x00, '0',0x00,
	'0',0x00, '0',0x00, '0',0x00, '0',0x00
};

static __code uint8_t str_desc_conf[32] = {
	0x20,		/* Length: 30 + 2 */
	0x03,		/* Type: String descriptor */
	'N',0x00, 'e',0x00, 't',0x00, 'B',0x00, 'S',0x00, 'D',0x00,
	' ',0x00, 'F',0x00, 'i',0x00, 'r',0x00, 'm',0x00, 'w',0x00,
	'a',0x00, 'r',0x00, 'e',0x00,
};

static __code uint8_t str_desc_iface[52] = {
	0x34,		/* Length: 50 + 2 */
	0x03,		/* Type: String descriptor */
	'N',0x00, 'e',0x00, 't',0x00, 'B',0x00, 'S',0x00, 'D',0x00,
	' ',0x00, 'F',0x00, 'i',0x00, 'r',0x00, 'm',0x00, 'w',0x00,
	'a',0x00, 'r',0x00, 'e',0x00, ' ',0x00, 'I',0x00, 'n',0x00,
	't',0x00, 'e',0x00, 'r',0x00, 'f',0x00, 'a',0x00, 'c',0x00,
	'e',0x00
};

static void control_read_data(uint8_t*, uint8_t);
static void control_read_zero_packet(void);
static int8_t clear_endpoint_feature(void);
static int8_t set_endpoint_feature(void);
static int8_t send_configuration(void);
static int8_t send_interface(void);
static int8_t send_descriptor(void);
static int8_t send_device_status(void);
static int8_t send_interface_status(void);
static int8_t send_endpoint_status(void);
static int8_t set_address(void);
static int8_t set_configuration(void);
static uint8_t ti3410_set_dtr(void);
static uint8_t ti3410_set_rts(void);
static uint8_t ti3410_set_break(void);
static uint8_t ti3410_set_crtscts(void);
static uint8_t ti3410_set_dlldlh(void);
static uint8_t ti3410_set_databits(void);
static uint8_t ti3410_set_stopbits(void);
static uint8_t ti3410_set_parity(void);
static uint8_t ti3410_set_parity_odd(void);
static uint8_t ti3410_set_param(void);
static uint8_t ti3410_reset_bulk(void);

void
unconfigure(void)
{
	configuration = 0;

	/* disable DMA */
	TI3410_DMACDR1 = 0;
	TI3410_DMACSR1 = 0;
	TI3410_DMACDR3 = 0;
	TI3410_DMACSR3 = 0;

	/* disable endpoints */
	TI3410_IEPCNF_1 = 0;
	TI3410_IEPBCTX_1 = TI3410_EPBCT_NAK;
	TI3410_IEPBCTY_1 = TI3410_EPBCT_NAK;
	TI3410_OEPCNF_1 = 0;
	TI3410_OEPBCTX_1 = TI3410_EPBCT_NAK;
	TI3410_OEPBCTY_1 = TI3410_EPBCT_NAK;
	TI3410_IEPCNF_2 = 0;
	TI3410_IEPBCTX_2 = TI3410_EPBCT_NAK;
	TI3410_IEPBCTY_2 = TI3410_EPBCT_NAK;
	TI3410_OEPCNF_2 = 0;
	TI3410_OEPBCTX_2 = TI3410_EPBCT_NAK;
	TI3410_OEPBCTY_2 = TI3410_EPBCT_NAK;
	TI3410_IEPCNF_3 = 0;
	TI3410_IEPBCTX_3 = TI3410_EPBCT_NAK;
	TI3410_IEPBCTY_3 = TI3410_EPBCT_NAK;
	TI3410_OEPCNF_3 = 0;
	TI3410_OEPBCTX_3 = TI3410_EPBCT_NAK;
	TI3410_OEPBCTY_3 = TI3410_EPBCT_NAK;

	/* reset UART */
	TI3410_UART_MCR |= TI3410_UART_MCR_URST;
	/* wait for the reset to complete */
	while (TI3410_UART_MCR & TI3410_UART_MCR_URST);

	/* clear TI3410 UART interrupt mask */
	TI3410_UART_MASK = 0;

	/* setup UART */
	TI3410_UART_FCRL = 0;
	TI3410_UART_LCR = 0;
	TI3410_UART_MCR = 0;
}

void
configure(void)
{
	configuration = TI3410_ACTIVE_CONFIGURATION;

	/* setup endpoints */

	/* bulk in endpoint 1 */
	TI3410_IEPBBAX_1 = BULKINLOC_X_C;
	TI3410_IEPBBAY_1 = BULKINLOC_Y_C;
	TI3410_IEPSIZXY_1 = 64;
	TI3410_IEPBCTX_1 = TI3410_EPBCT_NAK;
	TI3410_IEPBCTY_1 = TI3410_EPBCT_NAK;
	TI3410_IEPCNF_1 = TI3410_EPCNF_USBIE | TI3410_EPCNF_DBUF
	                | TI3410_EPCNF_UBME;
	/* bulk out endpoint 1 */
	TI3410_OEPBBAX_1 = BULKOUTLOC_X_C;
	TI3410_OEPBBAY_1 = BULKOUTLOC_Y_C;
	TI3410_OEPSIZXY_1 = 64;
	TI3410_OEPBCTX_1 = 0;
	TI3410_OEPBCTY_1 = 0;
	TI3410_OEPCNF_1 = TI3410_EPCNF_USBIE | TI3410_EPCNF_DBUF
	                | TI3410_EPCNF_UBME;

	/* interrupt (host in) endpoint 2 */
	TI3410_IEPBBAX_2 = 0; /* initialized on send */
	TI3410_IEPBBAY_2 = 0;
	TI3410_IEPSIZXY_2 = INTR_BUFSIZE;
	TI3410_IEPBCTX_2 = TI3410_EPBCT_NAK;
	TI3410_IEPBCTY_2 = TI3410_EPBCT_NAK;
	TI3410_IEPCNF_2 = TI3410_EPCNF_USBIE | TI3410_EPCNF_UBME;

	/* output endpoint 2 in unused */
	TI3410_OEPCNF_2 = 0;
	TI3410_OEPBCTX_2 = TI3410_EPBCT_NAK;
	TI3410_OEPBCTY_2 = TI3410_EPBCT_NAK;
	/* endpoint 3 is unused */
	TI3410_IEPCNF_3 = 0;
	TI3410_IEPBCTX_3 = TI3410_EPBCT_NAK;
	TI3410_IEPBCTY_3 = TI3410_EPBCT_NAK;
	TI3410_OEPCNF_3 = 0;
	TI3410_OEPBCTX_3 = TI3410_EPBCT_NAK;
	TI3410_OEPBCTY_3 = TI3410_EPBCT_NAK;

	/* reset UART */
	TI3410_UART_MCR |= TI3410_UART_MCR_URST;
	/* wait for the reset to complete */
	while (TI3410_UART_MCR & TI3410_UART_MCR_URST);

	/* setup UART */
	TI3410_UART_FCRL = 0;
	TI3410_UART_LCR = 0;
	TI3410_UART_MCR = 0;

#ifdef INTERRUPT_ON_LSR
	usb_interrupt(TI3410_INTR_LSR, &TI3410_UART_LSR, 1);
#endif
	/* enable (reset) the FIFO */
	TI3410_UART_LCR = TI3410_UART_LCR_FEN;
	/* if the LSR has any bits set, which stop DMA3, clear them */
	TI3410_UART_LSR = TI3410_UART_LSR;
	/* let the driver know the current MSR */
	usb_interrupt(TI3410_INTR_MSR, &TI3410_UART_MSR, 1);
	/* if the MSR has any bits set clear them */
	TI3410_UART_MSR = TI3410_UART_MSR;

	/* set TI3410 UART interrupt mask */
	TI3410_UART_MASK = TI3410_UART_MASK_MIE | TI3410_UART_MASK_SIE |
			   TI3410_UART_MASK_TRI;

	/* setup DMA */
	TI3410_DMACSR1 = 0;
	TI3410_DMACDR1 = 1 | TI3410_DMACDR1_CNT | TI3410_DMACDR1_EN
                       | TI3410_DMACDR1_INE;
	/* 31ms (maximum) timeout on read */
	TI3410_DMACSR3 = TI3410_DMACSR3_TEN | (31<<2);
	TI3410_DMACDR3 = 1 | TI3410_DMACDR3_CNT | TI3410_DMACDR3_EN
                       | TI3410_DMACDR3_INE;
}

void
do_control_read_data(void)
{
	uint8_t i;

	/* busy preparing data */
	TI3410_IEPBCNT_0 |= TI3410_EPBCT_NAK;

	/*
	 * if there is no more data to send and the last send was
	 * not a full packet then we already sent the last short packet
	 */
	if (read_data_size == 0 && read_packet_size != 8 && end_with_short_pkt) {
		TI3410_IEPCNFG_0 |= TI3410_EPCNF_STALL;
		return;
	}

	/* fill the buffer */
	read_packet_size = (read_data_size>8) ? 8 : read_data_size;
	for(i = 0; i != read_packet_size; ++i)
		TI3410_IEP_0_BUF[i] = *(read_data++);

	/* set size and ack */
	TI3410_IEPBCNT_0 = read_packet_size;
	read_data_size -= read_packet_size;

	/* accept the OUT packet for the status stage */
	if (read_data_size == 0) TI3410_OEPBCNT_0 = 0;
}

static void
control_read_data(uint8_t *dta, uint8_t size)
{
	read_data = dta;
	read_data_size = size;
	/*
	 * determine if a zero packet may be sent
	 * i.e. the host asked for more data than
	 * the firmware will send, or the firmware
	 * explicitly sends a zero sized packet.
	 */
	if (wLengthHi != 0 || wLengthLo > size || size == 0)
		end_with_short_pkt = 1;
	else
		end_with_short_pkt = 0;
	do_control_read_data();
}

static void
control_read_zero_packet(void)
{
	read_packet_size = 8; /* forces the zero packet */
	control_read_data((uint8_t *)NULL, 0);
}

static int8_t
clear_endpoint_feature(void)
{
	if (configuration != TI3410_ACTIVE_CONFIGURATION)
		return 0;

	/* bulk in endpoint 1 */
	if (wIndexLo == 0x81 && wIndexHi == 0
	   && wValueLo == 0 && wValueHi == 0) { /* HALT feature */
		/* halt the DMA */
		TI3410_DMACDR3 &= ~TI3410_DMACDR3_INE;
		TI3410_DMACDR3 &= ~TI3410_DMACDR3_EN;
		/* set the buffers to NAK */
		TI3410_IEPBCTX_1 = TI3410_EPBCT_NAK;
		TI3410_IEPBCTY_1 = TI3410_EPBCT_NAK;
		/* reset the X/Y buffer selection bit, the T/R bit must be 0 */
		TI3410_DMACDR3 &= ~(TI3410_DMACDR3_XY | TI3410_DMACDR3_TR);
		/* disable the FIFO */
		TI3410_UART_LCR &= ~TI3410_UART_LCR_FEN;
		/* enable (reset) the FIFO */
		TI3410_UART_LCR |= TI3410_UART_LCR_FEN;
		/* reset the data toggle and STALL */
		TI3410_IEPCNF_1 &= ~(TI3410_EPCNF_TOGLE | TI3410_EPCNF_STALL);
		/* start the DMA */
		TI3410_DMACDR3 |= TI3410_DMACDR3_INE;
		TI3410_DMACDR3 |= TI3410_DMACDR3_EN;
		control_read_zero_packet();
		return 1;
	}

	/* bulk out endpoint 1 */
	if (wIndexLo == 0x01 && wIndexHi == 0
	   && wValueLo == 0 && wValueHi == 0) { /* HALT feature */
		/* halt the DMA */
		TI3410_DMACDR1 &= ~TI3410_DMACDR1_INE;
		TI3410_DMACDR1 &= ~TI3410_DMACDR1_EN;
		/* set the buffers to NAK */
		TI3410_OEPBCTX_1 = TI3410_EPBCT_NAK;
		TI3410_OEPBCTY_1 = TI3410_EPBCT_NAK;
		/* reset the X/Y buffer selection bit, the T/R bit must be 0 */
		TI3410_DMACDR1 &= ~(TI3410_DMACDR1_XY | TI3410_DMACDR1_TR);
		/* reset the data toggle and STALL */
		TI3410_OEPCNF_1 &= ~(TI3410_EPCNF_TOGLE | TI3410_EPCNF_STALL);
		/* reset the buffers to ACK */
		TI3410_OEPBCTX_1 = 0;
		TI3410_OEPBCTY_1 = 0;
		/* start the DMA */
		TI3410_DMACDR1 |= TI3410_DMACDR1_INE;
		TI3410_DMACDR1 |= TI3410_DMACDR1_EN;
		control_read_zero_packet();
		return 1;
	}

	/* interrupt endpoint 2 */
	if (wIndexLo == 0x82 && wIndexHi == 0
	   && wValueLo == 0 && wValueHi == 0) { /* HALT feature */
		/* reset the data toggle */
		TI3410_IEPCNF_2 &= ~TI3410_EPCNF_TOGLE;
		/* clear STALL condition */
		TI3410_IEPCNF_2 &= ~TI3410_EPCNF_STALL;
		control_read_zero_packet();
		return 1;
	}

	return 0;
}

static int8_t
set_endpoint_feature(void)
{
	if (configuration != TI3410_ACTIVE_CONFIGURATION)
		return 0;

	/* bulk in endpoint 1 */
	if (wIndexLo == 0x81 && wIndexHi == 0
	   && wValueLo == 0 && wValueHi == 0) { /* HALT feature */
		/* set STALL condition */
		TI3410_IEPCNF_1 |= TI3410_EPCNF_STALL;
		control_read_zero_packet();
		return 1;
	}

	/* bulk out endpoint 1 */
	if (wIndexLo == 0x01 && wIndexHi == 0
	   && wValueLo == 0 && wValueHi == 0) { /* HALT feature */
		/* set STALL condition */
		TI3410_OEPCNF_1 |= TI3410_EPCNF_STALL;
		control_read_zero_packet();
		return 1;
	}

	/* interrupt endpoint 2 */
	if (wIndexLo == 0x82 && wIndexHi == 0
	   && wValueLo == 0 && wValueHi == 0) { /* HALT feature */
		/* set STALL condition */
		TI3410_IEPCNF_2 |= TI3410_EPCNF_STALL;
		control_read_zero_packet();
		return 1;
	}

	return 0;
}

static int8_t
send_configuration(void)
{
	control_read_data(&configuration, 1);
	return 1;
}

static int8_t
send_interface(void)
{
	if (configuration == 1 && wIndexLo == 0 && wIndexHi == 0) {
		static uint8_t i = 0;
		control_read_data(&i, 1);
		return 1;
	}

	return 0;
}

static int8_t
send_descriptor(void)
{
	uint8_t *descr;
	uint8_t *descr_end;
	uint8_t index = wValueLo, sendlen, len;

	/*
	 *  this firmware currently never sends more than 255 bytes
	 *  of information, so ignore the high byte
	 */
	if (wLengthHi != 0)
		len = 0xff;
	else
		len = wLengthLo;

	switch (wValueHi) {
	case 0x01: /* device */
		if (index == 0) {
			sendlen = MIN(len, device_descriptor[0]);
			control_read_data(device_descriptor, sendlen);
			return 1;
		}
		break;
	case 0x02: /* configuration */
		if (index == 1) {
			sendlen = MIN(len, configuration_descriptor[2]);
			control_read_data(configuration_descriptor, sendlen);
			return 1;
		}
		break;
	case 0x03: /* string */
		if ((wIndexLo == 0 && wIndexHi == 0) /* all languages */
		   || (wIndexLo == 0x09 && wIndexHi == 0x04)) { /* english */
			switch (index) {
			case 0: sendlen = MIN(len, str_desc_lang[0]);
				control_read_data(str_desc_lang, sendlen);
				return 1;
			case 1: sendlen = MIN(len, str_desc_mfact[0]);
				control_read_data(str_desc_mfact, sendlen);
				return 1;
			case 2: sendlen = MIN(len, str_desc_prod[0]);
				control_read_data(str_desc_prod, sendlen);
				return 1;
			case 3: sendlen = MIN(len, str_desc_serial[0]);
				control_read_data(str_desc_serial, sendlen);
				return 1;
			case 4: sendlen = MIN(len, str_desc_conf[0]);
				control_read_data(str_desc_conf, sendlen);
				return 1;
			case 5: sendlen = MIN(len, str_desc_iface[0]);
				control_read_data(str_desc_iface, sendlen);
				return 1;
			}
		}
		break;
	case 0x04: /* interface */
	case 0x05: /* endpoint */
		descr = configuration_descriptor;
		descr_end = descr + sizeof(configuration_descriptor);
		while (descr < descr_end
		      && (descr[1] != wValueHi || descr[2] != index))
			descr += descr[0];
		if (descr < descr_end) {
			sendlen = MIN(len, descr[0]);
			control_read_data(descr, sendlen);
			return 1;
		}
		break;
	case 0x06: /* device qualifier */
		break;
	case 0x07: /* other speed */
		break;
	case 0x08: /* power */
		break;
	default:
		break;
	}

	return 0;
}

static int8_t
send_device_status(void)
{
	if (configuration == TI3410_ACTIVE_CONFIGURATION) {
		/* no remote wakeup or self-powered */
		status[0] = 0; status[1] = 0;
		control_read_data(status, 2);
		return 1;
	}

	return 0;
}

static int8_t
send_interface_status(void)
{
	/* always zero */
	status[0] = 0; status[1] = 0;
	control_read_data(status, 2);

	return 1;
}

static int8_t
send_endpoint_status(void)
{
	if (configuration != TI3410_ACTIVE_CONFIGURATION)
		return 0;

	status[0] = 0; status[1] = 0;
	/* bulk in endpoint 1 */
	if (wIndexLo == 0x81 && wIndexHi == 0) {
		if (TI3410_IEPCNF_1 & TI3410_EPCNF_STALL)
			status[0] = 1; /* HALT feature enabled */
			control_read_data(status, 2);
			return 1;
	}

	/* bulk out endpoint 1 */
	if (wIndexLo == 0x01 && wIndexHi == 0) {
		if (TI3410_OEPCNF_1 & TI3410_EPCNF_STALL)
			status[0] = 1; /* HALT feature enabled */
			control_read_data(status, 2);
			return 1;
	}

	/* interrupt endpoint 2 */
	if (wIndexLo == 0x82 && wIndexHi == 0) {
		if (TI3410_IEPCNF_2 & TI3410_EPCNF_STALL)
			status[0] = 1; /* HALT feature enabled */
			control_read_data(status, 2);
			return 1;
	}

	return 0;
}

static int8_t
set_address(void)
{
	/* XXX handle addr = 0 => default state pg 256/257 of USB 2.0 */
	if (configuration == 0 && wValueLo <= 127 && wValueHi == 0) {
		TI3410_FUNADR = wValueLo;
		control_read_zero_packet();
		return 1;
	}

	return 0;
}

static int8_t
set_configuration(void)
{
	if (wValueHi != 0) return 0;

	switch (wValueLo) {
	case 0:	unconfigure(); return 1;
	case 1: configure(); return 1;
	default:
		break;
	}

	return 0;
}

static uint8_t
ti3410_set_dtr(void)
{
	if (configuration != TI3410_ACTIVE_CONFIGURATION)
		return 0;

	if (wIndexHi != 0 || wIndexLo != 0)
		return 0;
	if (wValueHi != 0 || wValueLo != 0)
		TI3410_UART_MCR |= TI3410_UART_MCR_DTR;
	else
		TI3410_UART_MCR &= ~TI3410_UART_MCR_DTR;
	control_read_zero_packet();
	return 1;
}

static uint8_t
ti3410_set_rts(void)
{
	if (configuration != TI3410_ACTIVE_CONFIGURATION)
		return 0;

	if (wIndexHi != 0 || wIndexLo != 0)
		return 0;
	if (wValueHi != 0 || wValueLo != 0)
		TI3410_UART_MCR |= TI3410_UART_MCR_RTS;
	else
		TI3410_UART_MCR &= ~TI3410_UART_MCR_RTS;
	control_read_zero_packet();
	return 1;
}

static uint8_t
ti3410_set_break(void)
{
	if (configuration != TI3410_ACTIVE_CONFIGURATION)
		return 0;

	if (wIndexHi != 0 || wIndexLo != 0)
		return 0;
	if (wValueHi != 0 || wValueLo != 0)
		TI3410_UART_LCR |= TI3410_UART_LCR_BRK;
	else
		TI3410_UART_LCR &= ~TI3410_UART_LCR_BRK;
	control_read_zero_packet();
	return 1;
}

static uint8_t
ti3410_set_crtscts(void)
{
	if (configuration != TI3410_ACTIVE_CONFIGURATION)
		return 0;

	if (wIndexHi != 0 || wIndexLo != 0)
		return 0;
	if (wValueHi != 0 || wValueLo != 0)
		TI3410_UART_FCRL |= TI3410_UART_FCRL_RTSCTS;
	else
		TI3410_UART_FCRL &= ~TI3410_UART_FCRL_RTSCTS;
	control_read_zero_packet();
	return 1;
}

static uint8_t
ti3410_set_dlldlh(void)
{
	if (configuration != TI3410_ACTIVE_CONFIGURATION)
		return 0;

	if (wIndexHi != 0 || wIndexLo != 0)
		return 0;
	TI3410_UART_DLL = wValueLo;
	TI3410_UART_DLH = wValueHi;
	control_read_zero_packet();
	return 1;
}

static uint8_t
ti3410_set_databits(void)
{
	if (configuration != TI3410_ACTIVE_CONFIGURATION)
		return 0;

	if (wIndexHi != 0 || wIndexLo != 0 || wValueHi != 0)
		return 0;
	switch (wValueLo) {
	case 5:	TI3410_UART_LCR &= TI3410_UART_LCR_WLMASK;
		TI3410_UART_LCR |= TI3410_UART_LCR_WL5;
		break;
	case 6:	TI3410_UART_LCR &= TI3410_UART_LCR_WLMASK;
		TI3410_UART_LCR |= TI3410_UART_LCR_WL6;
		break;
	case 7:	TI3410_UART_LCR &= TI3410_UART_LCR_WLMASK;
		TI3410_UART_LCR |= TI3410_UART_LCR_WL7;
		break;
	case 8:	TI3410_UART_LCR &= TI3410_UART_LCR_WLMASK;
		TI3410_UART_LCR |= TI3410_UART_LCR_WL8;
		break;
	default:
		return 0;
	}
	control_read_zero_packet();
	return 1;
}

static uint8_t
ti3410_set_stopbits(void)
{
	if (configuration != TI3410_ACTIVE_CONFIGURATION)
		return 0;

	if (wIndexHi != 0 || wIndexLo != 0)
		return 0;
	if (wValueHi != 0 || wValueLo != 0)
		TI3410_UART_LCR |= TI3410_UART_LCR_STP;
	else
		TI3410_UART_LCR &= ~TI3410_UART_LCR_STP;
	control_read_zero_packet();
	return 1;
}

static uint8_t
ti3410_set_parity(void)
{
	if (configuration != TI3410_ACTIVE_CONFIGURATION)
		return 0;

	if (wIndexHi != 0 || wIndexLo != 0)
		return 0;
	if (wValueHi != 0 || wValueLo != 0)
		TI3410_UART_LCR |= TI3410_UART_LCR_PRTY;
	else
		TI3410_UART_LCR &= ~TI3410_UART_LCR_PRTY;
	control_read_zero_packet();
	return 1;
}

static uint8_t
ti3410_set_parity_odd(void)
{
	if (configuration != TI3410_ACTIVE_CONFIGURATION)
		return 0;

	if (wIndexHi != 0 || wIndexLo != 0)
		return 0;
	if (wValueHi != 0 || wValueLo != 0)
		TI3410_UART_LCR &= ~TI3410_UART_LCR_EPRTY; /* odd parity */
	else
		TI3410_UART_LCR |= TI3410_UART_LCR_EPRTY;  /* even parity */
	control_read_zero_packet();
	return 1;
}

static uint8_t
ti3410_set_param(void)
{
	if (configuration != TI3410_ACTIVE_CONFIGURATION)
		return 0;

	if (wIndexHi != 0 || wIndexLo != 0)
		return 0;
	/*
	 * force the FIFO (and RDR register) to be enabled
	 * reads will not work without it
	 */
	TI3410_UART_LCR = wValueLo | TI3410_UART_LCR_FEN;
	TI3410_UART_FCRL = 0;
	if (wValueHi & TI3410_PARAM_CRTSCTS)
		TI3410_UART_FCRL |= TI3410_UART_FCRL_RTSCTS;
	else
		TI3410_UART_FCRL &= ~TI3410_UART_FCRL_RTSCTS;
	if (wValueHi & TI3410_PARAM_CDTRCTS)
		TI3410_UART_FCRL |= TI3410_UART_FCRL_DTRCTS;
	else
		TI3410_UART_FCRL &= ~TI3410_UART_FCRL_DTRCTS;
	control_read_zero_packet();
	return 1;
}

void
handle_setup_intr(void)
{
	int8_t handled = 0;

	/* servicing interrupt */
	TI3410_USBCTL |= TI3410_USBCTL_SIR;

stpow:
	do {
		TI3410_USBSTA |= TI3410_USBSTA_STPOW;
		bmRequestType = TI3410_setup_bmRequestType;
		bRequest = TI3410_setup_bRequest;
		wValueLo = TI3410_setup_wValueLo;
		wValueHi = TI3410_setup_wValueHi;
		wIndexLo = TI3410_setup_wIndexLo;
		wIndexHi = TI3410_setup_wIndexHi;
		wLengthLo = TI3410_setup_wLengthLo;
		wLengthHi = TI3410_setup_wLengthHi;
	} while (TI3410_USBSTA & TI3410_USBSTA_STPOW);

	/* data direction */
	if (bmRequestType & UT_READ)
		TI3410_USBCTL |= TI3410_USBCTL_DIR;
	else
		TI3410_USBCTL &= ~TI3410_USBCTL_DIR;

	if (bmRequestType == UT_WRITE_ENDPOINT
	    && bRequest == UR_CLEAR_FEATURE
	    && wLengthLo == 0 && wLengthHi == 0)
		handled = clear_endpoint_feature();

	if (bmRequestType == UT_READ_DEVICE
	    && bRequest == UR_GET_CONFIG
	    && wValueLo == 0 && wValueHi == 0
	    && wIndexLo == 0 && wIndexHi == 0
	    && wLengthLo == 1 && wLengthHi == 0)
		handled = send_configuration();

	if (bmRequestType == UT_READ_DEVICE
	    && bRequest == UR_GET_DESCRIPTOR)
		handled = send_descriptor();

	if (bmRequestType == UT_READ_INTERFACE
	    && bRequest == UR_GET_INTERFACE
	    && wValueLo == 0 && wValueHi == 0
	    && wLengthLo == 1 && wLengthHi == 0)
		handled = send_interface();

	if (bmRequestType == UT_READ_DEVICE
	    && bRequest == UR_GET_STATUS
	    && wValueLo == 0 && wValueHi == 0
	    && wIndexLo == 0 && wIndexHi == 0
	    && wLengthLo == 2 && wLengthHi == 0)
		handled = send_device_status();

	if (bmRequestType == UT_READ_INTERFACE
	    && bRequest == UR_GET_STATUS
	    && wValueLo == 0 && wValueHi == 0
	    && wIndexLo == 0 && wIndexHi == 0
	    && wLengthLo == 2 && wLengthHi == 0)
		handled = send_interface_status();

	if (bmRequestType == UT_READ_INTERFACE
	    && bRequest == UR_GET_STATUS
	    && wValueLo == 0 && wValueHi == 0
	    && wIndexLo == 0 && wIndexHi == 0
	    && wLengthLo == 2 && wLengthHi == 0)
		handled = send_endpoint_status();

	if (bmRequestType == UT_WRITE_DEVICE
	    && bRequest == UR_SET_ADDRESS
	    && wIndexLo == 0 && wIndexHi == 0
	    && wLengthLo == 1 && wLengthHi == 0)
		handled = set_address();

	if (bmRequestType == UT_WRITE_DEVICE
	    && bRequest == UR_SET_CONFIG
	    && wIndexLo == 0 && wIndexHi == 0
	    && wLengthLo == 0 && wLengthHi == 0)
		handled = set_configuration();

	if (bmRequestType == UT_WRITE_ENDPOINT
	    && bRequest == UR_SET_FEATURE
	    && wLengthLo == 0 && wLengthHi == 0)
		handled = set_endpoint_feature();

	if (bmRequestType == UT_WRITE_VENDOR_INTERFACE
	    && bRequest == TI3410_RQ_SET_DTR
	    && wLengthLo == 0 && wLengthHi == 0)
		handled = ti3410_set_dtr();

	if (bmRequestType == UT_WRITE_VENDOR_INTERFACE
	    && bRequest == TI3410_RQ_SET_RTS
	    && wLengthLo == 0 && wLengthHi == 0)
		handled = ti3410_set_rts();

	if (bmRequestType == UT_WRITE_VENDOR_INTERFACE
	    && bRequest == TI3410_RQ_SET_BREAK
	    && wLengthLo == 0 && wLengthHi == 0)
		handled = ti3410_set_break();

	if (bmRequestType == UT_WRITE_VENDOR_INTERFACE
	    && bRequest == TI3410_RQ_SET_CRTSCTS
	    && wLengthLo == 0 && wLengthHi == 0)
		handled = ti3410_set_crtscts();

	if (bmRequestType == UT_WRITE_VENDOR_INTERFACE
	    && bRequest == TI3410_RQ_SET_DLLDLH
	    && wLengthLo == 0 && wLengthHi == 0)
		handled = ti3410_set_dlldlh();

	if (bmRequestType == UT_WRITE_VENDOR_INTERFACE
	    && bRequest == TI3410_RQ_SET_DATABITS
	    && wLengthLo == 0 && wLengthHi == 0)
		handled = ti3410_set_databits();

	if (bmRequestType == UT_WRITE_VENDOR_INTERFACE
	    && bRequest == TI3410_RQ_SET_STOPBITS
	    && wLengthLo == 0 && wLengthHi == 0)
		handled = ti3410_set_stopbits();

	if (bmRequestType == UT_WRITE_VENDOR_INTERFACE
	    && bRequest == TI3410_RQ_SET_PARITY
	    && wLengthLo == 0 && wLengthHi == 0)
		handled = ti3410_set_parity();

	if (bmRequestType == UT_WRITE_VENDOR_INTERFACE
	    && bRequest == TI3410_RQ_SET_PARITY_ODD
	    && wLengthLo == 0 && wLengthHi == 0)
		handled = ti3410_set_parity_odd();

	if (bmRequestType == UT_WRITE_VENDOR_INTERFACE
	    && bRequest == TI3410_RQ_SET_PARAM
	    && wLengthLo == 0 && wLengthHi == 0)
		handled = ti3410_set_param();

	if (TI3410_USBSTA & TI3410_USBSTA_STPOW)
		goto stpow;

	/* for all unhandled requests initiate a STALL handshake */
	if (!handled) {
		TI3410_IEPCNFG_0 |= TI3410_EPCNF_STALL;
	} else {
		/* clear count and ack for the status stage */
		TI3410_OEPBCNT_0 = 0;
	}

	/* done servicing interrupt */
	TI3410_USBCTL &= ~TI3410_USBCTL_SIR;

	/* setup transaction complete */
	TI3410_USBSTA |= TI3410_USBSTA_SETUP;
}

void
setup_descriptors(void)
{
	/* set up the serial number string */
#define FRSTHALF(b) ( (b) & 0x0f )
#define SCNDHALF(b) (((b) & 0xf0) >> 4)
#define SETSERIALNUM(i) \
	if (FRSTHALF(TI3410_SERNUM ## i) < 10) \
		str_desc_serial[4+4*i] = FRSTHALF(TI3410_SERNUM ## i) + '0'; \
	else \
		str_desc_serial[4+4*i] = FRSTHALF(TI3410_SERNUM ## i) + 'A'-10;\
	if (SCNDHALF(TI3410_SERNUM ## i) < 10) \
		str_desc_serial[2+4*i] = SCNDHALF(TI3410_SERNUM ## i) + '0'; \
	else \
		str_desc_serial[2+4*i] = SCNDHALF(TI3410_SERNUM ## i) + 'A'-10;
		
	SETSERIALNUM(0);
	SETSERIALNUM(1);
	SETSERIALNUM(2);
	SETSERIALNUM(3);
	SETSERIALNUM(4);
	SETSERIALNUM(5);
	SETSERIALNUM(6);
	SETSERIALNUM(7);
}
