/* **********************************************************
 * Copyright (C) 1998-2000 VMware, Inc.
 * All Rights Reserved
 * **********************************************************/
#ifdef VMX86_DEVEL
char rcsId_memtrack[] = "$Id: memtrack.c,v 1.3 2003/02/16 15:32:33 bad Exp $";
#else
#define FILECODE "F(304)"
#endif 

/*
 * memtrack.c --
 *
 *      Utility module for tracking memory allocated and/or 
 *      locked by the monitor. 
 *     
 *      
 */

#include "hostif.h"	/* Must come before any linux header files */

#ifdef linux
# include "driver-config.h" /* Versioned symbols from linux/string.h */
# include <linux/string.h> /* memset() in the kernel */
#elif __FreeBSD__
# include <sys/param.h>
# include <sys/systm.h>
# include <string.h>
#elif __NetBSD__
# include <sys/param.h>
# include <sys/systm.h>
#else
# undef PAGE_SIZE        /* Redefined in ntddk.h, and we use that defn. */
# undef PAGE_SHIFT
# include <ntddk.h>
#endif

#include "x86.h"
#include "vm_types.h"
#include "vm_assert.h"
#include "memtrack.h"
#include "vmx86.h"


/*
 * MemTrack - We track memory using a two-level page table
 * like structure. We allocate the first level in the granularity 
 * of pages and pack as many MemTrackEntires as possible in a page. 
 * We currently handle 4 gig's worth of pages.  -- edward
 */

#define MAX_TRACE_PAGE_DIR \
   CEILING(1 << (32 - PAGE_SHIFT), PAGE_SIZE / sizeof (MemTrackEntry))

#define HASH_TABLE_SIZE    16384
#define HASH_TABLE_ENTRIES_PER_PAGE (PAGE_SIZE /  (sizeof(void*)))
#define HASH_TABLE_PAGES   (HASH_TABLE_SIZE / HASH_TABLE_ENTRIES_PER_PAGE)


typedef struct MemTrack {
   int numEntriesPerPage;    /* Number of entry per page */
   int numPages;             /* Number of pages stored in the tracker */
   char *mapPages[MAX_TRACE_PAGE_DIR];
   MemTrackEntry **hashTablePages[HASH_TABLE_PAGES]; /* VPN to entry hashtable */
} MemTrack;


static INLINE MemTrackEntry **
HASH_VPN(MemTrack *memtrack,
         VPN vpn) 
{
   uint32 hash = vpn % HASH_TABLE_SIZE;
   int page = hash / HASH_TABLE_ENTRIES_PER_PAGE;
   int offset = hash % HASH_TABLE_ENTRIES_PER_PAGE;
   return memtrack->hashTablePages[page] + offset;
}


/*
 *----------------------------------------------------------------------
 *
 * MemTrack_Init --
 *
 *      Allocate and initialize the memory tracker.
 *
 * Results:
 *      Handle used to access the memtracker.
 *
 * Side effects:
 *      memory allocation.
 *
 *----------------------------------------------------------------------
 */

void *
MemTrack_Init()
{
   MemTrack *st;
   int i;
   st = HostIF_AllocKernelMem(sizeof(MemTrack), FALSE);

   if (st == NULL) { 
      Warning("MemTrack_Init failed\n");
      return NULL;
   }
   
   memset(st,0,sizeof(MemTrack));
   
   for (i=0;i<HASH_TABLE_PAGES;i++) { 
      VA page = (VA)HostIF_AllocPage(FALSE);
      if (page == 0) { 
         Warning("MemTrack_Init failed on hashTablPages %d\n",i);
         /* LEAK */
         return NULL;
      }
      if (page & (PAGE_SIZE-1)) { 
         Warning("MemTrack_Init unaligned 0x%08x\n", (uint32)st->hashTablePages[i]);
         /* LEAK */
         return NULL;
      }
      st->hashTablePages[i] = (MemTrackEntry **)page;
      memset((char*)page, 0, PAGE_SIZE);
      
   }
   
   ASSERT(sizeof (MemTrackEntry) <= PAGE_SIZE);
   st->numEntriesPerPage = PAGE_SIZE / sizeof (MemTrackEntry);
   st->numPages = 0;
   memset(st->mapPages, 0, sizeof(st->mapPages));
   return (void *) st;
}

/*
 *----------------------------------------------------------------------
 *
 * MemTrack_Add --
 *
 *      Add the specified address to the memory tracker. 
 *
 * Results:
 *      A pointer to the allocated element, or NULL on error.
 *
 * Side effects:
 *      memory allocation.
 *
 *----------------------------------------------------------------------
 */

MemTrackEntry *
MemTrack_Add(void *s,  /* memtracker handler */
             VPN vpn,  /* VPN of entry */
             MPN mpn)  /* MPN of entry */
{
  MemTrack *st = (MemTrack *) s;
  int ind = st->numPages;
  int dirPage = ind / st->numEntriesPerPage;
  int offset = ind % st->numEntriesPerPage;
  MemTrackEntry *ptr;
  MemTrackEntry **head;
  /*
   * Allocate a directory page if needed.
   */
  if (dirPage >= MAX_TRACE_PAGE_DIR) return NULL;

  if (st->mapPages[dirPage] == NULL) {
     st->mapPages[dirPage] = HostIF_AllocPage(FALSE);
     if (st->mapPages[dirPage] == NULL) return NULL;
  }

  /*
   * Store the page in the tracker 
   */
  ptr = (MemTrackEntry *) st->mapPages[dirPage] + offset;
  memset(ptr, 0, sizeof *ptr);
  ptr->vpn = vpn;  
  ptr->mpn = mpn;  
  ptr->hashChain = NULL;
  
  head = HASH_VPN(st,vpn);
  if (*head == NULL) {
     *head = ptr;
  } else {
     MemTrackEntry *nextPtr = *head;
     while (nextPtr->hashChain != NULL) {
        nextPtr = nextPtr->hashChain;
     }
     nextPtr->hashChain = ptr;
  }
  st->numPages++;
  
  return ptr;
}

/*
 *----------------------------------------------------------------------
 *
 * MemTrack_LookupVPN --
 *
 *      Lookup the specified address in the memory tracker. 
 *
 * Results:
 *      A pointer to the allocated element, or NULL if not there;
 *
 * Side effects:
 *      memory allocation.
 *
 *----------------------------------------------------------------------
 */

MemTrackEntry *
MemTrack_LookupVPN(void *s,  /* memtracker handler */
        VPN vpn)  /* Value to lookup */
{
  MemTrack *st = (MemTrack *) s;
  MemTrackEntry *nextPtr;
    
  nextPtr = *HASH_VPN(st,vpn);
  while (nextPtr != NULL) {
    if (nextPtr->vpn == vpn) {
      return nextPtr;
    }
    nextPtr = nextPtr->hashChain;
  }
  return NULL;
}


/*
 *----------------------------------------------------------------------
 *
 * MemTrack_Scan --
 *
 *      Scan the stored pages using a provide search function. 
 *
 * Results:
 *      The return result of the searchFunc
 *
 * Side effects:
 *      None
 *
 *----------------------------------------------------------------------
 */

void *
MemTrack_Scan(void *s,  /* Handle for memtracker */
        void *arg,  /* Argument to searchFunc */
        void *(*searchFunc)(void *arg, MemTrackEntry *ptr))
{
  MemTrack *st = (MemTrack *) s;
  int ind;
  int dirPage;
  MemTrackEntry *ptr;
  void *ret;

  /* 
   * Call searchFunc with every address stored in the tracker
   * or until it returns a non-zero value.
   */
  for (ind = 0; ind < st->numPages; ind++) {
    dirPage = ind / st->numEntriesPerPage;
    ptr = (MemTrackEntry *) st->mapPages[dirPage] + 
          (ind % st->numEntriesPerPage);
    ret = searchFunc(arg, ptr);
    if (ret) return ret;
  }
  return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * MemTrack_Cleanup --
 *
 *      Cleanup all resources allocated for the memtracker. For
 *      all pages in the tracker call the user provided free function.
 *
 * Results:
 *      Number of pages in the tracker. 
 *
 * Side effects:
 *      Memory free
 *
 *----------------------------------------------------------------------
 */

int 
MemTrack_Cleanup(void *s,   /* Mem tracker handle */
                 void (*CleanUp)(void *clientData,MemTrackEntry*),
                 void *clientData)   /* free functions. */
{
  MemTrack *st = (MemTrack *) s;
  int ind;
  int dirPage;
  MemTrackEntry *ptr;
  int count = 0;
  int i;

  for (ind = 0; ind < st->numPages; ind++) {
    dirPage = ind / st->numEntriesPerPage;
    ptr = (MemTrackEntry *) st->mapPages[dirPage] + 
          (ind % st->numEntriesPerPage);
    CleanUp(clientData,ptr);
    count++;
  }
  /*
   * Free any directory page number and then free the tracker
   * structure itself 
   */
  for (dirPage = 0; dirPage < MAX_TRACE_PAGE_DIR; dirPage++) {
     if (st->mapPages[dirPage]) { 
        HostIF_FreePage(st->mapPages[dirPage]);
        st->mapPages[dirPage] = NULL;
     }
  }
  
  for (i=0;i<HASH_TABLE_PAGES;i++) { 
     if (st->hashTablePages[i]) {
        HostIF_FreePage(st->hashTablePages[i]);
        st->hashTablePages[i] = NULL;
     }
  }

  HostIF_FreeKernelMem(st);
  return count;
}

