/*	$NetBSD: $	*/

/*
 * Copyright (c) 1995 Allen Briggs.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
/*
 * Device-specific routines for handling Nubus-based video cards.
 */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: $");

#include <sys/param.h>

#include <sys/device.h>
#include <sys/ioctl.h>
#include <sys/file.h>
#include <sys/malloc.h>
#include <sys/mman.h>
#include <sys/proc.h>
#include <sys/systm.h>

#include <machine/bus.h>
#include <machine/cpu.h>
#include <machine/grfioctl.h>
#include <machine/viareg.h>
#include <machine/video.h>

#include <mac68k/nubus/nubus.h>
#include <mac68k/dev/grfvar.h>
#include <dev/wsfb/genfbvar.h>

#include "opt_wsfb.h"
#include "opt_genfb.h"

struct nubus_genfb_softc {
	struct genfb_softc sc_gen;
	nubus_slot	sc_slot;

	bus_addr_t		sc_basepa;	/* base of video space */
	bus_addr_t		sc_fbofs;	/* offset to framebuffer */

	bus_space_tag_t		sc_tag;
	bus_space_handle_t	sc_handle;
	bus_space_handle_t	sc_regh;

	struct	grfmode curr_mode;	/* hardware desc(for ioctl)	*/
	uint32_t	card_id;	/* DrHW value for nubus cards	*/
	bus_size_t	cli_offset;	/* Offset to clear interrupt	*/
					/* for cards where that's suff.	*/
	uint32_t	cli_value;	/* Value to write at cli_offset */
	nubus_dir	board_dir;	/* Nubus dir for curr board	*/
};

static void	load_image_data(void *, struct image_data *);

static void	intr_generic_write1(void *);
static void	intr_generic_write4(void *);
static void	intr_generic_or4(void *);

static void	intr_cb264(void *);
static void	intr_cb364(void *);
static void	intr_cmax(void *);
static void	intr_cti(void *);
static void	intr_radius(void *);
static void	intr_radius24(void *);
static void	intr_supermacgfx(void *);
static void	intr_lapis(void *);
static void	intr_formac(void *);
static void	intr_vimage(void *);
static void	intr_gvimage(void *);
static void	intr_radius_gsc(void *);
static void	intr_radius_gx(void *);
static void	intr_relax_200(void *);
static void	intr_mvc(void *);
static void	intr_viltro_340(void *);

static int	nubus_genfb_match(device_t, cfdata_t, void *);
static void	nubus_genfb_attach(device_t, device_t, void *);
static int	nubus_genfb_ioctl(void *, void *, unsigned long, void *, int,
		    struct lwp *);
static paddr_t	nubus_genfb_mmap(void *, void *, off_t, int);

CFATTACH_DECL_NEW(genfb_nubus, sizeof(struct nubus_genfb_softc),
    nubus_genfb_match, nubus_genfb_attach, NULL, NULL);

static void
load_image_data(void *	data, struct image_data *image)
{
	char *d = (char*)data;

	memcpy(&image->size,       d     , 4);
	memcpy(&image->offset,     d +  4, 4);
	memcpy(&image->rowbytes,   d +  8, 2);
	memcpy(&image->top,        d + 10, 2);
	memcpy(&image->left,       d + 12, 2);
	memcpy(&image->bottom,     d + 14, 2);
	memcpy(&image->right,      d + 16, 2);
	memcpy(&image->version,    d + 18, 2);
	memcpy(&image->packType,   d + 20, 2);
	memcpy(&image->packSize,   d + 22, 4);
	memcpy(&image->hRes,       d + 26, 4);
	memcpy(&image->vRes,       d + 30, 4);
	memcpy(&image->pixelType,  d + 34, 2);
	memcpy(&image->pixelSize,  d + 36, 2);
	memcpy(&image->cmpCount,   d + 38, 2);
	memcpy(&image->cmpSize,    d + 40, 2);
	memcpy(&image->planeBytes, d + 42, 4);
}


static int
nubus_genfb_match(device_t parent, struct cfdata *cf, void *aux)
{
	struct nubus_attach_args *na = (struct nubus_attach_args *)aux;

	if (na->category != NUBUS_CATEGORY_DISPLAY)
		return 0;

	if (na->type != NUBUS_TYPE_VIDEO)
		return 0;

	if (na->drsw != NUBUS_DRSW_APPLE)
		return 0;

	/*
	 * If we've gotten this far, then we're dealing with a real-live
	 * Apple QuickDraw-compatible display card resource.  Now, how to
	 * determine that this is an active resource???  Dunno.  But we'll
	 * proceed like it is.
	 */

	return 1;
}

static void
nubus_genfb_attach(device_t parent, device_t self, void *aux)
{
	struct nubus_genfb_softc *sc = device_private(self);
	struct nubus_attach_args *na = (struct nubus_attach_args *)aux;
	struct image_data image_store, image;
	char cardname[CARD_NAME_LEN];
	nubus_dirent dirent;
	nubus_dir dir, mode_dir;
	int mode;
	struct genfb_ops ops;

	memcpy(&sc->sc_slot, na->fmt, sizeof(nubus_slot));

	sc->sc_gen.sc_dev = self;
	sc->sc_tag = na->na_tag;
	sc->card_id = na->drhw;
	sc->sc_basepa = (bus_addr_t)NUBUS_SLOT2PA(na->slot);
	sc->sc_fbofs = 0;

	if (bus_space_map(sc->sc_tag, sc->sc_basepa, NBMEMSIZE,
	    0, &sc->sc_handle)) {
		aprint_normal(": nubus_genfb_attach: failed to map "
		    "slot %d\n", na->slot);
		return;
	}

	nubus_get_main_dir(&sc->sc_slot, &dir);

	if (nubus_find_rsrc(sc->sc_tag, sc->sc_handle,
	    &sc->sc_slot, &dir, na->rsrcid, &dirent) <= 0) {
bad:
		bus_space_unmap(sc->sc_tag, sc->sc_handle, NBMEMSIZE);
		return;
	}

	nubus_get_dir_from_rsrc(&sc->sc_slot, &dirent, &sc->board_dir);

	if (nubus_find_rsrc(sc->sc_tag, sc->sc_handle,
	    &sc->sc_slot, &sc->board_dir, NUBUS_RSRC_TYPE, &dirent) <= 0)
		if ((na->rsrcid != 128) ||
		    (nubus_find_rsrc(sc->sc_tag, sc->sc_handle,
		    &sc->sc_slot, &dir, 129, &dirent) <= 0))
			goto bad;

	mode = NUBUS_RSRC_FIRSTMODE;
	if (nubus_find_rsrc(sc->sc_tag, sc->sc_handle,
	    &sc->sc_slot, &sc->board_dir, mode, &dirent) <= 0) {
		aprint_normal(": probe failed to get board rsrc.\n");
		goto bad;
	}

	nubus_get_dir_from_rsrc(&sc->sc_slot, &dirent, &mode_dir);

	if (nubus_find_rsrc(sc->sc_tag, sc->sc_handle,
	    &sc->sc_slot, &mode_dir, VID_PARAMS, &dirent) <= 0) {
		aprint_normal(": probe failed to get mode dir.\n");
		goto bad;
	}

	if (nubus_get_ind_data(sc->sc_tag, sc->sc_handle, &sc->sc_slot,
	    &dirent, (void *)&image_store, sizeof(struct image_data)) <= 0) {
		aprint_normal(": probe failed to get indirect mode data.\n");
		goto bad;
	}

	/* Need to load display info (and driver?), etc... (?) */

	load_image_data((void *)&image_store, &image);

	/* XXX This has to go in favour of the code below... but uh-huh!
	 * If I remove it the machine hangs! */
	{
	prop_dictionary_t dict;
	dict = device_properties(sc->sc_gen.sc_dev);
	prop_dictionary_set_uint32(dict, "width", image.right - image.left);
	prop_dictionary_set_uint32(dict, "height", image.bottom - image.top);
	prop_dictionary_set_uint32(dict, "depth", image.rowbytes / (image.bottom - image.top));
	prop_dictionary_set_uint32(dict, "address", sc->sc_handle.base);
	prop_dictionary_set_uint32(dict, "linebytes", image.rowbytes);
	genfb_init(&sc->sc_gen);
	}

	sc->sc_gen.sc_fbaddr = (void *)mac68k_video.mv_kvaddr;
	sc->sc_gen.sc_fboffset = sc->sc_basepa;
	sc->sc_gen.sc_width = image.right - image.left;
	sc->sc_gen.sc_height = image.bottom - image.top;
	sc->sc_gen.sc_depth = image.rowbytes / sc->sc_gen.sc_height;
	sc->sc_gen.sc_stride = image.rowbytes;
	sc->sc_gen.sc_fbsize = image.rowbytes * sc->sc_gen.sc_height;

	strncpy(cardname, nubus_get_card_name(sc->sc_tag, sc->sc_handle,
	    &sc->sc_slot), CARD_NAME_LEN);
	cardname[CARD_NAME_LEN-1] = '\0';
	aprint_normal(": %s\n", cardname);

	if (sc->card_id == NUBUS_DRHW_TFB) {
		/*
		 * This is the Toby card, but apparently some manufacturers
		 * (like Cornerstone) didn't bother to get/use their own
		 * value here, even though the cards are different, so we
		 * try to differentiate here.
		 */
		if (strncmp(cardname, "Samsung 768", 11) == 0)
			sc->card_id = NUBUS_DRHW_SAM768;
		else if (strncmp(cardname, "Toby frame", 10) != 0)
			aprint_normal_dev(sc->sc_gen.sc_dev,
				"This display card pretends to be a TFB!\n");
	}

	switch (sc->card_id) {
	case NUBUS_DRHW_TFB:
	case NUBUS_DRHW_M2HRVC:
	case NUBUS_DRHW_PVC:
		sc->cli_offset = 0xa0000;
		sc->cli_value = 0;
		add_nubus_intr(na->slot, intr_generic_write1, sc);
		break;
	case NUBUS_DRHW_WVC:
		sc->cli_offset = 0xa00000;
		sc->cli_value = 0;
		add_nubus_intr(na->slot, intr_generic_write1, sc);
		break;
	case NUBUS_DRHW_COLORMAX:
		add_nubus_intr(na->slot, intr_cmax, sc);
		break;
	case NUBUS_DRHW_SE30:
		/* Do nothing--SE/30 interrupts are disabled */
		break;
	case NUBUS_DRHW_MDC:
		sc->cli_offset = 0x200148;
		sc->cli_value = 1;
		add_nubus_intr(na->slot, intr_generic_write4, sc);

		/* Enable interrupts; to disable, write 0x7 to this
		 * location */
		bus_space_write_4(sc->sc_tag, sc->sc_handle, 0x20013C, 5);
		break;
	case NUBUS_DRHW_CB264:
		add_nubus_intr(na->slot, intr_cb264, sc);
		break;
	case NUBUS_DRHW_CB364:
		add_nubus_intr(na->slot, intr_cb364, sc);
		break;
	case NUBUS_DRHW_RPC8:
		sc->cli_offset = 0xfdff8f;
		sc->cli_value = 0xff;
		add_nubus_intr(na->slot, intr_generic_write1, sc);
		break;
	case NUBUS_DRHW_RPC8XJ:
		sc->cli_value = 0x66;
		add_nubus_intr(na->slot, intr_radius, sc);
		break;
	case NUBUS_DRHW_RPC24X:
	case NUBUS_DRHW_BOOGIE:
		sc->cli_value = 0x64;
		add_nubus_intr(na->slot, intr_radius, sc);
		break;
	case NUBUS_DRHW_RPC24XP:
		add_nubus_intr(na->slot, intr_radius24, sc);
		break;
	case NUBUS_DRHW_RADGSC:
		add_nubus_intr(na->slot, intr_radius_gsc, sc);
		break;
	case NUBUS_DRHW_RDCGX:
		add_nubus_intr(na->slot, intr_radius_gx, sc);
		break;
	case NUBUS_DRHW_FIILX:
	case NUBUS_DRHW_FIISXDSP:
	case NUBUS_DRHW_FUTURASX:
		sc->cli_offset = 0xf05000;
		sc->cli_value = 0x80;
		add_nubus_intr(na->slot, intr_generic_write1, sc);
		break;
	case NUBUS_DRHW_SAM768:
		add_nubus_intr(na->slot, intr_cti, sc);
		break;
	case NUBUS_DRHW_SUPRGFX:
		add_nubus_intr(na->slot, intr_supermacgfx, sc);
		break;
	case NUBUS_DRHW_SPECTRM8:
		sc->cli_offset = 0x0de178;
		sc->cli_value = 0x80;
		add_nubus_intr(na->slot, intr_generic_or4, sc);
		break;
	case NUBUS_DRHW_LAPIS:
		add_nubus_intr(na->slot, intr_lapis, sc);
		break;
	case NUBUS_DRHW_RELAX200:
		add_nubus_intr(na->slot, intr_relax_200, sc);
		break;
	case NUBUS_DRHW_BAER:
	case NUBUS_DRHW_FORMAC:
		add_nubus_intr(na->slot, intr_formac, sc);
		break;
	case NUBUS_DRHW_ROPS24LXI:
	case NUBUS_DRHW_ROPS24XLTV:
	case NUBUS_DRHW_ROPS24MXTV:
		sc->cli_offset = 0xfb0010;
		sc->cli_value = 0x00;
		add_nubus_intr(na->slot, intr_generic_write4, sc);
		break;
	case NUBUS_DRHW_ROPSPPGT:
		sc->cli_offset = 0xf50010;
		sc->cli_value = 0x02;
		add_nubus_intr(na->slot, intr_generic_write4, sc);
		break;
	case NUBUS_DRHW_VIMAGE:
		add_nubus_intr(na->slot, intr_vimage, sc);
		break;
	case NUBUS_DRHW_GVIMAGE:
		add_nubus_intr(na->slot, intr_gvimage, sc);
		break;
	case NUBUS_DRHW_MC2124NB:
		sc->cli_offset = 0xfd1000;
		sc->cli_value = 0x00;
		add_nubus_intr(na->slot, intr_generic_write4, sc);
		break;
	case NUBUS_DRHW_MICRON:
		sc->cli_offset = 0xa00014;
		sc->cli_value = 0;
		add_nubus_intr(na->slot, intr_generic_write4, sc);
		break;
	case NUBUS_DRHW_MVC:
		add_nubus_intr(na->slot, intr_mvc, sc);
		break;
	case NUBUS_DRHW_VILTRO340:
		add_nubus_intr(na->slot, intr_viltro_340, sc);
		break;
	default:
		aprint_normal_dev(sc->sc_gen.sc_dev,
			"Unknown video card ID 0x%x; not installing interrupt routine",
		    sc->card_id);
		break;
	}

	ops.genfb_ioctl = nubus_genfb_ioctl;
	ops.genfb_mmap = nubus_genfb_mmap;

	genfb_attach(&sc->sc_gen, &ops);
}

static int
nubus_genfb_ioctl(void *v, void *vs, unsigned long cmd, void *data,
    int flag, struct lwp *l)
{
	switch (cmd) {
	case WSDISPLAYIO_GTYPE:
		*(unsigned int *)data = WSDISPLAY_TYPE_UNKNOWN;
		return 0;
	}

	return EPASSTHROUGH;
}

static paddr_t
nubus_genfb_mmap(void *v, void *vs, off_t offset, int prot)
{
	return -1;
}

/*
 * Interrupt handlers
 */

/*
 * Generic routine to clear interrupts for cards where it simply takes
 * a MOV.B to clear the interrupt.  The offset and value of this byte
 * varies between cards.
 */
/*ARGSUSED*/
static void
intr_generic_write1(void *vsc)
{
	struct nubus_genfb_softc *sc = (struct nubus_genfb_softc *)vsc;

	bus_space_write_1(sc->sc_tag, sc->sc_handle,
	    sc->cli_offset, (uint8_t)sc->cli_value);
}

/*
 * Generic routine to clear interrupts for cards where it simply takes
 * a MOV.L to clear the interrupt.  The offset and value of this byte
 * varies between cards.
 */
/*ARGSUSED*/
static void
intr_generic_write4(void *vsc)
{
	struct nubus_genfb_softc *sc = (struct nubus_genfb_softc *)vsc;

	bus_space_write_4(sc->sc_tag, sc->sc_handle,
	    sc->cli_offset, sc->cli_value);
}

/*
 * Generic routine to clear interrupts for cards where it simply takes
 * an OR.L to clear the interrupt.  The offset and value of this byte
 * varies between cards.
 */
/*ARGSUSED*/
static void
intr_generic_or4(void *vsc)
{
	struct nubus_genfb_softc *sc = (struct nubus_genfb_softc *)vsc;
	unsigned long scratch;

	scratch = bus_space_read_4(sc->sc_tag, sc->sc_handle,
	    sc->cli_offset);
	scratch |= 0x80;
	bus_space_write_4(sc->sc_tag, sc->sc_handle, sc->cli_offset,
	    scratch);
}

/*
 * Routine to clear interrupts for the Radius PrecisionColor 8xj card.
 */
/*ARGSUSED*/
static void
intr_radius(void *vsc)
{
	struct nubus_genfb_softc *sc = (struct nubus_genfb_softc *)vsc;
	uint8_t c;

	c = sc->cli_value;

	c |= 0x80;
	bus_space_write_1(sc->sc_tag, sc->sc_handle, 0xd00403, c);
	c &= 0x7f;
	bus_space_write_1(sc->sc_tag, sc->sc_handle, 0xd00403, c);
}

/*
 * Routine to clear interrupts for the Radius PrecisionColor 24Xp card.
 * Is this what the 8xj routine is doing, too?
 */
/*ARGSUSED*/
static void
intr_radius24(void *vsc)
{
	struct nubus_genfb_softc *sc = (struct nubus_genfb_softc *)vsc;
	uint8_t c;

	c = 0x80 | bus_space_read_1(sc->sc_tag, sc->sc_handle, 0xfffd8);
	bus_space_write_1(sc->sc_tag, sc->sc_handle, 0xd00403, c);
	c &= 0x7f;
	bus_space_write_1(sc->sc_tag, sc->sc_handle, 0xd00403, c);
}

/*
 * Routine to clear interrupts on Samsung 768x1006 video controller.
 * This controller was manufactured by Cornerstone Technology, Inc.,
 * now known as Cornerstone Imaging.
 *
 * To clear this interrupt, we apparently have to set, then clear,
 * bit 2 at byte offset 0x80000 from the card's base.
 *	Information for this provided by Brad Salai <bsalai@servtech.com>
 */
/*ARGSUSED*/
static void
intr_cti(void *vsc)
{
	struct nubus_genfb_softc *sc = (struct nubus_genfb_softc *)vsc;
	uint8_t c;

	c = bus_space_read_1(sc->sc_tag, sc->sc_handle, 0x80000);
	c |= 0x02;
	bus_space_write_1(sc->sc_tag, sc->sc_handle, 0x80000, c);
	c &= 0xfd;
	bus_space_write_1(sc->sc_tag, sc->sc_handle, 0x80000, c);
}

/*ARGSUSED*/
static void
intr_cb264(void *vsc)
{
	struct nubus_genfb_softc *sc;
	volatile char *slotbase;

	sc = (struct nubus_genfb_softc *)vsc;
	slotbase = (volatile char *)(sc->sc_handle.base); /* XXX evil hack */
	__asm volatile(
		"	movl	%0,%%a0				\n"
		"	movl	%%a0@(0xff6028),%%d0		\n"
		"	andl	#0x2,%%d0			\n"
		"	beq	_mv_intr0			\n"
		"	movql	#0x3,%%d0			\n"
		"_mv_intr0:					\n"
		"	movl	%%a0@(0xff600c),%%d1		\n"
		"	andl	#0x3,%%d1			\n"
		"	cmpl	%%d1,%%d0			\n"
		"	beq	_mv_intr_fin			\n"
		"	movl	%%d0,%%a0@(0xff600c)		\n"
		"	nop					\n"
		"	tstb	%%d0				\n"
		"	beq	_mv_intr1			\n"
		"	movl	#0x0002,%%a0@(0xff6040)		\n"
		"	movl	#0x0102,%%a0@(0xff6044)		\n"
		"	movl	#0x0105,%%a0@(0xff6048)		\n"
		"	movl	#0x000e,%%a0@(0xff604c)		\n"
		"	movl	#0x001c,%%a0@(0xff6050)		\n"
		"	movl	#0x00bc,%%a0@(0xff6054)		\n"
		"	movl	#0x00c3,%%a0@(0xff6058)		\n"
		"	movl	#0x0061,%%a0@(0xff605c)		\n"
		"	movl	#0x0012,%%a0@(0xff6060)		\n"
		"	bra	_mv_intr_fin			\n"
		"_mv_intr1:					\n"
		"	movl	#0x0002,%%a0@(0xff6040)		\n"
		"	movl	#0x0209,%%a0@(0xff6044)		\n"
		"	movl	#0x020c,%%a0@(0xff6048)		\n"
		"	movl	#0x000f,%%a0@(0xff604c)		\n"
		"	movl	#0x0027,%%a0@(0xff6050)		\n"
		"	movl	#0x00c7,%%a0@(0xff6054)		\n"
		"	movl	#0x00d7,%%a0@(0xff6058)		\n"
		"	movl	#0x006b,%%a0@(0xff605c)		\n"
		"	movl	#0x0029,%%a0@(0xff6060)		\n"
		"_mv_intr_fin:					\n"
		"	movl	#0x1,%%a0@(0xff6014)"
		: : "g" (slotbase) : "a0","d0","d1");
}

/*
 * Support for the Colorboard 364 might be more complex than it needs to
 * be.  If we can find more information about this card, this might be
 * significantly simplified.  Contributions welcome...  :-)
 */
/*ARGSUSED*/
static void
intr_cb364(void *vsc)
{
	struct nubus_genfb_softc *sc;
	volatile char *slotbase;

	sc = (struct nubus_genfb_softc *)vsc;
	slotbase = (volatile char *)(sc->sc_handle.base); /* XXX evil hack */
	__asm volatile(
		"	movl	%0,%%a0				\n"
		"	movl	%%a0@(0xfe6028),%%d0		\n"
		"	andl	#0x2,%%d0			\n"
		"	beq	_cb364_intr4			\n"
		"	movql	#0x3,%%d0			\n"
		"	movl	%%a0@(0xfe6018),%%d1		\n"
		"	movl	#0x3,%%a0@(0xfe6018)		\n"
		"	movw	%%a0@(0xfe7010),%%d2		\n"
		"	movl	%%d1,%%a0@(0xfe6018)		\n"
		"	movl	%%a0@(0xfe6020),%%d1		\n"
		"	btst	#0x06,%%d2			\n"
		"	beq	_cb364_intr0			\n"
		"	btst	#0x00,%%d1			\n"
		"	beq	_cb364_intr5			\n"
		"	bsr	_cb364_intr1			\n"
		"	bra	_cb364_intr_out			\n"
		"_cb364_intr0:					\n"
		"	btst	#0x00,%%d1			\n"
		"	bne	_cb364_intr5			\n"
		"	bsr	_cb364_intr1			\n"
		"	bra	_cb364_intr_out			\n"
		"_cb364_intr1:					\n"
		"	movl	%%d0,%%a0@(0xfe600c)		\n"
		"	nop					\n"
		"	tstb	%%d0				\n"
		"	beq	_cb364_intr3			\n"
		"	movl	#0x0002,%%a0@(0xfe6040)		\n"
		"	movl	#0x0105,%%a0@(0xfe6048)		\n"
		"	movl	#0x000e,%%a0@(0xfe604c)		\n"
		"	movl	#0x00c3,%%a0@(0xfe6058)		\n"
		"	movl	#0x0061,%%a0@(0xfe605c)		\n"
		"	btst	#0x06,%%d2			\n"
		"	beq	_cb364_intr2			\n"
		"	movl	#0x001c,%%a0@(0xfe6050)		\n"
		"	movl	#0x00bc,%%a0@(0xfe6054)		\n"
		"	movl	#0x0012,%%a0@(0xfe6060)		\n"
		"	movl	#0x000e,%%a0@(0xfe6044)		\n"
		"	movl	#0x00c3,%%a0@(0xfe6064)		\n"
		"	movl	#0x0061,%%a0@(0xfe6020)		\n"
		"	rts					\n"
		"_cb364_intr2:					\n"
		"	movl	#0x0016,%%a0@(0xfe6050)		\n"
		"	movl	#0x00b6,%%a0@(0xfe6054)		\n"
		"	movl	#0x0011,%%a0@(0xfe6060)		\n"
		"	movl	#0x0101,%%a0@(0xfe6044)		\n"
		"	movl	#0x00bf,%%a0@(0xfe6064)		\n"
		"	movl	#0x0001,%%a0@(0xfe6020)		\n"
		"	rts					\n"
		"_cb364_intr3:					\n"
		"	movl	#0x0002,%%a0@(0xfe6040)		\n"
		"	movl	#0x0209,%%a0@(0xfe6044)		\n"
		"	movl	#0x020c,%%a0@(0xfe6048)		\n"
		"	movl	#0x000f,%%a0@(0xfe604c)		\n"
		"	movl	#0x0027,%%a0@(0xfe6050)		\n"
		"	movl	#0x00c7,%%a0@(0xfe6054)		\n"
		"	movl	#0x00d7,%%a0@(0xfe6058)		\n"
		"	movl	#0x006b,%%a0@(0xfe605c)		\n"
		"	movl	#0x0029,%%a0@(0xfe6060)		\n"
		"	oril	#0x0040,%%a0@(0xfe6064)		\n"
		"	movl	#0x0000,%%a0@(0xfe6020)		\n"
		"	rts					\n"
		"_cb364_intr4:					\n"
		"	movq	#0x00,%%d0			\n"
		"_cb364_intr5:					\n"
		"	movl	%%a0@(0xfe600c),%%d1		\n"
		"	andl	#0x3,%%d1			\n"
		"	cmpl	%%d1,%%d0			\n"
		"	beq	_cb364_intr_out			\n"
		"	bsr	_cb364_intr1			\n"
		"_cb364_intr_out:				\n"
		"	movl	#0x1,%%a0@(0xfe6014)		\n"
		"_cb364_intr_quit:"
		: : "g" (slotbase) : "a0","d0","d1","d2");
}

/*
 * Interrupt clearing routine for SuperMac GFX card.
 */
/*ARGSUSED*/
static void
intr_supermacgfx(void *vsc)
{
	struct nubus_genfb_softc *sc = (struct nubus_genfb_softc *)vsc;

	bus_space_read_1(sc->sc_tag, sc->sc_handle, 0xE70D3);
}

/*
 * Routine to clear interrupts for the Sigma Designs ColorMax card.
 */
/*ARGSUSED*/
static void
intr_cmax(void *vsc)
{
	struct nubus_genfb_softc *sc = (struct nubus_genfb_softc *)vsc;

	bus_space_read_4(sc->sc_tag, sc->sc_handle, 0xf501c);
	bus_space_read_4(sc->sc_tag, sc->sc_handle, 0xf5018);
}

/*
 * Routine to clear interrupts for the Lapis ProColorServer 8 PDS card
 * (for the SE/30).
 */
/*ARGSUSED*/
static void
intr_lapis(void *vsc)
{
	struct nubus_genfb_softc *sc = (struct nubus_genfb_softc *)vsc;

	bus_space_write_1(sc->sc_tag, sc->sc_handle, 0xff7000, 0x08);
	bus_space_write_1(sc->sc_tag, sc->sc_handle, 0xff7000, 0x0C);
}

/*
 * Routine to clear interrupts for the Formac ProNitron 80.IVb
 * and Color Card II
 */
/*ARGSUSED*/
static void
intr_formac(void *vsc)
{
	struct nubus_genfb_softc *sc = (struct nubus_genfb_softc *)vsc;

	bus_space_read_1(sc->sc_tag, sc->sc_handle, 0xde80db);
	bus_space_read_1(sc->sc_tag, sc->sc_handle, 0xde80d3);
}

/*
 * Routine to clear interrupts for the Vimage by Interware Co., Ltd.
 */
/*ARGSUSED*/
static void
intr_vimage(void *vsc)
{
	struct nubus_genfb_softc *sc = (struct nubus_genfb_softc *)vsc;

	bus_space_write_1(sc->sc_tag, sc->sc_handle, 0x800000, 0x67);
	bus_space_write_1(sc->sc_tag, sc->sc_handle, 0x800000, 0xE7);
}

/*
 * Routine to clear interrupts for the Grand Vimage by Interware Co., Ltd.
 */
/*ARGSUSED*/
static void
intr_gvimage(void *vsc)
{
	struct nubus_genfb_softc *sc = (struct nubus_genfb_softc *)vsc;

	bus_space_read_1(sc->sc_tag, sc->sc_handle, 0xf00000);
}

/*
 * Routine to clear interrupts for the Radius GS/C
 */
/*ARGSUSED*/
static void
intr_radius_gsc(void *vsc)
{
	struct nubus_genfb_softc *sc = (struct nubus_genfb_softc *)vsc;

	bus_space_read_1(sc->sc_tag, sc->sc_handle, 0xfb802);
	bus_space_write_1(sc->sc_tag, sc->sc_handle, 0xfb802, 0xff);
}

/*
 * Routine to clear interrupts for the Radius GS/C
 */
/*ARGSUSED*/
static void
intr_radius_gx(void *vsc)
{
	struct nubus_genfb_softc *sc = (struct nubus_genfb_softc *)vsc;

	bus_space_write_1(sc->sc_tag, sc->sc_handle, 0x600000, 0x00);
	bus_space_write_1(sc->sc_tag, sc->sc_handle, 0x600000, 0x20);
}

/*
 * Routine to clear interrupts for the Relax 19" model 200.
 */
/*ARGSUSED*/
static void
intr_relax_200(void *vsc)
{
	struct nubus_genfb_softc *sc = (struct nubus_genfb_softc *)vsc;

	/* The board ROM driver code has a tst.l here. */
	bus_space_read_4(sc->sc_tag, sc->sc_handle, 0x000D0040);
}

/*
 * Routine to clear interrupts for the Apple Mac II Monochrome Video Card.
 */
/*ARGSUSED*/
static void
intr_mvc(void *vsc)
{
	struct nubus_genfb_softc *sc = (struct nubus_genfb_softc *)vsc;

	bus_space_write_4(sc->sc_tag, sc->sc_handle, 0x00040000, 0);
	bus_space_write_4(sc->sc_tag, sc->sc_handle, 0x00020000, 0);
}

/*
 * Routine to clear interrupts for the VillageTronic Mac Picasso 340.
 */
/*ARGSUSED*/
static void
intr_viltro_340(void *vsc)
{
	struct nubus_genfb_softc *sc = (struct nubus_genfb_softc *)vsc;

	/* Yes, two read accesses to the same spot. */
	bus_space_read_1(sc->sc_tag, sc->sc_handle, 0x0500);
	bus_space_read_1(sc->sc_tag, sc->sc_handle, 0x0500);
}
