/******************************************************************************/
/* Copyright (c) 2008 Patrick Mahoney <pat@polycrystal.org>.		      */
/* All rights reserved.							      */
/* 									      */
/* Redistribution and use in source and binary forms, with or without	      */
/* modification, are permitted provided that the following conditions	      */
/* are met:								      */
/* 1. Redistributions of source code must retain the above copyright	      */
/*    notice, this list of conditions and the following disclaimer.	      */
/* 2. Redistributions in binary form must reproduce the above copyright	      */
/*    notice, this list of conditions and the following disclaimer in the     */
/*    documentation and/or other materials provided with the distribution.    */
/* 3. Neither the name of the author nor the names of contributors	      */
/*    may be used to endorse or promote products derived from this software   */
/*    without specific prior written permission.			      */
/* 									      */
/* THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND    */
/* ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE      */
/* IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE */
/* ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE   */
/* FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL */
/* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS    */
/* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)      */
/* HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT */
/* LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY  */
/* OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF     */
/* SUCH DAMAGE.								      */
/******************************************************************************/
/*  Modified by Don Hayford, Feb 2009
 */

#include "jpeg_mangle.h"

/*
 * See JPEG spec ISO/IEC 10918-1:1993(E) from http://www.wotsit.org/
 *
 * From ISO/IEC 10918-1:1993(E) K.3.3.  This is the default Huffman
 * table used by USB Video Class devices using the MJPEG format but
 * lacking a Huffman table.
 */

enum {
	JPEG_START_OF_IMAGE = 0xffd8,
	JPEG_START_OF_FRAME = 0xffc0,
	JPEG_DEFINE_HUFFMAN = 0xffc4,
	JPEG_START_OF_SCAN = 0xffda
};

enum {
	JPEG_DC_LUM_ID = 0x00,
	JPEG_AC_LUM_ID = 0x10,
	JPEG_DC_CHROM_ID = 0x01,
	JPEG_AC_CHROM_ID = 0x11
};
	
#ifdef NEVERDEFINED
GQuark
jpeg_mangle_error_quark(void)
{
	static GQuark q = 0;
	if (q == 0)
		q = g_quark_from_static_string("jpeg-mangle-error-quark");
	return q;
}
#endif

#define JPEG_DEFINE_HUFFMAN_BYTES (JPEG_DEFINE_HUFFMAN >> 8),	\
	    (JPEG_DEFINE_HUFFMAN & 0xff)

static unsigned char jpeg_default_huffman_tables[] = {
	/* DC Luminance table */
	JPEG_DEFINE_HUFFMAN_BYTES,
	0x00, 0x1F, /* length of this table, including these bytes */
	JPEG_DC_LUM_ID,
	/* DC Luminance lengths */
	0x00, 0x01, 0x05, 0x01, 0x01, 0x01, 0x01, 0x01,
	0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	/* DC Luminance values */
	0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0A, 0x0B,

	/* AC Luminance table */
	JPEG_DEFINE_HUFFMAN_BYTES,
	0x00, 0xB5,		/* length */
	JPEG_AC_LUM_ID,
	/* AC Luminance lengths */
	0x00, 0x02, 0x01, 0x03, 0x03, 0x02, 0x04, 0x03,
        0x05, 0x05, 0x04, 0x04, 0x00, 0x00, 0x01, 0x7D,
	/* AC Luminance values */
	0x01, 0x02, 0x03, 0x00, 0x04, 0x11, 0x05, 0x12,
	0x21, 0x31, 0x41, 0x06, 0x13, 0x51, 0x61, 0x07,
	0x22, 0x71, 0x14, 0x32, 0x81, 0x91, 0xa1, 0x08,
	0x23, 0x42, 0xb1, 0xc1, 0x15, 0x52, 0xd1, 0xf0,
	0x24, 0x33, 0x62, 0x72, 0x82, 0x09, 0x0A, 0x16,
	0x17, 0x18, 0x19, 0x1A, 0x25, 0x26, 0x27, 0x28,
	0x29, 0x2A, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39,
	0x3A, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49,
	0x4A, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59,
	0x5A, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69,
	0x6A, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79,
	0x7A, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89,
	0x8A, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98,
	0x99, 0x9A, 0xA2, 0xA3, 0xA4, 0xA5, 0xA6, 0xA7,
	0xA8, 0xA9, 0xAA, 0xB2, 0xB3, 0xB4, 0xB5, 0xB6,
	0xB7, 0xB8, 0xB9, 0xBA, 0xC2, 0xC3, 0xC4, 0xC5,
	0xC6, 0xC7, 0xC8, 0xC9, 0xCA, 0xD2, 0xD3, 0xD4,
	0xD5, 0xD6, 0xD7, 0xD8, 0xD9, 0xDA, 0xE1, 0xE2,
	0xE3, 0xE4, 0xE5, 0xE6, 0xE7, 0xE8, 0xE9, 0xEA,
	0xF1, 0xF2, 0xF3, 0xF4, 0xF5, 0xF6, 0xF7, 0xF8,
	0xF9, 0xFA,

	/* DC Chrominance table */
	JPEG_DEFINE_HUFFMAN_BYTES,
	0x00, 0x1F,		/* length */
	JPEG_DC_CHROM_ID,
	/* DC Chrominance lengths */
	0x00, 0x03, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
	0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
	/* DC Chrominance values */
	0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0A, 0x0B,

	/* AC Chrominance table */
	JPEG_DEFINE_HUFFMAN_BYTES,
	0x00, 0xB5,		/* length */
	JPEG_AC_CHROM_ID,
	/* AC Chrominance lengths */
	0x00, 0x02, 0x01, 0x02, 0x04, 0x04, 0x03, 0x04,
	0x07, 0x05, 0x04, 0x04, 0x00, 0x01, 0x02, 0x77,
	/* AC Chrominance values */
	0x00, 0x01, 0x02, 0x03, 0x11, 0x04, 0x05, 0x21,
	0x31, 0x06, 0x12, 0x41, 0x51, 0x07, 0x61, 0x71,
	0x13, 0x22, 0x32, 0x81, 0x08, 0x14, 0x42, 0x91,
	0xA1, 0xB1, 0xC1, 0x09, 0x23, 0x33, 0x52, 0xF0,
	0x15, 0x62, 0x72, 0xD1, 0x0A, 0x16, 0x24, 0x34,
	0xE1, 0x25, 0xF1, 0x17, 0x18, 0x19, 0x1A, 0x26,
	0x28, 0x29, 0x2A, 0x35, 0x36, 0x37, 0x38, 0x39,
	0x3A, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x27,
	0x49, 0x4A, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58,
	0x59, 0x5A, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68,
	0x69, 0x6A, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78,
	0x79, 0x7A, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
	0x88, 0x89, 0x8A, 0x92, 0x93, 0x94, 0x95, 0x96,
	0x97, 0x98, 0x99, 0x9A, 0xA2, 0xA3, 0xA4, 0xA5,
	0xA6, 0xA7, 0xA8, 0xA9, 0xAA, 0xB2, 0xB3, 0xB4,
	0xB5, 0xB6, 0xB7, 0xB8, 0xB9, 0xBA, 0xC2, 0xC3,
	0xC4, 0xC5, 0xC6, 0xC7, 0xC8, 0xC9, 0xCA, 0xD2,
	0xD3, 0xD4, 0xD5, 0xD6, 0xD7, 0xD8, 0xD9, 0xDA,
	0xE2, 0xE3, 0xE4, 0xE5, 0xE6, 0xE7, 0xE8, 0xE9,
	0xEA, 0xF2, 0xF3, 0xF4, 0xF5, 0xF6, 0xF7, 0xF8,
	0xF9, 0xFA
};

#define GET_WORD(p) (((p)[0] << 8) | (p)[1])

/* Returns the offset in bytes to the position after which the huffman
 * table should be inserted.  E.g. offset of 5 means the mangled jpeg
 * stream should looke like (in byte indicies):
 *
 * [0 1 2 3 4]-[huffman table]-[5 6 7 8]
 */
bool
jpeg_mangle_get_huffman_offset(unsigned char *buf, size_t len, size_t *offset)
{
	unsigned char *p;

	if(buf == NULL || offset == NULL)
		{
		perror("bad pointers - jpeg mangle");
		return EXIT_FAILURE;
	}	
	if (GET_WORD(buf) != JPEG_START_OF_IMAGE) {
		perror("no JPEG image detected");
		return FALSE;
	}
	for (p = buf; p < (buf + len - 1); p += 2) {
		switch (GET_WORD(p)) {
		case JPEG_START_OF_SCAN:
			*offset = (p - buf);
			return TRUE;
		case JPEG_DEFINE_HUFFMAN:
			perror("Huffman table already defined");
			return FALSE;
		default:
			break;
		}
	}
	perror("offset not found");
	return FALSE;
}

const unsigned char *
jpeg_mangle_get_default_huffman_tables(void)
{
	return jpeg_default_huffman_tables;
}

size_t
jpeg_mangle_get_sizeof_default_huffman_tables(void)
{
	return sizeof(jpeg_default_huffman_tables);
}
