/* $NetBSD: $ */

#include <sys/cdefs.h>

__RCSID("$NetBSD: $");

#include <sys/types.h>
#include <sys/stat.h>

#include <err.h>
#include <grp.h>
#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

/*
-L use lstat
-q be quiet
-Q be really quiet

-a everything
-n  use numbers
-s	size
-b blocks
-f flags
-m[Augos] mode: all, user, group, other, setuid
-t type
-T[Aamc]  time: all, ccess, modify, change
-u user
-g group
-l number of links
--foo  Specific field of struct stat, by name.
*/

struct stat_context
{
	int f_use_names;
	int f_use_lstat;
	int f_quiet;

	int f_report_type;	// Subset of st_mode
	int f_report_mode;
	#define REPORT_MODE_ALL    0x0F
	#define REPORT_MODE_USER   0x01
	#define REPORT_MODE_GROUP  0x02
	#define REPORT_MODE_OTHER  0x04
	#define REPORT_MODE_SETUID 0x08
	#define REPORT_TYPE        0x10
	int f_report_nlinks;
	int f_report_uid;
	int f_report_gid;
	int f_report_time;
	#define REPORT_TIME_ALL    0xf
	#define REPORT_TIME_ATIME  0x1
	#define REPORT_TIME_MTIME  0x2
	#define REPORT_TIME_CTIME  0x4
	int f_report_size;
	int f_report_blocks;
	int f_report_flags;

	int nrep_mode;
	int maxfilenamelen;
};
static struct stat_context c;


#if 1
// XXX to allow ls/print.c to be used:
int f_inode;
int f_size;
int f_type;
int termwidth;
int blocksize;
int f_nonprint;
int f_sectime;
int f_longform;
int f_flags;
int f_accesstime;
int f_statustime;
int f_typedir;
#endif

static struct {
	char *field_name;
	char short_opt;
	int  *pf_report;
	int  on_value;
} struct_map[] = {
	{ "st_dev",       0,   NULL, 0 },
	{ "st_ino",       0,   NULL, 0 },
	{ "st_mode",      0,   &c.f_report_mode, REPORT_MODE_ALL|REPORT_TYPE },
	{ "",             't', &c.f_report_mode, REPORT_TYPE },
	{ "st_nlink",     'l', &c.f_report_nlinks, 1 },
	{ "st_uid",       'u', &c.f_report_uid, 1 },
	{ "st_gid",       'g', &c.f_report_gid, 1 },
	{ "st_rdev",      0,   NULL, 0 },
	{ "st_atimespec", 0,   &c.f_report_time, REPORT_TIME_ATIME },
	{ "st_mtimespec", 0,   &c.f_report_time, REPORT_TIME_MTIME },
	{ "st_ctimespec", 0,   &c.f_report_time, REPORT_TIME_CTIME },
	{ "st_size",      's', &c.f_report_size, 1 },
	{ "st_blocks",    'b', &c.f_report_blocks, 1 },
	{ "st_blksize",   0,   NULL, 0 },
	{ "st_flags",     'f', &c.f_report_flags, 1 },
	{ "st_gen",       0,   NULL, 0 },
	{ NULL, NULL, 0 }
};

static void x_usage(void);
static void usage(void);
static void long_usage(void);
static int decode_struct_field(char *);
static void decode_short_opt(char);
static int stat_file(struct stat_context *, char *filename);
static void print_header(struct stat_context *);

// XXX from ls:
char *flags_to_string(u_long, const char *);
void printtime(time_t); // XAX don't use this.  use something else.

int main(int argc, char **argv)
{
	int f_do_default = 1;
	int rv, error = 0;
	int ii;
	int ch;

	memset(&c, 0, sizeof(c));
	c.f_use_names = 1;

	while((ch = getopt(argc, argv, "nLqQabsftuglm:T:-:")) != -1)
	{
		switch(ch)
		{
		case 'n':
			c.f_use_names = 0;
			break;
		case 'L':
			c.f_use_lstat = 1;
			break;
		case 'q':
			c.f_quiet = 1;
			break;
		case 'Q':
			c.f_quiet = 2;
			break;
		case 'a':	// Everything.
			f_do_default = 0;
			c.f_report_mode = REPORT_MODE_ALL|REPORT_TYPE;
			c.f_report_nlinks = 1;
			c.f_report_uid = 1;
			c.f_report_gid = 1;
			c.f_report_time = REPORT_TIME_ALL;
			c.f_report_size = 1;
			c.f_report_blocks = 1;
			c.f_report_flags = 1;
			break;
		case 's':
		case 'b':
		case 'f':
		case 't':
		case 'u':
		case 'g':
		case 'l':
			f_do_default = 0;
			decode_short_opt(ch);
			break;
		case 'm':
			f_do_default = 0;
			for (ii = 0; ii < strlen(optarg); ii++)
			{
				switch(optarg[ii])
				{
				case 'A':
					c.f_report_mode = REPORT_MODE_ALL;
					break;
				case 'u':
					c.f_report_mode |= REPORT_MODE_USER;
					break;
				case 'g':
					c.f_report_mode |= REPORT_MODE_GROUP;
					break;
				case 'o':
					c.f_report_mode |= REPORT_MODE_OTHER;
					break;
				case 's':
					c.f_report_mode |= REPORT_MODE_SETUID;
					break;
				default:
					warnx("%c: Unknown mode type", optarg[ii]);
					error = 1;
					break;
				}
			}
			break;
		case 'T':
			f_do_default = 0;
			for (ii = 0; ii < strlen(optarg); ii++)
			{
				switch(optarg[ii])
				{
				case 'A':
					c.f_report_time = REPORT_TIME_ALL;
					break;
				case 'a':
					c.f_report_time |= REPORT_TIME_ATIME;
					break;
				case 'm':
					c.f_report_time |= REPORT_TIME_MTIME;
					break;
				case 'c':
					c.f_report_time |= REPORT_TIME_CTIME;
					break;
				default:
					warnx("%c: Unknown time type", optarg[ii]);
					error = 1;
					break;
				}
			}
			break;
		case '-':
			if (strcmp(optarg, "help") == 0)
				long_usage();
			f_do_default = 0;
			if (decode_struct_field(optarg))
				error = 1;
			break;
		default:
		case '?':
			usage();
		}
	}

	if (f_do_default)
	{
		// No output selection options selected.  Use default.
		c.f_report_mode = REPORT_MODE_ALL|REPORT_TYPE;
		c.f_report_uid = 1;
		c.f_report_gid = 1;
		c.f_report_size = 1;
	}

	if (error || optind >= argc)
		usage();

	c.nrep_mode = 0;
	if (c.f_report_mode & REPORT_MODE_USER)
		c.nrep_mode++;
	if (c.f_report_mode & REPORT_MODE_GROUP)
		c.nrep_mode++;
	if (c.f_report_mode & REPORT_MODE_OTHER)
		c.nrep_mode++;

	c.maxfilenamelen = strlen(argv[optind]);

	for (ii = optind + 1; ii < argc; ii++)
	{
		if (strlen(argv[ii]) > c.maxfilenamelen)
			c.maxfilenamelen = strlen(argv[ii]);
	}
	print_header(&c);

	rv = 0;
	for (ii = optind; ii < argc; ii++)
	{
		if (stat_file(&c, argv[ii]))
			rv = 1;
	}
	exit(rv);
}

static int decode_struct_field(char *field_name)
{
	int ii;

	for(ii = 0; struct_map[ii].field_name != NULL; ii++)
	{
		if (strcmp(field_name, struct_map[ii].field_name) == 0)
		{
			if (struct_map[ii].pf_report == NULL)
			{
				warnx("Field %s is not supported", field_name);
				return 1;
			}
			*(struct_map[ii].pf_report) = struct_map[ii].on_value;
			return 0;
		}
	}
	warnx("Unknown field %s", field_name);
	return 1;
}

static void decode_short_opt(char short_opt)
{
	int ii;

	for(ii = 0; struct_map[ii].field_name != NULL; ii++)
	{
		if (short_opt == struct_map[ii].short_opt)
		{
			if (struct_map[ii].pf_report == NULL)
			{
				// Shouldn't happen.
				errx(1, "Option %c has NULL flag pointer!", short_opt);
			}
			*(struct_map[ii].pf_report) = struct_map[ii].on_value;
			return;
		}
	}
	// Shouldn't happen.
	errx(1, "Option %c not in mapping table!", short_opt);
}

static void x_usage()
{
	fprintf(stderr, "Usage: %s [-LqQansbftugl] [-mA | -m[u][g][o][s]] "
	                "[-TA | -T[a][m][c]] file [...]\n", getprogname());
	fprintf(stderr, "       %s [-LqQ] [--fieldname] file [...]\n",
	                getprogname());
}
static void usage()
{
	x_usage();
	exit(1);
}
static void long_usage()
{
#define PFLAG(flag,desc) printf("%-10.10s %s\n", flag, desc)
	x_usage();
	PFLAG("--help", "Display this information");
	PFLAG("-L", "Use lstat() instead of stat()");
	PFLAG("-q", "Be quiet");
	PFLAG("-Q", "Be really quiet");
	PFLAG("-n", "Use numeric display");
	PFLAG("-a", "Show everything");
	PFLAG("-s", "Display size of file");
	PFLAG("-b", "Display blocks used");
	PFLAG("-f", "Display file flags");
	PFLAG("-u", "Display uid");
	PFLAG("-g", "Display gid");
	PFLAG("-l", "Display number of hard links");
	PFLAG("-t", "Display type of file");
	PFLAG("-m <arg>", "Display file mode.  arg is any combination of:");
	PFLAG("",   "A: All");
	PFLAG("",   "u: user modes");
	PFLAG("",   "g: group modes");
	PFLAG("",   "o: other modes");
	PFLAG("",   "s: setuid modes");
	PFLAG("-T <arg>", "Display file times.  arg is any combination of:");
	PFLAG("",   "A: All");
	PFLAG("",   "a: access time");
	PFLAG("",   "m: modification time");
	PFLAG("",   "c: creation time");
	PFLAG("--field", "Display a particular field of the stat struct");
	PFLAG("file...", "files to stat");
	printf("\n");
	printf("Default options are -t -u -g -s -mA\n");

	exit(1);
}

static int stat_file(struct stat_context *pc, char *filename)
{
	struct stat st;
	int rv;
	int ii;

	if (pc->f_use_lstat)
		rv = lstat(filename, &st);
	else
		rv = stat(filename, &st);

	if (rv == -1)
	{
		if (!pc->f_quiet)
			warn("%s", filename);
		return 1;
	}

	if (pc->f_quiet < 2)
	{
		for (ii = strlen(filename); ii < pc->maxfilenamelen; ii++)
			printf(" ");
		printf("%s: ", filename);
	}

	if (pc->f_report_mode & REPORT_TYPE)
	{
	
		if (pc->f_use_names)
		{
			const char *s_type;
// XXX pick a better way to output stuff.
			switch (st.st_mode & S_IFMT)
			{
			case S_IFIFO:
				s_type = "FIFO";
				break;
			case S_IFCHR:
				s_type = "CHR";
				break;
			case S_IFDIR:
				s_type = "DIR";
				break;
			case S_IFBLK:
				s_type = "BLK";
				break;
			case S_IFREG:
				s_type = "REG";
				break;
			case S_IFLNK:
				s_type = "LNK";
				break;
			case S_IFSOCK:
				s_type = "SOCK";
				break;
			case S_IFWHT:
				s_type = "WHT";
				break;
			}
			printf("%-5s\t", s_type);
		}
		else
		{
			printf("0x%-3x\t", st.st_mode & S_IFMT);
		}
	}
	if (pc->f_report_mode & (REPORT_MODE_USER|REPORT_MODE_GROUP|REPORT_MODE_OTHER))
	{
		mode_t mode_mask;
		mode_t mode_rep;
		if (pc->f_report_mode & REPORT_MODE_USER)
			mode_mask |= (S_IRUSR|S_IWUSR|S_IXUSR);
		if (pc->f_report_mode & REPORT_MODE_GROUP)
			mode_mask |= (S_IRGRP|S_IWGRP|S_IXGRP);
		if (pc->f_report_mode & REPORT_MODE_OTHER)
			mode_mask |= (S_IROTH|S_IWOTH|S_IXOTH);
		if (pc->f_report_mode & REPORT_MODE_SETUID)
			mode_mask |= (S_ISUID|S_ISGID|S_ISVTX);
		mode_rep = st.st_mode & mode_mask;
		if (pc->f_use_names)
		{
			char s_mode[12];
			strmode(mode_rep, s_mode);
			if (pc->f_report_mode & REPORT_MODE_USER)
			{
				printf("%c%c%c", s_mode[1], s_mode[2], s_mode[3]);
			}
			if (pc->f_report_mode & REPORT_MODE_GROUP)
			{
				printf("%c%c%c", s_mode[4], s_mode[5], s_mode[6]);
			}
			if (pc->f_report_mode & REPORT_MODE_OTHER)
			{
				printf("%c%c%c", s_mode[7], s_mode[8], s_mode[9]);
			}
			if (pc->nrep_mode == 1)
				printf("   ");
			printf("\t");
		}
		else
		{
			printf("0x%-9x\t", mode_rep);
		}
	}

	if (pc->f_report_nlinks)
	{
		printf("%5d\t", st.st_nlink);
	}
	if (pc->f_report_uid)
	{
		if (pc->f_use_names)
			printf("%s\t", user_from_uid(st.st_uid, 0));
		else
			printf("%-5d\t", st.st_uid);
	}
	if (pc->f_report_gid)
	{
		if (pc->f_use_names)
			printf("%s\t", group_from_gid(st.st_gid, 0));
		else
			printf("%-5d\t", st.st_gid);
	}
	if (pc->f_report_time & REPORT_TIME_ATIME)
	{
		if (pc->f_use_names)
			printtime(st.st_atimespec.tv_sec);
		else
			printf("%ld", st.st_atimespec.tv_sec);
		printf("\t");
	}
	if (pc->f_report_time & REPORT_TIME_MTIME)
	{
		if (pc->f_use_names)
			printtime(st.st_mtimespec.tv_sec);
		else
			printf("%ld", st.st_atimespec.tv_sec);
		printf("\t");
	}
	if (pc->f_report_time & REPORT_TIME_CTIME)
	{
		if (pc->f_use_names)
			printtime(st.st_ctimespec.tv_sec);
		else
			printf("%ld", st.st_atimespec.tv_sec);
		printf("\t");
	}
	if (pc->f_report_size)
	{
		printf("%-9lld\t", st.st_size);
	}
	if (pc->f_report_blocks)
	{
		printf("%-9lld\t", st.st_blocks);
	}
	if (pc->f_report_flags)
	{
		if (pc->f_use_names)
		{
			printf("%s\t", flags_to_string(st.st_flags, "-"));
		}
		else
		{
			printf("%d\t", st.st_flags);
		}
	}

	printf("\n");
	return 0;
}

static void print_header(struct stat_context *pc)
{
	if (!pc->f_quiet)
	{
		int ii;
		for (ii = pc->maxfilenamelen ; ii >= 0; ii--)
		{
			printf(" ");
		}
		printf(" ");
		if (pc->f_report_mode & REPORT_TYPE)
			printf("%-5s\t", "TYPE");
		if (!pc->f_use_names || pc->nrep_mode == 3)
			printf("%-9s\t", "MODE");
		else if (pc->nrep_mode == 1 || pc->nrep_mode == 2)
			printf("%-6s\t", "MODE");

		if (pc->f_report_nlinks)
			printf("%-5s\t", "LINKS");
		if (pc->f_report_uid)
			printf("%-5s\t", "UID");
		if (pc->f_report_gid)
			printf("%-5s\t", "GID");
		if (pc->f_report_time & REPORT_TIME_ATIME)
			printf("%-12s\t", "ATIME");
		if (pc->f_report_time & REPORT_TIME_MTIME)
			printf("%-12s\t", "MTIME");
		if (pc->f_report_time & REPORT_TIME_CTIME)
			printf("%-12s\t", "CTIME");
		if (pc->f_report_size)
			printf("%-9s\t", "SIZE");
		if (pc->f_report_blocks)
			printf("%-9s\t", "BLOCKS");
		if (pc->f_report_flags)
			printf("FLAGS");
		printf("\n");
	}
}
