/* $NetBSD$ */

/*-
 * Copyright (c) 2010 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */


#include <assert.h>

#include <Python.h>
#include <errno.h>
#include <sys/sysctl.h>

/* 
 * create Python object of type similar to the MIB node 'name'
 * This is basically the laziest way to do this, as we outsource the
 * type recognition to the sysctl(3) API. Fantastic! :-)
 */
static PyObject *
node_to_object(const char *name, void *val, size_t len)
{
	PyObject *o;

	struct sysctlnode *node;
	int rv;
	unsigned long long ull;
	unsigned int ui;

	assert (name != NULL);
	assert (val != NULL);
	assert (len >= 1);

	node = NULL;
	rv = sysctlgetmibinfo(name, NULL, NULL, NULL, NULL, &node, SYSCTL_VERSION);
	if (rv != 0) {
		free(val);
		PyErr_SetFromErrno(PyExc_OSError);
		return NULL;
	}

	o = NULL;

	switch (SYSCTL_TYPE(node->sysctl_flags)) {
	case CTLTYPE_NODE:
		o = NULL;
		break;
	case CTLTYPE_INT:
		o = Py_BuildValue("i", *(unsigned int *)val);
		break;
	case CTLTYPE_STRING:
		o = PyString_FromStringAndSize(val, len - 1);
		break;
	case CTLTYPE_QUAD:
		ull = *(uint64_t *)val;
		o = PyLong_FromUnsignedLongLong(ull);
		break;
	case CTLTYPE_STRUCT:
		o = NULL;
		PyErr_SetString(PyExc_ValueError, "sysctl node type CTLTYPE_STRUCT unsupported");
		break;
	case CTLTYPE_BOOL:
		o = NULL;
		PyErr_SetString(PyExc_ValueError, "sysctl node type CTLTYPE_BOOL unsupported");
		break;
	default:
		o = NULL;
		PyErr_SetString(PyExc_ValueError, "Unknown sysctl node type operation requested");
	}

	free(node);

	return o;
}

/* 
 * SYSCTL_TYPEMASK is used as an error type. This is an implementation
 * detail. 
 */
static uint32_t
nodetype(char *nodepath)
{

	int rv;
	char cname[SYSCTL_NAMELEN];
	int csz;

	struct sysctlnode *rnode = NULL;

	assert (nodepath != NULL);

	csz = SYSCTL_NAMELEN;

	rv = sysctlgetmibinfo(nodepath, NULL, NULL, cname, &csz, &rnode, SYSCTL_VERSION);

	if (rv == -1 || rnode == NULL) {
		return SYSCTL_TYPEMASK;
	}

	return SYSCTL_TYPE(rnode->sysctl_flags);

}

/*
 * Write a copy of the provided string to the given sysctl path.
 * The node should already exist.
 *
 * Note: value is not NULL terminated, and sz does not include
 * trailing NULL.
 *
 * Implementation detail: The node is deleted ( and its current value
 * discarded ),  and a new one, with requested string is assigned.
 *
 * XXX: This is a workaround for sysctl(3) not allowing strings longer
 * than the current value. 
 * See: http://mail-index.netbsd.org/tech-kern/2009/10/22/msg006329.html
 */

static bool
write_sysctl_string(char *name, char *value, size_t sz)
{


	int rv;
	int mib[CTL_MAXNAME];
	u_int miblen;
	struct sysctlnode *rnode, node;
	size_t nodelen;

	char cname[SYSCTL_NAMELEN];
	int csz = SYSCTL_NAMELEN;

	/* Sanity checks */

	assert (name != NULL);
	assert (value != NULL);
	assert (sz > 0);

	if (sz == 0 || value[sz] != '\0') {
		return false;
	}

	/* check length of node name*/
	if (strlen(name) >= SYSCTL_NAMELEN) {
		return false;
	}

	/* Check the type of node */
	if (CTLTYPE_STRING != nodetype(name)) {
		return false;
	}

	memset(mib, 0, CTL_MAXNAME * sizeof mib[0]);
	miblen = CTL_MAXNAME;
	rnode = NULL;
	csz = SYSCTL_NAMELEN;

	/* Traverse to the leaf, if possible. */
	rv = sysctlgetmibinfo(name, mib, &miblen, cname, &csz, &rnode, SYSCTL_VERSION);

	if (rv != 0 || miblen == 0 || rnode == NULL || csz == 0) { /* Couldn't find node */
		/* XXX: what does the sysctl(3) api say about rnode ? */
		free(rnode);
		PyErr_SetFromErrno(PyExc_OSError);
		return false;

	}

	/* Save the leaf node name into cname */
	/* Note: csz include trailing '\0' */
	snprintf(cname, csz + 1, rnode->sysctl_name);

	/* Delete Node */

	if (rv == 0 && rnode != NULL) { /* The node is already present */

		/* Destroy leaf node */
		mib[miblen - 1] = CTL_DESTROY;

		rv = sysctl(&mib[0], miblen, NULL, NULL, rnode, sizeof *rnode);

		if (rv != 0) {
			free(rnode);
			PyErr_SetFromErrno(PyExc_OSError);
			return false;
		}

	}

	free(rnode);

	/* Re-create new node with new string cname */

	/* Add cname */
	mib[miblen - 1] = CTL_CREATE;
	nodelen = sizeof node;
	memset(&node, 0, nodelen);
	node.sysctl_num = CTL_CREATE;
	node.sysctl_flags = SYSCTL_VERSION | CTLFLAG_READWRITE | CTLFLAG_OWNDATA | CTLTYPE_STRING;
	node.sysctl_size = sz;
	node.sysctl_data = value;

	snprintf(node.sysctl_name, csz + 1, cname);

	rv = sysctl(&mib[0], miblen, &node, &nodelen, &node, nodelen);

	if (rv != 0) {
		return false;
	}

	return true;
}

static PyObject *
read_sysctl(PyObject *self, PyObject *args)
{
	PyObject *o;
	void *val;
	size_t len;
	const char *name;
	int rv;

	if (!PyArg_ParseTuple(args, "s", &name))
		return NULL;

getbyname:
	val = NULL;
	len = 0;

	rv = sysctlbyname(name, NULL, &len, NULL, 0);

	if (rv == -1) {
		PyErr_SetFromErrno(PyExc_OSError);
		return NULL;
	}

	val = malloc(len);
	if (val ==NULL) {
		PyErr_NoMemory();
		return NULL;
	}

	rv = sysctlbyname(name, val, &len, NULL, 0);

	if (rv == ENOMEM) { /* Try again */
		assert (!"retry implementation untested");
		const size_t ARBITRARYLEN = 4096; /* XXX: Tune this */
		free(val);
		if (len > ARBITRARYLEN) {
			PyErr_SetFromErrno(PyExc_OSError);
			return NULL;
		}
		goto getbyname;
	}

	if (rv == -1) {
		free(val);
		PyErr_SetFromErrno(PyExc_OSError);
		return NULL;
	}

	o = node_to_object(name, val, len);

	free(val);

	return o;
}

static PyObject *
write_sysctl(PyObject *self, PyObject *args)
{

	PyObject *o;
	PyObject *value;

	char *name = NULL;
	void *oldval;
	size_t len;

	int rv;

	int intval;
	char *strval;

	if (!PyArg_ParseTuple(args, "s|O: Incorrect values passed to sysctl.write", &name, &value)) {
		return NULL;
	}

	/* Arrange to obtain the oldvalue */
	oldval = NULL;
	len = 0;

	rv = sysctlbyname(name, NULL, &len, NULL, 0);

	if (rv == -1) {
		PyErr_SetFromErrno(PyExc_OSError);
		return NULL;
	}

	oldval = malloc(len);
	if (oldval == NULL) {
		PyErr_NoMemory();
		return NULL;
	}

	if (PyInt_CheckExact(value)) {
		if (!PyArg_ParseTuple(args, "s|i: conversion to int failed.", &name, &intval)) {
			free(oldval);
			PyErr_SetFromErrno(PyExc_OSError);
			return NULL;
		}

		rv = sysctlbyname(name, oldval, &len, &intval, sizeof intval);

		if (rv == -1) {
			free(oldval);
			PyErr_SetFromErrno(PyExc_OSError);
			return NULL;
		}

		/* Update PyObject to return */

		o = node_to_object(name, oldval, len);

	}
	else if (PyString_CheckExact(value)) {
		if (!PyArg_ParseTuple(args, "s|s#: conversion to string failed.", &name, &strval, &len)) {
			free(oldval);
			return NULL;
		}

		/* 
		 * Strings are handled specially. We just use
		 * read_sysctl() to record the previous string value.
		 * We ignore and free(oldval)
		 */

		/* Obtain the old value to return */

		PyObject *readargs;

		readargs = Py_BuildValue("(s)", name);

		if (readargs == NULL) {
			PyErr_SetString(PyExc_ValueError, "BuildValue() failed\n");
			free(oldval);
			return NULL;
		}

		/* Update PyObject to return */

		o = read_sysctl(self, readargs);

		if (o == NULL) { /* XXX: Doesn't matter if read failed
				  * ? */
			(void) o;
		}

		/* Now unconditionally write the new one */
		if (!write_sysctl_string(name, strval, len)) {
			free(oldval);
			PyErr_SetFromErrno(PyExc_OSError);
			return NULL;
		}

	}
	else {

		PyErr_SetString(PyExc_ValueError, "TODO - Implement");
		return NULL;
	}

	free(oldval);
	return o;

}

/*
 * Get the leaf name from the requested new node in cname. This name
 * may be passed to sysctl() for creation.
 * Returns the size of the cname string (including terminating '\0'
 */

static size_t
getnewleafname(const char *name, char *cname)
{
	int mib[CTL_MAXNAME];
	u_int miblen = CTL_MAXNAME;
	size_t csz;
	int rv;

	/* Traverse to the leaf, if possible. */
	miblen = CTL_MAXNAME;
	csz = SYSCTL_NAMELEN;
	rv = sysctlgetmibinfo(name, mib, &miblen, cname, &csz, NULL, SYSCTL_VERSION);

	if (rv == 0) {
		return 0;
	}

	return csz;
}

/*
 * Get the common prefix of name from what's already in the MIB and
 * what's been requested 
 */
static void
getprefixname(const char *name, char *pname)
{

	int csz;
	char cname[SYSCTL_NAMELEN]; /* Canonical name */

	csz = getnewleafname(name, cname);
	snprintf(pname, strlen(name) - csz, name);

	return;
}


static bool
create_node(const char *name, int ctl_type, uint32_t flags, void *value, size_t vlen)
{

	int mib[CTL_MAXNAME];
	u_int miblen = CTL_MAXNAME;

	char cname[SYSCTL_NAMELEN]; /* Canonical name */
	char pname[SYSCTL_NAMELEN]; /* The Canonical prefix */
	int csz, psz;

	int rv;

	struct sysctlnode node;
	size_t nodelen;

	/* Check for NULL ptr dereference */
	assert (value != NULL || vlen == 0);

	/* Sanity check ctl_type */
	switch (ctl_type) {
	case CTLTYPE_NODE:
	case CTLTYPE_INT:
	case CTLTYPE_STRING:
	case CTLTYPE_BOOL:
		break;

	case CTLTYPE_QUAD: /* XXX: TODO - implement */
	default:
		assert (!"Unknown type requested.");
		return false;
	}

	/* Sanity check flags */
	if (!((flags && CTLFLAG_READWRITE) ||
	      (flags && CTLFLAG_OWNDATA))) {
		assert (!"Unknown flag type requested");
		return false;
	}

	/* Get the suffix/leaf name */
	csz = getnewleafname(name, cname);

	if (csz == 0) {
		PyErr_SetString(PyExc_ValueError, "Could not obtain leaf name from given sysctl path.\n");
		return false;
	}

	/* Get the canonical prefix */
	getprefixname(name, pname);

	/* Setup mib for this canonical prefix node */
	rv = sysctlnametomib(pname, mib, &miblen);
	if (rv != 0 && miblen != 0) { /* Couldn't find dirname */
		PyErr_SetFromErrno(PyExc_OSError);
		return false;
	}

	mib[miblen] = CTL_CREATE;

	nodelen = sizeof node;
	memset(&node, 0, nodelen);
	node.sysctl_num = CTL_CREATE;
	node.sysctl_flags = SYSCTL_VERSION | flags | ctl_type;
	node.sysctl_data = value;
	node.sysctl_size = vlen;

	snprintf(node.sysctl_name, csz + 1, cname);

	rv = sysctl(&mib[0], miblen + 1, &node, &nodelen, &node, nodelen);

	if (rv != 0) {
		PyErr_SetFromErrno(PyExc_OSError);
		return false;
	}

	return true;
}


/* Create a node */
static PyObject *
create_sysctl(PyObject *self, PyObject *args)
{

   	PyObject *o;
	PyObject *value = NULL;
	Py_ssize_t vlen;

	const char *name = NULL;
	const char *typename = NULL;
	size_t typelen = 0;

	int len, rv;

	/* XXX: Royal mess... needs more thought */
	if (!PyArg_ParseTuple(args, "s|s#O: Incorrect values passed to sysctl.write", &name, &typename, &typelen, &value)) {
		return NULL;
	}

	if (name == NULL) {
		PyErr_SetString(PyExc_ValueError, "Invalid Arguments");
		return NULL;
	}

	/* XXX: Arrange to obtain the oldvalue */
	len = 0;

	if ((typename == NULL && typelen == 0)
	    || (!strncmp(typename, "CTLTYPE_NODE", typelen)) ) {
		/* default type is CTLTYPE_NODE */
		if (value != NULL) {
			char errvalstr[sizeof "Value provided for type node:  " + SYSCTL_NAMELEN];
			sprintf(errvalstr, "Value provided for type node: %s\n", name);
			PyErr_SetString(PyExc_ValueError, errvalstr);
			return NULL;
		}
		if (!create_node(name, CTLTYPE_NODE, CTLFLAG_READWRITE, NULL, 0)) {
			return NULL;
		} else {
			Py_RETURN_NONE;
		}
	}

	if (!strncmp(typename, "CTLTYPE_INT", typelen)) {
		int intval = 0;

		if (value != NULL) { /* Obtain value passed in */
			if (!PyInt_CheckExact(value)) {
				PyErr_SetString(PyExc_TypeError, "Value passed is of wrong type");
				return NULL;
			}
			if ((-1 == (intval = PyInt_AsLong(value))) && PyErr_Occurred() != NULL) {
				PyErr_SetString(PyExc_TypeError, "Error converting object to type int\n");
				return NULL;
			}
		}

		if (!create_node(name, CTLTYPE_INT, CTLFLAG_READWRITE, &intval, sizeof intval)) {
			return NULL;
		} else {
			Py_RETURN_NONE;
		}
	}


	if (!strncmp(typename, "CTLTYPE_STRING", typelen)) {
		char *strval;
		if (value != NULL) { /* Obtain value passed in */
			if (!PyString_CheckExact(value)) {
				PyErr_SetString(PyExc_TypeError, "Value passed is of wrong type");
				return NULL;
			}
			if (-1 == PyString_AsStringAndSize(value, &strval, &vlen)) {
				PyErr_SetString(PyExc_TypeError, "Error decoding string from buffer \n");
				return NULL;
			}
		} else {  /* Default empty string */
			strval = "";
			vlen = 1;
		}

		if (!create_node(name, CTLTYPE_STRING, CTLFLAG_READWRITE | CTLFLAG_OWNDATA, strval, vlen)) {
			printf("Create STRING node failed\n");
			return NULL;
		} else {
			Py_RETURN_NONE;
		}
	}

	if (!strncmp(typename, "CTLTYPE_QUAD", typelen)) {
		/* XXX: What's the right C type for QUAD ? */
		/* XXX: create node */
		/* XXX: TODO: Please implement */
		return NULL;
	}


	if (!strncmp(typename, "CTLTYPE_BOOL", typelen)) {
		if (!PyBool_Check(value)) {
			PyErr_SetString(PyExc_TypeError, "Value passed is of wrong type");
			return NULL;
		}
		if (!create_node(name, CTLTYPE_BOOL, CTLFLAG_READWRITE, NULL, 0)) {
			return NULL;
		}
		else {
			Py_RETURN_NONE;
		}
	} else {
		/* Note that CTLTYPE_STRUCT is unsupported right now */
		PyErr_SetString(PyExc_TypeError, "Unsupported node type requested");
		return NULL;
	}


	Py_RETURN_NONE;
}

/* Delete leaf */
static PyObject *
destroy_sysctl(PyObject *self, PyObject *args)
{
	int rv;
	int mib[CTL_MAXNAME];
	u_int miblen;
	size_t len;

	struct sysctlnode *rnode = NULL;

	const char *name;


	if (!PyArg_ParseTuple(args, "s", &name)) {
		return NULL;
	}

	/* Find the MIB of name  */
	memset(mib, 0, CTL_MAXNAME * sizeof mib[0]);
	miblen = CTL_MAXNAME;
	rv = sysctlgetmibinfo(name, mib, &miblen, NULL, NULL, &rnode, SYSCTL_VERSION);

	if (rv != 0 || miblen == 0 || rnode == NULL) { /* Couldn't find node */
		PyErr_SetFromErrno(PyExc_OSError);
		return NULL;

	}


	mib[miblen - 1] = CTL_DESTROY;
	if (0 != sysctl(&mib[0], miblen, NULL, NULL, rnode, sizeof *rnode)) {
		PyErr_SetFromErrno(PyExc_OSError);
		free(rnode);
		return NULL;
	}

	free(rnode);

	Py_RETURN_NONE;
}

static PyMethodDef sysctl_methods[] = {
	{ "read", read_sysctl, METH_VARARGS, 
	  "read value from sysctl node." },
	{ "write", write_sysctl, METH_VARARGS, 
	  "write value to sysctl node." },
	{ "create", create_sysctl, METH_VARARGS,
	  "create a sysctl node." },
	{ "destroy", destroy_sysctl, METH_VARARGS,
	  "destroy a sysctl node." },
	{ NULL, NULL, 0, NULL }
};


PyMODINIT_FUNC
initsysctl(void)
{
	(void) Py_InitModule("sysctl", sysctl_methods);
}
