/* $Id: data.c,v 1.1.1.1 2008/09/10 09:32:57 agcrooks Exp $ */

/*
 * Copyright (c) 2005 Manuel Freire.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote
 *    products derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <string.h>
#include <time.h>

#include <openssl/evp.h>

#include "constants.h"
#include "error.h"

#include "data.h"

#define EXPBIAS 6

bpg_time_t
bpg_time_get(void)
{
	bpg_time_t t;

	t = (bpg_time_t)time(NULL);

	if (t == sizeof(time_t) - 1) {
		BPG_err(BPGERR_GET_TIME);
		return (bpg_time_t)-1;
		/* XXX - bpg_time_t is unsigned */
	}

	return t;
}

S2K *
S2K_new(void)
{
	S2K *s2k;

	if ((s2k = malloc(sizeof(S2K))) == NULL) {
		BPG_err(BPGERR_MALLOC);
		return NULL;
	}

	s2k->type = 0;
	s2k->hash_algo = 0;
	s2k->count = 0;
	
	return s2k;
}

void
S2K_free(S2K *s2k)
{

	if (s2k)
		free(s2k);
}

int
S2K_read(S2K *s2k, unsigned char *from, size_t flen)
{
	unsigned c;
	int len;

	if (s2k == NULL || from == NULL) {
		BPG_err(BPGERR_INVALID_ARGUMENT);
		return -1;
	}

	if (flen < 1) {
		BPG_err(BPGERR_S2K_TOO_SHORT);
		return -1;
	}

	s2k->type = from[0];

	switch (s2k->type) {
	case S2KTYPE_SIMPLE:
		len = 2;

		if (flen < len) {
			BPG_err(BPGERR_S2K_TOO_SHORT);
			return -1;
		}

		s2k->hash_algo = from[1];

		break;
	case S2KTYPE_SALTED:
		len = 10;

		if (flen < len) {
			BPG_err(BPGERR_S2K_TOO_SHORT);
			return -1;
		}

		s2k->hash_algo = from[1];

		memcpy(s2k->salt, &from[2], 8);

		break;
	case S2KTYPE_ITERATED_SALTED:
		len = 11;

		if (flen < len) {
			BPG_err(BPGERR_S2K_TOO_SHORT);
			return -1;
		}

		s2k->hash_algo = from[1];

		memcpy(s2k->salt, &from[2], 8);

		c = from[10];
		s2k->count = ((int32_t)16 + (c & 15)) << ((c >> 4) + EXPBIAS);

		break;
	default:
		BPG_err(BPGERR_S2K_UNKNOWN);
		return -1;
	}

	return len;
}

/*ARGSUSED0 */
int
S2K_write(S2K *s2k, BPG_BUF *to)
{
	printf("S2K_write Function Unimplemented!\n");
	return 0;
}

void
S2K_print(S2K *s2k, FILE *f)
{
	int i;

	if (s2k != NULL && f != NULL) {
		fprintf(f, "<S2K>\n");

		switch (s2k->type) {
		case S2KTYPE_SIMPLE:
			fprintf(f, "\tType: Simple\n");
			break;
		case S2KTYPE_SALTED:
			fprintf(f, "\tType: Salted\n");
			break;
		case S2KTYPE_ITERATED_SALTED:
			fprintf(f, "\tType: Iterated and Salted\n");
			break;
		default:
			fprintf(f, "\tType: Unknown (%d)\n", s2k->type);
		}

		fprintf(f, "\tHash algorithm: %d\n", s2k->hash_algo);

		if (s2k->type == S2KTYPE_SALTED ||
		    s2k->type == S2KTYPE_ITERATED_SALTED) {
			fprintf(f, "\tSalt: ");
			for (i = 0; i < 8; i ++)
				fprintf(f, "%02x", s2k->salt[i]);
			fprintf(f, "\n");
		}

		if (s2k->type == S2KTYPE_ITERATED_SALTED)
			fprintf(f, "\tCount: %d\n", s2k->count);

		fprintf(f, "</S2K>\n");
	}
}

/* Obtain a key from a passphrase as defined in a string-to-key specifier. */
int
S2K_getkey(S2K *s2k, char *passphrase, uint8_t symm_algo, BPG_BUF *key)
{
	size_t hash_size, key_size;	/* In bytes. */
	int hashed_octets, hlen;
	BPG_BUF *hash_input;
	EVP_MD_CTX *hctx;
	const EVP_MD *htype;

	if (s2k == NULL || passphrase == NULL || key == NULL) {
		BPG_err(BPGERR_NULL_POINTER);
		return -1;
	}

	hash_input = BPG_BUF_new(0);

	/*
	 * Init digest context.
	 */

	hctx = EVP_MD_CTX_create();

	switch (s2k->hash_algo) {
	case HA_SHA1:
		htype = (const EVP_MD *)EVP_sha1();
		hash_size = SHA1_SIZE;
		break;
	default:
		BPG_err(BPGERR_ALGORITHM_NOT_SUPPORTED);
		return -1;
	}

	EVP_DigestInit(hctx, htype);

	BPG_BUF_resize(key, hash_size);

	/*
	 * Get key size.
	 */

	switch (symm_algo) {
	case SKA_CAST5:
		key_size = CAST5_KEYSIZE;
		break;
	default:
		BPG_err(BPGERR_ALGORITHM_NOT_SUPPORTED);
		return -1;
	}

	if (key_size < hash_size)
		hash_size = key_size;
	else {
		/* TODO: make this function compatible with hash sizes smaller
		 * than key sizes. */
		BPG_err(BPGERR_NOT_IMPLEMENTED);
		return -1;
	}

	/*
	 * Hash the passphrase.
	 */

	switch (s2k->type) {
	case S2KTYPE_SIMPLE:
		BPG_BUF_resize(hash_input, strlen(passphrase));
		memcpy(hash_input->body, passphrase, strlen(passphrase));

		break;
	case S2KTYPE_SALTED:
	case S2KTYPE_ITERATED_SALTED:
		BPG_BUF_resize(hash_input, strlen(passphrase) + 8);
		memcpy(hash_input->body, s2k->salt, 8);
		memcpy(&hash_input->body[8], passphrase, strlen(passphrase));

		break;
	default:
		BPG_err(BPGERR_S2K_UNKNOWN);
		return -1;
	}

	if (s2k->type == S2KTYPE_ITERATED_SALTED) {
		EVP_DigestUpdate(hctx, hash_input->body, hash_input->len);
		hashed_octets = hash_input->len;
		while (hashed_octets < s2k->count) {
			if (hashed_octets < s2k->count - hash_input->len)
				hlen = hash_input->len;
			else
				hlen = s2k->count - hashed_octets;

			EVP_DigestUpdate(hctx, hash_input->body,
				(unsigned)hlen);
			hashed_octets += hlen;
		}
		while (hashed_octets < s2k->count);
	} else
		EVP_DigestUpdate(hctx, hash_input->body, hash_input->len);

	/*
	 * Extract the key.
	 */

	EVP_DigestFinal(hctx, key->body, &(key->len));
	BPG_BUF_resize(key, key_size);

	BPG_BUF_free(hash_input);

	return 0;
}

