#include <lib/libsa/stand.h>

#include <sys/cdefs.h>
#include <sys/types.h>
#if 0
#include <arc/arc/arcbios.h>
#else
#include "arcbios.h"
#endif

#ifdef __STDC__
#include <machine/stdarg.h>
#else
#include <machine/varargs.h>
#endif

#include <stdio.h>

#define STDIN 0
#define STDOUT 1

#define NFDS	20
#define BUFSIZ	8192

struct FILE {
	unsigned char *p;
	int avail;
	int eof;
	int openmode;
	unsigned char buffer[BUFSIZ];
};

int errno;
FILE iob[NFDS];
FILE *stdin = &iob[STDIN];
FILE *stdout = &iob[STDOUT];

#define FILENO(file)	((file) - (iob))

void freadline(FILE *file);
void ffilbuf(FILE *file);

FILE *
fopen(char *name, char *mode)
{
	int fd;
	int openmode;

	switch (*mode) {
	case 'r':
		openmode = arc_OpenReadOnly;
		break;
	case 'w':
		openmode = arc_SupersedeWriteOnly;
		break;
	default:
		errno = arc_EINVAL;
		return (NULL);
	}
	errno = Bios_Open(name, openmode, &fd);
	if (errno != arc_ESUCCESS)
		return (NULL);
	if (fd >= NFDS) {
		Bios_Close(fd);
		errno = arc_EMFILE;
		return (NULL);
	}
	iob[fd].p = NULL;
	iob[fd].avail = 0;
	iob[fd].eof = 0;
	iob[fd].openmode = openmode;
	return (&iob[fd]);
}

void
fclose(FILE *file)
{
	int fd = FILENO(file);

	switch (file->openmode) {
	case arc_OpenReadOnly:
		break;
	case arc_SupersedeWriteOnly:
		fflush(file);
		break;
	}
	Bios_Close(fd);
	file->p = NULL;
	file->avail = 0;
	file->eof = 0;
	file->openmode = 0;
}

int
fileno(FILE *file)
{
	return (FILENO(file));
}

void
fflush(FILE *file)
{
	int fd = FILENO(file);
	int i, outcount;
	arc_status_t rv;

	for (i = 0; i < file->avail; i += outcount) {
		rv = Bios_Write(fd, &file->buffer[i], file->avail - i,
				&outcount);
		if (rv != arc_ESUCCESS) {
			errno = rv;
			file->avail -= i;
			bcopy(&file->buffer[i], file->buffer, file->avail);
			file->p = &file->buffer[file->avail];
			return;
		}
	}
	file->avail = 0;
	file->p = file->buffer;
}

void
fputc(int c, FILE *file)
{
	int fd = FILENO(file);
#if 0
	if (fd != STDOUT)
		fputc(c, stdout);
#endif
	if (fd == STDOUT && c == '\n') {
		fputc('\r', file);
		fflush(file);
	} else if (file->avail >= BUFSIZ) {
		fflush(file);
	} else if (file->p == NULL) {
		file->p = iob[fd].buffer;
	}
	*file->p++ = c;
	file->avail++;
}

void
fwrite(char *buffer, int size, int count, FILE *file)
{
	size *= count;
	while (--size >=0)
		fputc(*buffer++, file);
}

FILE *putchar_file = &iob[STDOUT];

void
putchar(int c)
{
	fputc(c, putchar_file);
}

void
#ifdef __STDC__
fprintf(FILE *file, const char *fmt, ...)
#else
fprintf(file, fmt, va_alist)
	FILE *file;
	char *fmt;
#endif
{
	FILE *saved_file;
	va_list ap;

#ifdef __STDC__
	va_start(ap, fmt);
#else
	va_start(ap);
#endif
	saved_file = putchar_file;
	putchar_file = file;
	vprintf(fmt, ap);
	putchar_file = saved_file;

	va_end(ap);
}

void
freadline(FILE *file)
{
	int fd = FILENO(file);
	int incount;
	unsigned char c;
	arc_status_t rv;

	file->p = file->buffer;
	file->avail = 0;
	for (;;) {
		fflush(stdout);
		rv = Bios_Read(fd, &c, 1, &incount);
		if (rv != arc_ESUCCESS || incount == 0) {
			errno = rv;
			file->eof = file->avail == 0;
			return;
		}
		if (c == '\b') {
			if (file->avail > 0) {
				--file->avail;
				fputc('\b', stdout);
				fputc(' ', stdout);
				fputc('\b', stdout);
			}
			continue;
		}
		if (c == '\r' || c == '\n')
			break;
		if (file->avail < BUFSIZ - 1) {
			file->buffer[file->avail++] = c;
			fputc(c, stdout);
		}
	}
	file->buffer[file->avail++] = '\n';
	fputc('\r', stdout);
	fputc('\n', stdout);
	fflush(stdout);
}

void
ffilbuf(FILE *file)
{
	int fd = FILENO(file);
	arc_status_t rv;

	if (file->avail > 0 && file->p < &file->buffer[file->avail]) {
		file->avail -= file->p - file->buffer;
		bcopy(file->p, file->buffer, file->avail);
		return;
	}
	if (fd == STDIN) {
		freadline(file);
		return;
	}
	rv = Bios_Read(fd, file->buffer, BUFSIZ, &file->avail);
	if (rv != arc_ESUCCESS) {
		errno = rv;
		file->eof = 1;
		return;
	}
	file->p = file->buffer;
	file->eof = file->avail == 0;
}

int
fgetc(FILE *file)
{
	if (file->p == NULL) {
		ffilbuf(file);
	} else if (file->p >= &file->buffer[file->avail]) {
		ffilbuf(file);
	}
	if (file->eof)
		return (EOF);
	return (*file->p++);
}

char *
fgets(char *s, int size, FILE *file)
{
	int i = 0;
	int c;

	if (size <= 0)
		return (NULL);
	if (size == 1) {
		*s = '\0';
		return (NULL);
	}
	--size;
	for (;;) {
		if (i >= size || (c = fgetc(file)) == EOF) {
			s[i] = '\0';
			return (i == 0 ? NULL : s);
		}
		s[i++] = c;
		if (c == '\n') {
			s[i] = '\0';
			return (s);
		}
	}
}
