// -*- c++ -*-

/*
 *  Copyright (C) 2001-2002, Richard J. Moore <rich@kde.org>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this library; see the file COPYING.LIB.  If not, write to
 *  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 *  Boston, MA 02111-1307, USA.
 */

#ifndef KJSEMBEDJSOBJECTPROXY_H
#define KJSEMBEDJSOBJECTPROXY_H

#include <qguardedptr.h>
#include <kjs/object.h>
#include <kjsembed/jsbinding.h>

namespace KJS { class Interpreter; };

namespace KJSEmbed {

/**
 * A JS object that provides a binding to a QObject. The JS object
 * created lets scripts access all the properties defined by the
 * object in a similar way to the DCOP/QObject bridge.
 *
 * The following example creates a @ref KJS::ObjectImp that provides
 * a binding to the properties of a @ref QLineEdit . This binding is
 * then used to create a property 'edit' for the object 'jsparent'.
 *
 * <pre>
 *   QLineEdit *edit = new QLineEdit();
 *   KJSEmbed::JSObjectProxy *proxy = new KJSEmbed::JSObjectProxy( js, edit );
 *   jsparent.put( js->globalExec(), "edit", proxy );
 * </pre>
 *
 * It is 
 *
 *
 * @version $Id: jsobjectproxy.h,v 1.3 2002/03/08 18:53:34 rich Exp $
 * @author Richard Moore, rich@kde.org
 */
class JSObjectProxy : public KJS::ObjectImp
{
public:
    /**
     * Create a JS binding to the target object. The binding will allow scripts to
     * access any QObject that is descended from the specified root. If the specified
     * root is 0 then access is granted to all objects.
     */
    JSObjectProxy( KJS::Interpreter *js, QObject *target, QObject *root );

    /**
     * Create a JS binding to the target object. The binding will allow scripts to
     * access any QObject that is descended the target and no others.
     */
    JSObjectProxy( KJS::Interpreter *js, QObject *target );

    ~JSObjectProxy() {}

    /**
     * The interpreter in which this proxy lives.
     */
    KJS::Interpreter *jscript() const { return js; }

    /**
     * Adds methods for traversing the QObject tree to the specified
     * @ref KJS::Object . Only QObjects descended from the root specified
     * in the constructor can be reached through JS proxies created with
     * these bindings.
     */
    virtual void addBindings( KJS::ExecState *state, KJS::Object &object );

    /**
     * Reimplemented to return the value of the specified property if present.
     */
    virtual KJS::Value get( KJS::ExecState *state, const KJS::UString &p ) const;

    /**
     * Reimplemented to set the value of the specified property if possible.
     */
    virtual void put( KJS::ExecState *state, const KJS::UString &p,
		      const KJS::Value &v, int attr = KJS::None );

    /**
     * Reimplemented to return the name and class of the target.
     */
    virtual KJS::UString toString( KJS::ExecState *exec ) const;

    //
    // Security
    //

    /**
     * Returns true if the specified interpreter can access this proxy.
     */
    virtual bool isAllowed( KJS::Interpreter *interp ) const;

    /**
     * Returns true if scripts are allowed to see the specified QObject.
     */
    virtual bool isAllowed( QObject *obj ) const;

    /**
     * Returns true if scripts are allowed to see the specified property.
     */
    virtual bool isAllowed( QObject *obj, const char *prop ) const;

private:
    KJS::Interpreter *js;
    QGuardedPtr<QObject> obj;
    QGuardedPtr<QObject> root;
    class Private *d;

    /**
     * @internal
     * Provides the implementation of a JS method.
     */
    class MethodImp : public KJS::ObjectImp
    {
    public:
	enum MethodId { MethodParent, MethodChildCount, MethodChildAt, MethodFindChild };

	MethodImp( MethodId id, const JSObjectProxy *parent );
	~MethodImp() {}

	/**
	 * Reimplemented to specify that we implement the call operation.
	 */
	virtual bool implementsCall() const { return true; }

	virtual KJS::Value call( KJS::ExecState *exec, KJS::Object &self,
				 const KJS::List &args );

    private:
	MethodId id;
	const JSObjectProxy *proxy;
	class Private *d;
    };

    friend class MethodImp;
};

}; // namespace KJSEmbed

#endif // KJSEMBEDJSOBJECTPROXY_H

// Local Variables:
// c-basic-offset: 4
// End:
