#include "field.h"
#include "field.moc"

#include <qwhatsthis.h>
#include <klocale.h>

#include "base/defines.h"
#include "settings.h"
#include "highscores.h"
#include "misc_ui.h"
#include "board.h"


Field::Field(QWidget *parent)
: MPSimpleBoard(parent), BaseField(this)
{
// column 1
    // previous player name
	prevName = new QLabel(" ", this);
	prevName->setAlignment(AlignRight);
	QWhatsThis::add(prevName, i18n("Previous player's name"));
	top->addWidget(prevName, 0, 0);

    // score LCD
    LCDList *sc = new LCDList(i18n("Score"), this);
    showScore = new LCD(6, sc);
    sc->append(showScore);
    QWhatsThis::add(sc, i18n("<qt>Display the current score.<br/>"
                             "It turns <font color=\"blue\">blue</font>"
                             " if it is a highscore "
                             "and <font color=\"red\">red</font> "
                             "if it is the best local score.</qt>"));
    lcds->addWidget(sc, 1, 0);
    lcds->setRowStretch(2, 1);

    // removed LCD
    showRemoved =
        new LCDList(i18n(Factory::self()->mainData().removedLabel), this);
    lcds->addWidget(showRemoved, 3, 0);
    lcds->setRowStretch(4, 1);

	// level LCD
    showLevel = new LCDList(i18n("Level"), this);
    showLevel->append(new LCD(2, showLevel));
    lcds->addWidget(showLevel, 5, 0);
    lcds->setRowStretch(6, 1);
    connect(board, SIGNAL(updateLevelSignal(int)),
			showLevel->lcd(0), SLOT(display(int)));

// column 2
    // gift received shower
	giftReceived = new GiftShower(this);
	QWhatsThis::add(giftReceived, i18n("Light when a \"gift\" is received "
									   "from previous player."));
	top->addWidget(giftReceived, 0, 1, AlignCenter);

    // previous player height
	prevHeight = new KGameProgress(0, ((const Board *)board)->matrix().height(),
							   0, KGameProgress::Vertical, this);
	prevHeight->setFixedWidth(10);
	prevHeight->setBackgroundColor(lightGray);
	prevHeight->setTextEnabled(FALSE);
	prevHeight->setBarColor(blue);
	QWhatsThis::add(prevHeight, i18n("Previous player's height"));
	top->addWidget(prevHeight, 1, 1, AlignHCenter);

// column 3
	// pending gift shower
	top->addWidget(((Board *)board)->giftPool(), 0, 2, AlignCenter);

    // shadow piece
	shadow = new Shadow(board, this);
	QWhatsThis::add(shadow, i18n("Shadow of the current piece"));
	top->addWidget(shadow, 2, 2);

// column 4
	// gift sent shower
	giftSent = new GiftShower(this);
	QWhatsThis::add(giftSent, i18n("Light when you send a \"gift\" to the "
								   "next player."));
	top->addWidget(giftSent, 0, 3, AlignCenter);

    // next player height
	nextHeight = new KGameProgress(0,((const Board *)board)->matrix().height(),
                                   0, KGameProgress::Vertical, this);
	nextHeight->setFixedWidth(10);
	nextHeight->setBackgroundColor(lightGray);
	nextHeight->setTextEnabled(FALSE);
	nextHeight->setBarColor(blue);
	QWhatsThis::add(nextHeight, i18n("Next player's height"));
	top->addWidget(nextHeight, 1, 3, AlignHCenter);

// column 5
    // next player name
	nextName = new QLabel(" ", this);
	QWhatsThis::add(nextName, i18n("Next player's name"));
	top->addWidget(nextName, 0, 4);

    // next piece shower
	QVBoxLayout *vbl = new QVBoxLayout(10);
	top->addLayout(vbl, 1, 4);
	vbl->addStretch(1);

	labShowNext = new QLabel(i18n("Next tile"), this);
	labShowNext->setAlignment(AlignCenter);
	vbl->addWidget(labShowNext, 0);
	showNext = new ShowNextPiece(board, this);
    _snRootPixmap = new KCanvasRootPixmap(showNext);
    _snRootPixmap->start();
	vbl->addWidget(showNext, 0);
	vbl->addStretch(4);

    connect(board, SIGNAL(scoreUpdated(int)), SLOT(scoreUpdatedSlot(int)));
}

Field::~Field()
{}

QSizePolicy Field::sizePolicy() const
{
	return QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);
}

void Field::showOpponents(bool show)
{
    Board *b = static_cast<Board *>(board);
	if (show) {
		prevHeight->show();
		nextHeight->show();
		prevName->show();
		nextName->show();
		giftReceived->show();
		giftSent->show();
		b->giftPool()->show();
	} else {
		prevHeight->hide();
		nextHeight->hide();
		prevName->hide();
		nextName->hide();
		giftReceived->hide();
		giftSent->hide();
		b->giftPool()->hide();
	}
}

void Field::settingsChanged()
{
    BaseField::settingsChanged();
     QColor color = AppearanceSettingsWidget::readFadeColor();
    _snRootPixmap->setFadeEffect(AppearanceSettingsWidget::readFadeStrength(),
                                 color);
    showNext->canvas()->setBackgroundColor(color);
    showNextPiece(GameSettingsWidget::readShowNext());
    showShadow(GameSettingsWidget::readShadow());
}

void Field::showNextPiece(bool show)
{
   if (show) {
       showNext->show();
       labShowNext->show();
   } else {
       showNext->hide();
       labShowNext->hide();
   }
}

void Field::showShadow(bool show)
{
   if (show) shadow->show();
   else shadow->hide();
}

void Field::_init(bool AI, bool multi, bool server, bool first,
                         const QString &name)
{
    BaseField::init(AI, multi, server, first, name);
	showOpponents(multi);
	static_cast<Board *>(board)->setType(AI);
}

void Field::_initFlag(QDataStream &s)
{
	ServerInitData sid;
	s >> sid;
    GTInitData data;
    data.seed = sid.seed;
    data.initLevel = sid.initLevel;
    prevName->setText(sid.prevName);
	nextName->setText(sid.nextName);

    BaseField::init(data);
    shadow->setDisplay(true);
}

void Field::_playFlag(QDataStream &s)
{
	ServerPlayData spd;
	s >> spd;
	prevHeight->setValue(spd.prevHeight);
	nextHeight->setValue(spd.nextHeight);
	if (spd.gift) {
		giftReceived->set();
		static_cast<Board *>(board)->giftPool()->put(spd.gift);
	}
}

void Field::_pauseFlag(bool p)
{
    pause(p);
    shadow->setDisplay(!p);
}

void Field::_dataOut(QDataStream &s)
{
    _cpd.height = board->firstClearLine();
	_cpd.end    = static_cast<Board *>(board)->isGameOver();
    _cpd.gift   = static_cast<Board *>(board)->gift();
	s << _cpd;
	if (_cpd.gift) giftSent->set();
}

KExtHighscores::Score Field::currentScore() const
{
    KExtHighscores::Score score;
    score.setType(_cpd.end ? KExtHighscores::Lost : KExtHighscores::Won);
    score.setData("score", board->score());
    score.setData("level", board->level());
    score.setData("removed", board->nbRemoved());
    return score;
}

void Field::_gameOverDataOut(QDataStream &s)
{
	s << currentScore();
}

void Field::pMoveLeft()    { static_cast<Board *>(board)->pMoveLeft();    };
void Field::pMoveRight()   { static_cast<Board *>(board)->pMoveRight();   };
void Field::pDropDown()    { static_cast<Board *>(board)->pDropDown();    };
void Field::pOneLineDown() { static_cast<Board *>(board)->pOneLineDown(); };
void Field::pRotateLeft()  { static_cast<Board *>(board)->pRotateLeft();  };
void Field::pRotateRight() { static_cast<Board *>(board)->pRotateRight(); };
