#***************************************************************************
#                            kalyptusCxxHTML.pm -  Generates JNI wrappers for C++ headers
#                             -------------------
#    begin                : Fri Oct 20 12:00:00 2000
#    copyright            : (C) 2000-2001 Lost Highway Ltd. All Rights Reserved.
#    email                : Richard_Dale@tipitina.demon.co.uk
#    author               : Richard Dale.
#***************************************************************************/

#/***************************************************************************
# *                                                                         *
# *   This program is free software; you can redistribute it and/or modify  *
# *   it under the terms of the GNU General Public License as published by  *
# *   the Free Software Foundation; either version 2 of the License, or     *
# *   (at your option) any later version.                                   *
# *                                                                         *
#***************************************************************************/

package kalyptusCxxToJava;

use File::Path;
use File::Basename;

use Carp;
use Ast;
use kdocAstUtil;
use kdocUtil; 
use Iter;
use kalyptusDataDict;

use strict;
no strict "subs";

use vars qw/ @clist $host $who $now $gentext %functionId $qtdocTop $kdedocTop
	$lib $rootnode $outputdir $opt $debug $typeprefix $signalCount $eventHandlerCount $constructorCount *JNISOURCE *CLASS *INTERFACE /;




BEGIN
{
	@clist = ();


	# Page footer

	$who = kdocUtil::userName();
	$host = kdocUtil::hostName();
	$now = localtime;
	$gentext = "$who\@$host on $now, using kalyptus $main::Version.";

	$qtdocTop = <<EOF;
                             -------------------
    begin                : $now
    copyright            : (C) 2000-2002 Lost Highway Ltd. All rights reserved.
    email                : Lost_Highway\@tipitina.demon.co.uk
    generated by         : $gentext
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
***************************************************************************/

EOF

	$kdedocTop = <<EOF;
                             -------------------
    begin                : $now
    copyright            : (C) 2000-2002 Lost Highway Ltd.
    email                : Richard_Dale\@tipitina.demon.co.uk
    generated by         : $gentext
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
***************************************************************************/

EOF

}

sub cplusplusToJava
{
	my ( $cplusplusType )  = @_;

	if ( $cplusplusType =~ /bool/ && kalyptusDataDict::ctypemap($cplusplusType) eq "int" ) {
		return "boolean";
	} elsif ( $cplusplusType =~ /bool\s*\*/ ) {
		return "boolean[]";
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /^void\s*\*/ ) {
		return "int";
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /^\s*(unsigned )?int\s*\*/
				|| kalyptusDataDict::ctypemap($cplusplusType) =~ /^qt_QIntValueList\*/ )
	{
		return "int[]";
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /^\s*double\s*\*/ ) {
		return "double[]";
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /^\s*(unsigned )?short\s*\*/ ) {
		return "short[]";
	} elsif ( 	kalyptusDataDict::ctypemap($cplusplusType) =~ /qt_QStringList\s*\*/
				|| kalyptusDataDict::ctypemap($cplusplusType) =~ /qt_QStrList\s*\*/
				|| kalyptusDataDict::ctypemap($cplusplusType) =~ /kde_QCStringList\s*\*/
				|| kalyptusDataDict::ctypemap($cplusplusType) =~ /qt_QObjectList\s*\*/
				|| kalyptusDataDict::ctypemap($cplusplusType) =~ /qt_QDomNodeList\s*\*/
				|| kalyptusDataDict::ctypemap($cplusplusType) =~ /qt_QWidgetList\s*\*/
				|| kalyptusDataDict::ctypemap($cplusplusType) =~ /kde_KURLList\s*\*/
				|| kalyptusDataDict::ctypemap($cplusplusType) =~ /kde_KMainWindow\s*\*/
				|| kalyptusDataDict::ctypemap($cplusplusType) =~ /kde_KFileItemList\s*\*/
				|| kalyptusDataDict::ctypemap($cplusplusType) =~ /kde_KFileViewItemList\s*\*/
				|| kalyptusDataDict::ctypemap($cplusplusType) =~ /kde_DOMNodeList\s*\*/
				|| kalyptusDataDict::ctypemap($cplusplusType) =~ /kde_StyleSheetList\s*\*/
				|| kalyptusDataDict::ctypemap($cplusplusType) =~ /kde_MediaList\s*\*/
				|| kalyptusDataDict::ctypemap($cplusplusType) =~ /kde_OfferList\s*\*/
				|| kalyptusDataDict::ctypemap($cplusplusType) =~ /qt_QCanvasItemList\s*\*/ ) {
		return "ArrayList"
	} elsif ( $cplusplusType =~ /(DOM::)?DOMString/ || $cplusplusType =~ /QString/ || $cplusplusType =~ /QCString/ || kalyptusDataDict::ctypemap($cplusplusType) =~ /^char\s*\*/ ) {
		return "String"
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /qt_QChar\s*\*/ ) {
		return "char"
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /qt_QTime\s*\*/ ) {
		return "Date"
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /qt_QDateTime\s*\*/ || kalyptusDataDict::ctypemap($cplusplusType) =~ /qt_QDate\s*\*/ ) {
		return "Calendar"
	} elsif ( $cplusplusType =~ /QPaintDevice/ ) {
		return "QPaintDeviceInterface"
	} elsif ( $cplusplusType =~ /QByteArray/ ) {
		return "byte[]"
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /qt_([^\*\s]*)(.*)$/ ) {
		if ( kalyptusDataDict::interfacemap($1) ne () ) {
			return $1."Interface";
        } else {
			return $1;
		}
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /kde_([^\*\s]*)(.*)$/ ) {
		if ( kalyptusDataDict::interfacemap($1) ne () ) {
			return $1."Interface";
        } else {
			return $1;
		}
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /(const)? char *\s*/ ) {
		return "String";
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /(unsigned )?(.*)/ ) {
		return $2;
	} else {
		return kalyptusDataDict::ctypemap($cplusplusType);
	}

}

sub cplusplusToJNI
{
	my ( $cplusplusType )  = @_;

	if ( $cplusplusType =~ /bool/ && kalyptusDataDict::ctypemap($cplusplusType) eq "int" ) {
		return "jboolean";
	} elsif ( $cplusplusType =~ /bool\s*\*/ ) {
		return "jbooleanArray";
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /^\s*void\s*\*/ ) {
		return "jint";
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /^\s*(unsigned )?int\s*\*/
				|| kalyptusDataDict::ctypemap($cplusplusType) =~ /^qt_QIntValueList\*/ )
	{
		return "jintArray";
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /^\s*double\s*\*/ ) {
		return "jdoubleArray";
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /^\s*(unsigned )?short\s*\*/ ) {
		return "jshortArray";
	} elsif ( 	kalyptusDataDict::ctypemap($cplusplusType) =~ /qt_QStringList\s*\*/
				|| kalyptusDataDict::ctypemap($cplusplusType) =~ /qt_QStrList\s*\*/
				|| kalyptusDataDict::ctypemap($cplusplusType) =~ /kde_QCStringList\s*\*/
				|| kalyptusDataDict::ctypemap($cplusplusType) =~ /qt_QObjectList\s*\*/
				|| kalyptusDataDict::ctypemap($cplusplusType) =~ /qt_QDomNodeList\s*\*/
				|| kalyptusDataDict::ctypemap($cplusplusType) =~ /kde_DOMNodeList\s*\*/
				|| kalyptusDataDict::ctypemap($cplusplusType) =~ /kde_KFileItemList\s*\*/
				|| kalyptusDataDict::ctypemap($cplusplusType) =~ /kde_KFileItemViewList\s*\*/
				|| kalyptusDataDict::ctypemap($cplusplusType) =~ /kde_OfferList\s*\*/
				|| kalyptusDataDict::ctypemap($cplusplusType) =~ /kde_MediaList\s*\*/
				|| kalyptusDataDict::ctypemap($cplusplusType) =~ /kde_StyleSheetList\s*\*/
				|| kalyptusDataDict::ctypemap($cplusplusType) =~ /kde_KURLList\s*\*/
				|| kalyptusDataDict::ctypemap($cplusplusType) =~ /qt_QWidgetList\s*\*/
				|| kalyptusDataDict::ctypemap($cplusplusType) =~ /qt_QCanvasItemList\s*\*/ ) {
		return "jobjectArray"
	} elsif ( $cplusplusType =~ /(DOM::)?DOMString/ || $cplusplusType =~ /QString/ || $cplusplusType =~ /QCString/ || kalyptusDataDict::ctypemap($cplusplusType) =~ /^char\s*\*/ ) {
		return "jstring"
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /qt_QDateTime\s*\*/ || kalyptusDataDict::ctypemap($cplusplusType) =~ /qt_QTime\s*\*/ || kalyptusDataDict::ctypemap($cplusplusType) =~ /qt_QDate\s*\*/ ) {
		return "jobject"
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /qt_QChar\s*\*/  ) {
		return "jchar"
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /qt_QByteArray\s*\*/  ) {
		return "jbyteArray"
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /qt_([^\*]*)(.*)$/ ) {
		return "jobject";
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /kde_([^\*]*)(.*)$/ ) {
		return "jobject";
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /(const)? char *\s*/ ) {
		return "jstring";
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /(unsigned\s+)(.*)/ ) {
		return "j".$2;
	} elsif ( $cplusplusType eq "void" || $cplusplusType eq "" ) {
		return "void";
	} else {
		return "j".kalyptusDataDict::ctypemap($cplusplusType);
	}

}

sub cplusplusToJNISignature
{
	my ( $cplusplusType )  = @_;

	if ( cplusplusToJava( $cplusplusType ) =~ /Calendar/ ) {
		return "Ljava_util_Calendar_2"
	} elsif ( cplusplusToJava( $cplusplusType ) =~ /Date/ ) {
		return "Ljava_util_Date_2"
	} elsif ( 	cplusplusToJava( $cplusplusType ) =~ /ArrayList/ ) {
		return "Ljava_util_ArrayList_2"
	} elsif ( cplusplusToJava( $cplusplusType ) =~ /String/ ) {
		return "Ljava_lang_String_2";
	} elsif ( cplusplusToJava( $cplusplusType ) =~ /boolean\[\]/ ) {
		return "_3Z";
	} elsif ( cplusplusToJava( $cplusplusType ) =~ /byte\[\]/ ) {
		return "_3B";
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /void\s*\*/ ) {
		return "I";
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /^\s*(unsigned )?int\s*\*/
				|| kalyptusDataDict::ctypemap($cplusplusType) =~ /^qt_QIntValueList\*/ )
	{
		return "_3I";
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /^\s*double\s*\*/ ) {
		return "_3D";
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /^\s*(unsigned )?short\s*\*/ ) {
		return "_3S";
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /^\s*qt_QChar\s*\*/ ) {
		return "C";
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /qt_([^\*\s]*)(.*)$/ ) {
		if ( kalyptusDataDict::interfacemap($1) ne () ) {
			return "Lorg_kde_qt_$1"."Interface_2";
        } else {
			return "Lorg_kde_qt_$1"."_2";
		}
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /kde_([^\*\s]*)(.*)$/ ) {
		if ( kalyptusDataDict::interfacemap($1) ne () ) {
			return "Lorg_kde_koala_$1"."Interface_2";
        } else {
			return "Lorg_kde_koala_$1"."_2";
		}
	} elsif ( cplusplusToJava( $cplusplusType ) =~ /^boolean\s*/ ) {
		return "Z";
	} elsif ( cplusplusToJava( $cplusplusType ) =~ /^byte\s*/ ) {
		return "B";
	} elsif ( cplusplusToJava( $cplusplusType ) =~ /^char\s*/ ) {
		return "C";
	} elsif ( cplusplusToJava( $cplusplusType ) =~ /^short\s*/ ) {
		return "S";
	} elsif ( cplusplusToJava( $cplusplusType ) =~ /^int\s*/ ) {
		return "I";
	} elsif ( cplusplusToJava( $cplusplusType ) =~ /^long\s*/ ) {
		return "J";
	} elsif ( cplusplusToJava( $cplusplusType ) =~ /^float\s*/ ) {
		return "F";
	} elsif ( cplusplusToJava( $cplusplusType ) =~ /^double\s*/ ) {
		return "D";
	} else {
		return "";
	}

}

sub writeDoc
{
	( $lib, $rootnode, $outputdir, $opt ) = @_;

	$debug = $main::debug;

	mkpath( $outputdir ) unless -f $outputdir;


	# Document all compound nodes
	Iter::LocalCompounds( $rootnode, sub { writeClassDoc( shift ); } );

}




=head2 writeClassDoc

	Write documentation for one compound node.

=cut

sub writeClassDoc
{
	my( $node ) = @_;

	print "Enter: $node->{astNodeName}\n" if $debug;

	my $typeName = $node->{astNodeName}."*";

	if ( kalyptusDataDict::ctypemap($typeName) eq () ) {
		$typeprefix = ($typeName =~ /^Q/ ? "qt_" : "kde_");
		kalyptusDataDict::setctypemap($typeName, $typeprefix.$node->{astNodeName}."*");
		print "'$typeName' => '$typeprefix$typeName',\n";
	} elsif ( kalyptusDataDict::ctypemap($typeName) =~ /^qt_/ ) {
		$typeprefix = "qt_";
	} elsif ( kalyptusDataDict::ctypemap($typeName) =~ /^kde_/ ) {
		$typeprefix = "kde_";
	}

	my $docnode = $node->{DocNode};
	my @list = ();
	my $version = undef;
	my $author = undef;


	if( $#{$node->{Kids}} < 0 || $node->{Access} eq "private" || exists $node->{Tmpl} ) {
		return;
	}

#	If the interface has the same name as the class, then junk the .java implementation and .cpp
#	output into /dev/null
	my $file;
	if ( kalyptusDataDict::interfacemap($node->{astNodeName}) ne $node->{astNodeName} ) {
		$file = "$outputdir/".join("__", kdocAstUtil::heritage($node)).".java";
	} else {
		$file = "/dev/null";
	}

	open( CLASS, ">$file" ) || die "Couldn't create $file\n";

	if ( kalyptusDataDict::interfacemap($node->{astNodeName}) ne $node->{astNodeName} ) {
		$file = "$outputdir/".join("__", kdocAstUtil::heritage($node)).".cpp";
	} else {
		$file = "/dev/null";
	}
	open( JNISOURCE, ">$file" ) || die "Couldn't create $file\n";

	if ( kalyptusDataDict::interfacemap($node->{astNodeName}) ne () ) {
		$file = "$outputdir/".join("__", kdocAstUtil::heritage($node)).".java";
		my $interfaceName = kalyptusDataDict::interfacemap($node->{astNodeName});
		$file =~ s/$node->{astNodeName}/$interfaceName/;
		open( INTERFACE, ">$file" ) || die "Couldn't create $file\n";
		print INTERFACE "/***************************************************************************\n";
		print INTERFACE "                            ", kalyptusDataDict::interfacemap($node->{astNodeName}), ".java -  description\n";
		if ( $typeprefix eq "qt_" ) {
			print INTERFACE $qtdocTop;
			print INTERFACE "package org.kde.qt;\n\n";
        } else {
			print INTERFACE $kdedocTop;
			print INTERFACE "package org.kde.koala;\n\n";
			print INTERFACE "import org.kde.qt.*;\n";
 		}
 		print INTERFACE "public interface ", kalyptusDataDict::interfacemap($node->{astNodeName}), " {\n";

	}

	my $short = "";
	my $extra = "";

	print CLASS "/***************************************************************************\n";
	print CLASS "                            ", $node->{astNodeName}, ".java -  description\n";
	if ( $typeprefix eq "qt_" ) {
		print CLASS $qtdocTop;
    } else {
		print CLASS $kdedocTop;
	}

	print JNISOURCE "/***************************************************************************\n";
	print JNISOURCE "                            ", $node->{astNodeName}, ".cpp -  description\n";
	if ( $typeprefix eq "qt_" ) {
		print JNISOURCE $qtdocTop;
		print CLASS "package org.kde.qt;\n\n";
    } else {
		print JNISOURCE $kdedocTop;
		print CLASS "package org.kde.koala;\n\n";
		print CLASS "import org.kde.qt.*;\n";
	}

	my $sourcename = $node->{Source}->{astNodeName};

	if ( $sourcename =~ m!.*(dom|kabc|kdeprint|kdesu|kio|kjs|kparts|ktexteditor|libkmid)/([^/]*$)! ) {
		$sourcename = $1."/".$2;
	} else {
		$sourcename =~ s!.*/([^/]*$)!$1!;
	}

	print JNISOURCE "#define _BOOL_H_\n";
	print JNISOURCE "#include <",$sourcename , ">\n";
	print JNISOURCE "#include <qstring.h>\n";
	print JNISOURCE "#include <qcstring.h>\n\n";

	print JNISOURCE "#include <qtjava/QtSupport.h>\n";

	if ( $typeprefix eq "qt_" ) {
		print JNISOURCE "#include <qtjava/", $node->{astNodeName}, ".h>\n";
	} else {
		print JNISOURCE "#include <kdejava/KDESupport.h>\n";
		print JNISOURCE "#include <kdejava/", $node->{astNodeName}, ".h>\n";
	}

		# ancestors
	my @ancestors = ();
	Iter::Ancestors( $node, $rootnode, undef, undef,
		sub { # print
			my ( $ances, $name, $type, $template ) = @_;
			push @ancestors, $name;
			},
			undef
		);

	print CLASS "import java.util.*;\n";
	print CLASS "import java.lang.Error;\n\n";

	%functionId = ();
	$signalCount = 0;
	$eventHandlerCount = 0;
	$constructorCount = 0;

	# Add any enums found to the %ctypemap
	Iter::MembersByType ( $node,
		sub { print CLASS "", $_[0], ""; print JNISOURCE "", $_[0], "";  },
		sub {	my ($node, $kid ) = @_;
                 preParseMember( $node, $kid );
               },
		sub { print CLASS ""; print JNISOURCE ""; }
	);


	if ( ! exists $node->{Pure} && $constructorCount > 0 ) {
		print JNISOURCE "\nclass ", $node->{astNodeName}, "JBridge : public ", kalyptusDataDict::addNamespace($node->{astNodeName}), "\n{\npublic:\n";

		Iter::MembersByType ( $node,
			sub { print HEADER "", $_[0], ""; print CLASS "", $_[0], ""; },
			sub {	my ($node, $kid ) = @_;
                    generateBridgeClass( $node, $kid );
                 },
			sub { ; }
			);

		generateBridgeEventHandlers($node);

		print JNISOURCE "};\n\n";
	}

	if ( defined $docnode ) {
		print CLASS "/**\n";
		if ( defined $docnode->{Text} ) {
			printJavadocComment( $docnode, \*CLASS, "" );
		}

		if ($signalCount > 0) {
			print CLASS " See {\@link ", $node->{astNodeName}, "Signals} for signals emitted by ",$node->{astNodeName}, ".\n\n";
   		}

		if ($eventHandlerCount > 0) {
			print CLASS " See {\@link ", $node->{astNodeName}, "EventHandling} for event handlers to override in subclasses of ", $node->{astNodeName}, ".\n\n";
   		}

		exists $docnode->{Author} && print CLASS " \@author ", $docnode->{Author}, "\n";
		exists $docnode->{Version} && print CLASS " \@version ", $docnode->{Version}, "\n";
		exists $docnode->{ClassShort} && print CLASS " \@short ", $docnode->{ClassShort}, "\n";

		print CLASS "*/\n";
	} elsif ($signalCount > 0 || $eventHandlerCount > 0) {
		print CLASS "/**\n";
		if ($signalCount > 0) {
			print CLASS " See {\@link ", $node->{astNodeName}, "Signals} for signals emitted by ",$node->{astNodeName}, ".\n\n";
   		}

		if ($eventHandlerCount > 0) {
			print CLASS " See {\@link ", $node->{astNodeName}, "EventHandling} for event handlers to override in subclasses of ", $node->{astNodeName}, ".\n\n";
   		}
		print CLASS "*/\n";
	}

	if ( $#ancestors < 0 ) {
		print CLASS "public class  ", $node->{astNodeName}, " implements QtSupport";

		if ( kalyptusDataDict::interfacemap($node->{astNodeName}) ne () ) {
			print CLASS ", ".kalyptusDataDict::interfacemap($node->{astNodeName});
		}

		print CLASS " {\n\tprivate long _qt;\n";
		print CLASS "\tprivate boolean _allocatedInJavaWorld = true;\n\n";
	} else {
		print CLASS "public class ", $node->{astNodeName}, " extends ";
		my $ancestor;
		foreach $ancestor ( @ancestors ) {
			if ( kalyptusDataDict::interfacemap($ancestor) eq () ) {
				print CLASS "$ancestor ";
				last;
           	} elsif ($ancestor eq @ancestors[$#ancestors] ) {
				print CLASS @ancestors[$#ancestors], " ";
			}
		}

		if ( $#ancestors >= 1 ) {
			print CLASS "implements ";
			foreach $ancestor ( @ancestors ) {
				if ( kalyptusDataDict::interfacemap($ancestor) ne () ) {
					print CLASS " ".kalyptusDataDict::interfacemap($ancestor);
				}
			}
        }

		if ( kalyptusDataDict::interfacemap($node->{astNodeName}) ne () ) {
			print CLASS ", ".kalyptusDataDict::interfacemap($node->{astNodeName});
		}

		print CLASS " {\n";
	}

	if ( $#ancestors >= 0 ) {
		print CLASS "\tprotected ", $node->{astNodeName}, "(Class dummy){super((Class) null);}\n\n";
    } else {
		print CLASS "\tprotected ", $node->{astNodeName}, "(Class dummy){}\n\n";
	}

	Iter::MembersByType ( $node,
		sub { print CLASS "", $_[0], ""; print JNISOURCE "", $_[0], "";  },
		sub {	my ($node, $kid ) = @_;
                 parseMember( $node, $kid, $#ancestors );
               },
		sub { print CLASS ""; print JNISOURCE ""; }
	);


	Iter::MembersByType ( $node,
		sub { print CLASS "", $_[0], ""; print JNISOURCE "", $_[0], "";  },
		sub {	my ($node, $kid ) = @_;
                 generateClassMethodForEnum( $node, $kid );
               },
		sub { print CLASS ""; print JNISOURCE ""; }
	);

	print CLASS "}\n";

	if ( kalyptusDataDict::interfacemap($node->{astNodeName}) ne () ) {
		print INTERFACE "}\n";
		close INTERFACE;
    }

	if ( $signalCount > 0 ) {
		$file = "$outputdir/".join("__", kdocAstUtil::heritage($node)).".java";
		$file =~ s/$node->{astNodeName}/$node->{astNodeName}Signals/;
		open( INTERFACE, ">$file" ) || die "Couldn't create $file\n";

		if ( $typeprefix eq "qt_" ) {
			print INTERFACE "package org.kde.qt;\n\n";
        } else {
			print INTERFACE "package org.kde.koala;\n";
			print INTERFACE "import org.kde.qt.*;\n\n";
 		}

		print INTERFACE "\n/\*\* {\@link ",
			$node->{astNodeName}, "} emits these signals \*/\n";
		print INTERFACE "public interface ", $node->{astNodeName}, "Signals {\n";
		generateSignals($node);
		print INTERFACE "}\n\n";
		close INTERFACE;
	}

	if ( $eventHandlerCount > 0 ) {
		$file = "$outputdir/".join("__", kdocAstUtil::heritage($node)).".java";
		$file =~ s/$node->{astNodeName}/$node->{astNodeName}EventHandling/;
		open( INTERFACE, ">$file" ) || die "Couldn't create $file\n";

		if ( $typeprefix eq "qt_" ) {
			print INTERFACE "package org.kde.qt;\n\n";
        } else {
			print INTERFACE "package org.kde.koala;\n";
			print INTERFACE "import org.kde.qt.*;\n\n";
 		}

		print INTERFACE "\n/\*\* Override these methods in subclasses of {\@link ",
			$node->{astNodeName}, "} to customise event handling behaviour \*/\n";
		print INTERFACE "public interface ", $node->{astNodeName}, "EventHandling {\n";
		generateEventHandlers($node);
		print INTERFACE "}\n\n";
		close INTERFACE;
	}

	close CLASS;
	print JNISOURCE "\n";
	close JNISOURCE;


}



sub generateSignals
{
	my( $node ) = @_;
	my %allmem = ();
	my $key;

	my $m;
	my $name;

	kdocAstUtil::allMembers( \%allmem, $node );

	foreach $key (keys (%allmem)) {
		$m = $allmem{$key};
		$name = $m->{astNodeName} ;
		my $type = $m->{NodeType};
		my $docnode = $m->{DocNode};

	if( $m->{Access} eq "signals" ) {
		my $cplusplusparams = $m->{Params};
		my $javaparams;
		$cplusplusparams =~ s/\s+/ /g;
		$cplusplusparams =~ s/\s*([,\*\&])\s*/$1 /g;
		$cplusplusparams =~ s/^\s*$//;
		$cplusplusparams =~ s/^\s*void\s*$//;
		my $argId = 0;
		my @cargs = split(",", $cplusplusparams);
		$cplusplusparams = "";
		foreach my $arg ( @cargs ) {
			my $argType;
			my $cargType;
			my $javaargType;
			$arg =~ s/\s*([^\s].*[^\s])\s*/$1/;
			$arg =~ s/(\w+)\[\]/\* $1/;

			if ( $arg =~ /^(.*)\s+(\w+)\s*$/ ) {
				$argType = $1;
				$arg = $2;
			} else {
				$argType = $arg;
				$argId++;
				$arg = "arg".$argId;
			}
			$arg =~ s/^id$/identifier/;
			$argType =~ s/\s*([^\s].*[^\s])\s*/$1/;
			$argType =~ s/\s*const//g;
			$argType =~ s/^\s*//;
			$argType =~ s/([\*\&])\s*([\*\&])/$1$2/;
			$cargType = kalyptusDataDict::ctypemap($argType);
			$javaargType = cplusplusToJava( $argType );
			if ( $javaargType eq "" ) {
				$javaargType = $argType;
				$javaargType =~ s/\&/\*/;
				$javaargType =~ s/^.*::.*$/int/;
			}

			if ( kalyptusDataDict::ctypemap($argType) =~ /qt_QStr(ing)?List\s*\*/ ) {
				$javaparams .= " String[] $arg,";
			} else {
				$javaparams .= " $javaargType $arg,";
			}
		}

		$javaparams =~ s/,$/ /;

		if ( defined $docnode ) {
			if ( defined $docnode->{Text} ) {
				print INTERFACE "\n/** ";
				printJavadocComment( $docnode, \*INTERFACE, $node->{astNodeName} );
				print INTERFACE "*/\n";
			}
		}

		print INTERFACE "\tvoid $name($javaparams);\n";
	}
	}

}

sub generateEventHandlers
{
	my( $node ) = @_;
	my %allmem = ();
	my $key;

	my $m;
	my $name;

	kdocAstUtil::allMembers( \%allmem, $node );

	foreach $key (keys (%allmem)) {
		$m = $allmem{$key};
		$name = $m->{astNodeName} ;
		my $type = $m->{NodeType};
		my $docnode = $m->{DocNode};

	if( $type eq "method" && $m->{Access} eq "protected"  && $name =~ /.*Event$/
		&& $name !~ /qwsEvent/ && $name !~ /x11Event/ && $name !~ /winEvent/ && $name !~ /macEvent/  && $name !~ /movableDropEvent/ ) {

		my $cparams = $m->{Params};
		my $javaparams;
		$cparams =~ s/=\s*[-"\w]*//g;
		$cparams =~ s/\s+/ /g;
		$cparams =~ s/\s*([,\*\&])\s*/$1 /g;
		$cparams =~ s/^\s*void\s*$//;
		my $argId = 0;
		my @cargs = split(",", $cparams);
		$cparams = "";
		foreach my $arg ( @cargs ) {
			my $argType;
			my $cargType;
			$arg =~ s/\s*([^\s].*[^\s])\s*/$1/;
			if ( $arg =~ /(.*)\s+(\w+)$/ ) {
				$argType = $1;
				$arg = $2;
			} else {
				$argType = $arg;
				$argId++;
				$arg = "arg".$argId;
			}
			$argType =~ s/\*//;
			$javaparams .= $argType." ".$arg.", ";
			$cparams .= $arg.", ";
		}
		$cparams =~ s/, $//;
		$javaparams =~ s/, $//;

		if ( defined $docnode ) {
			if ( defined $docnode->{Text} ) {
				print INTERFACE "\n/** ";
				printJavadocComment( $docnode, \*INTERFACE, $node->{astNodeName} );
				print INTERFACE "*/\n";
			}
		}

		print INTERFACE "\tvoid $name($javaparams);\n";
	}
	}

}

sub generateBridgeEventHandlers
{
	my( $node ) = @_;
	my %allmem = ();
	my $key;

	my $m;
	my $name;

	kdocAstUtil::allMembers( \%allmem, $node );

	foreach $key (keys (%allmem)) {
		$m = $allmem{$key};
		$name = $m->{astNodeName} ;
		my $type = $m->{NodeType};
		my $docnode = $m->{DocNode};
		my $cparams = $m->{Params};
		my $parent = $m->{Parent};
		my $cplusplusparams;

		if ($m->{Access} eq "signals") {
			$signalCount++;
		}
	
		$cparams =~ s/=\s*[-"\w]*//g;
		$cparams =~ s/\s+/ /g;
		$cparams =~ s/\s*([,\*\&])\s*/$1 /g;
		$cparams =~ s/^\s*void\s*$//;
		$cparams =~ s/^\s*$//;
		my $argId = 0;
		my @cargs = split(",", $cparams);
		$cparams = "";
		foreach my $arg ( @cargs ) {
			my $argType;
			my $cargType;
			$arg =~ s/\s*([^\s].*[^\s])\s*/$1/;
			if ( $arg =~ /(.*)\s+(\w+)$/ ) {
				$argType = $1;
				$arg = $2;
			} else {
				$argType = $arg;
				$argId++;
				$arg = "arg".$argId;
			}
			$cplusplusparams .= $argType." ".$arg.", ";
			$cparams .= $arg.", ";
		}
		$cparams =~ s/, $//;
		$cplusplusparams =~ s/, $//;
	
		if( $type eq "method" && $m->{Access} eq "protected"  && $name =~ /.*Event$/
			&& $name !~ /qwsEvent/ && $name !~ /x11Event/ && $name !~ /winEvent/ && $name !~ /macEvent/ && $name !~ /movableDropEvent/ ) {
			$eventHandlerCount++;

			my $eventType = $cplusplusparams;
			$eventType =~ s/(.*)\*.*$/$1/;
			$eventType =~ s/^([^Q].*)/org.kde.koala.$1/;
			$eventType =~ s/^(Q.*)/org.kde.qt.$1/;
			print JNISOURCE "\tvoid $name(", $cplusplusparams, ") {\n",
			"\t\tif (! QtSupport::eventDelegate(this, \"", $name, "\", $cparams, \"$eventType\")) {\n",
			"\t\t\t", $parent->{astNodeName}, "::", $name, "($cparams);\n",
			"\t\t}\n",
			"\t\treturn;\n\t}\n";
		} elsif( $type eq "method" && $m->{Flags} =~ "v" ) {
			;
		}
	}

}


sub preParseMember
{
	my( $class, $m ) = @_;
	my $name = $m->{astNodeName};

	if( $m->{NodeType} eq "method" && $m->{ReturnType} !~ /~/) {
		if ( $functionId{$name} eq "" ) {
			$functionId{$name} = 0;
		} else {
			$functionId{$name}++;
		}

		# If there are any default parameters (ie '=' found), then an extra method call will be generated
		if ( $m->{Params} =~ /=/ ) {
			$functionId{$name}++;
		}

		# A JBridge class will only be generated if there is at least one
		# public or protected constructor
		if ( $name eq $class->{astNodeName} && $m->{Access} ne "private" ) {
			$constructorCount++;
		}
    }

	if( $m->{NodeType} eq "enum" ) {
		# Add a C++ to C type mapping for this enum - an int in C
		$name =~ s/\s//g;
		kalyptusDataDict::setctypemap($name, 'int');
		$name = $class->{astNodeName}."::".$name;
		kalyptusDataDict::setctypemap($name, 'int');
	}
}

sub generateBridgeClass
{
	my( $class, $m ) = @_;
	my $name;
    my $function;

	$name = $m->{astNodeName} ;
	my $type = $m->{NodeType};
	my $docnode = $m->{DocNode};

	if( $type eq "method" && $m->{Access} ne "private" && $m->{Access} ne "private_slots" && $m->{Access} ne "signals" ) {
		if ( $m->{Params} =~ /Impl/ ) {
			return;
		}

		my $returnType = $m->{ReturnType};
		my $cparams = $m->{Params};
		my $cplusplusparams;
		if ( $cparams =~ /\s*\)\s*:\s*/ ) {
			$cparams =~ s/(.*)\s*\)\s*:\s*.*$/$1/;
		}

		$cparams =~ s/=\s*(("[^"]*")|(\'.\')|(([-\w:.]*)\s*(\|\s*[-\w]*)*(\(\w*\))?))//g;
		$cparams =~ s/\s+/ /g;
		$cparams =~ s/\s*([,\*\&])\s*/$1 /g;
		$cparams =~ s/^\s*void\s*$//;
		$cparams =~ s/^\s*$//;
		my $argId = 0;
		my @cargs = split(",", $cparams);
		$cparams = "";
		foreach my $arg ( @cargs ) {
			my $argType;
			my $cargType;
			$arg =~ s/\s*([^\s].*[^\s])\s*/$1/;
			if ( $arg =~ /(.*)\s+(\w+)$/ ) {
				$argType = $1;
				$arg = $2;
			} else {
				$argType = $arg;
				$argId++;
				$arg = "arg".$argId;
			}
			$cplusplusparams .= $argType." ".$arg.", ";
			$cparams .= $arg.", ";
		}
		$cparams =~ s/, $//;
		$cplusplusparams =~ s/, $//;

		my $flags = $m->{Flags};

		if ( !defined $flags ) {
			warn "Method ".$m->{astNodeName}.  " has no flags\n";
		}

		my $extra = "";
		$extra .= "static " if $flags =~ "s";
		if ( $name =~ /operator/ || $extra =~ /friend|inline/ ) {
			return;
		}


		if ( $name eq $class->{astNodeName} ) {
			if ( $returnType =~ "~" ) {
				print JNISOURCE "\t~", $name, "JBridge() {QtSupport::qtKeyDeleted(this);}\n";
			} else {
				print JNISOURCE $extra,
					"\t", $name, "JBridge(", $cplusplusparams, ") : $name($cparams) {}\n";
			}
#		} elsif( $type eq "method" && $m->{Access} eq "protected"  && $name =~ /.*Event$/ ) {
#			;
		} elsif( $m->{Access} =~ /^protected/ ){
			if ( $returnType =~ "void" ) {
				print JNISOURCE "\tvoid protected_$name(", $cplusplusparams, ") {\n",
				"\t\t", $class->{astNodeName}, "::$name($cparams);\n",
				"\t\treturn;\n\t}\n";
			} else {
				print JNISOURCE "\t$returnType protected_$name(", $cplusplusparams, ") {\n",
				"\t\treturn ($returnType) ", $class->{astNodeName}, "::$name($cparams);\n\t}\n";
			}
		}
	}

}

sub parseMember
{
	my( $class, $m, $ancestorCount ) = @_;
	my $name;
    my $jniFunction;
	my $jniPackage;
    my $stringargs;
	my $cplusplusparams = $m->{Params};
	my $javaparams;
	my $jniparams;
	my $jnireturntype;
	my $javaaccess;
	my $javaargs;

	my $defaultstringargs;
	my $defaultcplusplusparams;
	my $defaultjavaparams;
	my $defaultjniFunction;
	my $defaultjniparams;
	my $defaultjavaargs;


	$name = $m->{astNodeName} ;
	my $type = $m->{NodeType};

	my $docnode = $m->{DocNode};

	if ( $m->{ReturnType} =~ /~/ ) {
		$name = "~".$name;
	}

	if( $type eq "method" && $m->{Access} ne "private" && $m->{Access} ne "private_slots" && $m->{Access} ne "signals" ) {
		if ( $m->{Params} =~ /\.\.\./ || $m->{Params} =~ /Impl/ ) {
			return;
		}

		my $returnType = $m->{ReturnType};

		$returnType =~ s/friend\s*//;
		$returnType =~ s/const\s*//;
		$returnType =~ s/inline\s*//;
		$returnType =~ s/Q_EXPORT\s*//;
		$returnType =~ s/\s*([,\*\&])\s*/$1/;
		$returnType =~ s/^\s*//;
		$returnType =~ s/\s*$//;

		if ( $returnType ne "" && cplusplusToJava($returnType) eq () ) {
			$returnType =~ s/^.*::.*$/int/;
		} else {
			$jnireturntype = cplusplusToJNI($returnType);
			if ( $jnireturntype =~ /jobjectArray/ ) {
				$jnireturntype = "jobject";
			}
			$returnType = cplusplusToJava($returnType);
		}

		$jniFunction = $class->{astNodeName};
		$jniFunction =~ s/_/_1/g;

		if ( $jniFunction =~ /^Q/ ) {
			$jniPackage = "Java_org_kde_qt_";
		} else {
			$jniPackage = "Java_org_kde_koala_";
		}

		$jniFunction = "\nJNIEXPORT $jnireturntype JNICALL\n".$jniPackage.$jniFunction."_$name";
		if ( $functionId{$name} > 0 ) {
			$jniFunction .= "__";
		}

		if ( $cplusplusparams =~ /\s*\)\s*:\s*/ ) {
			$cplusplusparams =~ s/(.*)\s*\)\s*:\s*.*$/$1/;
		}

		$cplusplusparams =~ s/\s+/ /g;
		$cplusplusparams =~ s/\s*([,\*\&])\s*/$1 /g;
		$cplusplusparams =~ s/^\s*$//;
		$cplusplusparams =~ s/^\s*void\s*$//;
		my $argId = 0;
		my @cargs = split(",", $cplusplusparams);
		$cplusplusparams = "";
		foreach my $arg ( @cargs ) {
			my $argType;
			my $cargType;
			my $javaargType;
			$arg =~ s/\s*([^\s].*[^\s])\s*/$1/;
			$arg =~ s/(\w+)\[\]/\* $1/;

			if ( $arg =~ /^\s*$/ ) {
				next;
			}

			# A '<arg> = <value>' default parameter
			if ( $arg =~ s/=\s*(("[^"]*")|(\'.\')|(([-\w:.]*)\s*(\|\s*[-\w]*)*(\(\w*\))?))// ) {
				my $param = $1;
				if ( $defaultcplusplusparams eq "" ) {
					$defaultcplusplusparams = $cplusplusparams;
					$defaultjavaparams = $javaparams;
					$defaultstringargs = $stringargs;
					$defaultjniFunction = $jniFunction;
					$defaultjniparams = $jniparams;
					$defaultjavaargs = $javaargs;
				}

				if ( $param =~ /(.*)::(.*)/ && kalyptusDataDict::ctypemap($param) eq ()) {
					$defaultcplusplusparams .= "$1::$2, ";
				} else {
					if ( kalyptusDataDict::ctypemap($param) eq () ) {
						if ( $param =~ /^[A-Z].*[^\)]$/ && $param !~ /NULL/ ) {
							$defaultcplusplusparams .= $class->{astNodeName}."::".$param.", ";
						} else {
							$defaultcplusplusparams .= $param.", ";
                    	}
					} else {
						$defaultcplusplusparams .= $param.", ";
					}
				}
			}

			if ( $arg =~ /^(.*)\s+(\w+)\s*$/ ) {
				$argType = $1;
				$arg = $2;
			} else {
				$argType = $arg;
				$argId++;
				$arg = "arg".$argId;
			}
			$arg =~ s/^id$/identifier/;
			$argType =~ s/\s*([^\s].*[^\s])\s*/$1/;
			$argType =~ s/\s*const//g;
			$argType =~ s/^\s*//;
			$argType =~ s/([\*\&])\s*([\*\&])/$1$2/;
			$cargType = kalyptusDataDict::ctypemap($argType);
			$javaargType = cplusplusToJava( $argType );
			if ( $javaargType eq "" ) {
				$javaargType = $argType;
				$javaargType =~ s/\&/\*/;
				$javaargType =~ s/^.*::.*$/int/;
			}

			$arg =~ s/^super$/superClass/;
			$jniparams .= ", ".cplusplusToJNI( $argType )." ".$arg;

			if ( kalyptusDataDict::ctypemap($argType) =~ /qt_QStr(ing)?List\s*\*/ ) {
				$javaparams .= " String[] $arg,";
			} else {
				$javaparams .= " $javaargType $arg,";
			}
			$javaargs .= " $arg,";
			if ( $functionId{$name} > 0 ) {
				# QStringList should be converted to String[] as arguments, but ArrayLists
				# as return types. Hence, this hack.
				if ( kalyptusDataDict::ctypemap($argType) =~ /qt_QStr(ing)?List\s*\*/ ) {
					$jniFunction .= "_3Ljava_lang_String_2";
				} else {
					$jniFunction .= cplusplusToJNISignature($argType);
				}
			}

			if ( $cargType eq "" ) {
				print "'$argType' => '$typeprefix$argType',\n";
				if ( kalyptusDataDict::ctypemap($argType) =~ /^qt_/ || kalyptusDataDict::ctypemap($argType) =~ /^kde_/ ) {
					$argType =~ s/\&//;
					$cplusplusparams .= "($argType) QtSupport::getQt(env, $arg), ";
				} else {
					$argType =~ s/\&//;
					$cplusplusparams .= "(".$argType.") ".$arg.", ";
				}
			} else {
				if ( $javaargType =~ /String/ ) {
					if ( kalyptusDataDict::ctypemap($argType) =~ /qt_QString\s*\*/ ) {
						$stringargs = "\nstatic QString * _qstring_$arg = 0;".$stringargs;
						if ( $argType =~ /[\*]/ ) {
							$cplusplusparams .= "($argType) QtSupport::toQString(env, $arg, &_qstring_$arg), ";
						} else {
							$cplusplusparams .= "($argType) * (QString *) QtSupport::toQString(env, $arg, &_qstring_$arg), ";
						}
					} elsif ( kalyptusDataDict::ctypemap($argType) =~ /qt_QCString\s*\*/ ) {
						$stringargs = "\nstatic QCString * _qcstring_$arg = 0;".$stringargs;
						if ( $argType =~ /[\*]/ ) {
							$cplusplusparams .= "($argType) QtSupport::toQCString(env, $arg, &_qcstring_$arg), ";
						} else {
							$cplusplusparams .= "($argType) * (QCString *) QtSupport::toQCString(env, $arg, &_qcstring_$arg), ";
						}
					} elsif ( kalyptusDataDict::ctypemap($argType) =~ /kde_DOMString\s*(\*)/ ) {
						$stringargs = "\nstatic DOM::DOMString * _domstring_$arg = 0;".$stringargs;
						if ( $argType =~ /[\*]/ ) {
							$cplusplusparams .= "(DOM::DOMString *) KDESupport::toDOMString(env, $arg, &_domstring_$arg), ";
						} else {
							$cplusplusparams .= "(DOM::DOMString) * (DOM::DOMString *) KDESupport::toDOMString(env, $arg, &_domstring_$arg), ";
						}
					} else {
						$stringargs = "\nstatic QCString * _qstring_$arg = 0;".$stringargs;
						$cplusplusparams .= "($argType) QtSupport::toCharString(env, $arg, &_qstring_$arg), ";
					}
				} elsif ( $javaargType =~ /byte\[\]/ ) {
						$stringargs = "\nstatic QByteArray * _qbyteArray_$arg = 0;".$stringargs;
						if ( $argType =~ /[\*]/ ) {
							$cplusplusparams .= "($argType) QtSupport::toQByteArray(env, $arg, &_qbyteArray_$arg), ";
						} else {
							$cplusplusparams .= "($argType) * (QByteArray *) QtSupport::toQByteArray(env, $arg, &_qbyteArray_$arg), ";
						}
				} elsif ( kalyptusDataDict::ctypemap($argType) =~ /qt_QChar\s*\*/ ) {
					$stringargs = "\nstatic QChar * _qchar_$arg = 0;".$stringargs;
					if ( $argType =~ /[\*]/ ) {
						$cplusplusparams .= "($argType) QtSupport::toQChar(env, $arg, &_qchar_$arg), ";
      				} else {
						$cplusplusparams .= "($argType) * (QChar *) QtSupport::toQChar(env, $arg, &_qchar_$arg), ";
					}
				} elsif ( $javaargType =~ /Date/ ) {
					if ( kalyptusDataDict::ctypemap($argType) =~ /qt_QTime\s*\*/ ) {
						$stringargs = "\nstatic QTime * _qtime_$arg = 0;".$stringargs;
						if ( $argType =~ /[\*]/ ) {
							$cplusplusparams .= "($argType) QtSupport::toQTime(env, $arg, &_qtime_$arg), ";
                        } else {
							$cplusplusparams .= "($argType) * (QTime *) QtSupport::toQTime(env, $arg, &_qtime_$arg), ";
						}
      				}
				} elsif ( $javaargType =~ /Calendar/ ) {
					if ( kalyptusDataDict::ctypemap($argType) =~ /qt_QDateTime\s*\*/ ) {
						$stringargs = "\nstatic QDateTime * _qdate_$arg = 0;".$stringargs;
						if ( $argType =~ /[\*]/ ) {
							$cplusplusparams .= "($argType) QtSupport::toQDateTime(env, $arg, &_qdate_$arg), ";
                        } else {
							$cplusplusparams .= "($argType) * (QDateTime *) QtSupport::toQDateTime(env, $arg, &_qdate_$arg), ";
						}
					} elsif ( kalyptusDataDict::ctypemap($argType) =~ /qt_QDate\s*\*/ ) {
						$stringargs = "\nstatic QDate * _qdate_$arg = 0;".$stringargs;
						if ( $argType =~ /[\*]/ ) {
							$cplusplusparams .= "($argType) QtSupport::toQDate(env, $arg, &_qdate_$arg), ";
                        } else {
							$cplusplusparams .= "($argType) * (QDate *) QtSupport::toQDate(env, $arg, &_qdate_$arg), ";
						}
					}
				} elsif ( $javaargType =~ /ArrayList/ && kalyptusDataDict::ctypemap($argType) =~ /qt_(QStrList)\s*\*/ ) {
						$stringargs = "\nstatic $1 * _qlist_$arg = 0;".$stringargs;
						if ( $argType =~ /[\*]/ ) {
							$cplusplusparams .= "($argType) QtSupport::toQStrList(env, $arg, &_qlist_$arg), ";
						} else {
							$cplusplusparams .= "($argType) * (QStrList *) QtSupport::toQStrList(env, $arg, &_qlist_$arg), ";
						}
				} elsif ( $javaargType =~ /ArrayList/ && kalyptusDataDict::ctypemap($argType) =~ /qt_(QStringList)\s*\*/ ) {
						$stringargs = "\nstatic $1 * _qlist_$arg = 0;".$stringargs;
						if ( $argType =~ /[\*]/ ) {
							$cplusplusparams .= "($argType) QtSupport::toQStringList(env, $arg, &_qlist_$arg), ";
						} else {
							$cplusplusparams .= "($argType) * (QStringList *) QtSupport::toQStringList(env, $arg, &_qlist_$arg), ";
						}
				} elsif ( $javaargType =~ /ArrayList/ && kalyptusDataDict::ctypemap($argType) =~ /kde_(KFileItemList)\s*\*/ ) {
						$stringargs = "\nstatic $1 * _qlist_$arg = 0;".$stringargs;
						if ( $argType =~ /[\*]/ ) {
							$cplusplusparams .= "($argType) KDESupport::toKFileItemList(env, $arg, &_qlist_$arg), ";
						} else {
							$cplusplusparams .= "($argType) * (KFileItemList *) KDESupport::toKFileItemList(env, $arg, &_qlist_$arg), ";
						}
				} elsif ( $javaargType =~ /ArrayList/ && kalyptusDataDict::ctypemap($argType) =~ /kde_(QCStringList)\s*\*/ ) {
						$stringargs = "\nstatic $1 * _qlist_$arg = 0;".$stringargs;
						if ( $argType =~ /[\*]/ ) {
							$cplusplusparams .= "($argType) KDESupport::toQCStringList(env, $arg, &_qlist_$arg), ";
						} else {
							$cplusplusparams .= "($argType) * (QCStringList *) KDESupport::toQCStringList(env, $arg, &_qlist_$arg), ";
						}
				} elsif ( $javaargType =~ /ArrayList/ && kalyptusDataDict::ctypemap($argType) =~ /kde_(KURLList)\s*\*/ ) {
						$stringargs = "\nstatic KURL::List * _qlist_$arg = 0;".$stringargs;
						if ( $argType =~ /[\*]/ ) {
							$cplusplusparams .= "($argType) KDESupport::toKURLList(env, $arg, &_qlist_$arg), ";
						} else {
							$cplusplusparams .= "($argType) * (KURL::List *) KDESupport::toKURLList(env, $arg, &_qlist_$arg), ";
						}
				} elsif ( $javaargType =~ /^int\[\]/  ) {
						if ( kalyptusDataDict::ctypemap($argType) =~ /^\s*(unsigned )?int\s*\*/ ) {
							$cplusplusparams .= " ($argType) QtSupport::toIntPtr(env, $arg), ";
						} elsif ( kalyptusDataDict::ctypemap($argType) =~ /^qt_QIntValueList\*/ ) {
							$stringargs = "\nstatic QValueList<int> * _qlist_$arg = 0;".$stringargs;
							$cplusplusparams .= " ($argType) QtSupport::toQIntValueList(env, $arg, &_qlist_$arg), ";
						}
				} elsif ( $javaargType =~ /boolean\[\]/  ) {
					$cplusplusparams .= " ($argType) QtSupport::toBooleanPtr(env, $arg), ";
				} elsif ( $javaargType =~ /double\[\]/  ) {
					$cplusplusparams .= " ($argType) QtSupport::toDoublePtr(env, $arg), ";
				} elsif ( $javaargType =~ /short\[\]/  ) {
					$cplusplusparams .= " ($argType) QtSupport::toShortPtr(env, $arg), ";
				} elsif ( $argType =~ /QPaintDevice/  ) {
					$cplusplusparams .= "($argType) QtSupport::paintDevice(env, $arg), ";
				} elsif ( kalyptusDataDict::ctypemap($argType) =~ /^qt_/ || kalyptusDataDict::ctypemap($argType) =~ /^kde_/ ) {
					$argType = kalyptusDataDict::addNamespace($argType);

					if ( $argType =~ /^([^\*\&]*)&?$/ ) {
						$cplusplusparams .= "($argType) *($1 *) QtSupport::getQt(env, $arg), ";
					} else {
						$cplusplusparams .= "($argType) QtSupport::getQt(env, $arg), ";
					}
				} elsif ( $argType =~ /^[A-Z][^:]*$/ && $cargType eq "int" && kalyptusDataDict::ctypemap($class->{astNodeName}."::".$argType) ne "" ) {
					$cplusplusparams .= "(".$class->{astNodeName}."::".$argType.") $arg, ";
				} elsif ( $argType =~ /^\s*WFlags\s*$/ ) {
					$cplusplusparams .= "(QWidget::WFlags) $arg, ";
				} elsif ( $argType =~ /^\s*ArrowType\s*$/ ) {
					$cplusplusparams .= "(Qt::ArrowType) $arg, ";
				} elsif ( $argType =~ /^\s*Orientation\s*$/ ) {
					$cplusplusparams .= "(Qt::Orientation) $arg, ";
				} elsif ( $argType =~ /^\s*BrushStyle\s*$/ ) {
					$cplusplusparams .= "(Qt::BrushStyle) $arg, ";
				} elsif ( $argType =~ /^\s*BGMode\s*$/ ) {
					$cplusplusparams .= "(Qt::BGMode) $arg, ";
				} elsif ( $argType =~ /^\s*PenCapStyle\s*$/ ) {
					$cplusplusparams .= "(Qt::PenCapStyle) $arg, ";
				} elsif ( $argType =~ /^\s*PenStyle\s*$/ ) {
					$cplusplusparams .= "(Qt::PenStyle) $arg, ";
				} elsif ( $argType =~ /^\s*PenJoinStyle\s*$/ ) {
					$cplusplusparams .= "(Qt::PenJoinStyle) $arg, ";
				} elsif ( $argType =~ /^\s*RasterOp\s*$/ ) {
					$cplusplusparams .= "(Qt::RasterOp) $arg, ";
				} elsif ( $argType =~ /^\s*TextFormat\s*$/ ) {
					$cplusplusparams .= "(Qt::TextFormat) $arg, ";
				} elsif ( $argType =~ /^\s*QDragMode\s*$/ ) {
					$cplusplusparams .= "(QDragObject::DragMode) $arg, ";
				} elsif ( $argType =~ /^\s*GUIStyle\s*$/ ) {
					$cplusplusparams .= "(Qt::GUIStyle) $arg, ";
				} elsif ( $argType =~ /^\s*Type\s*$/ ) {
					$cplusplusparams .= "(QEvent::Type) $arg, ";
				} elsif ( $argType =~ /^([^\*\&]*)&$/ ) {
						$cplusplusparams .= "($argType) * ($1 *) $arg, ";
				} else {
						$cplusplusparams .= "($argType) $arg, ";
				}
			}

		}
		$cplusplusparams =~ s/\s*,\s*$//;
		$javaparams =~ s/\s*,\s*$//;
		$javaargs =~ s/\s*,\s*$//;

		$defaultcplusplusparams =~ s/\s*,\s*$//;
		$defaultjavaparams =~ s/\s*,\s*$//;
		$defaultjavaargs =~ s/\s*,\s*$//;


		if ( $returnType =~ /String/ && $m->{ReturnType} =~ /QString(\s*\*)?/ ) {
			$stringargs = "\n\tQString $1 _qstring;".$stringargs;
		} elsif ( $returnType =~ /String/ && $m->{ReturnType} =~ /QCString(\s*\*)?/ ) {
			$stringargs = "\n\tQCString $1 _qcstring;".$stringargs;
		} elsif ( $returnType =~ /String/ && $m->{ReturnType} =~ /(DOM::)?DOMString(\s*\*)?/ ) {
			$stringargs = "\n\tDOM::DOMString _domstring;".$stringargs;
		} elsif ( $returnType =~ /String/ && $m->{ReturnType} =~ /(const\s*)?char\s*\*\s*/  ) {
			$stringargs = "\n\t".$m->{ReturnType}." _qcstring;".$stringargs;
		} elsif ( $returnType =~ /String/  ) {
			$stringargs = "\n\t".$m->{ReturnType}." _qcstring;".$stringargs;
		} elsif (  $returnType =~ /byte\[\]/ && $m->{ReturnType} =~ /QByteArray(\s*\*)?/ ) {
			$stringargs = "\n\tQByteArray $1 _qbyteArray;".$stringargs;
		} elsif (  $returnType =~ /int\[\]/ && $m->{ReturnType} =~ /QValueList/ ) {
			$stringargs = "\n\tQValueList<int> _qintArray;".$stringargs;
		}

		if ( $m->{ReturnType} =~ /(QSqlIndex|QSqlRecord|QSqlQuery|QSqlError)\s*$/ ) {
			$stringargs = "\n\t$1  _qsql;".$stringargs;
		}

		if ( $m->{ReturnType} =~ /(QChar)(\s*\*)?\s*$/ ) {
			$stringargs = "\n\t$1 $2 _qchar;".$stringargs;
		}

		if (  $returnType =~ /Date/ && $m->{ReturnType} =~ /(QTime)\s*(\*)?&?\s*$/ ) {
			$stringargs = "\n\t$1 $2 _qtime;".$stringargs;
		}

		if (  $returnType =~ /Calendar/ && $m->{ReturnType} =~ /(QDateTime|QDate)\s*(\*)?&?\s*$/ ) {
			$stringargs = "\n\t$1 $2 _qdate;".$stringargs;
		}

		if (  $returnType =~ /ArrayList/ && $m->{ReturnType} =~ /(NodeList|StyleSheetList|MediaList)(\s*\*)?/ ) {
			$stringargs = "\n\tDOM::$1 $2 _qlist;".$stringargs;
		}

		if (  $returnType =~ /ArrayList/ && $m->{ReturnType} =~ /(QStrList|QStringList|QCStringList|QCanvasItemList|QWidgetList|QObjectList|QDomNodeList|DOM::NodeList|DOM::StyleSheetList|DOM::MediaList|const KFileItemList|KFileItemList|KFileViewItemList|KURL::List|OfferList)(\s*\*)?/ ) {
			$stringargs = "\n\t$1 $2 _qlist;".$stringargs;
		}

		my $flags = $m->{Flags};

		if ( !defined $flags ) {
			warn "Method ".$m->{astNodeName}.  " has no flags\n";
		}


		my $extra = "";
		$extra .= "static " if $flags =~ "s";
		if ( $name =~ /operator/ ) {
			return;
		}

		if ( $m->{Access} =~ /protected/ && $name ne $class->{astNodeName}  ) {
			if ( $class->{Pure} ) {
				return;
			}

#			if ( $name =~ /.*Event$/ ) {
#				return;
#			} 

			$name = "protected_".$name;
		}

		$m->{Access} =~ /([^_]*)(.*)?\s*/;
		$javaaccess = $1;
		if ( $extra =~ /static/ ) {
			$javaaccess .= " static";
		}
		$javaaccess .= " native";

		if ( $name eq $class->{astNodeName} ) {
			if ( exists $class->{Pure} ) {
				return;
            }
			# Constructor
			print CLASS "\tprivate native void new$name($javaparams);\n";
			if ( defined $docnode && defined $docnode->{Text} ) {
					print CLASS "\n/** ";
					printJavadocComment( $docnode, \*CLASS, "" );
					print CLASS "*/\n";
			}

			print CLASS "\tpublic $name($javaparams) {\n";
			if ($ancestorCount >= 0) {
				print CLASS "\t\tsuper((Class) null);\n";
            }
			print CLASS "\t\tnew$name($javaargs);\n\t}\n";
			$jniFunction =~ s/$name\_$name(__)?/$name\_new$name$1/;
			print JNISOURCE "$jniFunction(JNIEnv *env, jobject obj$jniparams)\n{$stringargs\n",
				"\tif (QtSupport::getQt(env, obj) == 0) {\n\t\tQtSupport::setQt(env, obj, new $name", "JBridge($cplusplusparams));\n",
				"\t\tQtSupport::setObjectForQtKey(env, obj, QtSupport::getQt(env, obj));\n\t}\n\treturn;\n}\n";

            if ( $defaultcplusplusparams ne "" ) {
				print CLASS "\tprivate native void new$name($defaultjavaparams);\n";
				print CLASS "\tpublic $name($defaultjavaparams) {\n";
				if ($ancestorCount >= 0) {
					print CLASS "\t\tsuper((Class) null);\n";
            	}
				print CLASS "\t\tnew$name($defaultjavaargs);\n\t}\n";
				$defaultjniFunction =~ s/$name\_$name(__)?/$name\_new$name$1/;
				print JNISOURCE "$defaultjniFunction(JNIEnv *env, jobject obj$defaultjniparams)\n{$defaultstringargs\n",
				"\tif (QtSupport::getQt(env, obj) == 0) {\n\t\tQtSupport::setQt(env, obj, new $name", "JBridge($defaultcplusplusparams));\n",
				"\t\tQtSupport::setObjectForQtKey(env, obj, QtSupport::getQt(env, obj));\n\t}\n\treturn;\n}\n";
 			}
		}
		elsif ( $m->{ReturnType} =~ /~/ ) {
			# Destructor
			if ( defined $docnode && defined $docnode->{Text} ) {
					print CLASS "\n/** ";
					printJavadocComment( $docnode, \*CLASS, "" );
					print CLASS "*/\n";
			}
			print CLASS "\tprotected native void finalize() throws InternalError;\n";
			print JNISOURCE "\nJNIEXPORT void JNICALL\n", $jniPackage,  $class->{astNodeName}, "_finalize(JNIEnv *env, jobject obj)\n{";

			if ( $class->{astNodeName} =~ /^QWidget$/  || defined kdocAstUtil::findOverride( $rootnode, $class, "parentWidget" ) ) {
				print JNISOURCE "\n\tif (QtSupport::allocatedInJavaWorld(env, obj) && ((", kalyptusDataDict::addNamespace($class->{astNodeName}), "*)QtSupport::getQt(env, obj))->parentWidget(TRUE) == (QWidget *) 0) {\n";
			} elsif ( $class->{astNodeName} =~ /^QObject$/  || defined kdocAstUtil::findOverride( $rootnode, $class, "parent" ) ) {
				print JNISOURCE "\n\tif (QtSupport::allocatedInJavaWorld(env, obj) && ((", kalyptusDataDict::addNamespace($class->{astNodeName}), "*)QtSupport::getQt(env, obj))->parent() == (QObject *) 0) {\n";
			} else {
				print JNISOURCE "\n\tif (QtSupport::allocatedInJavaWorld(env, obj)) {\n";
			}
			print JNISOURCE "\t\tdelete (", kalyptusDataDict::addNamespace($class->{astNodeName}), "*)QtSupport::getQt(env, obj);\n";
			print JNISOURCE "\t\tQtSupport::setQt(env, obj, 0);\n\t}\n\treturn;\n}\n";

			print CLASS "\tpublic native void dispose();\n";
			print JNISOURCE "\nJNIEXPORT void JNICALL\n", $jniPackage,  $class->{astNodeName}, "_dispose(JNIEnv *env, jobject obj)\n{";
			print JNISOURCE "\n\t$jniPackage",  $class->{astNodeName}, "_finalize(env, obj);\n\treturn;\n}\n";
			
			print CLASS "\tpublic native boolean isDisposed();\n";
			print JNISOURCE "\nJNIEXPORT jboolean JNICALL\n", $jniPackage,  $class->{astNodeName}, "_isDisposed(JNIEnv *env, jobject obj)\n{";
			print JNISOURCE "\n\treturn (QtSupport::getQt(env, obj) == 0);\n}\n";

		} else {
			# Class or instance method
			my $selfstring;
			if ( defined $docnode && defined $docnode->{Text} ) {
					print CLASS "\n/** ";
					printJavadocComment( $docnode, \*CLASS, "" );
					print CLASS "*/\n";
			}

			if ( $javaaccess =~ /static/ ) {
				if ( exists $class->{Pure} || $constructorCount == 0 ) {
	 				$selfstring = kalyptusDataDict::addNamespace($class->{astNodeName})."::";
				} else {
	 				$selfstring = $class->{astNodeName}."JBridge::";
				}
			} else {
				if ( exists $class->{Pure} || $constructorCount == 0 || $javaaccess =~ /public/ ) {
	 				$selfstring = "((".kalyptusDataDict::addNamespace($class->{astNodeName})."*) QtSupport::getQt(env, obj))->";
				} else {
	 				$selfstring = "((".$class->{astNodeName}."JBridge*) QtSupport::getQt(env, obj))->";
				}
			}

			if ( $javaaccess !~ /static/ && $javaaccess =~ /public/ && kalyptusDataDict::interfacemap($class->{astNodeName}) ne () ) {
				if ( defined $docnode->{Text} ) {
					print INTERFACE "\n/** ";
					printJavadocComment( $docnode, \*INTERFACE, "" );
					print INTERFACE "*/\n";
				}
				print INTERFACE "\t$returnType ", $m->{astNodeName}, "($javaparams);\n";
			}

			print CLASS "\t$javaaccess $returnType ", $m->{astNodeName}, "($javaparams);\n";
 			print JNISOURCE "$jniFunction(JNIEnv *env, ", ($javaaccess =~ /static/ ? "jclass obj" : "jobject obj"), "$jniparams)\n{$stringargs\n";

			if ( $returnType eq "void" ) {
				print JNISOURCE "\t$selfstring$name\($cplusplusparams\);\n\treturn;\n}\n" ;
				if ( $defaultcplusplusparams ne "" ) {
					if ( kalyptusDataDict::interfacemap($class->{astNodeName}) ne () && $javaaccess !~ /static/ && $javaaccess =~ /public/ ) {
						print INTERFACE "\t$returnType $name($defaultjavaparams);\n";
					}
					print CLASS "\t$javaaccess void ", $m->{astNodeName}, "($defaultjavaparams);\n";
					print JNISOURCE "$defaultjniFunction(JNIEnv *env, ", ($javaaccess =~ /static/ ? "jclass obj" : "jobject obj"), "$defaultjniparams)\n{$defaultstringargs\n";
					print JNISOURCE "\t$selfstring$name\($defaultcplusplusparams\);\n\treturn;\n}\n" ;
				}
			} elsif ( $returnType =~ /String/ ) {
				if ( $m->{ReturnType} =~ /QString\s*[\*\&]?\s*$/ ) {
					print JNISOURCE "\t_qstring = $selfstring$name\($cplusplusparams\);\n";
					if ( $m->{ReturnType} =~ /\*/ ) {
						print JNISOURCE "\treturn QtSupport::fromQString(env, _qstring);\n}\n";
                    } else {
						print JNISOURCE "\treturn QtSupport::fromQString(env, &_qstring);\n}\n";
					}
					if ( $defaultcplusplusparams ne "" ) {
						if ( kalyptusDataDict::interfacemap($class->{astNodeName}) ne () && $javaaccess !~ /static/ && $javaaccess =~ /public/ ) {
							print INTERFACE "\t$returnType ", $m->{astNodeName}, "($defaultjavaparams);\n";
						}
						print CLASS "\t$javaaccess $returnType ", $m->{astNodeName}, "($defaultjavaparams);\n";
						print JNISOURCE "$defaultjniFunction(JNIEnv *env, ", ($javaaccess =~ /static/ ? "jclass obj" : "jobject obj"), "$defaultjniparams)\n{$defaultstringargs\n";
						if ( $m->{ReturnType} =~ /\*/ ) {
							print JNISOURCE "\tQString * _qstring;\n\t_qstring = $selfstring$name($defaultcplusplusparams);\n";
							print JNISOURCE "\treturn QtSupport::fromQString(env, _qstring);\n}\n";
	                    } else {
							print JNISOURCE "\tQString _qstring;\n\t_qstring = $selfstring$name($defaultcplusplusparams);\n";
							print JNISOURCE "\treturn QtSupport::fromQString(env, &_qstring);\n}\n";
						}
					}
				} elsif ( $m->{ReturnType} =~ /QCString\s*[\*\&]?\s*$/ ) {
					print JNISOURCE "\t_qcstring = $selfstring$name($cplusplusparams);\n";
					if ( $m->{ReturnType} =~ /\*/ ) {
						print JNISOURCE "\treturn QtSupport::fromQCString(env, _qcstring);\n}\n";
                    } else {
						print JNISOURCE "\treturn QtSupport::fromQCString(env, &_qcstring);\n}\n";
					}
				} elsif ( $m->{ReturnType} =~ /(DOM::)?DOMString\s*([\*\&]?)\s*$/ ) {
					print JNISOURCE "\t_domstring = $selfstring$name\($cplusplusparams\);\n";
					if ( $m->{ReturnType} =~ /\*/ ) {
						print JNISOURCE "\treturn KDESupport::fromDOMString(env, _domstring);\n}\n";
					} else {
						print JNISOURCE "\treturn KDESupport::fromDOMString(env, &_domstring);\n}\n";
					}
				} else {
					print JNISOURCE "\t_qcstring = (const char *)$selfstring$name\($cplusplusparams\);\n";
					print JNISOURCE "\treturn QtSupport::fromCharString(env, (char *) _qcstring);\n}\n";
				}
			} elsif ( $returnType =~ /byte\[\]/ ) {
					print JNISOURCE "\t_qbyteArray = $selfstring$name\($cplusplusparams\);\n";
					if ( $m->{ReturnType} =~ /\*/ ) {
						print JNISOURCE "\treturn QtSupport::fromQByteArray(env, _qbyteArray);\n}\n";
                    } else {
						print JNISOURCE "\treturn QtSupport::fromQByteArray(env, &_qbyteArray);\n}\n";
					}
			} elsif ( $returnType =~ /int\[\]/ ) {
					print JNISOURCE "\t_qintArray = $selfstring$name\($cplusplusparams\);\n";
					print JNISOURCE "\treturn QtSupport::fromQIntValueList(env, &_qintArray);\n}\n";
			} elsif ( $returnType =~ /Date/ ) {
				if ( $m->{ReturnType} =~ /(QTime)\s*([\*\&])?\s*$/ ) {
					print JNISOURCE "\t_qtime = $selfstring$name($cplusplusparams);\n";
					if ( $2 eq "\*" ) {
						print JNISOURCE "\treturn (jobject) QtSupport::from$1(env, ($1 *) _qtime);\n}\n";
					} else {
						print JNISOURCE "\treturn (jobject) QtSupport::from$1(env, ($1 *) &_qtime);\n}\n";
					}
				}
			} elsif ( $m->{ReturnType} =~ /(QChar)\s*([\*\&])?\s*$/ ) {
				print JNISOURCE "\t_qchar = $selfstring$name($cplusplusparams);\n";
				if ( $2 eq "\*" ) {
					print JNISOURCE "\treturn (jchar) QtSupport::fromQChar(env, (QChar *) _qchar);\n}\n";
				} else {
					print JNISOURCE "\treturn (jchar) QtSupport::fromQChar(env, (QChar *) &_qchar);\n}\n";
				}
			} elsif ( $returnType =~ /Calendar/ ) {
				if ( $m->{ReturnType} =~ /(QDateTime|QDate)\s*([\*\&])?\s*$/ ) {
					print JNISOURCE "\t_qdate = $selfstring$name($cplusplusparams);\n";
					if ( $2 eq "\*" ) {
						print JNISOURCE "\treturn (jobject) QtSupport::from$1(env, ($1 *) _qdate);\n}\n";
					} else {
						print JNISOURCE "\treturn (jobject) QtSupport::from$1(env, ($1 *) &_qdate);\n}\n";
					}
				}
			} elsif ( $returnType =~ /ArrayList/ ) {
				if ( $m->{ReturnType} =~ /(QStrList|QStringList|QCanvasItemList|QWidgetList|QDomNodeList|QObjectList)\s*([\*\&])?\s*$/ ) {
					print JNISOURCE "\t_qlist = $selfstring$name($cplusplusparams);\n";
					if ( $2 eq "\*" ) {
						print JNISOURCE "\treturn (jobject) QtSupport::arrayWith$1(env, ($1 *) _qlist);\n}\n";
					} else {
						print JNISOURCE "\treturn (jobject) QtSupport::arrayWith$1(env, ($1 *) &_qlist);\n}\n";
					}
				} elsif ( $m->{ReturnType} =~ /(NodeList|StyleSheetListMediaList)\s*([\*\&])?\s*$/ ) {
					print JNISOURCE "\t_qlist = $selfstring$name($cplusplusparams);\n";
					if ( $2 eq "\*" ) {
						print JNISOURCE "\treturn (jobject) KDESupport::arrayWith$1(env, (DOM::$1 *) _qlist);\n}\n";
					} else {
						print JNISOURCE "\treturn (jobject) KDESupport::arrayWith$1(env, (DOM::$1 *) &_qlist);\n}\n";
					}
				} elsif ( $m->{ReturnType} =~ /(QCStringList|DOM::NodeList|DOM::StyleSheetList|DOM::MediaList|KFileItemList|KFileViewItemList|OfferList|KURL::List)\s*([\*\&])?\s*$/ ) {
					print JNISOURCE "\t_qlist = $selfstring$name($cplusplusparams);\n";
					if ( $2 eq "\*" ) {
						print JNISOURCE "\treturn (jobject) KDESupport::arrayWith$1(env, ($1 *) _qlist);\n}\n";
					} else {
						print JNISOURCE "\treturn (jobject) KDESupport::arrayWith$1(env, ($1 *) &_qlist);\n}\n";
					}
			   }
			} elsif ( $m->{ReturnType} =~ /(QSqlIndex|QSqlRecord|QSqlQuery|QSqlError)\s*$/ ) {
				print JNISOURCE "\t_qsql = $selfstring$name($cplusplusparams);\n";
				print JNISOURCE "\treturn (jobject) QtSupport::objectForQtKey(env, (void *)new $1(_qsql), \"org.kde.qt.$1\", TRUE);\n}\n" ;
			} elsif ( $m->{ReturnType} =~ /^\s*(inline)?\s*(const)?\s*?QBrush\s*$/ ) {
				print JNISOURCE "\tQBrush _b= $selfstring$name($cplusplusparams);\n" ;
				print JNISOURCE "\treturn (jobject) QtSupport::objectForQtKey(env, (void *)new QBrush(_b.color(),_b.style()), \"org.kde.qt.QBrush\", TRUE);\n}\n" ;
			} elsif ( $m->{ReturnType} =~ /^\s*(inline)?\s*(const)?\s*?QColorGroup\s*$/ ) {
				print JNISOURCE "\tQColorGroup _c= $selfstring$name($cplusplusparams);\n" ;
				print JNISOURCE "\treturn (jobject) QtSupport::objectForQtKey(env, (void *)new QColorGroup(_c.foreground(),_c.background(),_c.light(),_c.dark(),_c.mid(),_c.text(),_c.base()), \"org.kde.qt.QColorGroup\", TRUE);\n}\n" ;
			} elsif ( $m->{ReturnType} =~ /^\s*(inline)?\s*(const)?\s*?QDateTime\s*$/ ) {
				print JNISOURCE "\tQDateTime _dt= $selfstring$name($cplusplusparams);\n" ;
				print JNISOURCE "\treturn (jobject) QtSupport::objectForQtKey(env, (void *)new QDateTime (_dt.date(),_dt.time()), \"org.kde.qt.QDateTime\", TRUE);\n}\n" ;
			} elsif ( $m->{ReturnType} =~ /^\s*(inline)?\s*(const)?\s*?QDate\s*$/ ) {
				print JNISOURCE "\tQDate _d= $selfstring$name($cplusplusparams);\n" ;
				print JNISOURCE "\treturn (jobject) QtSupport::objectForQtKey(env, (void *)new QDate(_d.year(),_d.month(),_d.day(), \"org.kde.qt.QDate\", TRUE));\n}\n" ;
			} elsif ( $m->{ReturnType} =~ /^\s*(inline)?\s*(const)?\s*?QPen\s*$/ ) {
				print JNISOURCE "\tQPen _b= $selfstring$name($cplusplusparams);\n" ;
				print JNISOURCE "\treturn (jobject) QtSupport::objectForQtKey(env, (void *)new QPen(_b.color(),_b.width(),_b.style()), \"org.kde.qt.QPen\", TRUE);\n}\n" ;
			} elsif ( $m->{ReturnType} =~ /^\s*(inline)?\s*(const)?\s*?QPoint\s*\&?\s*$/ ) {
				print JNISOURCE "\tQPoint _p= $selfstring$name($cplusplusparams);\n" ;
				print JNISOURCE "\treturn (jobject) QtSupport::objectForQtKey(env, (void *)new QPoint(_p.x(),_p.y()), \"org.kde.qt.QPoint\", TRUE);\n}\n" ;
			} elsif ( $m->{ReturnType} =~ /^\s*(inline)?\s*(const)?\s*?QRect\s*$/ ) {
				print JNISOURCE "\tQRect _r= $selfstring$name($cplusplusparams);\n";
				print JNISOURCE "\treturn (jobject) QtSupport::objectForQtKey(env, (void *)new QRect(_r.left(),_r.top(),_r.width(),_r.height()), \"org.kde.qt.QRect\", TRUE);\n}\n" ;
			} elsif ( $m->{ReturnType} =~ /^\s*(inline)?\s*(const)?\s*?QSizePolicy\s*$/ ) {
				print JNISOURCE "\tQSizePolicy _s= $selfstring$name($cplusplusparams);\n" ;
				print JNISOURCE "\treturn (jobject) QtSupport::objectForQtKey(env, (void *)new QSizePolicy(_s.horData(),_s.verData(),_s.hasHeightForWidth()), \"org.kde.qt.QSizePolicy\", TRUE);\n}\n" ;
			} elsif ( $m->{ReturnType} =~ /^\s*(inline)?\s*(const)?\s*?QSize\s*$/ ) {
				print JNISOURCE "\tQSize _s= $selfstring$name($cplusplusparams);\n" ;
				print JNISOURCE "\treturn (jobject) QtSupport::objectForQtKey(env, (void *)new QSize(_s.width(),_s.height()), \"org.kde.qt.QSize\", TRUE);\n}\n" ;
			} elsif ( $m->{ReturnType} =~ /^\s*(inline)?\s*(const)?\s*?QStyle\s*$/ ) {
				print JNISOURCE "\tQStyle * _s= \&$selfstring$name($cplusplusparams);\n" ;
				print JNISOURCE "\treturn (jobject) _s;\n}\n" ;
			} elsif ( $m->{ReturnType} =~ /^\s*(inline)?\s*(const)?\s*?QTime\s*$/ ) {
				print JNISOURCE "\tQTime _t= $selfstring$name($cplusplusparams);\n" ;
				print JNISOURCE "\treturn (jobject) QtSupport::objectForQtKey(env, (void *)new QTime(_t.hour(),_t.minute(),_t.second(),_t.msec()), \"java.util.Date\", TRUE);\n}\n" ;
			} elsif ( $m->{ReturnType} =~ /^\s*(inline)?\s*(const)?\s*?QWMatrix\s*$/ ) {
				print JNISOURCE "\tQWMatrix _m= $selfstring$name($cplusplusparams);\n" ;
				print JNISOURCE "\treturn (jobject) QtSupport::objectForQtKey(env, (void *)new QWMatrix(_m.m11(),_m.m12(),_m.m21(),_m.m22(),_m.dx(),_m.dy()), \"org.kde.qt.QWMatrix\", TRUE);\n}\n" ;
			} elsif ( $m->{ReturnType} =~ /^\s*(inline)?\s*int\s*\*\s*$/ ) {
				print JNISOURCE "\treturn ($jnireturntype) QtSupport::fromIntPtr(env, $selfstring$name($cplusplusparams));\n}\n" ;
			} elsif ( $m->{ReturnType} =~ /^\s*(inline)?\s*double\s*\*\s*$/ ) {
				print JNISOURCE "\treturn ($jnireturntype) QtSupport::fromDoublePtr(env, $selfstring$name($cplusplusparams));\n}\n" ;
			} elsif ( $m->{ReturnType} =~ /^\s*(inline)?\s*short\s*\*\s*$/ ) {
				print JNISOURCE "\treturn ($jnireturntype) QtSupport::fromShortPtr(env, $selfstring$name($cplusplusparams));\n}\n" ;
			} elsif ( $jnireturntype =~ /jobject/ ) {
				print JNISOURCE "\treturn ($jnireturntype) QtSupport::objectForQtKey(env, (void *)";
				my $fullyQualifiedReturnType = ($returnType =~ /^Q/ ? "org.kde.qt.$returnType" : "org.kde.koala.$returnType");
				if ( $m->{ReturnType} =~ /\&/ ) {
					print JNISOURCE "($returnType *) &$selfstring$name($cplusplusparams), \"$fullyQualifiedReturnType\");\n}\n";
				} elsif ( $m->{ReturnType} =~ /\*/ ) {
					print JNISOURCE "$selfstring$name($cplusplusparams), \"$fullyQualifiedReturnType\");\n}\n";
				} elsif ($m->{ReturnType} =~ /(.*)\s*\&?/) {
					print JNISOURCE "new ", kalyptusDataDict::addNamespace($1), "($selfstring$name($cplusplusparams)), \"$fullyQualifiedReturnType\", TRUE);\n}\n";
				}
				if ( $defaultcplusplusparams ne "" ) {
					if ( kalyptusDataDict::interfacemap($class->{astNodeName}) ne () && $javaaccess !~ /static/ && $javaaccess =~ /public/ ) {
						print INTERFACE "\t$returnType ", $m->{astNodeName}, "($defaultjavaparams);\n";
					}
					print CLASS "\t$javaaccess $returnType ", $m->{astNodeName}, "($defaultjavaparams);\n";

					print JNISOURCE "$defaultjniFunction(JNIEnv *env, ", ($javaaccess =~ /static/ ? "jclass obj" : "jobject obj"), "$defaultjniparams)\n{$defaultstringargs\n";
					print JNISOURCE "\treturn ($jnireturntype) QtSupport::objectForQtKey(env, (void *)";
					if ( $m->{ReturnType} =~ /\*/ ) {
						print JNISOURCE "$selfstring$name($defaultcplusplusparams), \"$fullyQualifiedReturnType\");\n}\n";
					} else {
						print JNISOURCE "new $returnType($selfstring$name($defaultcplusplusparams)), \"$fullyQualifiedReturnType\", TRUE);\n}\n";
					}
				}
			} else {
				print JNISOURCE "\treturn ($jnireturntype) $selfstring$name($cplusplusparams);\n}\n" ;
				if ( $defaultcplusplusparams ne "" ) {
					if ( kalyptusDataDict::interfacemap($class->{astNodeName}) ne () && $javaaccess !~ /static/ && $javaaccess =~ /public/ ) {
						print INTERFACE "\t$returnType ", $m->{astNodeName}, "($defaultjavaparams);\n";
					}
					print CLASS "\t$javaaccess $returnType ", $m->{astNodeName}, "($defaultjavaparams);\n";
					print JNISOURCE "$defaultjniFunction(JNIEnv *env, ", ($javaaccess =~ /static/ ? "jclass obj" : "jobject obj"), "$defaultjniparams)\n{$defaultstringargs\n";
					print JNISOURCE "\treturn ($jnireturntype) $selfstring$name($defaultcplusplusparams);\n}\n";
				}
			}
		}
	}

}

sub generateClassMethodForEnum
{
	my( $class, $m ) = @_;

	if( $m->{NodeType} eq "enum" ) {
		my $enum = $m->{astNodeName};
		my @enums = split(",", $m->{Params});
		my $enumCount = 0;
		foreach my $enum ( @enums ) {
			$enum =~ s/\s//g;
			$enum =~ s/::/./g;
			if ( $enum =~ /(.*)=(.*)/ ) {
				print CLASS "\tpublic static final int $1 = $2;\n",;
			} else {
				print CLASS "\tpublic static final int $enum = $enumCount;\n";
				$enumCount++;
			}
		}
	}
}


=head2 printJavadocComment

	Parameters: docnode filehandle

	Converts a kdoc comment to javadoc format and prints it.
	@ref's are converted to @link's; @p's and @em's are converted
	to inline HTML.

=cut

sub printJavadocComment
{
	my( $docnode, $handle, $name ) = @_;

	my $node;
	foreach $node ( @{$docnode->{Text}} ) {
		next if $node->{NodeType} ne "DocText";
		my $line = $node->{astNodeName};
		$line =~ s/argc, ?argv/args/g;
		$line =~ s/int argc, ?char ?\* ?argv(\[\])?/String[] args/g;
		$line =~ s/int argc, ?char ?\*\* ?argv/String[] args/g;
		$line =~ s/(const )?QString&?/String/g;
		$line =~ s/QByteArray/byte[]/g;
		$line =~ s/0L/null/g;
		$line =~ s/const char/String/g;
		$line =~ s/bool /boolean /g;
		$line =~ s/SLOT\(([^\)]*)\) ?\)/SLOT("$1)")/g;
		$line =~ s/SIGNAL\(([^\)]*)\) ?\)/SIGNAL("$1)")/g;
		$line =~ s/::/./g;
		$line =~ s/->/./g;
		$line =~ s/\*//g;
		$line =~ s/\@ref\s+([\w]+)\.([\w]+)\s*(\([^\)]*\))(\.)?/{\@link $1#$2}$4/g;
		$line =~ s/\@ref\s+#([\w\.]+)(\(\))?/{\@link #$1}/g;
		$line =~ s/\@ref\s+([\w]+)\s*(\([^\)]*\))/{\@link #$1}/g;
		$line =~ s/\@ref\s+([\w]+)\.([\w]+)/{\@link $1#$2}/g;
		$line =~ s/\@ref\s+([\w]+)/{\@link $1}/g;
		$line =~ s!\@p\s+([\w\._]*)!<code>$1</code>!g;
		$line =~ s!\@em\s+([\w\._]*)!<em>$1</em>!g;

		if ($name ne "") {
			$line =~ s/\@link #/\@link $name\#/g;
		}

		print $handle $line, "\n";
	}
}


=head2 printIndexEntry

	Parameters: member node

	Prints an index entry for a single node.

	TODO: stub

=cut

sub printIndexEntry
{
	my ( @node ) = @_;
}



sub writeSrcHTML
{
	my ( $outfile, $infile ) = @_;

	open ( OUT, ">$outfile" ) || die "Couldn't open $outfile for".
			"writing.\n";


	close OUT;
}

1;




