/************************************
 Message history
 (c) 1999 Jeremy Wise
 (c) 2001 Gediminas Paulauskas
 GnomeICU
*************************************/

#include "common.h"
#include "gnomeicu.h"
#include "history.h"
#include "icons.h"
#include "util.h"
#include "icu_db.h"

struct _HistoryWindow {
	GtkWidget *window;

	GtkWidget *dialog_text;
	GtkWidget *incoming_text;
	GtkWidget *outgoing_text;

	char *filename;
	CONTACT_PTR contact;
};

#define HISTORY_WINDOW(x) ((HistoryWindow *) x)

/* only history_add_**** functions need to get/set/restore point in all 3
 * gtktexts.
 * history_show_entry just inserts anywhere.
 */

/*** Local function declarations ***/
static void clear_hist_cb(gint reply, gpointer data);
static void file_ok_cb(GtkWidget *widget, gpointer data);
static void save_hist_cb(GtkWidget *widget, gpointer data);
static void search_cb(GtkWidget *widget, gpointer data);
static void searchentry_changed(GtkEditable *widget, gpointer data);
static void search_show_cb(GtkWidget *widget, gpointer data);
static void show_clear_dialog(GtkWidget *widget, gpointer data);
static void text_pos_remember (GtkText *text, guint *histpoint, guint *origlen);
static void text_pos_restore (GtkText *text, guint histpoint, guint origlen);
static void add_to_history (UIN_T uin, const char *statement, time_t * timedate,
			    GdkColor * mesg_color, gboolean received);
static HistoryWindow* history_construct (CONTACT_PTR contact);
static void history_destroyed (GtkWidget *widget, gpointer data);
static GtkWidget* create_searchhbox (GtkBox *parent_box,
                                     GtkWidget *slave_text,
                                     HistoryWindow *hw);

static void insert_entry_into_widget (GtkWidget *widget,
                               GdkColor *mesg_color,
                               char *pdate, const char *text, guint length);
static void append_history_entry (HistoryWindow *hw, struct tm time,
                           const char *text, guint length, gboolean received);

#ifdef HAVE_ICUDB
static gint compare_dates(const gint *a, const gint *b);
#endif

/*** Variables ***/
static GdkColor recv_color = { 0, 65535, 0, 0 }; /* Red */
static GdkColor send_color = { 0, 0, 0, 65535 }; /* Blue */
	
/*** Global functions ***/
void history_add_incoming (UIN_T uin, const char *statement,
                           time_t *timedate)
{
	char *tmp;
	time_t my_time;

#ifdef TRACE_FUNCTION
	g_print ("history_add_incoming\n");
#endif

	g_return_if_fail (statement != NULL);
	if (statement[0] == '\0')
		return;

	/* No one strips \r before calling this function */
	tmp = stripr (statement);

	if ((timedate == NULL) || (*timedate > time (NULL)) ||
            (*timedate < 600000000)) { /* thats in 1989 */
          my_time = time (NULL);
	} else {
          my_time = *timedate;
	}

	add_to_history (uin, tmp, &my_time, &recv_color, TRUE);

	g_free (tmp);
}

void history_add_outgoing (UIN_T uin, const char *statement)
{
	time_t timedate;

#ifdef TRACE_FUNCTION
	g_print ("history_add_outgoing\n");
#endif

	g_return_if_fail (statement != NULL);
	if (statement[0] == '\0')
		return;

	time (&timedate);

	add_to_history (uin, statement, &timedate, &send_color, FALSE);
}

#ifndef HAVE_ICUDB
typedef struct _HistoryEntry HistoryEntry;

struct _HistoryEntry {
	struct tm time;
	gboolean received;
	char *text;
};
#endif

void insert_entry_into_widget (GtkWidget *widget,
                               GdkColor *mesg_color,
                               char *pdate, const char *text, guint length)
{
	gtk_text_freeze (GTK_TEXT (widget));
	gtk_text_insert (GTK_TEXT (widget), NULL, mesg_color,
			 NULL, pdate, -1);
	gtk_text_insert (GTK_TEXT (widget), NULL, NULL, NULL,
			 text, length);
	gtk_text_insert (GTK_TEXT (widget), NULL, NULL, NULL,
			 "\n\n", -1);
	gtk_text_thaw (GTK_TEXT (widget));
}

void append_history_entry (HistoryWindow *hw, struct tm time,
                           const char *text, guint length, gboolean received)
{
	char buf[40];
	char *pdate;

	g_return_if_fail (hw != NULL);
	
	strftime (buf, sizeof (buf), _("%c"), &time);

	pdate = g_strdup_printf ("*** %s [ %s ] ***\n", buf,
				 received ? _("Received") : _("Sent"));

	insert_entry_into_widget (hw->dialog_text,
	                          received ? &recv_color : &send_color,
	                          pdate, text, length);
	if (received) {
		insert_entry_into_widget (hw->incoming_text,
		                          &recv_color,
		                          pdate, text, length);
	} else {
		insert_entry_into_widget (hw->outgoing_text,
		                          &send_color,
		                          pdate, text, length);
	}

	g_free (pdate);
}

void history_display( GtkWidget *widget, gpointer data )
{
	gchar *filename;

	GSList *contact;
	GList *listed_history = NULL, *rev_list;

	struct tm *my_tm = NULL;

#ifdef HAVE_ICUDB
	DB_FILE db_file;
	datum firstkey, nextkey, key_data;
#else
	char *buffer;
	char *ptr;
	struct stat fs;
	int nchars;
	FILE *file;
#endif

#ifdef TRACE_FUNCTION
	g_print( "history_display\n" );
#endif

	contact = Contacts;
	while (contact != NULL && contact->data != data)
		contact = contact->next;

	if( contact == NULL )
		return;

	if( kontakt->message_history != NULL )
	{
  		gtk_widget_show(kontakt->message_history->window);
		gdk_window_raise(kontakt->message_history->window->window);
		return;
	}

	/* Open file for reading */
#ifdef HAVE_ICUDB

	filename = g_strdup_printf( "%s/.icq/history/%u.db", g_get_home_dir(),
				    kontakt->uin );

	db_file = icudb_open( filename, DB_READ );

	if( db_file == NULL )
	{
		gnome_error_dialog( _("No message history available.") );
		g_free( filename );
		return;
	}

#else
	filename = g_strdup_printf( "%s/.icq/history/%u", g_get_home_dir(),
				    kontakt->uin );

	file = fopen( filename, "r" );

	if( file == NULL )
	{
		gnome_error_dialog( _("No message history available.") );
		g_free( filename );
		return;
	}
#endif

	HISTORY_WINDOW (kontakt->message_history) = history_construct (kontakt);
	HISTORY_WINDOW (kontakt->message_history)->filename = g_strdup (filename);
	HISTORY_WINDOW (kontakt->message_history)->contact = kontakt;

#ifdef HAVE_ICUDB

	firstkey = icudb_firstkey ( db_file );

	/* First, create a list of keys from the history file */
	while ( firstkey.dptr ) {

		listed_history = g_list_prepend( listed_history, GINT_TO_POINTER(firstkey.dptr) );

		nextkey = icudb_nextkey ( db_file, firstkey );
		firstkey = nextkey;

		while( gtk_events_pending() )
			gtk_main_iteration();

	}

	/* Then sort the keys */
	listed_history = g_list_sort (listed_history, (GCompareFunc)compare_dates);
	rev_list = g_list_last( listed_history );

	firstkey.dsize = sizeof (time_t);

	while (rev_list != NULL) {
		if (!kontakt->message_history)
			break;

		firstkey.dptr = rev_list->data;
		key_data = icudb_fetch (db_file, firstkey);

		if (key_data.dptr != NULL) {
			my_tm = localtime (rev_list->data);
			append_history_entry (HISTORY_WINDOW(kontakt->message_history),
			                      *my_tm, key_data.dptr + 2,
			                      key_data.dsize - 2,
			                      MESG_ISRECV (key_data.dptr));

			icudb_free (key_data.dptr);
		}

		while (gtk_events_pending ())
			gtk_main_iteration ();

		icudb_free (rev_list->data);
		rev_list = rev_list->prev;
	}

	/* If window was destroyed, free the rest of the list */
	while (rev_list != NULL) {
		icudb_free (rev_list->data);
		rev_list = rev_list->prev;
	}

	/* Finally, close the db file */
	icudb_close ( db_file );

#else

	stat( filename, &fs );

	buffer = (char*)g_malloc( fs.st_size + 16 );

	nchars = fread( buffer, 1, fs.st_size, file );
	fclose( file );

	my_tm = g_new0 (struct tm, 1);

	ptr = strstr (buffer, "\n*** ");

	while (ptr)
	{
		HistoryEntry *he;
		char *endptr;
		char half[3];
		char recv[15];
		
		he = g_new(HistoryEntry, 1);
		
		endptr = strstr (ptr+5, " ***\n");

		sscanf (ptr+5, "%02d/%02d/%d %02d:%02d:%02d %s [ %s ]",
			     &my_tm->tm_mon,
			     &my_tm->tm_mday,
			     &my_tm->tm_year,
			     &my_tm->tm_hour,
			     &my_tm->tm_min,
			     &my_tm->tm_sec,
			     half,
			     recv);
		my_tm->tm_mon -= 1;
		my_tm->tm_year -= 1900;
		if (half[0] == 'P')
			my_tm->tm_hour += 12;
		he->time = *my_tm;
		he->received = (strcmp (recv, "Received") == 0);

		ptr = strstr (endptr+5, "\n*** ");
		if (ptr)
			/* additional -1 to cut the \n at the end of every
			 * entry */
			he->text = g_strndup (endptr+5, ptr-endptr-5-1);
		else
			he->text = g_strndup (endptr+5, strlen (endptr+5) - 1);

		listed_history = g_list_prepend (listed_history, he);

		while( gtk_events_pending() )
			gtk_main_iteration();
	}

	g_free (my_tm);

	rev_list = listed_history;

	while (rev_list != NULL) {
		HistoryEntry *he = rev_list->data;

		if (!kontakt->message_history)
			break;

		append_history_entry (HISTORY_WINDOW (kontakt->message_history),
		                      he->time, he->text,
		                      strlen (he->text), he->received);

		while (gtk_events_pending ())
			gtk_main_iteration ();

		g_free (he->text);
		g_free (he);

		rev_list = rev_list->next;
	}
	
	/* If window was destroyed, free the rest of the list */
	while (rev_list != NULL) {
		g_free (((HistoryEntry *) rev_list->data)->text);
		g_free (rev_list->data);

		rev_list = rev_list->next;
	}

	g_free (buffer);
#endif
	/* Only show the window after all contents were added -- it is much
	 * faster */
	gtk_widget_show (HISTORY_WINDOW (kontakt->message_history)->window);

	g_free (filename);
	g_list_free (listed_history);
}

/*** Local functions ***/
void save_hist_cb (GtkWidget *widget, gpointer data)
{
	GtkText *text = GTK_TEXT (data);
	gint length;

	length = gtk_text_get_length (text);

	if (length == 0) {
		gnome_error_dialog (_("No history to save."));
	} else {
		CONTACT_PTR contact;
		GtkFileSelection *filew;
		gchar *savename;

		filew = (GtkFileSelection *) gtk_file_selection_new (_("Save History"));
		gtk_window_set_wmclass (GTK_WINDOW (filew), "Save History",
					"GnomeICU");

		gtk_signal_connect_object (GTK_OBJECT (filew->cancel_button),
					   "clicked",
					   (GtkSignalFunc) gtk_widget_destroy,
					   GTK_OBJECT (filew));

		gtk_signal_connect (GTK_OBJECT (filew->ok_button),
				    "clicked", (GtkSignalFunc) file_ok_cb,
				    filew);

		gtk_object_set_data (GTK_OBJECT (filew->ok_button), "text",
				     text);

		contact = gtk_object_get_data (GTK_OBJECT (widget), "contact");
		savename = g_strconcat (contact->nick, ".hist", NULL);
		gtk_file_selection_set_filename (filew, savename);
		g_free (savename);

		gtk_widget_show (GTK_WIDGET (filew));
	}
}

void file_ok_cb (GtkWidget *widget, gpointer data)
{
	GtkWidget *filew = GTK_WIDGET (data);
	GtkWidget *text;
	gchar *writename;
	gchar *history_text;
	FILE *writefile;

#ifdef TRACE_FUNCTION
	g_print ("file_ok_cb");
#endif

	text = gtk_object_get_data (GTK_OBJECT (widget), "text");
	writename = gtk_file_selection_get_filename (GTK_FILE_SELECTION (filew));

	history_text = gtk_editable_get_chars (GTK_EDITABLE (text), 0, -1);

	writefile = fopen (writename, "w");
	if (writefile) {
		fprintf (writefile, "%s", history_text);
		fclose (writefile);
	} else {
		gnome_error_dialog (_("File save error."));
	}

	gtk_widget_destroy (filew);
	g_free (writename);
}

void show_clear_dialog (GtkWidget *widget, gpointer data)
{
	gnome_question_dialog (_("Clear all stored history?"), clear_hist_cb,
	                       data);
}

void clear_hist_cb (gint reply, gpointer data)
{
	if (reply == GNOME_YES) {
		HistoryWindow *hw = HISTORY_WINDOW(data);

		/* clear the text widgets */
		gtk_editable_delete_text (GTK_EDITABLE(hw->dialog_text), 0, -1);
		gtk_editable_delete_text (GTK_EDITABLE(hw->incoming_text), 0, -1);
		gtk_editable_delete_text (GTK_EDITABLE(hw->outgoing_text), 0, -1);
#ifdef HAVE_ICUDB
		icudb_close_dbpath (hw->filename);
#endif
		remove (hw->filename);
	}
}

void search_show_cb(GtkWidget *widget, gpointer data)
{
	GtkWidget *searchhbox;
	GtkWidget *searchhbox2;
	GtkWidget *searchhbox3;
	GtkWidget *searchentry;
	GtkNotebook *notebook;
	gint currenttab;

	HistoryWindow *hw = HISTORY_WINDOW (data);

	notebook = gtk_object_get_data (GTK_OBJECT(widget), "notebook");

	searchhbox = create_searchhbox (GTK_BOX (gtk_notebook_get_nth_page (notebook, 0)), hw->dialog_text, hw);
	searchhbox2 = create_searchhbox (GTK_BOX (gtk_notebook_get_nth_page (notebook, 1)), hw->incoming_text, hw);
	searchhbox3 = create_searchhbox (GTK_BOX (gtk_notebook_get_nth_page (notebook, 2)), hw->outgoing_text, hw);

	currenttab = gtk_notebook_get_current_page(GTK_NOTEBOOK(notebook));

	if( currenttab == 0){
	    searchentry = gtk_object_get_data(GTK_OBJECT(searchhbox), "searchentry" );
	} else if( currenttab == 1){
	    searchentry = gtk_object_get_data(GTK_OBJECT(searchhbox2), "searchentry" );
	} else {
	    searchentry = gtk_object_get_data(GTK_OBJECT(searchhbox3), "searchentry" );
	}

	gtk_widget_show_all (searchhbox);
	gtk_widget_show_all (searchhbox2);
	gtk_widget_show_all (searchhbox3);
	gtk_widget_grab_focus (searchentry);
	gtk_widget_hide (widget);
}

void searchentry_changed (GtkEditable * widget, gpointer data)
{
	GtkWidget *button;
	GtkWidget *label;

	button = gtk_object_get_data (GTK_OBJECT (widget), "searchbutton");
	label = gtk_object_get_data (GTK_OBJECT (button), "label");

	gtk_object_set_data (GTK_OBJECT (button), "searched",
			     GINT_TO_POINTER (FALSE));

	gtk_label_set_text (GTK_LABEL (label), _("Search"));
}

void search_cb(GtkWidget *widget, gpointer data)
{
	GtkWidget *searchentry;
	GtkWidget *label;
	GtkWidget *history;
	GtkWidget *searchbutton;
	gboolean searched;
	gchar *historytext;
	gchar *searchstring;
	gint searchfrom, foundat = 0;

	/* Get the widgets */
	searchbutton = gtk_object_get_data (GTK_OBJECT (widget), "searchbutton");
	searchentry = gtk_object_get_data (GTK_OBJECT (widget), "searchentry");
	label = gtk_object_get_data (GTK_OBJECT (widget), "label");
	history = gtk_object_get_data (GTK_OBJECT (widget), "history");
	searched = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (widget), "searched"));

	if (searched)
		foundat = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (widget), "foundat"));

	/* OK, lets search now! */

	searchstring = gtk_editable_get_chars (GTK_EDITABLE (searchentry), 0, -1);
	if (!searched) {
		searchfrom = 0;
		foundat = 0;
	} else {
		searchfrom = foundat;
	}

	historytext = gtk_editable_get_chars (GTK_EDITABLE (history), searchfrom, -1);
	foundat = (gint) strstr (historytext, searchstring);
	searched = TRUE;

	gtk_text_freeze (GTK_TEXT (history));
	if (foundat != 0) {
		foundat -= (gint) historytext;
		gtk_editable_set_position (GTK_EDITABLE (history),
					   foundat + searchfrom);
		gtk_editable_select_region (GTK_EDITABLE (history),
					    foundat + searchfrom,
					    foundat + strlen (searchstring) +
					    searchfrom);
		foundat += searchfrom + strlen (searchstring) + 1;
	} else {
		gnome_ok_dialog (_("No more occurrences found."));
		searched = FALSE;
	}


	gtk_text_thaw (GTK_TEXT (history));

	gtk_object_set_data (GTK_OBJECT (searchbutton), "searched",
			     GINT_TO_POINTER (searched));
	gtk_object_set_data (GTK_OBJECT (searchbutton), "foundat",
			     GINT_TO_POINTER (foundat));
	gtk_object_set_data (GTK_OBJECT (searchentry), "searched",
			     GINT_TO_POINTER (searched));
	gtk_object_set_data (GTK_OBJECT (searchentry), "foundat",
			     GINT_TO_POINTER (foundat));

	g_free (historytext);
	g_free (searchstring);

	if (searched) {
		gtk_label_set_text (GTK_LABEL (label), _("Search Again"));
	} else {
		gtk_label_set_text (GTK_LABEL (label), _("Search"));
	}
}

#ifdef HAVE_ICUDB
gint compare_dates(const gint *a,
		   const gint *b)
{
	return( *a - *b );
}
#endif

void text_pos_remember (GtkText *text, guint *histpoint, guint *origlen)
{
	*histpoint = gtk_text_get_point (text);
	*origlen = gtk_text_get_length (text);
	gtk_text_set_point (text, 0);
}

void text_pos_restore (GtkText *text, guint histpoint, guint origlen)
{
	gint difference = gtk_text_get_length (text) - origlen;
	gtk_text_set_point (text, histpoint + difference);
}

static void add_to_history (UIN_T uin, const char *statement, time_t * timedate,
			    GdkColor * mesg_color, gboolean received)
{
	gchar *filename;
#ifdef HAVE_ICUDB
	DB_FILE db_file;
	datum kdat, vdat;
#else
	int file;
	gchar *pdate;
	int cx, cy, cz;
	char *halves[] = { "AM", "PM" };
#endif

	int half = 0;
	struct tm *my_tm;
	GSList *contact;

	contact = Find_User (uin);
	if (contact == NULL)
		return;

	/* FIXME: should create directory if does not exist */
#ifdef HAVE_ICUDB
	filename = g_strdup_printf ("%s/.icq/history/%d.db", g_get_home_dir (), uin);
	if ((db_file = icudb_open (filename, DB_WRITE)) == NULL)
#else
	filename = g_strdup_printf ("%s/.icq/history/%d", g_get_home_dir (), uin);
	if ((file = open (filename, O_WRONLY | O_CREAT | O_APPEND, S_IRUSR | S_IWUSR)) == -1)
#endif /* HAVE_ICUDB */
	{
#ifdef HAVE_LIBGDBM
		g_warning ("gdbm open:%s:%s\n", filename,
			   gdbm_strerror (gdbm_errno));
#else
		g_warning ("open:%s\n", filename);
#endif /* HAVE_LIBGDBM */  
		g_free (filename);
		return;
	}
	g_free (filename);

	my_tm = localtime (timedate);

	if (my_tm->tm_hour > 12) {
		my_tm->tm_hour -= 12;
		half++;
	}

#ifdef HAVE_ICUDB
	/* Set up a key by which this message is referenced */
	kdat.dptr = (char *) timedate;
	kdat.dsize = sizeof (time_t);

	/* Set up the data to store the message, with two bytes at the beginning
	 * for flags. */
	vdat.dptr = g_strdup_printf ("  %s", statement);
	vdat.dsize = strlen (vdat.dptr);
	memset (vdat.dptr, 0, 2);

	if (received == TRUE)
		MESG_SETRECV (vdat.dptr);

	/* incoming: Until we read messages from the file, mark all incoming messages as read */
	/* outgoing: Mark this message as read, since we wrote it. */
	/* FIXME: why there is no difference? */
	MESG_SETREAD (vdat.dptr);

 tryagain:
	if (icudb_store (db_file, kdat, vdat, DB_INSERT)) {
#ifdef HAVE_LIBGDBM
          if (gdbm_errno == GDBM_CANNOT_REPLACE) {
            (*timedate)++;
            goto tryagain;
          }
          else
            g_warning ("gdbm icudb_store:%d:%s:%s\n",
                       *((time_t *)kdat.dptr), vdat.dptr + 2,
                       gdbm_strerror(gdbm_errno));
              
#else
          g_warning ("icudb_store:%s:%s\n",
                     kdat.dptr, vdat.dptr + 2);
#endif /* HAVE_LIBGDBM */  
        }
        
	g_free (vdat.dptr);

	icudb_close (db_file);

#else

	pdate =
	    g_strdup_printf ("*** %02d/%02d/%d %02d:%02d:%02d %s [ %s ] ***\n",
			     my_tm->tm_mon + 1,
			     my_tm->tm_mday,
			     my_tm->tm_year + 1900,
			     my_tm->tm_hour,
			     my_tm->tm_min,
			     my_tm->tm_sec,
			     halves[half],
			     received ? "Received" : "Sent");

	write (file, "\n", 1);
	write (file, pdate, strlen (pdate));

	g_free (pdate);

	cy = cz = 0;
	for (cx = 0; cx < strlen (statement); cx++) {
		cy++;
		if (statement[cx] == '\n' || cy == 70 ||
		    (cy >= 60 && statement[cx] == ' ')) {
			write (file, (statement + cz), cy);
			if (cy >= 60 && cy != 70) {
				write (file, "\n", 1);
			}
			if (cy == 70) {
				write (file, "-\n", 2);
			}
			cz += cy;
			cy = 0;
		}
	}

	if (cz != strlen (statement)) {
		write (file, (statement + cz), strlen (statement) - cz);
	}

	write (file, "\n", 1);

	close (file);
#endif /* HAVE_ICUDB */

	if (kontakt->message_history != NULL) {
		HistoryWindow *hw = HISTORY_WINDOW(kontakt->message_history);
		GtkWidget *other_text;

		int dialog_len, other_len;
		guint dialog_point, other_point;

		other_text = received ? hw->incoming_text : hw->outgoing_text;

		text_pos_remember (GTK_TEXT (hw->dialog_text), &dialog_point, &dialog_len);
		text_pos_remember (GTK_TEXT (other_text), &other_point, &other_len);

		append_history_entry (hw, *my_tm, statement,
				      strlen(statement),
				      received);

		text_pos_restore (GTK_TEXT (hw->dialog_text), dialog_point, dialog_len);
		text_pos_restore (GTK_TEXT (other_text), other_point, other_len);
	}
}

static HistoryWindow* history_construct (CONTACT_PTR contact)
{
	HistoryWindow *hw;
	gchar *dialog_name;

	GtkWidget *window;
	GtkWidget *scrolledwindow;
	GtkWidget *text;
	GtkWidget *incoming_text;
	GtkWidget *outgoing_text;
	GtkWidget *close_button;
	GtkWidget *notebook;
	GtkWidget *dialog_vbox;
	GtkWidget *incoming_vbox;
	GtkWidget *outgoing_vbox;
	GtkWidget *functionbutton;

#ifdef TRACE_FUNCTION
	g_print("history_construct\n");
#endif
	
	hw = g_new0 (HistoryWindow, 1);

	dialog_name = g_strdup_printf( _("GnomeICU: Message History: %s"), contact->nick );
	window = gnome_dialog_new ( dialog_name , NULL);
	g_free( dialog_name );

	gtk_object_set_data( GTK_OBJECT( window ), "contact", contact );
	gtk_window_set_default_size ( GTK_WINDOW( window ), 400, 300);
	gtk_window_set_position (GTK_WINDOW (window), GTK_WIN_POS_CENTER);
	gtk_window_set_policy (GTK_WINDOW (window), TRUE, TRUE, FALSE);
	gtk_window_set_wmclass (GTK_WINDOW (window), "window", "GnomeICU");

	set_window_icon( window, "gnomeicu-hist.png" );

	hw->window = window;

	/* Notebook */

	notebook = gtk_notebook_new ();
	gtk_notebook_set_tab_pos (GTK_NOTEBOOK (notebook), GTK_POS_TOP);
	gtk_object_set_data (GTK_OBJECT (window), "notebook", notebook);
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (window)->vbox), notebook, TRUE, TRUE, 0);
	gtk_widget_show (notebook);

	dialog_vbox = gtk_vbox_new (0, 5);
	gtk_container_set_border_width (GTK_CONTAINER (dialog_vbox), 5);
	gtk_widget_show (dialog_vbox);

	incoming_vbox = gtk_vbox_new (0, 5);
	gtk_container_set_border_width (GTK_CONTAINER (incoming_vbox), 5);
	gtk_widget_show (incoming_vbox);

	outgoing_vbox = gtk_vbox_new (0, 5);
	gtk_container_set_border_width (GTK_CONTAINER (outgoing_vbox), 5);
	gtk_widget_show (outgoing_vbox);

	gtk_notebook_append_page (GTK_NOTEBOOK (notebook), dialog_vbox,
				  gtk_label_new (_("Dialog")));
	gtk_notebook_append_page (GTK_NOTEBOOK (notebook), incoming_vbox,
				  gtk_label_new (_("Incoming")));
	gtk_notebook_append_page (GTK_NOTEBOOK (notebook), outgoing_vbox,
				  gtk_label_new (_("Outgoing")));

	/* Dialog Notebook Page */

	scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
	gtk_widget_show (scrolledwindow);
	gtk_box_pack_start (GTK_BOX (dialog_vbox), scrolledwindow, TRUE, TRUE, 0);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow), GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);

	hw->dialog_text = text = gtk_text_new (NULL, NULL);
	gtk_text_set_word_wrap( GTK_TEXT( text ), TRUE );
	gtk_widget_set_usize( text, 225, 200 );
	gtk_widget_show (text);
	gtk_container_add (GTK_CONTAINER (scrolledwindow), text);

	/* Incoming Notebook Page */
	scrolledwindow = gtk_scrolled_window_new (NULL, NULL);

	gtk_widget_show (scrolledwindow);
	gtk_box_pack_start (GTK_BOX (incoming_vbox), scrolledwindow, TRUE,
			    TRUE, 0);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow),
					GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
	hw->incoming_text =
		incoming_text = gtk_text_new (NULL, NULL);
	gtk_text_set_word_wrap (GTK_TEXT (incoming_text), TRUE);
	gtk_widget_show (incoming_text);
	gtk_container_add (GTK_CONTAINER (scrolledwindow), incoming_text);
	
	/* Outgoing Notebook Page */
	scrolledwindow = gtk_scrolled_window_new (NULL, NULL);

	gtk_widget_show (scrolledwindow);
	gtk_box_pack_start (GTK_BOX (outgoing_vbox), scrolledwindow, TRUE,
			    TRUE, 0);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow),
					GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);

	hw->outgoing_text = outgoing_text = gtk_text_new (NULL, NULL);
	gtk_text_set_word_wrap (GTK_TEXT (outgoing_text), TRUE);
	gtk_widget_show (outgoing_text);
	gtk_container_add (GTK_CONTAINER (scrolledwindow), outgoing_text);

   	/* Function button */
	gnome_dialog_append_button_with_pixmap (GNOME_DIALOG (window),
						_("Functions >>"),
						GNOME_STOCK_PIXMAP_EXEC);
	functionbutton = g_list_last (GNOME_DIALOG (window)->buttons)->data;

	gtk_signal_connect (GTK_OBJECT (functionbutton), "clicked",
	                    GTK_SIGNAL_FUNC (search_show_cb), hw);

	gtk_object_set_data (GTK_OBJECT (functionbutton), "notebook", notebook);

	gtk_widget_show (functionbutton);
   	/* End Search */

	gnome_dialog_append_button (GNOME_DIALOG (window), GNOME_STOCK_BUTTON_CLOSE);
	close_button = g_list_last (GNOME_DIALOG (window)->buttons)->data;

	gtk_signal_connect_object (GTK_OBJECT (close_button), "clicked",
	                           GTK_SIGNAL_FUNC (gtk_widget_destroy),
	                           GTK_OBJECT (window));

	gtk_window_set_default (GTK_WINDOW (window), close_button);

	gtk_signal_connect (GTK_OBJECT (window), "destroy",
			    GTK_SIGNAL_FUNC (history_destroyed),
			    contact);
	
	return hw;
}

static void history_destroyed (GtkWidget *widget, gpointer data)
{
	CONTACT_PTR contact = (CONTACT_PTR) data;

#ifdef TRACE_FUNCTION
	g_print ("history_destroyed\n");
#endif

	g_free (HISTORY_WINDOW (contact->message_history)->filename);
	g_free (contact->message_history);
	contact->message_history = NULL;
}

/* Creates one "searchbox":
 * |         | [Search] [Save] [Clear]
 */
static GtkWidget* create_searchhbox (GtkBox *parent_box,
                                     GtkWidget *slave_text,
                                     HistoryWindow *hw)
{
	GtkWidget *searchhbox;
	GtkWidget *searchbutton;
	GtkWidget *searchbutton_label;
	GtkWidget *searchentry;
	GtkWidget *savebutton;
	GtkWidget *clearbutton;
	GtkWidget *clearbutton_label;
	GtkWidget *savebutton_label;

	searchhbox = gtk_hbox_new (FALSE, 8);
	gtk_box_pack_end (parent_box, searchhbox, FALSE, TRUE, 0);

	searchentry = gtk_entry_new ();
	gtk_signal_connect (GTK_OBJECT (searchentry), "changed",
			    GTK_SIGNAL_FUNC (searchentry_changed), NULL);

	gtk_box_pack_start (GTK_BOX (searchhbox), searchentry, TRUE, TRUE, 0);

	clearbutton = gtk_button_new ();
	gtk_box_pack_end (GTK_BOX (searchhbox), clearbutton, FALSE, TRUE, 0);

	savebutton = gtk_button_new ();
	gtk_box_pack_end (GTK_BOX (searchhbox), savebutton, FALSE, TRUE, 0);

	searchbutton = gtk_button_new ();
	gtk_box_pack_end (GTK_BOX (searchhbox), searchbutton, FALSE, TRUE, 0);

	clearbutton_label = gtk_label_new (_("Clear"));
	gtk_container_add (GTK_CONTAINER (clearbutton), clearbutton_label);

	savebutton_label = gtk_label_new (_("Save As"));
	gtk_container_add (GTK_CONTAINER (savebutton), savebutton_label);

	searchbutton_label = gtk_label_new (_("Search"));
	gtk_container_add (GTK_CONTAINER (searchbutton), searchbutton_label);

	gtk_object_set_data (GTK_OBJECT (searchbutton), "searchentry", searchentry);
	gtk_object_set_data (GTK_OBJECT (searchbutton), "searchbutton", searchbutton);
	gtk_object_set_data (GTK_OBJECT (searchbutton), "history", slave_text);
	gtk_object_set_data (GTK_OBJECT (searchbutton), "label", searchbutton_label);
	gtk_object_set_data (GTK_OBJECT (searchbutton), "searched", GINT_TO_POINTER (FALSE));

	gtk_object_set_data (GTK_OBJECT (searchentry), "searchentry", searchentry);
	gtk_object_set_data (GTK_OBJECT (searchentry), "searchbutton", searchbutton);
	gtk_object_set_data (GTK_OBJECT (searchentry), "history", slave_text);
	gtk_object_set_data (GTK_OBJECT (searchentry), "label", searchbutton_label);
	gtk_object_set_data (GTK_OBJECT (searchentry), "searched", GINT_TO_POINTER (FALSE));

	gtk_object_set_data (GTK_OBJECT (savebutton), "contact", hw->contact);

	gtk_signal_connect (GTK_OBJECT (searchbutton), "clicked",
			    GTK_SIGNAL_FUNC (search_cb), NULL);

	gtk_signal_connect (GTK_OBJECT (searchentry), "activate",
			    GTK_SIGNAL_FUNC (search_cb), NULL);

	gtk_signal_connect (GTK_OBJECT (clearbutton), "clicked",
			    GTK_SIGNAL_FUNC (show_clear_dialog), hw);

	gtk_signal_connect (GTK_OBJECT (savebutton), "clicked",
			    GTK_SIGNAL_FUNC (save_hist_cb), slave_text);

	gtk_object_set_data (GTK_OBJECT (searchhbox), "searchentry", searchentry);
	return searchhbox;
}
