/* smb-method.h - VFS modules for SMB
 *
 *  Copyright (C) 2001 Red Hat Inc,
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * Author: Alexander Larsson <alexl@redhat.com>
 *
 * Some inspiration comes from the smbfs implementation in midnight commander
 * by Wayne Roberts <wroberts1@home.com>. 
 */

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <string.h>
#include <time.h>

#include <libgnomevfs/gnome-vfs.h>
#include <libgnomevfs/gnome-vfs-mime.h>

#include <libgnomevfs/gnome-vfs-method.h>
#include <libgnomevfs/gnome-vfs-module.h>
#include <libgnomevfs/gnome-vfs-module-shared.h>
#include <libgnomevfs/gnome-vfs-module-callback-module-api.h>
#include <libgnomevfs/gnome-vfs-standard-callbacks.h>

#include "smb-method.h"

#include <sys/types.h>

#include "samba/include/config.h"
/* don't load crap in "samba/include/includes.h" we don't use and which 
   conflicts with definitions in other includes */
#undef HAVE_LIBREADLINE
#define NO_CONFIG_H
#define BOOL_DEFINED
#undef MAX
#undef MIN

#include "samba/include/includes.h"
#undef strcpy

#define ROOT_CACHE_TIME (5*60)

typedef struct _SmbConnection SmbConnection;
typedef struct _SmbAuthInfo SmbAuthInfo;
typedef struct _SmbHandle SmbHandle;
typedef struct _SmbDirHandle SmbDirHandle;
typedef struct _SmbDirContent SmbDirContent;
typedef struct _SmbBrowseContent SmbBrowseContent;
typedef struct _SmbVirtualFile SmbVirtualFile;
typedef struct _SmbVirtualRoot SmbVirtualRoot;

typedef enum {
	SMB_VIRTUAL_TYPE_FILE,
	SMB_VIRTUAL_TYPE_DIRECTORY,
	SMB_VIRTUAL_TYPE_SHARE
} SmbVirtualFileType;

struct _SmbVirtualFile {
	char *name;
	char *mime_type;

	/* Common data: */
	SmbVirtualFileType virtual_type;
	GnomeVFSFilePermissions permissions;
	union {
		struct {
			/* Data for shares: */
			char *server;
			char *share;
			SmbConnection *connection;
		} share;

		struct {
			/* Data for non-dirs: */
			char *contents;
			GnomeVFSFileSize size;
		} file;

		struct {
			/* Data for dirs: */
			GList *dir_contents;
		} directory;
	} u;
};

struct _SmbVirtualRoot {
	char *name; /* NULL for smb: root */
	time_t cache_time;
	SmbVirtualFile *root;
};

struct _SmbAuthInfo {
	char *user;
	char *password;
};

/* One connection per server, share and auth-info */
struct _SmbConnection {
	struct cli_state *cli;
	char *server;
	char *share;
	/* authinfo, maybe group in struct? */
	char *user;
	char *password;
	uint refcount;
};

struct _SmbHandle {
	SmbConnection *connection;
	gboolean is_data;
	char *file_data;
	int fnum;
	GnomeVFSFileOffset offset;
};

struct _SmbDirContent {
	char *name;
	time_t mtime;
	time_t atime;
	time_t ctime;
	uint16 mode;
	uid_t uid;
	gid_t gid;
	SMB_OFF_T size;
};

struct _SmbDirHandle {
	gboolean is_files;
	char *server;
	GList *contents;
	const GnomeVFSDirectoryFilter *filter;
};

/* Global vars: */
static GPrivate *dir_key = NULL;
static GPrivate *static_key = NULL;
static GPrivate *get_info_key = NULL;
static GMutex *samba_lock = NULL;

extern pstring global_myname;

static GHashTable *virtual_hash = NULL;
static GHashTable *server_auth_infos = NULL;

#undef DEBUG_SMB_ENABLE
#undef DEBUG_SMB_LOCKS

#ifdef DEBUG_SMB_ENABLE
#define DEBUG_SMB(x) g_print x
#else
#define DEBUG_SMB(x) 
#endif

#ifdef DEBUG_SMB_LOCKS
#define LOCK_SAMBA() 	{g_mutex_lock (samba_lock); g_print ("LOCK %s\n", G_GNUC_PRETTY_FUNCTION);}
#define UNLOCK_SAMBA() 	{g_print ("UNLOCK %s\n", G_GNUC_PRETTY_FUNCTION); g_mutex_unlock (samba_lock);}
#else
#define LOCK_SAMBA() 	g_mutex_lock (samba_lock)
#define UNLOCK_SAMBA() 	g_mutex_unlock (samba_lock)
#endif



/* FIXME / TODO:
 *
 * Free unused virtual roots + connections after a while
 * Free stuff on shutdown 
 */

static void smb_connection_unref (SmbConnection *connection);
static GnomeVFSResult gnome_vfs_result_from_cli (struct cli_state *cli);


static SmbVirtualFile *
smb_virtual_file_new (SmbVirtualFileType type)
{
	SmbVirtualFile *file;

	file = g_new0 (SmbVirtualFile, 1);
	file->virtual_type = type;
}

static SmbVirtualFile *
smb_virtual_file_shallow_copy (SmbVirtualFile *orig)
{
	SmbVirtualFile *file;

	file = g_new0 (SmbVirtualFile, 1);

	file->name = g_strdup (orig->name);
	file->mime_type = g_strdup (orig->mime_type);
	file->permissions = orig->permissions;
	
	file->virtual_type = orig->virtual_type;

	switch (file->virtual_type) {
	case SMB_VIRTUAL_TYPE_SHARE:
		file->u.share.server = g_strdup (orig->u.share.server);
		file->u.share.share = g_strdup (orig->u.share.share);
		break;
	case SMB_VIRTUAL_TYPE_FILE:
		file->u.file.contents = NULL; /* Shallow copy */
		file->u.file.size = orig->u.file.size;
		break;
	case SMB_VIRTUAL_TYPE_DIRECTORY:
		file->u.directory.dir_contents = NULL; /* Shallow copy */
		break;
	}
	return file;
}

static GList *
smb_virtual_file_list_shallow_copy (GList *list)
{
	GList *new = NULL;

	while (list) {
		new = g_list_prepend (new, smb_virtual_file_shallow_copy (list->data));
		list = list->next;
	}

	return g_list_reverse (new);
}


static void
smb_virtual_file_free (SmbVirtualFile *virt)
{
	GList *l;
	
	g_free (virt->name);
	g_free (virt->mime_type);

	switch (virt->virtual_type) {
	case SMB_VIRTUAL_TYPE_SHARE:
		g_free (virt->u.share.server);
		g_free (virt->u.share.share);
		if (virt->u.share.connection)
			smb_connection_unref (virt->u.share.connection);
		virt->u.share.connection = NULL;
		break;
	case SMB_VIRTUAL_TYPE_FILE:
		g_free (virt->u.file.contents);
		break;
	case SMB_VIRTUAL_TYPE_DIRECTORY:
		l = virt->u.directory.dir_contents;
		while (l) {
			smb_virtual_file_free (l->data);
			l = l->next;
		}
		g_list_free (virt->u.directory.dir_contents);
		break;
	}

	g_free (virt);
}

/* Path passed must be absolute */
static GnomeVFSResult
smb_virtual_file_lookup (SmbVirtualFile  *root,
			 const char      *path,
			 SmbVirtualFile **result,
			 char           **remainder_of_path)
{
	const char *end;
	int part_len;
	SmbVirtualFile *dir;
	SmbVirtualFile *file;
	GList *l;
	
	*result = NULL;
	*remainder_of_path = NULL;
	
	g_return_val_if_fail (root->virtual_type == SMB_VIRTUAL_TYPE_DIRECTORY, GNOME_VFS_ERROR_BAD_PARAMETERS);
	
	if (path == NULL) {
		*result = root;
		return GNOME_VFS_OK;
	}
	
	g_return_val_if_fail (*path == '/', GNOME_VFS_ERROR_BAD_PARAMETERS);

	
	if (*path != '/')
		return GNOME_VFS_ERROR_BAD_PARAMETERS;

	while (*path == '/')
		path++;

	dir = root;

	while (TRUE) {
		if (*path == 0) {
			*result = dir;
			return GNOME_VFS_OK;
		}

		end = strchr (path, '/');
		
		if (end)
			part_len = end - path;
		else
			part_len = strlen (path);

		l = dir->u.directory.dir_contents;
		while (l) {
			file = l->data;

			if (strncmp (file->name, path, part_len) == 0) {
				path += part_len;
				
				while (*path == '/')
					path++;
				
				if (file->virtual_type == SMB_VIRTUAL_TYPE_DIRECTORY) {
					dir = file;
					continue;
				}

				if (file->virtual_type != SMB_VIRTUAL_TYPE_SHARE &&
				    *path != 0) {
					return GNOME_VFS_ERROR_NOT_A_DIRECTORY;
				}
				
				if (file->virtual_type == SMB_VIRTUAL_TYPE_SHARE &&
				    *path != 0) {
					*remainder_of_path = g_strconcat ("/", path, NULL);
				}

				*result = file;
				return GNOME_VFS_OK;
			}
			l = l->next;
		}
		return GNOME_VFS_ERROR_NOT_FOUND;
	}
	
	g_assert_not_reached();
	return GNOME_VFS_ERROR_INTERNAL;
}

static char *
unix_filename_to_dos (const char *filename)
{
	int len, i;
	char *dos_filename;
	
	dos_filename = g_strdup (filename);
	len = strlen (dos_filename);
	for (i = 0; i < len; i++)
		if (dos_filename[i] == '/')
			dos_filename[i] = '\\';
	
	return dos_filename;
}

static GnomeVFSResult
smb_server_connection_new_from_ip (struct in_addr *ip,
				   const char *fallback_server_name,
				   const char *share,
				   const char *user,
				   const char *password,
				   SmbConnection **connection)
{
	struct cli_state *cli;
	struct nmb_name called, calling;
	char servername[256];
	pstring workgroup;
	SmbConnection *smbcon;
	const char *server;
	GnomeVFSResult res;

	DEBUG_SMB (("smb_server_connection_new_from_ip (ip: %s, share: %s, user: %s, password: %s)\n",
		    inet_ntoa (*ip), (share)?share:"NULL", user, password));

	if (name_status_find (0x20, *ip, servername)) {
		server = servername;
	} else {
		server = fallback_server_name;
	}
	
	make_nmb_name (&calling, global_myname, 0x0);
	make_nmb_name (&called , server, 0x20);

	DEBUG_SMB (("smb_server_connection_new_from_ip: caller.name: %s, called.name: %s\n",
		    calling.name, called.name));
	
	/* Read the workgroup from the config file */
	pstrcpy (workgroup, lp_workgroup ());

	cli = cli_initialise (NULL);
	
	if (!cli) {
		g_warning ("Couldn't initialize cli");
		return GNOME_VFS_ERROR_INTERNAL;
	}	
	if (!cli_set_port (cli, SMB_PORT)) {
		g_warning ("Couldn't set port");
		res = gnome_vfs_result_from_cli (cli);
		cli_shutdown (cli);
		free (cli);
		return res;
	}
	
	if (!cli_connect (cli, server, ip)) {
		g_warning ("Couldn't connect to server");
		cli_shutdown (cli);
		free (cli);
		return GNOME_VFS_ERROR_HOST_NOT_FOUND;
	}

	cli->protocol = PROTOCOL_NT1;

	if (!cli_session_request (cli, &calling, &called)) {
		g_warning ("Couldn't request session");
		res = gnome_vfs_result_from_cli (cli);
		/* TODO: smbclient tries some alternative names here */ 
		cli_shutdown (cli);
		free (cli);
		return res;
	}
	
	if (!cli_negprot (cli)) {
		g_warning ("Couldn't negotiate protocol");
		cli_shutdown (cli);
		free (cli);
		return GNOME_VFS_ERROR_LOGIN_FAILED;
	}

	if (!user)
	  user = "GUEST";
	if (!password)
	  password = "";

	if (!cli_session_setup (cli,
				/* Username: */
				(char *)user,
				/* Password: */
				(char *)password, strlen (password),
				/* NT Password: */
				(char *)password, strlen (password),
				/* Workgroup: */
				workgroup)) {
		/* Try anon login */
		if (!cli_session_setup (cli,
					"",
					"", 0,
					"", 0,
					workgroup)) { 
			cli_shutdown (cli);
			free (cli);
			return GNOME_VFS_ERROR_LOGIN_FAILED;
		}
		/* Anon login ok */
		user = "";
		password = "";
	}

	/* Connect to the share */
	if (share && !cli_send_tconX (cli,
				      (char *)share,
				      "?????",
				      (char *)password, strlen (password)+1)) {
		g_warning ("couldn't send tconX: res: %s",
			   cli_errstr (cli));
		res = gnome_vfs_result_from_cli (cli);
		cli_shutdown (cli);
		free(cli);
		return res;
	}

	smbcon = g_new (SmbConnection, 1);
	smbcon->cli = cli;
	smbcon->refcount = 1;
	smbcon->server = g_strdup (server);
	if (share)
		smbcon->share = g_strdup (share);
	else
		smbcon->share = NULL;
	smbcon->user = g_strdup (user);
	smbcon->password = g_strdup (password);

	*connection = smbcon;
	
	return GNOME_VFS_OK;
}


static GnomeVFSResult
smb_server_connection_new (const char *server,
			   const char *share,
			   const char *user,
			   const char *password,
			   SmbConnection **connection)
{
	struct in_addr ip;

	DEBUG_SMB (("smb_server_connection_new (server: %s, share: %s, user: %s, password: %s)\n",
		    server, (share)?share:"NULL", user, password));

	
	if (!resolve_name (server, &ip, 0x20)) {
		DEBUG_SMB (("smb_server_connection_new: Couldn't resolve name %s\n", server));
		return GNOME_VFS_ERROR_HOST_NOT_FOUND;
	}

	return smb_server_connection_new_from_ip (&ip,
						  server,
						  share,
						  user,
						  password,
						  connection);
}

static void
smb_connection_last_refcount (SmbConnection *connection)
{
	cli_shutdown (connection->cli);
	g_free (connection->cli);
	g_free (connection->server);
	g_free (connection->share);
	g_free (connection->user);
	g_free (connection->password);

	g_free (connection);
}

static void
smb_connection_unref (SmbConnection *connection)
{
	connection->refcount -= 1;
	if (connection->refcount == 0)
		smb_connection_last_refcount (connection); 
}

static SmbConnection *
smb_connection_ref (SmbConnection *connection)
{
	connection->refcount += 1;
	return connection;
}



static void
smb_dir_content_destroy (SmbDirContent *content)
{
	g_free (content->name);
	g_free (content);
}

static void
smb_dir_handle_destroy (SmbDirHandle *handle)
{
	GList *list;
	SmbDirContent *dir_content;
	SmbVirtualFile *file;
	if (!handle)
		return;
	
	list = handle->contents;
	while (list) {
		if (handle->is_files) {
			dir_content = (SmbDirContent *)list->data;
			smb_dir_content_destroy (dir_content);
		} else {
			file = (SmbVirtualFile *)list->data;
			smb_virtual_file_free (file);
		}
		list = g_list_next (list);
	}
	g_free (handle->server);
	g_list_free (handle->contents);
}

static gboolean
get_a_master_browser (struct in_addr *ip)
{
	struct in_addr *bcast, *ip_list;
	int iface;
	int count;
	int fd;

	fd = open_socket_in (SOCK_DGRAM,
			     0, 3,
                             interpret_addr(lp_socket_address()),
			     TRUE);

	if (fd < 0)
		return FALSE;

	set_socket_options (fd, "SO_BROADCAST");

	for (iface = iface_count () -1; iface >= 0; iface--) {
		bcast = iface_n_bcast (iface);
		ip_list = name_query (fd,
				      "\01\02__MSBROWSE__\02", 1,
				      TRUE,
				      TRUE,
				      *bcast,
				      &count);
		if (ip_list) {
			close (fd);
			*ip = ip_list[0];
			free (ip_list);
			return TRUE;
		}
	}
	close (fd);
	return FALSE;
}

static char *
get_root_dir_data (void)
{
  return g_strdup ("[Desktop Entry]\n"
		   "Encoding=Legacy-Mixed\n"
		   "Name=Network neighborhood\n"
		   "Type=Directory\n"
		   "Icon=gnome-computer.png\n");
}

static char *
get_workgroup_dir_data (const char *name)
{
  return g_strdup_printf ("[Desktop Entry]\n"
			  "Encoding=Legacy-Mixed\n"
			  "Name=Workgroup %s\n"
			  "Type=Directory\n"
			  "Icon=gnome-computer.png\n",
			  name);
}

static char *
get_server_dir_data (const char *name)
{
  return g_strdup_printf ("[Desktop Entry]\n"
			  "Encoding=Legacy-Mixed\n"
			  "Name=Shares on %s\n"
			  "Type=Directory\n"
			  "Icon=gnome-computer.png\n",
			  name);
}


static char *
get_workgroup_data (const char *name)
{
  return g_strdup_printf ("[Desktop Entry]\n"
			  "Encoding=Legacy-Mixed\n"
			  "Name=%s\n"
			  "Type=Link\n"
			  "URL=smb://%s/\n"
			  "Icon=gnome-computer.png\n",
			  name, name);
}

static char *
get_computer_data (const char *name)
{
  return g_strdup_printf ("[Desktop Entry]\n"
			  "Encoding=Legacy-Mixed\n"
			  "Name=%s\n"
			  "Type=Link\n"
			  "URL=smb://%s/\n"
			  "Icon=gnome-computer.png\n",
			  name, name);
}

static GnomeVFSResult
gnome_vfs_result_from_cli (struct cli_state *cli)
{
	int err;
	GnomeVFSResult res;
	guint8 errclass;
	guint32 errnum;
	guint32 nt_rpc_error;
	
	err = cli_error (cli, &errclass, &errnum, &nt_rpc_error);
	if (errclass == ERRSRV && errnum == ERRbadpw)
		res = GNOME_VFS_ERROR_LOGIN_FAILED;
	else
		res = gnome_vfs_result_from_errno_code (err);

	return res;
}



/***************** virtual tree handling *****************/

/* Pffft. Stupid SAMBA, no user-data */
static SmbVirtualFile *global_build_root_helper_dir;
static void
build_root_helper (const char *workgroup,
		   uint32 m,
		   const char *master)
{
	SmbVirtualFile *file;
	SmbVirtualFile *dir;

	file = smb_virtual_file_new (SMB_VIRTUAL_TYPE_FILE);
	file->name = g_strdup (workgroup);
	file->mime_type = g_strdup("application/x-gnome-app-info");
	file->permissions =
		GNOME_VFS_PERM_USER_READ |
		GNOME_VFS_PERM_GROUP_READ |
		GNOME_VFS_PERM_OTHER_READ;

	file->u.file.contents = get_workgroup_data (workgroup);
	file->u.file.size = strlen (file->u.file.contents);
	
	dir = global_build_root_helper_dir;
	
	dir->u.directory.dir_contents =
		g_list_prepend (dir->u.directory.dir_contents, file);
}

static GnomeVFSResult
build_root_tree (SmbVirtualFile **out)
{
	SmbVirtualFile *root;
	GnomeVFSResult res;
	struct in_addr ip;
	SmbConnection *connection;
	SmbVirtualFile *file;

	DEBUG_SMB (("build_root_tree ()\n"));
	
	*out = NULL;

	if (!get_a_master_browser (&ip)) {
		g_warning ("Couldn't find a master browser");
		return GNOME_VFS_ERROR_HOST_NOT_FOUND;
	}

	res = smb_server_connection_new_from_ip (&ip,
						 "*SMBSERVER",
						 "IPC$",
						 NULL,
						 NULL,
						 &connection);
	if (res) {
		g_warning ("build_root_tree: Failed to connect");
		return res;
	}

	root = smb_virtual_file_new (SMB_VIRTUAL_TYPE_DIRECTORY);
	root->name = NULL;
	root->mime_type = g_strdup("x-directory/normal");

	global_build_root_helper_dir = root;
	
	cli_NetServerEnum (connection->cli,
			   connection->cli->server_domain,
			   SV_TYPE_DOMAIN_ENUM,
			   build_root_helper);

	smb_connection_unref (connection);

	file = smb_virtual_file_new (SMB_VIRTUAL_TYPE_FILE);
	file->name = g_strdup (".directory");
	file->mime_type = g_strdup("application/x-gnome-app-info");
	file->permissions =
		GNOME_VFS_PERM_USER_READ |
		GNOME_VFS_PERM_GROUP_READ |
		GNOME_VFS_PERM_OTHER_READ;

	file->u.file.contents = get_root_dir_data ();
	file->u.file.size = strlen (file->u.file.contents);

	root->u.directory.dir_contents =
		g_list_prepend (root->u.directory.dir_contents, file);

	*out = root;
	return GNOME_VFS_OK;
}

/* Pffft. Stupid SAMBA, no user-data */
static SmbVirtualFile *global_build_server_helper_dir;
static const char *global_build_server_helper_server;
static void
build_workgroup_helper (const char *server,
			uint32 m,
			const char *comment)
{
	SmbVirtualFile *file;
	SmbVirtualFile *dir;

	file = smb_virtual_file_new (SMB_VIRTUAL_TYPE_FILE);
	file->name = g_strdup (server);
	file->mime_type = g_strdup("application/x-gnome-app-info");

	file->u.file.contents = get_computer_data (server);
	file->u.file.size = strlen (file->u.file.contents);
	file->permissions =
		GNOME_VFS_PERM_USER_READ |
		GNOME_VFS_PERM_GROUP_READ |
		GNOME_VFS_PERM_OTHER_READ;
	
	dir = global_build_server_helper_dir;
	
	dir->u.directory.dir_contents =
		g_list_prepend (dir->u.directory.dir_contents, file);
}

static void
build_server_helper (const char *share,
		     uint32 m,
		     const char *comment)
{
	SmbVirtualFile *file;
	SmbVirtualFile *dir;
	int len;

	if (m != STYPE_DISKTREE)
		return;

	if (m & STYPE_HIDDEN)
		return;

	len = strlen (share);
	if (len == 0 | share[len-1]=='$')
		return;
	
	dir = global_build_server_helper_dir;

	file = smb_virtual_file_new (SMB_VIRTUAL_TYPE_SHARE);
	file->name = g_strdup (share);
	file->mime_type = g_strdup("x-directory/normal");
	file->permissions =
		GNOME_VFS_PERM_USER_ALL |
		GNOME_VFS_PERM_GROUP_ALL |
		GNOME_VFS_PERM_OTHER_ALL;

	file->u.share.server = g_strdup (global_build_server_helper_server);
	file->u.share.share = g_strdup (share);
	file->u.share.connection = NULL;
	
	dir->u.directory.dir_contents =
		g_list_prepend (dir->u.directory.dir_contents, file);
}
static GnomeVFSResult
build_server_tree (SmbVirtualFile **out, const char *name)
{
	SmbVirtualFile *root;
	SmbVirtualFile *file;
	GnomeVFSResult res;
	int error;
	int ret;
	struct in_addr ip;
	SmbConnection *connection;
	gboolean found_workgroup;
	SmbAuthInfo *auth_info;
	char *user, *password;
	GnomeVFSModuleCallbackAuthenticationIn in_args;
	GnomeVFSModuleCallbackAuthenticationOut out_args;
	gboolean got_shares;
	
	DEBUG_SMB (("build_server_tree (%s)\n", name));

	*out = NULL;
	
	root = smb_virtual_file_new (SMB_VIRTUAL_TYPE_DIRECTORY);
	root->name = NULL;
	root->mime_type = g_strdup("x-directory/normal");

	global_build_server_helper_dir = root;
	global_build_server_helper_server = name;
	
	/* FIXME: Handle user + passwords here */
	   
	DEBUG_SMB (("Enumerating servers in workgroup %s\n", name));
	/* Look for any servers in the workgroup named name */
	if (find_master_ip ((char *)name, &ip)) {
		res = smb_server_connection_new_from_ip (&ip,
							 "*SMBSERVER",
							 "IPC$",
							 NULL /* user */,
							 NULL /* pwd */,
							 &connection);
		if (!res) {
			cli_NetServerEnum (connection->cli,
					   connection->cli->server_domain,
					   SV_TYPE_ALL,
					   build_workgroup_helper);
			smb_connection_unref (connection);
		} else {
		  DEBUG_SMB (("b_s_tree: Couldn't open for server listing, res: %d", res));
		}
	}
	DEBUG_SMB (("Done enumerating servers in workgroup %s\n", name));

	found_workgroup = root->u.directory.dir_contents != NULL;

	DEBUG_SMB (("Enumerating shares in server %s\n", name));

	user = NULL;
	password = NULL;
	
	auth_info = g_hash_table_lookup (server_auth_infos, name);

	if (auth_info) {
		user = g_strdup (auth_info->user);
		password = g_strdup (auth_info->password);
	}

	memset (&in_args, 0, sizeof (in_args));

	in_args.uri = g_strdup_printf ("smb://%s", name);
	got_shares = FALSE;
	
	while (TRUE) {
		/* Look for shares in the server named name */
		res = smb_server_connection_new (name,
						 "IPC$",
						 user /* user */,
						 password /* pwd */,
						 &connection);
		if (!res) {
			ret = cli_RNetShareEnum (connection->cli,
						 build_server_helper);
			if (ret == -1) {
				error = cli_error(connection->cli, NULL, NULL, NULL);
				if (error == EACCES) {
					memset (&out_args, 0, sizeof (out_args));
					res = gnome_vfs_module_callback_invoke (GNOME_VFS_MODULE_CALLBACK_AUTHENTICATION,
										&in_args, sizeof (in_args), 
										&out_args, sizeof (out_args));
					if (!res || (out_args.username == NULL)) {
						break;
					} else {
						g_free (user);
						g_free (password);
						
						user = out_args.username;
						password = out_args.password;
						
						smb_connection_unref (connection);
						
						continue;
					}
				}
			} else {
				got_shares = TRUE;
			}
			
			smb_connection_unref (connection);
		} else {
			DEBUG_SMB (("b_s_tree: Couldn't open %s for share listing, res %d\n", name, res));
		}
		break;
	}
	DEBUG_SMB (("Done enumerating shares in server %s\n", name));

	g_free (in_args.uri);

	if (!got_shares) {
		if (auth_info == NULL || auth_info->user != user)
			g_free (user);
		if (auth_info == NULL || auth_info->password != password)
			g_free (password);
	} else if (user != NULL || password != NULL) {
		if (auth_info == NULL) {
			auth_info = g_new (SmbAuthInfo, 1);
			auth_info->user = user;
			auth_info->password = password;
			g_hash_table_insert (server_auth_infos, g_strdup (name), auth_info);
		}

		if (auth_info->user != user) {
			g_free (auth_info->user);
			auth_info->user = user;
		}
		
		if (auth_info->password != password) {
			g_free (auth_info->password);
			auth_info->password = password;
		}
	}
	
	/* Add .directory file */
	file = smb_virtual_file_new (SMB_VIRTUAL_TYPE_FILE);
	file->name = g_strdup (".directory");
	file->mime_type = g_strdup("application/x-gnome-app-info");

	if (found_workgroup)
		file->u.file.contents = get_workgroup_dir_data (name);
	else
		file->u.file.contents = get_server_dir_data (name);
		
	file->u.file.size = strlen (file->u.file.contents);

	root->u.directory.dir_contents =
		g_list_prepend (root->u.directory.dir_contents, file);

	
	*out = root;
	return GNOME_VFS_OK;
}

static guint
virt_root_hash (gconstpointer key)
{
	const SmbVirtualRoot *root = key;

	if (root->name)
		return g_str_hash (root->name);
	else
		return 0;
}

gint
virt_root_equal (gconstpointer v, gconstpointer v2)
{
	const SmbVirtualRoot *root1 = v;
	const SmbVirtualRoot *root2 = v2;

	if (root1->name == NULL &&
	    root2->name == NULL)
		return TRUE;

	if (root1->name == NULL ||
	    root2->name == NULL)
		return FALSE;
	
	return g_str_equal (root1->name, root2->name);
}

static GnomeVFSResult
lookup_uri (GnomeVFSURI     *uri,
	    SmbVirtualFile **file,
	    char           **path_remainder,
	    SmbConnection  **connection,
	    gboolean         open_connection_for_share_file)
{
	char *server;
	char *path;
	SmbVirtualRoot *root;
	SmbVirtualRoot key;
	time_t now;
	GnomeVFSResult res;
	char *user;
	char *password;
	
	DEBUG_SMB (("lookup_uri (uri: %s)\n", gnome_vfs_uri_to_string (uri, 0)));
	
	if (virtual_hash == NULL) {
		virtual_hash = g_hash_table_new (virt_root_hash,
						 virt_root_equal);
	}

	*file = NULL;
	*path_remainder = NULL;
	if (connection)
		*connection = NULL;

	server = gnome_vfs_unescape_string (gnome_vfs_uri_get_host_name (uri), NULL);

	key.name = server;
	
	root = g_hash_table_lookup (virtual_hash, &key);

	DEBUG_SMB (("lookup_uri - Mapped %s to %p\n", server, root));
	
	now = time (NULL);

	if (root == NULL || root->root == NULL ||
	    ((now - root->cache_time) > ROOT_CACHE_TIME)) {
		DEBUG_SMB (("lookup_uri - rebuild %s tree\n", server));
		if (root == NULL) {
			DEBUG_SMB (("lookup_uri - allocate new virtual root for %s\n", server));
			root = g_new (SmbVirtualRoot, 1);
			root->name = g_strdup (server);
			g_hash_table_insert (virtual_hash, root, root);
		} else {
			if (root->root)
				smb_virtual_file_free (root->root);
		}
		root->cache_time = now;
		root->root = NULL;

		if (server)
			res = build_server_tree (&root->root, server);
		else
			res = build_root_tree (&root->root);

		if (res) {
			g_free (server);
			return res;
		}
	}

	g_free (server);
	
	path = gnome_vfs_unescape_string (gnome_vfs_uri_get_path (uri), NULL);
	if (path && *path == 0) {
		DEBUG_SMB (("Hmmm. gnome_vfs_uri_get_path (uri) returned an empty string...\n"));
		g_free (path);
		path = NULL;
	}
	
	DEBUG_SMB (("lookup_uri - looking up '%s' (%p) in root\n", path, path));
	res = smb_virtual_file_lookup (root->root, path, file, path_remainder);
	DEBUG_SMB (("lookup_uri - smb_virtual_file_lookup returned %d\n", res));

	g_free (path);

	if (res)
		return res;

	if (connection &&
	    (*file)->virtual_type == SMB_VIRTUAL_TYPE_SHARE &&
	    ((path_remainder != NULL) ||
	     open_connection_for_share_file)) {
		if ((*file)->u.share.connection == NULL) {
			GnomeVFSModuleCallbackAuthenticationIn in_args;
			GnomeVFSModuleCallbackAuthenticationOut out_args;
			gboolean previous_attempt_failed;

			memset (&in_args, 0, sizeof (in_args));
			
			in_args.uri = g_strdup_printf ("smb://%s/%s",
						       (*file)->u.share.server,
						       (*file)->u.share.share);

			user = gnome_vfs_unescape_string (gnome_vfs_uri_get_user_name (uri), NULL);
			password = gnome_vfs_unescape_string (gnome_vfs_uri_get_password (uri), NULL);

			/* Don't consider it failed if it was an anon try */
			previous_attempt_failed = (user == NULL) ? FALSE : TRUE;

			while (TRUE) {
				res = smb_server_connection_new ((*file)->u.share.server,
								 (*file)->u.share.share,
								 user,
								 password,
								 &(*file)->u.share.connection);
				if (res == GNOME_VFS_ERROR_LOGIN_FAILED) {
					memset (&out_args, 0, sizeof (out_args));
					res = gnome_vfs_module_callback_invoke (GNOME_VFS_MODULE_CALLBACK_AUTHENTICATION,
										&in_args, sizeof (in_args), 
										&out_args, sizeof (out_args));
					if (!res) {
						break;
					} else if (out_args.username == NULL) {
						res = GNOME_VFS_ERROR_ACCESS_DENIED;
						break;
					} else {
						g_free (user);
						g_free (password);
						
						user = out_args.username;
						password = out_args.password;
					}
					previous_attempt_failed = TRUE;
				} else {
					/* Error, or logged in ok */
					break;
				}
			}

			g_free (in_args.uri);
			g_free (user);
			g_free (password);
			
			if (res) {
				(*file)->u.share.connection = NULL;
				g_free (*path_remainder);
				return res;

			}
		}
		*connection = smb_connection_ref ((*file)->u.share.connection);
	}

	return res;
}

static GnomeVFSResult
do_open (GnomeVFSMethod *method,
	 GnomeVFSMethodHandle **method_handle,
	 GnomeVFSURI *uri,
	 GnomeVFSOpenMode mode,
	 GnomeVFSContext *context)
{
	GnomeVFSResult res;
	SmbConnection *connection;
	SmbHandle *handle;
	SmbVirtualFile *file;
	char *path_remainder;
	int fnum;
	char *filename;
	char *dos_filename;
	int flags;
	char *server;
	char *path;
	char *end;
	const char *user;
	const char *password;
	
	LOCK_SAMBA();
	
	DEBUG_SMB (("do_open() %s mode %d\n",
		    gnome_vfs_uri_to_string (uri, 0), mode));
	
	if ((mode & GNOME_VFS_OPEN_READ) &&
	    (mode & GNOME_VFS_OPEN_WRITE))
		flags = O_RDWR;
	else if (mode & GNOME_VFS_OPEN_READ)
		flags = O_RDONLY;
	else if (mode & GNOME_VFS_OPEN_WRITE)
		flags = O_WRONLY;
	else
		return GNOME_VFS_ERROR_INVALID_OPEN_MODE;

	res = lookup_uri (uri, &file, &path_remainder, &connection, TRUE);

	if (res) {
		UNLOCK_SAMBA ();
		return res;
	}

	if (file->virtual_type == SMB_VIRTUAL_TYPE_DIRECTORY) {
		UNLOCK_SAMBA ();
		return GNOME_VFS_ERROR_IS_DIRECTORY;
	}

	if (file->virtual_type == SMB_VIRTUAL_TYPE_FILE) {
		handle = g_new (SmbHandle, 1);
		handle->is_data = TRUE;
		handle->offset = 0;
		handle->file_data = g_strdup (file->u.file.contents);
	} else { /* SMB_VIRTUAL_TYPE_SHARE */
		dos_filename = unix_filename_to_dos ((path_remainder)?path_remainder:"/");
		fnum = cli_open (connection->cli, dos_filename, flags, DENY_NONE);
		g_free (path_remainder);
		g_free (dos_filename);
		if (fnum == -1) {
			res = gnome_vfs_result_from_cli (connection->cli);
			smb_connection_unref (connection);
			UNLOCK_SAMBA();
			return res;
		}
		
		handle = g_new (SmbHandle, 1);
		handle->is_data = FALSE;
		handle->connection = connection;
		handle->fnum = fnum;
		handle->offset = 0;
	}

	UNLOCK_SAMBA();

	*method_handle = (GnomeVFSMethodHandle *)handle;

	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_close (GnomeVFSMethod *method,
	  GnomeVFSMethodHandle *method_handle,
	  GnomeVFSContext *context)

{
	SmbHandle *handle = (SmbHandle *)method_handle;
	size_t n;

	LOCK_SAMBA();
	DEBUG_SMB (("do_close() %p\n", method_handle));

	if (handle->is_data) {
		g_free (handle->file_data);
	} else {
		cli_close  (handle->connection->cli, handle->fnum);
		smb_connection_unref (handle->connection);
	}
	
	g_free (handle);
	
	UNLOCK_SAMBA();

	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_read (GnomeVFSMethod *method,
	 GnomeVFSMethodHandle *method_handle,
	 gpointer buffer,
	 GnomeVFSFileSize num_bytes,
	 GnomeVFSFileSize *bytes_read,
	 GnomeVFSContext *context)
{
	SmbHandle *handle = (SmbHandle *)method_handle;
	size_t n;
	int link_len;

	LOCK_SAMBA();
	DEBUG_SMB (("do_read() %p\n", method_handle));

	if (handle->is_data) {
		link_len = strlen (handle->file_data);
		if (handle->offset >= link_len)
			n = 0;
		else {
			n = MIN (num_bytes, link_len - handle->offset);
			memcpy (buffer, handle->file_data + handle->offset, n);
		}
	} else {
		n = cli_read (handle->connection->cli,
			      handle->fnum,
			      buffer,
			      handle->offset,
			      num_bytes);
	}

	UNLOCK_SAMBA();

	if (n < 0) {
		*bytes_read = 0;
		return gnome_vfs_result_from_cli (handle->connection->cli);
	}
	
	*bytes_read = n;
	
	if (n == 0)
		return GNOME_VFS_ERROR_EOF;
	
	handle->offset += n;
	
	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_write (GnomeVFSMethod *method,
	  GnomeVFSMethodHandle *method_handle,
	  gconstpointer buffer,
	  GnomeVFSFileSize num_bytes,
	  GnomeVFSFileSize *bytes_written,
	  GnomeVFSContext *context)


{
	SmbHandle *handle = (SmbHandle *)method_handle;
	ssize_t n;
	int link_len;

	LOCK_SAMBA();
	DEBUG_SMB (("do_write() %p\n", method_handle));

	if (handle->is_data) {
	  return GNOME_VFS_ERROR_NOT_SUPPORTED;
	} else {
		n = cli_write (handle->connection->cli,
			       handle->fnum,
			       0,
			       (char *)buffer,
			       handle->offset,
			       num_bytes);
	}

	UNLOCK_SAMBA();

	if (n < 0) {
		*bytes_written = 0;
		return gnome_vfs_result_from_cli (handle->connection->cli);
	}
	
	*bytes_written = n;
	
	if (n == 0)
		return GNOME_VFS_ERROR_EOF;
	
	handle->offset += n;
	
	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_create (GnomeVFSMethod *method,
	   GnomeVFSMethodHandle **method_handle,
	   GnomeVFSURI *uri,
	   GnomeVFSOpenMode mode,
	   gboolean exclusive,
	   guint perm,
	   GnomeVFSContext *context)

{
	GnomeVFSResult res;
	SmbVirtualFile *file;
	SmbConnection *connection;
	SmbHandle *handle;
	int fnum;
	char *path_remainder;
	char *dos_filename;
	int flags;
	char *server;
	char *path;

	LOCK_SAMBA();
	
	DEBUG_SMB (("do_create() %s mode %d\n",
		    gnome_vfs_uri_to_string (uri, 0), mode));
	
	if ((mode & GNOME_VFS_OPEN_READ) &&
	    (mode & GNOME_VFS_OPEN_WRITE))
		flags = O_RDWR;
	else if (mode & GNOME_VFS_OPEN_READ)
		flags = O_RDONLY;
	else if (mode & GNOME_VFS_OPEN_WRITE)
		flags = O_WRONLY;
	else
		return GNOME_VFS_ERROR_INVALID_OPEN_MODE;


	flags |= O_CREAT;
	if (exclusive)
		flags |= O_EXCL;
	
	res = lookup_uri (uri, &file, &path_remainder, &connection, TRUE);

	if (res) {
		UNLOCK_SAMBA ();
		if (res == GNOME_VFS_ERROR_NOT_FOUND)
			res = GNOME_VFS_ERROR_ACCESS_DENIED;
		return res;
	}

	if (file->virtual_type == SMB_VIRTUAL_TYPE_DIRECTORY) {
		UNLOCK_SAMBA ();
		return GNOME_VFS_ERROR_IS_DIRECTORY;
	}

	if (file->virtual_type == SMB_VIRTUAL_TYPE_FILE) {
		UNLOCK_SAMBA ();
		return GNOME_VFS_ERROR_ACCESS_DENIED;
	}
	
	/* SMB_VIRTUAL_TYPE_SHARE */
	dos_filename = unix_filename_to_dos ((path_remainder)?path_remainder:"/");
	DEBUG_SMB (("tryping to cli_open '%s', flags: %d\n", dos_filename, flags));
	fnum = cli_open (connection->cli, dos_filename, flags, DENY_NONE);
	g_free (path_remainder);
	g_free (dos_filename);
	if (fnum == -1) {
		res = gnome_vfs_result_from_cli (connection->cli);
		smb_connection_unref (connection);
		UNLOCK_SAMBA();
		return res;
	}
	
	handle = g_new (SmbHandle, 1);
	handle->is_data = FALSE;
	handle->connection = connection;
	handle->fnum = fnum;
	handle->offset = 0;

	UNLOCK_SAMBA();

	*method_handle = (GnomeVFSMethodHandle *)handle;

	return GNOME_VFS_OK;
}

static void
get_info_helper (file_info *info, const char *dir)
{
	file_info *smb_info;

	smb_info = g_private_get (get_info_key);
	*smb_info = *info;
}

static GnomeVFSResult
do_get_file_info (GnomeVFSMethod *method,
		  GnomeVFSURI *uri,
		  GnomeVFSFileInfo *vfs_file_info,
		  GnomeVFSFileInfoOptions options,
		  GnomeVFSContext *context)

{
	GnomeVFSResult res;
	SmbVirtualFile *file;
	SmbConnection *connection;
	char *filename;
	char *dos_filename;
	file_info *smb_info;
	char *path_remainder;


	LOCK_SAMBA();
	DEBUG_SMB (("do_get_file_info() %s\n",
		    gnome_vfs_uri_to_string (uri, 0)));
	res = lookup_uri (uri, &file, &path_remainder, &connection, FALSE);

	if (res) {
		UNLOCK_SAMBA();
		return res;
	}

	if (file->virtual_type == SMB_VIRTUAL_TYPE_DIRECTORY ||
	    (file->virtual_type == SMB_VIRTUAL_TYPE_SHARE &&
	     path_remainder == NULL)) {
		vfs_file_info->name = g_strdup (file->name);
		vfs_file_info->mime_type = g_strdup (file->mime_type);
		vfs_file_info->type = GNOME_VFS_FILE_TYPE_DIRECTORY;
		vfs_file_info->flags = GNOME_VFS_FILE_FLAGS_NONE;
		vfs_file_info->permissions = file->permissions;
		vfs_file_info->valid_fields =
			GNOME_VFS_FILE_INFO_FIELDS_TYPE |
			GNOME_VFS_FILE_INFO_FIELDS_PERMISSIONS |
			GNOME_VFS_FILE_INFO_FIELDS_FLAGS |
			GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE;
	} else if (file->virtual_type == SMB_VIRTUAL_TYPE_FILE) {
		vfs_file_info->name = g_strdup (file->name);
		vfs_file_info->mime_type = g_strdup (file->mime_type);
		vfs_file_info->type = GNOME_VFS_FILE_TYPE_REGULAR;
		vfs_file_info->flags = GNOME_VFS_FILE_FLAGS_NONE;
		vfs_file_info->size = file->u.file.size;
		vfs_file_info->permissions = file->permissions;
		vfs_file_info->valid_fields =
			GNOME_VFS_FILE_INFO_FIELDS_TYPE |
			GNOME_VFS_FILE_INFO_FIELDS_PERMISSIONS |
			GNOME_VFS_FILE_INFO_FIELDS_FLAGS |
			GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE |
			GNOME_VFS_FILE_INFO_FIELDS_SIZE;
	} else { /* SHARE file */
		
		smb_info = g_private_get (get_info_key);
		if (!smb_info) {
			smb_info = g_new (file_info, 1);
			g_private_set (get_info_key, smb_info);
		}

		dos_filename = unix_filename_to_dos (path_remainder);
		
		if (cli_list (connection->cli, dos_filename,
			      aDIR | aSYSTEM | aHIDDEN,
			      get_info_helper) == -1) {
			smb_connection_unref (connection);
			g_free (dos_filename);
			g_free (path_remainder);
			UNLOCK_SAMBA();
			return GNOME_VFS_ERROR_NOT_FOUND;
		}
		
		smb_connection_unref (connection);
		
		g_free (dos_filename);
		
		vfs_file_info->name = g_strdup (g_basename (path_remainder));
		g_free (path_remainder);
		
		if (smb_info->mode & aDIR) {
			vfs_file_info->type = GNOME_VFS_FILE_TYPE_DIRECTORY;
			vfs_file_info->mime_type = g_strdup("x-directory/normal");
		} else {
			vfs_file_info->type = GNOME_VFS_FILE_TYPE_REGULAR;
			vfs_file_info->mime_type =
				g_strdup (gnome_vfs_mime_type_from_name_or_default (vfs_file_info->name,
										    GNOME_VFS_MIME_TYPE_UNKNOWN));
		}
		
		if (smb_info->mode & aRONLY)
			vfs_file_info->permissions =
				GNOME_VFS_PERM_USER_READ | GNOME_VFS_PERM_USER_EXEC |
				GNOME_VFS_PERM_GROUP_READ | GNOME_VFS_PERM_GROUP_EXEC |
				GNOME_VFS_PERM_OTHER_READ | GNOME_VFS_PERM_OTHER_EXEC;
		else
			vfs_file_info->permissions =
				GNOME_VFS_PERM_USER_ALL |
				GNOME_VFS_PERM_GROUP_ALL |
				GNOME_VFS_PERM_OTHER_ALL;
		
		vfs_file_info->size = smb_info->size;
		vfs_file_info->uid = smb_info->uid;
		vfs_file_info->gid = smb_info->gid;
		
		vfs_file_info->mtime = smb_info->mtime;
		vfs_file_info->atime = smb_info->atime;
		vfs_file_info->ctime = smb_info->ctime;
		
		vfs_file_info->flags = 0;
		
		vfs_file_info->valid_fields =
			GNOME_VFS_FILE_INFO_FIELDS_TYPE |
			GNOME_VFS_FILE_INFO_FIELDS_PERMISSIONS |
			GNOME_VFS_FILE_INFO_FIELDS_FLAGS |
			GNOME_VFS_FILE_INFO_FIELDS_SIZE |
			GNOME_VFS_FILE_INFO_FIELDS_ATIME |
			GNOME_VFS_FILE_INFO_FIELDS_MTIME |
			GNOME_VFS_FILE_INFO_FIELDS_CTIME |
			GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE;
	}

	UNLOCK_SAMBA();
	return GNOME_VFS_OK;
}

static gboolean
do_is_local (GnomeVFSMethod *method,
	     const GnomeVFSURI *uri)


{
	const char *server;
	const char *share;
	const char *tmp;
	const char *next_slash;

	server = gnome_vfs_uri_get_host_name (uri);

	/* Is this of the form smb:/// or smb:///workgroup
	 */
	if (server == NULL)
		return TRUE;

	share = gnome_vfs_uri_get_path (uri);
	
	/* Is this of the form smb://foo/ , where foo can be
	 * a server and/or a workgroup */ 
	if (share == NULL || share[0] == 0 ||
	    (share[0] == '/' && share[1] == 0))
		return TRUE;
	
	/* Is this a //workgroup/server/ ?
	 */
	tmp = share;
	if (*tmp == '/')
		tmp++;
	
	next_slash = strchr (tmp, '/');
	if (next_slash == NULL || next_slash[1] == 0) 
		return TRUE;

	return FALSE;
}

static void
open_dir_helper (file_info *info, const char *dir)
{
	SmbDirHandle *handle;
	SmbDirContent *content;
	GnomeVFSDirectoryFilterNeeds needs;
	GnomeVFSFileInfo *file_info;
	gboolean passed;

	handle = g_private_get (dir_key);

	if (handle->filter) {
		needs = gnome_vfs_directory_filter_get_needs (handle->filter);
		file_info = gnome_vfs_file_info_new ();
		
		if (needs & GNOME_VFS_DIRECTORY_FILTER_NEEDS_NAME) {
			file_info->name = g_strdup (info->name);
		}
			
		if (needs & GNOME_VFS_DIRECTORY_FILTER_NEEDS_TYPE) {
			if (content->mode & aDIR) 
				file_info->type = GNOME_VFS_FILE_TYPE_DIRECTORY;
			else
				file_info->type = GNOME_VFS_FILE_TYPE_REGULAR;
			file_info->valid_fields |=
				GNOME_VFS_FILE_INFO_FIELDS_TYPE;

		}
		
		if (needs & GNOME_VFS_DIRECTORY_FILTER_NEEDS_MIMETYPE) {
			if (content->mode & aDIR) 
				file_info->type = GNOME_VFS_FILE_TYPE_DIRECTORY;
			else
				file_info->type = GNOME_VFS_FILE_TYPE_REGULAR;
			file_info->valid_fields |=
				GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE;
		}
		
		if (needs & GNOME_VFS_DIRECTORY_FILTER_NEEDS_STAT) {
			if (info->mode & aRONLY)
				file_info->permissions =
					GNOME_VFS_PERM_USER_READ | GNOME_VFS_PERM_USER_EXEC |
					GNOME_VFS_PERM_GROUP_READ | GNOME_VFS_PERM_GROUP_EXEC |
					GNOME_VFS_PERM_OTHER_READ | GNOME_VFS_PERM_OTHER_EXEC;
			else
				file_info->permissions =
					GNOME_VFS_PERM_USER_ALL |
					GNOME_VFS_PERM_GROUP_ALL |
					GNOME_VFS_PERM_OTHER_ALL;
	
			file_info->size = info->size;
			file_info->uid = info->uid;
			info->gid = info->gid;
			
			file_info->mtime = info->mtime;
			file_info->atime = info->atime;
			file_info->ctime = info->ctime;
			
			file_info->flags = 0;
			
			file_info->valid_fields |=
				GNOME_VFS_FILE_INFO_FIELDS_PERMISSIONS |
				GNOME_VFS_FILE_INFO_FIELDS_FLAGS |
				GNOME_VFS_FILE_INFO_FIELDS_SIZE |
				GNOME_VFS_FILE_INFO_FIELDS_ATIME |
				GNOME_VFS_FILE_INFO_FIELDS_MTIME |
				GNOME_VFS_FILE_INFO_FIELDS_CTIME;
		}

		passed =
			gnome_vfs_directory_filter_apply (handle->filter,
							  file_info);
		gnome_vfs_file_info_unref (file_info);

		if (!passed)
			return;
	}
	
	content = g_new (SmbDirContent, 1);
	content->name = g_strdup (info->name);
	content->mtime = info->mtime;
	content->atime = info->atime;
	content->ctime = info->ctime;
	content->mode = info->mode;
	content->size = info->size;
	content->uid = info->uid;
	content->gid = info->gid;
	handle->contents = g_list_prepend (handle->contents, content);
}

static char *
dirname_to_mask (const char *dirname)
{
	char *mask;
	int dirname_len;
	int i;

	dirname_len = strlen (dirname);
	mask = g_malloc (dirname_len + 3);
	strcpy (mask, dirname);
	for (i = 0; i < dirname_len; i++)
		if (mask[i]=='/')
			mask[i] = '\\';

	if (mask[dirname_len-1] != '\\') {
		dirname_len++;
		mask[dirname_len-1] = '\\';
	}
	mask[dirname_len] = '*';
	mask[dirname_len+1] = 0;
	
	return mask;
}

static GnomeVFSResult
do_open_directory (GnomeVFSMethod *method,
		   GnomeVFSMethodHandle **method_handle,
		   GnomeVFSURI *uri,
		   GnomeVFSFileInfoOptions options,
		   const GnomeVFSDirectoryFilter *filter,
		   GnomeVFSContext *context)

{
	GnomeVFSResult res;
	SmbDirHandle *handle;
	SmbConnection *connection;
	SmbVirtualFile *file;
	char *path_remainder;
	char *mask;

	
	LOCK_SAMBA();
	DEBUG_SMB (("do_open_directory() %s\n",
		    gnome_vfs_uri_to_string (uri, 0)));
	
	res = lookup_uri (uri, &file, &path_remainder, &connection, TRUE);

	if (res) {
		UNLOCK_SAMBA ();
		return res;
	}

	if (file->virtual_type == SMB_VIRTUAL_TYPE_FILE) {
		UNLOCK_SAMBA ();
		return GNOME_VFS_ERROR_NOT_A_DIRECTORY;
	}

	if (file->virtual_type == SMB_VIRTUAL_TYPE_DIRECTORY) {
		handle = g_new (SmbDirHandle, 1);
		handle->is_files = FALSE;
		handle->contents = smb_virtual_file_list_shallow_copy (file->u.directory.dir_contents);
		handle->server = NULL;
		handle->filter = filter;
	} else {
		handle = g_new (SmbDirHandle, 1);
		handle->is_files = TRUE;
		handle->contents = NULL;
		handle->server = NULL;
		handle->filter = filter;
	
		g_private_set (dir_key, handle);

		mask = dirname_to_mask ((path_remainder)?path_remainder:"/");
		if (cli_list (connection->cli, mask,
			      aDIR | aSYSTEM | aHIDDEN,
			      open_dir_helper) == -1) {
			res = gnome_vfs_result_from_cli (connection->cli);
			smb_connection_unref (connection);
			g_free (mask);
			g_free (handle);
			g_free (path_remainder);
			g_private_set (dir_key, NULL);
			g_warning ("open_share_dir: cli_list returned -1, returning %d %s", res, gnome_vfs_result_to_string (res));
			UNLOCK_SAMBA();
			return res;
		}
		
		g_free (mask);
		g_free (path_remainder);
		g_private_set (dir_key, NULL);
		smb_connection_unref (connection);
	}
	
	UNLOCK_SAMBA();
	
	*method_handle = (GnomeVFSMethodHandle *)handle;
	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_close_directory (GnomeVFSMethod *method,
		    GnomeVFSMethodHandle *method_handle,
		    GnomeVFSContext *context)
{
	SmbDirHandle *handle = (SmbDirHandle *)method_handle;

	DEBUG_SMB (("do_close_directory() %p\n", handle));
	
	smb_dir_handle_destroy (handle);

	return GNOME_VFS_OK;
}


static GnomeVFSResult
share_read_dir (GnomeVFSMethod *method,
		GnomeVFSMethodHandle *method_handle,
		GnomeVFSFileInfo *file_info,
		GnomeVFSContext *context)
{
	SmbDirHandle *handle = (SmbDirHandle *)method_handle;
	SmbDirContent *content;
	GList *node;

	if (handle->contents) {
		node = handle->contents;
		content = (SmbDirContent *)node->data;
		handle->contents =
			g_list_remove_link (handle->contents, node);

		file_info->name = content->name;
		g_assert (file_info->name != NULL);

		if (content->mode & aDIR) {
			file_info->type = GNOME_VFS_FILE_TYPE_DIRECTORY;
			file_info->mime_type = g_strdup("x-directory/normal");
		} else {
			file_info->type = GNOME_VFS_FILE_TYPE_REGULAR;
			file_info->mime_type =
				g_strdup (gnome_vfs_mime_type_from_name_or_default (file_info->name,
										    GNOME_VFS_MIME_TYPE_UNKNOWN));
		}

		if (content->mode & aRONLY)
			file_info->permissions =
				GNOME_VFS_PERM_USER_READ | GNOME_VFS_PERM_USER_EXEC |
				GNOME_VFS_PERM_GROUP_READ | GNOME_VFS_PERM_GROUP_EXEC |
				GNOME_VFS_PERM_OTHER_READ | GNOME_VFS_PERM_OTHER_EXEC;
		else
			file_info->permissions =
				GNOME_VFS_PERM_USER_ALL |
				GNOME_VFS_PERM_GROUP_ALL |
				GNOME_VFS_PERM_OTHER_ALL;

		file_info->uid = content->uid;
		file_info->gid = content->gid;
		file_info->mtime = content->mtime;
		file_info->ctime = content->ctime;
		file_info->atime = content->atime;
		file_info->size = content->size;
		
		file_info->flags = GNOME_VFS_FILE_FLAGS_NONE;


		file_info->valid_fields |=
			GNOME_VFS_FILE_INFO_FIELDS_TYPE |
			GNOME_VFS_FILE_INFO_FIELDS_PERMISSIONS |
			GNOME_VFS_FILE_INFO_FIELDS_FLAGS |
			GNOME_VFS_FILE_INFO_FIELDS_SIZE |
			GNOME_VFS_FILE_INFO_FIELDS_ATIME |
			GNOME_VFS_FILE_INFO_FIELDS_MTIME |
			GNOME_VFS_FILE_INFO_FIELDS_CTIME |
			GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE;
		
		g_free (content);
		return GNOME_VFS_OK;
	} else {
		return GNOME_VFS_ERROR_EOF;
	}
}

static GnomeVFSResult
browse_read_dir (GnomeVFSMethod *method,
		GnomeVFSMethodHandle *method_handle,
		GnomeVFSFileInfo *file_info,
		GnomeVFSContext *context)
{
	SmbDirHandle *handle = (SmbDirHandle *)method_handle;
	SmbVirtualFile *file;
	GList *node;

	if (handle->contents) {
		node = handle->contents;
		file = (SmbVirtualFile *)node->data;
		handle->contents =
			g_list_remove_link (handle->contents, node);

		file_info->name = g_strdup (file->name);
		g_assert (file_info->name != NULL);
		file_info->permissions = file->permissions;
				
		file_info->flags = GNOME_VFS_FILE_FLAGS_NONE;
		file_info->valid_fields = GNOME_VFS_FILE_INFO_FIELDS_PERMISSIONS;
		
		if (file->virtual_type == SMB_VIRTUAL_TYPE_FILE) {
			file_info->type = GNOME_VFS_FILE_TYPE_REGULAR;
			file_info->mime_type = g_strdup(file->mime_type);
			file_info->size = file->u.file.size;
			file_info->valid_fields |= GNOME_VFS_FILE_INFO_FIELDS_SIZE;
		} else {
			/* It's a share or a directory */
			file_info->type = GNOME_VFS_FILE_TYPE_DIRECTORY;
			file_info->mime_type = g_strdup("x-directory/normal");
		} 

		file_info->valid_fields |=
			GNOME_VFS_FILE_INFO_FIELDS_TYPE |
			GNOME_VFS_FILE_INFO_FIELDS_FLAGS |
			GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE;

		smb_virtual_file_free (file);
		return GNOME_VFS_OK;
	} else {
		return GNOME_VFS_ERROR_EOF;
	}

}

static GnomeVFSResult
do_read_directory (GnomeVFSMethod *method,
		   GnomeVFSMethodHandle *method_handle,
		   GnomeVFSFileInfo *file_info,
		   GnomeVFSContext *context)
{
	SmbDirHandle *handle = (SmbDirHandle *)method_handle;
	DEBUG_SMB (("do_read_directory()\n"));
	if (handle->is_files) {
		return share_read_dir (method,
				       method_handle,
				       file_info,
				       context);
	} else {
		return browse_read_dir (method,
				       method_handle,
				       file_info,
				       context);
	}
}

static GnomeVFSResult
do_seek (GnomeVFSMethod *method,
	 GnomeVFSMethodHandle *method_handle,
	 GnomeVFSSeekPosition whence,
	 GnomeVFSFileOffset offset,
	 GnomeVFSContext *context)

{
	SmbHandle *handle = (SmbHandle *)method_handle;

	DEBUG_SMB (("do_seek() %d from %d\n", offset, whence));
	
	switch (whence) {
	case GNOME_VFS_SEEK_START:
		handle->offset = offset;
		break;
	case GNOME_VFS_SEEK_CURRENT:
		handle->offset += offset;
		break;
	case GNOME_VFS_SEEK_END:
		return GNOME_VFS_ERROR_NOT_SUPPORTED;
		break;
	}

	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_tell (GnomeVFSMethod *method,
	 GnomeVFSMethodHandle *method_handle,
	 GnomeVFSFileOffset *offset_return)


{
	SmbHandle *handle = (SmbHandle *)method_handle;

	*offset_return = handle->offset;

	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_unlink (GnomeVFSMethod *method,
	   GnomeVFSURI *uri,
	   GnomeVFSContext *context)
{
  	GnomeVFSResult res;
	SmbVirtualFile *file;
	SmbConnection *connection;
	char *path_remainder;
	char *dos_filename;
	file_info *smb_info;
	BOOL ret;

	LOCK_SAMBA();
	DEBUG_SMB (("do_unlink() %s\n",
		    gnome_vfs_uri_to_string (uri, 0)));
	
	res = lookup_uri (uri, &file, &path_remainder, &connection, TRUE);

	if (res) {
		UNLOCK_SAMBA ();
		return res;
	}

	if (file->virtual_type == SMB_VIRTUAL_TYPE_DIRECTORY) {
		UNLOCK_SAMBA ();
		return GNOME_VFS_ERROR_IS_DIRECTORY;
	}

	if (file->virtual_type == SMB_VIRTUAL_TYPE_FILE) {
		UNLOCK_SAMBA ();
		return GNOME_VFS_ERROR_ACCESS_DENIED;
	}

	dos_filename = unix_filename_to_dos ((path_remainder)?path_remainder:"/");
	ret = cli_unlink (connection->cli, dos_filename);
	g_free (path_remainder);
	g_free (dos_filename);
	if (!ret) {
		res = gnome_vfs_result_from_cli (connection->cli);
		smb_connection_unref (connection);
		UNLOCK_SAMBA();
		return res;
	}
		
	smb_connection_unref (connection);
	UNLOCK_SAMBA();
	
	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_check_same_fs (GnomeVFSMethod *method,
		  GnomeVFSURI *a,
		  GnomeVFSURI *b,
		  gboolean *same_fs_return,
		  GnomeVFSContext *context)
{
	char *server1;
	char *server2;
	char *path1;
	char *path2;
	char *p1, *p2;

	DEBUG_SMB (("do_check_same_fs()\n"));
	
	server1 =
	  gnome_vfs_unescape_string (gnome_vfs_uri_get_host_name (a),
				     NULL);
	server2 =
	  gnome_vfs_unescape_string (gnome_vfs_uri_get_host_name (b),
				     NULL);
	path1 =
	  gnome_vfs_unescape_string (gnome_vfs_uri_get_path (a), NULL);
	path2 =
	  gnome_vfs_unescape_string (gnome_vfs_uri_get_path (b), NULL);

	if (!server1 || !server2 || !path1 || !path2 ||
	    (strcmp (server1, server2) != 0)) {
		g_free (server1);
		g_free (server2);
		g_free (path1);
		g_free (path2);
		*same_fs_return = FALSE;
		return GNOME_VFS_OK;
	}

	p1 = path1;
	p2 = path2;
	if (*p1 == '/')
		p1++;
	if (*p2 == '/')
		p2++;

	/* Make sure both URIs are on the same share: */
	while (*p1 && *p2 && *p1 == *p2 && *p1 != '/') {
		p1++;
		p2++;
	}
	if (*p1 == 0 || *p2 == 0 || *p1 != *p2) {
		*same_fs_return = FALSE;
	} else {
		*same_fs_return = TRUE;
	}
	
	g_free (server1);
	g_free (server2);
	g_free (path1);
	g_free (path2);
	
	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_move (GnomeVFSMethod *method,
	 GnomeVFSURI *old_uri,
	 GnomeVFSURI *new_uri,
	 gboolean force_replace,
	 GnomeVFSContext *context)
{
  	GnomeVFSResult res;
	SmbConnection *connection;
	SmbVirtualFile *new_file, *old_file;
	char *filename1;
	char *filename2;
	char *dos_filename1;
	char *dos_filename2;
	file_info *smb_info;
	char *server1;
	char *server2;
	char *path1;
	char *path2;
	BOOL ret;

	LOCK_SAMBA();
	DEBUG_SMB (("do_move() %s %s\n",
		    gnome_vfs_uri_to_string (old_uri, 0),
		    gnome_vfs_uri_to_string (new_uri, 0)));


	res = lookup_uri (old_uri, &old_file, &filename1, &connection, FALSE);
	if (res) {
		UNLOCK_SAMBA ();
		return res;
	}
	
	if (old_file->virtual_type != SMB_VIRTUAL_TYPE_SHARE ||
	    filename1 == NULL) {
		UNLOCK_SAMBA ();
		return GNOME_VFS_ERROR_ACCESS_DENIED;
	}
	
	res = lookup_uri (new_uri, &new_file, &filename2, NULL, FALSE);
	if (res) {
		g_free (filename1);
		smb_connection_unref (connection);
		UNLOCK_SAMBA ();
		return res;
	}
	
	if (new_file != old_file || filename2 == NULL) {
		smb_connection_unref (connection);
		g_free (filename1);
		g_free (filename2);
		UNLOCK_SAMBA ();
		return GNOME_VFS_ERROR_ACCESS_DENIED;
	}

	dos_filename1 = unix_filename_to_dos (filename1);
	dos_filename2 = unix_filename_to_dos (filename2);
	ret = cli_rename (connection->cli, dos_filename1, dos_filename2);
	g_free (dos_filename1);
	g_free (dos_filename2);
	g_free (filename1);
	g_free (filename2);
	if (!ret) {
		res = gnome_vfs_result_from_cli (connection->cli);
		smb_connection_unref (connection);
		UNLOCK_SAMBA();
		return res;
	}
		
	smb_connection_unref (connection);
	UNLOCK_SAMBA();
	
	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_truncate_handle (GnomeVFSMethod *method,
		    GnomeVFSMethodHandle *method_handle,
		    GnomeVFSFileSize where,
		    GnomeVFSContext *context)

{
	g_warning ("do_truncate_handle\n");
	return GNOME_VFS_ERROR_NOT_SUPPORTED;
}

static GnomeVFSResult
do_get_file_info_from_handle (GnomeVFSMethod *method,
			      GnomeVFSMethodHandle *method_handle,
			      GnomeVFSFileInfo *file_info,
			      GnomeVFSFileInfoOptions options,
			      GnomeVFSContext *context)


{
	g_warning ("do_get_file_info_from_handle\n");
	return GNOME_VFS_ERROR_NOT_SUPPORTED;
}

static GnomeVFSResult
do_make_directory (GnomeVFSMethod *method,
		   GnomeVFSURI *uri,
		   guint perm,
		   GnomeVFSContext *context)
{
	GnomeVFSResult res;
	SmbConnection *connection;
	SmbVirtualFile *file;
	BOOL ret;
	char *filename;
	char *dos_filename;

	LOCK_SAMBA();
	
	DEBUG_SMB (("do_make_dir() %s\n",
		    gnome_vfs_uri_to_string (uri, 0)));
	

	res = lookup_uri (uri, &file, &filename, &connection, FALSE);
	if (res) {
		UNLOCK_SAMBA ();
		if (res == GNOME_VFS_ERROR_NOT_FOUND)
			res = GNOME_VFS_ERROR_ACCESS_DENIED;
		return res;
	}

	if (file->virtual_type != SMB_VIRTUAL_TYPE_SHARE ||
	    filename == NULL) {
		UNLOCK_SAMBA ();
		return GNOME_VFS_ERROR_ACCESS_DENIED;
	}

	dos_filename = unix_filename_to_dos (filename);
	ret = cli_mkdir (connection->cli, dos_filename);
	g_free (filename);
	g_free (dos_filename);
	if (!ret) {
		res = gnome_vfs_result_from_cli (connection->cli);
		smb_connection_unref (connection);
		UNLOCK_SAMBA();
		return res;
	}
	
	UNLOCK_SAMBA();

	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_remove_directory (GnomeVFSMethod *method,
		     GnomeVFSURI *uri,
		     GnomeVFSContext *context)
{
	GnomeVFSResult res;
	SmbConnection *connection;
	SmbVirtualFile *file;
	BOOL ret;
	char *filename;
	char *dos_filename;

	LOCK_SAMBA();
	
	DEBUG_SMB (("do_make_dir() %s\n",
		    gnome_vfs_uri_to_string (uri, 0)));
	
	res = lookup_uri (uri, &file, &filename, &connection, FALSE);
	if (res) {
		UNLOCK_SAMBA ();
		return res;
	}
	
	if (file->virtual_type != SMB_VIRTUAL_TYPE_SHARE ||
	    filename == NULL) {
		UNLOCK_SAMBA ();
		return GNOME_VFS_ERROR_ACCESS_DENIED;
	}

	dos_filename = unix_filename_to_dos (filename);
	ret = cli_rmdir (connection->cli, dos_filename);
	g_free (filename);
	g_free (dos_filename);
	if (!ret) {
		res = gnome_vfs_result_from_cli (connection->cli);
		smb_connection_unref (connection);
		UNLOCK_SAMBA();
		return res;
	}
	
	UNLOCK_SAMBA();

	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_set_file_info (GnomeVFSMethod *method,
		  GnomeVFSURI *uri,
		  const GnomeVFSFileInfo *info,
		  GnomeVFSSetFileInfoMask mask,
		  GnomeVFSContext *context)
{
  	GnomeVFSResult res;
	SmbConnection *connection;
	SmbVirtualFile *file;
	char *filename1;
	char *filename2;
	char *dos_filename1;
	char *dos_filename2;
	char *end;
	BOOL ret;

	DEBUG_SMB (("do_set_file_info mask: %d\n", mask));
	
	if (mask &
	    (GNOME_VFS_SET_FILE_INFO_PERMISSIONS |
	     GNOME_VFS_SET_FILE_INFO_OWNER |
	     GNOME_VFS_SET_FILE_INFO_TIME))
		return GNOME_VFS_ERROR_NOT_SUPPORTED;
	
	if (mask & GNOME_VFS_SET_FILE_INFO_NAME) {
		LOCK_SAMBA();
		DEBUG_SMB (("set_info: set new name: %s\n", info->name));
		
		res = lookup_uri (uri, &file, &filename1, &connection, FALSE);
		if (res) {
			UNLOCK_SAMBA ();
			return res;
		}

		if (file->virtual_type != SMB_VIRTUAL_TYPE_SHARE ||
		    filename1 == NULL) {
			UNLOCK_SAMBA ();
			return GNOME_VFS_ERROR_ACCESS_DENIED;
		}

		dos_filename1 = unix_filename_to_dos (filename1);
		filename2 = g_malloc (strlen (filename1) + strlen (info->name) + 1);

		strcpy (filename2, filename1);
		end = strrchr (filename2, '/');
		if (!end)
			strcpy (filename2, info->name);
		else
			strcpy (end+1, info->name);
		
		dos_filename2 = unix_filename_to_dos (filename2);
		DEBUG_SMB (("cli_rename:ing %s to %s\n", dos_filename1, dos_filename2));
		ret = cli_rename (connection->cli, dos_filename1, dos_filename2);
		g_free (filename1);
		g_free (filename2);
		g_free (dos_filename1);
		g_free (dos_filename2);
		if (!ret) {
			res = gnome_vfs_result_from_cli (connection->cli);
			smb_connection_unref (connection);
			UNLOCK_SAMBA();
			return res;
		}
		
		smb_connection_unref (connection);
		UNLOCK_SAMBA();
	}

	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_truncate (GnomeVFSMethod *method,
	     GnomeVFSURI *uri,
	     GnomeVFSFileSize where,
	     GnomeVFSContext *context)

{
	g_warning ("do_truncate\n");
	return GNOME_VFS_ERROR_NOT_SUPPORTED;
}

static GnomeVFSResult
do_find_directory (GnomeVFSMethod *method,
		   GnomeVFSURI *near_uri,
		   GnomeVFSFindDirectoryKind kind,
		   GnomeVFSURI **result_uri,
		   gboolean create_if_needed,
		   gboolean find_if_needed,
		   guint permissions,
		   GnomeVFSContext *context)
{
	g_warning ("do_find_directory\n");
	return GNOME_VFS_ERROR_NOT_SUPPORTED;
}

static GnomeVFSResult
do_create_symbolic_link (GnomeVFSMethod *method,
			 GnomeVFSURI *uri,
			 const char *target_reference,
			 GnomeVFSContext *context)
{
	g_warning ("do_create_symbolic_link\n");
	return GNOME_VFS_ERROR_NOT_SUPPORTED;
}


static GnomeVFSMethod method = {
	sizeof (GnomeVFSMethod),
	do_open,
	do_create,
	do_close,
	do_read,
	do_write,
	do_seek,
	do_tell,
	NULL, //do_truncate_handle,
	do_open_directory,
	do_close_directory,
	do_read_directory,
	do_get_file_info,
	NULL, //do_get_file_info_from_handle,
	do_is_local,
	do_make_directory,
	do_remove_directory,
	do_move,
	do_unlink,
	do_check_same_fs,
	do_set_file_info,
	NULL, //do_truncate,
	NULL, //do_find_directory,
	NULL //do_create_symbolic_link
};

GnomeVFSMethod *
vfs_module_init (const char *method_name, const char *args)
{
	TimeInit ();
	charset_initialise ();
	
	if (!lp_load ("/etc/samba/smb.conf", TRUE, FALSE, FALSE)) {
		DEBUG_SMB (("Couldn't load smb config file"));
	}

	server_auth_infos = g_hash_table_new (g_str_hash, g_str_equal);
	
	codepage_initialise (lp_client_code_page());

	load_interfaces ();

	get_myname((*global_myname)?NULL:global_myname);

	dir_key = g_private_new ((GDestroyNotify)smb_dir_handle_destroy);
	static_key = g_private_new (NULL);
	get_info_key = g_private_new ((GDestroyNotify)g_free);
	
	samba_lock = g_mutex_new();

	return &method;
}

void
vfs_module_shutdown (GnomeVFSMethod *method)
{
	DEBUG_SMB (("<-- smb module shutdown called -->\n"));
}

