/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "mozilla.h"
#include "dialog.h"
#include "glade.h"
#include "PromptService.h"
#include "themes.h"

#include "nsCOMPtr.h"
#include "nsIFactory.h"
#include "nsString.h"
#include "nsIServiceManager.h"
#include "nsXPComFactory.h"

#include "nsIPromptService.h"
#include "nsIUnicodeEncoder.h"

#include <gtk/gtkentry.h>
#include <gtk/gtktogglebutton.h>
#include <gtk/gtklist.h>
#include <libgnomeui/gnome-stock.h>
#include <libgnomeui/gnome-dialog.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-triggers.h>

/* local function prototypes */
static void set_title (GtkWidget *dialog, const PRUnichar *title);
static void set_label_text (GtkWidget *label, const PRUnichar *text);
static void set_check_button_text (GtkWidget *check_button, 
				   const PRUnichar *text);
static void set_check_button (GtkWidget *button, PRBool *value);
static void set_check_button_size_to_label (GtkWidget *check_button,
					    GtkWidget *label);
static void set_editable (GtkWidget *entry, PRUnichar **text);
static void get_check_button (GtkWidget *button, PRBool *value);
static void get_editable (GtkWidget *editable, PRUnichar **text);

/**
 * class CPromptService: an GNOME implementation of prompt dialogs for
 * Mozilla
 */
class CPromptService: public nsIPromptService
{
	public:
		CPromptService();
  		virtual ~CPromptService();

		NS_DECL_ISUPPORTS
		NS_DECL_NSIPROMPTSERVICE
  
};

NS_IMPL_ISUPPORTS1 (CPromptService, nsIPromptService)

/**
 * CPromptService::CPromptService: constructor
 */
CPromptService::CPromptService ()
{
	NS_INIT_REFCNT();
}

/**
 * CPromptService::~CPromptService: destructor
 */
CPromptService::~CPromptService ()
{
}

/**
 * CPromptService::Alert: show an alert box
 */
NS_IMETHODIMP CPromptService::Alert (nsIDOMWindow *parent, 
				     const PRUnichar *dialogTitle,
				     const PRUnichar *text)
{    
	GtkWidget *label;
	GtkWidget *dialog;
	GtkWidget *pixmap;
	GladeXML *gxml;
	PixmapData *icon =
		themes_get_pixmap ("gnome-warning.png", FALSE);

	/* build the dialog */
	gxml = glade_widget_new ("prompts.glade", "alert_dialog", 
				 &dialog, NULL);
	label = glade_xml_get_widget (gxml, "label");
	pixmap = glade_xml_get_widget (gxml, "pixmap");
	gtk_object_unref (GTK_OBJECT (gxml));

	/* set dynamic attributes */
	set_title (dialog, dialogTitle);
	set_label_text (label, text);
	gtk_pixmap_set (GTK_PIXMAP (pixmap), icon->pixmap, icon->mask);

	GtkWidget *parentWidget = mozilla_find_gtk_parent (parent);
	dialog_set_parent (dialog, parentWidget);

	/* make a suitable sound */
	gnome_triggers_vdo ("", "warning", NULL);

	/* run dialog and capture return values */
	gnome_dialog_run_and_close (GNOME_DIALOG (dialog));

	/* done */
	gtk_widget_destroy (dialog);
	return NS_OK;
}

/**
 * CPromptService::AlertCheck: show an alert box with a checkbutton,
 * (typically for things like "dont show this warning again")
 */
NS_IMETHODIMP CPromptService::AlertCheck (nsIDOMWindow *parent, 
					  const PRUnichar *dialogTitle,
					  const PRUnichar *text,
					  const PRUnichar *checkMsg, 
					  PRBool *checkValue)
{
	GtkWidget *label;
	GtkWidget *pixmap;
	GtkWidget *check_button;
	GtkWidget *dialog;
	GladeXML *gxml;
	PixmapData *icon =
		themes_get_pixmap ("gnome-warning.png", FALSE);

	/* build the dialog */
	gxml = glade_widget_new ("prompts.glade", "alert_check_dialog", 
				 &dialog, NULL);
	label = glade_xml_get_widget (gxml, "label");
	pixmap = glade_xml_get_widget (gxml, "pixmap");
	check_button = glade_xml_get_widget (gxml, "check_button");
	gtk_object_unref (GTK_OBJECT (gxml));

	/* set dynamic attributes */
	set_title (dialog, dialogTitle);
	set_label_text (label, text);
	gtk_pixmap_set (GTK_PIXMAP (pixmap), icon->pixmap, icon->mask);
	set_check_button_text (check_button, checkMsg);
	set_check_button (check_button, checkValue);
	set_check_button_size_to_label (check_button, label);

	GtkWidget *parentWidget = mozilla_find_gtk_parent (parent);
	dialog_set_parent (dialog, parentWidget);

	/* make a suitable sound */
	gnome_triggers_vdo ("", "warning", NULL);

	/* run dialog and capture return values */
	gnome_dialog_run_and_close (GNOME_DIALOG (dialog));
	get_check_button (check_button, checkValue);

	/* done */
	gtk_widget_destroy (dialog);
	return NS_OK;
}

/**
 * CPromptService::Confirm: for simple yes/no dialogs
 */
NS_IMETHODIMP CPromptService::Confirm (nsIDOMWindow *parent, 
				       const PRUnichar *dialogTitle,
				       const PRUnichar *text,
				       PRBool *_retval)
{
	GtkWidget *label;
	GtkWidget *pixmap;
	GtkWidget *dialog;
	GladeXML *gxml;
	PixmapData *icon =
		themes_get_pixmap ("gnome-question.png", FALSE);
	gint ret;

	/* build the dialog */
	gxml = glade_widget_new ("prompts.glade", "confirm_dialog", 
				 &dialog, NULL);
	label = glade_xml_get_widget (gxml, "label");
	pixmap = glade_xml_get_widget (gxml, "pixmap");
	gtk_object_unref (GTK_OBJECT (gxml));

	/* set dynamic attributes */
	set_title (dialog, dialogTitle);
	set_label_text (label, text);
	gtk_pixmap_set (GTK_PIXMAP (pixmap), icon->pixmap, icon->mask);

	GtkWidget *parentWidget = mozilla_find_gtk_parent (parent);
	dialog_set_parent (dialog, parentWidget);

	/* make a suitable sound */
	gnome_triggers_vdo ("", "generic", NULL);

	/* run dialog and capture return values */
	ret = gnome_dialog_run_and_close (GNOME_DIALOG (dialog));
	*_retval = (ret == 0);

	/* done */
	gtk_widget_destroy (dialog);
	return NS_OK;
}

/**
 * CPromptService::Confirm: for simple yes/no dialogs, with an additional
 * check button
 */
NS_IMETHODIMP CPromptService::ConfirmCheck (nsIDOMWindow *parent, 
					    const PRUnichar *dialogTitle,
					    const PRUnichar *text,
					    const PRUnichar *checkMsg, 
					    PRBool *checkValue,
					    PRBool *_retval)
{
	/* convert dynamic attributes */
	GtkWidget *parentWidget = mozilla_find_gtk_parent (parent);
	gchar *cdialogTitle = mozilla_unicode_to_locale (dialogTitle),
		*ctext = mozilla_unicode_to_locale (text),
		*ccheckMsg = mozilla_unicode_to_locale (checkMsg);

	/* run dialog and capture return values */
	*_retval = dialog_confirm_check (parentWidget, cdialogTitle, ctext,
					 ccheckMsg, checkValue);
	
	/* free converted strings */
	g_free (cdialogTitle);
	g_free (ctext);
	g_free (ccheckMsg);

	/* done */
	return NS_OK;
}

/**
 * CPromptService::ConfirmEx: For fancy confirmation dialogs
 */
NS_IMETHODIMP CPromptService::ConfirmEx (nsIDOMWindow *parent, 
					 const PRUnichar *dialogTitle,
					 const PRUnichar *text,
					 PRUint32 buttonFlags, 
					 const PRUnichar *button0Title,
					 const PRUnichar *button1Title,
					 const PRUnichar *button2Title,
					 const PRUnichar *checkMsg, 
					 PRBool *checkValue,
					 PRInt32 *buttonPressed)
{
	GtkWidget *dialog;
	GtkWidget *label;
	GtkWidget *pixmap;
	GtkWidget *check_button;
	GladeXML *gxml;
	PixmapData *icon =
		themes_get_pixmap ("gnome-question.png", FALSE);
	gint ret;
	gint i;

	/* build and show the dialog */
	gxml = glade_widget_new ("prompts.glade", "confirmex_dialog", 
				 &dialog, NULL);
	label = glade_xml_get_widget (gxml, "label");
	pixmap = glade_xml_get_widget (gxml, "pixmap");
	check_button = glade_xml_get_widget (gxml, "check_button");
	gtk_object_unref (GTK_OBJECT (gxml));
	
	/* set dynamic attributes */
	set_title (dialog, dialogTitle);
	set_label_text (label, text);
	gtk_pixmap_set (GTK_PIXMAP (pixmap), icon->pixmap, icon->mask);

	if (checkMsg)
	{
		set_check_button_text (check_button, checkMsg);
		set_check_button (check_button, checkValue);
		set_check_button_size_to_label (check_button, label);
	}
	else
	{
		gtk_widget_hide (GTK_WIDGET (check_button));
	}

	/* create buttons */
	for (i = 0; i < 4; i++)
	{
		switch (buttonFlags & 0xff)
		{
		case BUTTON_TITLE_OK:
			gnome_dialog_append_button (GNOME_DIALOG(dialog),
						    GNOME_STOCK_BUTTON_OK);
			break;
		case BUTTON_TITLE_CANCEL:
			gnome_dialog_append_button (GNOME_DIALOG(dialog),
						    GNOME_STOCK_BUTTON_CANCEL);
			break;
		case BUTTON_TITLE_YES:
			gnome_dialog_append_button (GNOME_DIALOG(dialog),
						    GNOME_STOCK_BUTTON_YES);
			break;
		case BUTTON_TITLE_NO:
			gnome_dialog_append_button (GNOME_DIALOG(dialog),
						    GNOME_STOCK_BUTTON_NO);
			break;
		case BUTTON_TITLE_SAVE:
			gnome_dialog_append_button (GNOME_DIALOG(dialog),
						    _("Save"));
			break;
		case BUTTON_TITLE_REVERT:
			gnome_dialog_append_button (GNOME_DIALOG(dialog),
						    _("Revert"));
			break;
		case BUTTON_TITLE_DONT_SAVE:
			gnome_dialog_append_button (GNOME_DIALOG(dialog),
						    _("Don't save"));
			break;
		case BUTTON_TITLE_IS_STRING:
			{
				gchar *tmpstr;
				if (i==0)
					tmpstr = mozilla_unicode_to_locale (
							button0Title);
				else if (i==1)
					tmpstr = mozilla_unicode_to_locale (
							button1Title);
				else if (i==2)
					tmpstr = mozilla_unicode_to_locale (
							button2Title);
				else
					break;
				gnome_dialog_append_button (
						GNOME_DIALOG(dialog), tmpstr);
				g_free (tmpstr);
			}
			break;
		default:
			break;
		}	

		buttonFlags >>= 8;
	}

	gnome_dialog_set_default (GNOME_DIALOG(dialog), 0);

	GtkWidget *parentWidget = mozilla_find_gtk_parent (parent);
	dialog_set_parent (dialog, parentWidget);

	/* make a suitable sound */
	gnome_triggers_vdo ("", "generic", NULL);

	/* run dialog and capture return values */
	ret = gnome_dialog_run_and_close (GNOME_DIALOG (dialog));

	/* handle return values */
	get_check_button (check_button, checkValue);
	*buttonPressed = ret;

	/* done */
	gtk_widget_destroy (dialog);
	return NS_OK;
}

/**
 * CPromptService::Prompt: show a prompt for text, with a checkbutton
 */
NS_IMETHODIMP CPromptService::Prompt (nsIDOMWindow *parent, 
				      const PRUnichar *dialogTitle,
				      const PRUnichar *text, 
				      PRUnichar **value,
				      const PRUnichar *checkMsg, 
				      PRBool *checkValue,
				      PRBool *_retval)
{
	GtkWidget *dialog;
	GtkWidget *entry;
	GtkWidget *label;
	GtkWidget *pixmap;
	GtkWidget *check_button;
	GladeXML *gxml;
	PixmapData *icon =
		themes_get_pixmap ("gnome-question.png", FALSE);
	gint ret;

	/* build and show the dialog */
	gxml = glade_widget_new ("prompts.glade", "prompt_dialog", 
				 &dialog, NULL);
	entry = glade_xml_get_widget (gxml, "entry");
	label = glade_xml_get_widget (gxml, "label");
	pixmap = glade_xml_get_widget (gxml, "pixmap");
	check_button = glade_xml_get_widget (gxml, "check_button");
	gtk_object_unref (GTK_OBJECT (gxml));

	/* set dynamic attributes */
	set_title (dialog, dialogTitle);
	set_label_text (label, text);
	gtk_pixmap_set (GTK_PIXMAP (pixmap), icon->pixmap, icon->mask);
	set_check_button_text (check_button, checkMsg);
	set_editable (entry, value);
	set_check_button (check_button, checkValue);
	set_check_button_size_to_label (check_button, label);

	/* make an enter keypress activate the "ok" button */
	gnome_dialog_editable_enters (GNOME_DIALOG (dialog), 
				      GTK_EDITABLE (entry));

	GtkWidget *parentWidget = mozilla_find_gtk_parent (parent);
	dialog_set_parent (dialog, parentWidget);

	/* make a suitable sound */
	/* NB: should be question, but this is missing in many
	 * of the current gnome sound packages that I've tried... */
	gnome_triggers_vdo ("", "generic", NULL);

	/* run dialog and capture return values */
	ret = gnome_dialog_run_and_close (GNOME_DIALOG (dialog));

	/* handle return values */
	get_check_button (check_button, checkValue);
	get_editable (entry, value);
	*_retval = (ret == 0);

	/* done */
	gtk_widget_destroy (dialog);
	return NS_OK;
}

/**
 * CPromptService::PromptUsernameAndPassword: show a prompt for username
 * and password with an additional check button.
 */
NS_IMETHODIMP CPromptService::PromptUsernameAndPassword 
                                        (nsIDOMWindow *parent, 
					 const PRUnichar *dialogTitle,
					 const PRUnichar *text,
					 PRUnichar **username,
					 PRUnichar **password,
					 const PRUnichar *checkMsg,
					 PRBool *checkValue,
					 PRBool *_retval)
{
	GtkWidget *dialog;
	GtkWidget *check_button;
	GtkWidget *label;
	GtkWidget *pixmap;
	GtkWidget *username_entry;
	GtkWidget *password_entry;
	GladeXML *gxml;
	PixmapData *icon =
		themes_get_pixmap ("gnome-question.png", FALSE);
	gint ret;

	/* build and show the dialog */
	gxml = glade_widget_new ("prompts.glade", "prompt_user_pass_dialog", 
				 &dialog, NULL);
	check_button = glade_xml_get_widget (gxml, "check_button");
	label = glade_xml_get_widget (gxml, "label");
	pixmap = glade_xml_get_widget (gxml, "pixmap");
	username_entry = glade_xml_get_widget (gxml, "username_entry");
	password_entry = glade_xml_get_widget (gxml, "password_entry");
	gtk_object_unref (GTK_OBJECT (gxml));

	/* set dynamic attributes */
	set_title (dialog, dialogTitle);
	set_check_button_text (check_button, checkMsg);
	set_editable (username_entry, username);
	set_editable (password_entry, password);
	set_check_button (check_button, checkValue);
	set_label_text (label, text);
	set_check_button_size_to_label (check_button, label);
	gtk_pixmap_set (GTK_PIXMAP (pixmap), icon->pixmap, icon->mask);

	/* make an enter keypress activate the "ok" button */
	gnome_dialog_editable_enters (GNOME_DIALOG (dialog),
				      GTK_EDITABLE (username_entry));
	gnome_dialog_editable_enters (GNOME_DIALOG (dialog),
				      GTK_EDITABLE (password_entry));

	GtkWidget *parentWidget = mozilla_find_gtk_parent (parent);
	dialog_set_parent (dialog, parentWidget);

	/* make a suitable sound */
	gnome_triggers_vdo ("", "question", NULL);

	/* run dialog and capture return values */
	ret = gnome_dialog_run_and_close (GNOME_DIALOG (dialog));

	/* handle return values */
	get_check_button (check_button, checkValue);
	get_editable (username_entry, username);
	get_editable (password_entry, password);
	*_retval = (ret == 0);

	/* done */
	gtk_widget_destroy (dialog);
	return NS_OK;
}

/**
 * CPromptService::PromptPassword: show a prompt for just
 * a password with an additional check button.
 */
NS_IMETHODIMP CPromptService::PromptPassword (nsIDOMWindow *parent, 
					      const PRUnichar *dialogTitle,
					      const PRUnichar *text,
					      PRUnichar **password,
					      const PRUnichar *checkMsg, 
					      PRBool *checkValue,
					      PRBool *_retval)
{
	GtkWidget *dialog;
	GtkWidget *label;
	GtkWidget *pixmap;
	GtkWidget *check_button;
	GtkWidget *password_entry;
	GladeXML *gxml;
	PixmapData *icon =
		themes_get_pixmap ("gnome-question.png", FALSE);
	gint ret;

	/* build and show the dialog */
	gxml = glade_widget_new ("prompts.glade", "prompt_pass_dialog", 
				 &dialog, NULL);
	check_button = glade_xml_get_widget (gxml, "check_button");
	label = glade_xml_get_widget (gxml, "label");
	pixmap = glade_xml_get_widget (gxml, "pixmap");
	password_entry = glade_xml_get_widget (gxml, "password_entry");
	gtk_object_unref (GTK_OBJECT (gxml));

	/* set dynamic attributes */
	set_title (dialog, dialogTitle);
	set_check_button_text (check_button, checkMsg);
	set_editable (password_entry, password);
	set_check_button (check_button, checkValue);
	set_label_text (label, text);
	set_check_button_size_to_label (check_button, label);
	gtk_pixmap_set (GTK_PIXMAP (pixmap), icon->pixmap, icon->mask);

	/* make an enter keypress activate the "ok" button */
	gnome_dialog_editable_enters (GNOME_DIALOG (dialog),
				      GTK_EDITABLE (password_entry));

	GtkWidget *parentWidget = mozilla_find_gtk_parent (parent);
	dialog_set_parent (dialog, parentWidget);

	/* make a suitable sound */
	gnome_triggers_vdo ("", "question", NULL);

	/* run dialog and capture return values */
	ret = gnome_dialog_run_and_close (GNOME_DIALOG (dialog));

	/* handle return values */
	get_check_button (check_button, checkValue);
	get_editable (password_entry, password);
	*_retval = (ret == 0);

	/* done */
	gtk_widget_destroy (dialog);
	return NS_OK;
}


/**
 * CPromptService::Select: FIXME
 */
NS_IMETHODIMP CPromptService::Select (nsIDOMWindow *parent, 
				      const PRUnichar *dialogTitle,
				      const PRUnichar *text, PRUint32 count,
				      const PRUnichar **selectList, 
				      PRInt32 *outSelection,
				      PRBool *_retval)
{
	GtkWidget *dialog;
	GtkWidget *list;
	GtkWidget *label;
	GtkWidget *pixmap;
	GladeXML *gxml;
	PixmapData *icon =
		themes_get_pixmap ("gnome-question.png", FALSE);
	gint ret;

	/* build and show the dialog */
	gxml = glade_widget_new ("prompts.glade", "select_dialog", 
				 &dialog, NULL);
	list   = glade_xml_get_widget (gxml, "list");
	pixmap = glade_xml_get_widget (gxml, "pixmap");
	label   = glade_xml_get_widget (gxml, "label");
	gtk_object_unref (GTK_OBJECT (gxml));

	/* set dynamic attributes */
	set_title (dialog, dialogTitle);
	set_label_text (label, text);
	gtk_pixmap_set (GTK_PIXMAP (pixmap), icon->pixmap, icon->mask);

	GList *gList = NULL;
	for (PRUint32 i = 0 ; i < count ; i++)
	{
		char *itemText = mozilla_unicode_to_locale (selectList[i]);
		GtkWidget *item = gtk_list_item_new_with_label (itemText);
		gList = g_list_append (gList, item);
		gtk_widget_show (item);
		g_free (itemText);
	}
	gtk_list_append_items (GTK_LIST(list), gList);
	gtk_list_select_item (GTK_LIST(list), 0);

	GtkWidget *parentWidget = mozilla_find_gtk_parent (parent);
	dialog_set_parent (dialog, parentWidget);

	/* make a suitable sound */
	gnome_triggers_vdo ("", "question", NULL);

	/* run dialog and capture return values */
	ret = gnome_dialog_run_and_close (GNOME_DIALOG (dialog));

	/* handle return values */
	if(g_list_length(GTK_LIST(list)->selection))
	{
		GtkWidget *selectedItem = GTK_WIDGET(GTK_LIST(list)->selection->data);
		*outSelection = gtk_list_child_position(GTK_LIST(list), selectedItem);
		*_retval = (ret == 0) ? PR_TRUE : PR_FALSE;
	}
	else
		*_retval = PR_FALSE;

	/* done */
	gtk_list_clear_items (GTK_LIST(list), 0 ,-1);
	gtk_widget_destroy (dialog);
	return NS_OK;
}

NS_DEF_FACTORY (CPromptService, CPromptService);

/**
 * NS_NewPromptServiceFactory:
 */ 
nsresult NS_NewPromptServiceFactory(nsIFactory** aFactory)
{
	NS_ENSURE_ARG_POINTER(aFactory);
	*aFactory = nsnull;

	nsCPromptServiceFactory *result = new nsCPromptServiceFactory;
	if (result == NULL)
	{
		return NS_ERROR_OUT_OF_MEMORY;
	}
    
	NS_ADDREF(result);
	*aFactory = result;
  
	return NS_OK;
}

/**
 * set_title: set a dialog title to a unicode string
 */
static void
set_title (GtkWidget *dialog, const PRUnichar *title)
{
	gchar *gtitle;

	/* convert the title */
	gtitle = mozilla_unicode_to_locale (title);

	/* set it */
	gtk_window_set_title (GTK_WINDOW (dialog), 
			      (gtitle == NULL ? N_("Galeon") : gtitle));

	/* and free */
	g_free (gtitle);
}

/**
 * set_label_text: set a labels text to a unicode string
 */
static void
set_label_text (GtkWidget *label, const PRUnichar *text)
{
	gchar *gtext;

	/* convert the title */
	gtext = mozilla_unicode_to_locale (text);

	/* set it */
	gtk_label_set_text (GTK_LABEL (label), gtext);

	/* and free */
	g_free (gtext);
}

/**
 * set_check_button_text: set a check buttons text to a unicode string
 */
static void
set_check_button_text (GtkWidget *check_button, const PRUnichar *text)
{
	gchar *gtext;

	/* convert the title */
	gtext = mozilla_unicode_to_locale (text);

	/* set it */
	gtk_label_set_text (GTK_LABEL (GTK_BIN (check_button)->child), gtext);

	/* and free */
	g_free (gtext);
}
	
/**
 * set_check_button: set a togglebutton to an initial state
 */
static void
set_check_button (GtkWidget *button, PRBool *value)
{
	/* check pointer is valid */
	if (value == NULL)
	{
		gtk_widget_hide (GTK_WIDGET (button));
		return;
	}

	/* set the value of the check button */
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button), *value);
}

/**
 * se_check_button_size_to_label: sync text widgets sizes
 */
static void
set_check_button_size_to_label (GtkWidget *check_button,
			        GtkWidget *label)
{
	GtkRequisition r, label_r;

	gtk_widget_size_request (check_button, &r);
	gtk_widget_size_request (label, &label_r);

	if (r.width <= label_r.width) return;
	
	gtk_widget_set_usize (label, r.width, 0);
}

/**
 * set_editable: set an editable to a unicode string
 */
static void
set_editable (GtkWidget *entry, PRUnichar **text)
{
	gchar *gtext;

	/* check there's something set */
	if (text == NULL || *text == NULL)
	{
		return;
	}

	/* convert the title */
	gtext = mozilla_unicode_to_locale (*text);

	/* set this string value in the widget */
	gtk_entry_set_text (GTK_ENTRY (entry), gtext);

	/* and free */
	g_free (gtext);
}

/**
 * get_check_button: get value of a toggle button and store it in a PRBool
 */
static void
get_check_button (GtkWidget *button, PRBool *value)
{
	/* check we can write */
	if (value == NULL)
	{
		return;
	}

	/* set the value from the check button */
	*value = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button));
}

/**
 * get_editable: get a string from an editable and store it as unicode
 */
static void
get_editable (GtkWidget *editable, PRUnichar **text)
{
	gchar *edited;

	/* check we can write */
	if (text == NULL)
	{
		return;
	}

	/* get the text */
	edited = gtk_editable_get_chars (GTK_EDITABLE (editable), 0, -1);

	/* decode and set it as the return value */
	*text = mozilla_locale_to_unicode (edited);

	/* free */
	g_free (edited);
}
