/*****************************************************************************\
  registry.cpp : Implimentation for the DeviceRegistry class

  Copyright (c) 1996 - 2001, Hewlett-Packard Co.
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:
  1. Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
  3. Neither the name of Hewlett-Packard nor the names of its
     contributors may be used to endorse or promote products derived
     from this software without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR IMPLIED
  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
  NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
  TO, PATENT INFRINGEMENT; PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
  OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
  ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
\*****************************************************************************/


// The purpose of this file is to facilitate addition and subtraction
// of supported devices from the system.

#include "header.h"

#ifdef APDK_DJ350
#include "dj6xx.h"
#include "dj600.h"
#include "dj350.h"
#endif
#ifdef APDK_DJ400
#include "dj400.h"
#endif
#if defined(APDK_DJ6xx) || defined(APDK_DJ6xxPhoto)
#include "dj6xx.h"
#include "dj660.h"
#include "dj690.h"
#endif
#ifdef APDK_DJ600
#include "dj6xx.h"
#include "dj600.h"
#endif
#ifdef APDK_DJ540
#include "dj6xx.h"
#include "dj540.h"
#endif
#ifdef APDK_DJ630
#include "dj630.h"
#endif
#ifdef APDK_DJ8xx
#include "dj8xx.h"
#endif
#ifdef APDK_DJ8x5
#include "dj8x5.h"
#endif
#ifdef APDK_DJ9xx
#include "dj9xx.h"
#endif
#if defined(APDK_DJ9xxVIP) || defined(APDK_PSP100)
#include "dj9xxvip.h"
#include "djgenericvip.h"
#endif
#ifdef APDK_PSP100
#include "psp100.h"
#endif
#if defined(APDK_APOLLO2XXX) || defined(APDK_APOLLO21XX) || defined(APDK_APOLLO2560)
#include "apollo2xxx.h"
#endif
#ifdef APDK_APOLLO21XX
#include "apollo21xx.h"
#endif
#ifdef APDK_APOLLO2560
#include "apollo2560.h"
#endif
#ifdef APDK_LJMONO
#include "ljmono.h"
#endif
#ifdef APDK_LJCOLOR
#include "ljcolor.h"
#endif
#ifdef APDK_PSCRIPT
#include "pscript.h"
#endif

APDK_BEGIN_NAMESPACE

extern PRINTER_TYPE DevIDMap[MAX_ID_STRING];
extern char *ModelString[MAX_ID_STRING];
extern PRINTER_TYPE BuiltIn[];
extern unsigned int BuiltInSize;


PRINTER_TYPE DevIDtoPT(unsigned int StringIndex)
{
    if (StringIndex >= MAX_ID_STRING)
        return UNSUPPORTED;

    return DevIDMap[StringIndex];
}


DeviceRegistry::DeviceRegistry()
    : device(UNSUPPORTED)
{
}


DeviceRegistry::~DeviceRegistry()
{
    DBG1("deleting DeviceRegistry\n");
}


DRIVER_ERROR DeviceRegistry::SelectDevice(const PRINTER_TYPE Model)
{
    if (Model > MAX_PRINTER_TYPE)
        return UNSUPPORTED_PRINTER;
    device = Model;

 return NO_ERROR;
}


DRIVER_ERROR DeviceRegistry::SelectDevice(char* model, int *pVIPVersion, char* pens, SystemServices* pSS)
// used by PrintContext constructor
// based on this 'model' string, we will search for the enum'd value
// and set this enum'd value in 'device'
{

#if defined(DEBUG) && (DBG_MASK & DBG_LVL1)
    printf("DR::SelectDevice: model= '%s'\n",model);
    printf("DR::SelectDevice: VIPver= %d\n",*pVIPVersion);
    printf("DR::SelectDevice: pens= '%s'\n",pens);
#endif

    unsigned int i=0; // counter for model type
    unsigned int j=0; // counter for string parsing
    char pen1 = '\0';   // black/color(for CCM)/photo(for 690) pen
    char pen2 = '\0';   // color/non-existent(for CCM) pen

    BOOL match=FALSE;

    DRIVER_ERROR err = NO_ERROR;

    while (!match && (i < MAX_ID_STRING))
    {
        if (! strncmp(model,ModelString[i],strlen(ModelString[i])) )
        {
            match=TRUE;
            device = DevIDtoPT(i);
        }
        else
        {
            i++;
        }
    }

    if (!match) // see if printer supports VIP, if so set compatible device
    {
//        if (*pVIPVersion == 1)
        if (*pVIPVersion > 0)
        {
            match = TRUE;
            device = eDJGenericVIP; // eDJ9xxVIP;
        }
    }

    if (!match)
    {
    // The devID model string did not have a match for a known printer
    // and the printer doesn't support VIP so let's look at the pen info for clues

        // if we don't have pen info (VSTATUS) it's presumably
        //  either sleek, DJ4xx or non-HP
        if ( pens[0] != '\0' )
        {
            // Venice (and Broadway?) printers return penID $X0$X0
            //  when powered off
            if(pens[1] == 'X')
            {
                DBG1("DR:(Unknown Model) Need to do a POWER ON to get penIDs\n");

                DWORD length=sizeof(Venice_Power_On);
                err = pSS->ToDevice(Venice_Power_On, &length);
                ERRCHECK;

                err = pSS->FlushIO();
                ERRCHECK;

                // give the printer some time to power up
                if (pSS->BusyWait((DWORD)1000) == JOB_CANCELED)
                return JOB_CANCELED;

                // we must re-query the devID
                err=GetPrinterModel(model,pVIPVersion,pens,pSS);
                ERRCHECK;
            }

            // Arggghh.  The pen(s) COULD be missing
            do
            {
                // get pen1 - penID format is $HB0$FC0
                pen1=pens[1];

                // get pen2 - if it exists
                j=2;
                BOOL NO_PEN2 = FALSE;
                while(pens[j] != '$')   // handles variable length penIDs
                {
                    j++;
                    if ( pens[j] == '\0' )
                    // never found a pen2
                    {
                        pen2 = '\0';
                        NO_PEN2 = TRUE;
                        break;
                    }
                }
                if (NO_PEN2 == FALSE)
                {
                    j++;
                    pen2 = pens[j];
                }

                if(pen1 == 'A' || pen2 == 'A')
                {
                    if(pen1 == 'A')
                    {
                        // 2-pen printer with both pens missing
                        if(pen2 == 'A')
                            pSS->DisplayPrinterStatus(DISPLAY_NO_PENS);

                        // 1-pen printer with missing pen
                        else if(pen2 == '\0')
                            pSS->DisplayPrinterStatus(DISPLAY_NO_PEN_DJ600);

                        // 2-pen printer with BLACK missing
                        else pSS->DisplayPrinterStatus(DISPLAY_NO_BLACK_PEN);
                    }
                    // 2-pen printer with COLOR missing
                    else if(pen2 == 'A')
                            pSS->DisplayPrinterStatus(DISPLAY_NO_COLOR_PEN);

                    if (pSS->BusyWait(500) == JOB_CANCELED)
                        return  JOB_CANCELED;

                    // we must re-query the devID
                    err=GetPrinterModel(model,pVIPVersion,pens,pSS);
                    ERRCHECK;
                }

            } while(pen1 == 'A' || pen2 == 'A');

            // now that we have pens to look at, let's do the logic
            //  to instantiate the 'best-fit' driver

            if(pen1 == 'H') // Hobbes (BLACK)
            {
                // check for a 850/855/870
                if(pen2 == 'M') device=UNSUPPORTED; // Monet (COLOR)

                else if (strncmp(model,"DESKJET 890",11) == 0)
                    device=UNSUPPORTED; // 890 has same pens as Venice!

                else if(pen2 == 'N') device=eDJ9xx; // Chinook (COLOR)

                // It must be a Venice derivative or will hopefully at
                // least recognize a Venice print mode
                else device=eDJ8xx;
            }
            else if(pen1 == 'C') // Candide (BLACK)
            {
                // check for 1-pen printer
                if(pen2 == '\0') device=eDJ600;
                // must be a 2-pen 6xx-derivative
                else device=eDJ6xx;
            }
            else if(pen1 == 'M') // Multi-dye load
            {
                // must be a 690-derivative
                device=eDJ6xxPhoto;
            }

            // check for 540-style pens?
            //  D = Kukla color, E = Triad black

            else device=UNSUPPORTED;
        }
    }


    // Early Venice printer do not yet have full bi-di so check
    // the model to avoid a communication problem.
    if ( ( (strncmp(model,ModelString[DEVID_MODEL_81X],strlen(ModelString[DEVID_MODEL_81X])) == 0)
        || (strncmp(model,ModelString[DEVID_MODEL_83X],strlen(ModelString[DEVID_MODEL_83X])) == 0)
        || (strncmp(model,ModelString[DEVID_MODEL_88X],strlen(ModelString[DEVID_MODEL_88X])) == 0)
        || (strncmp(model,ModelString[DEVID_MODEL_895],strlen(ModelString[DEVID_MODEL_895])) == 0)
         )
        && (pSS->IOMode.bUSB)
       )
    {
        DBG1("This printer has limited USB status\n");
        pSS->IOMode.bStatus = FALSE;
        pSS->IOMode.bDevID = FALSE;
    }

    if ( ( (strncmp(model,ModelString[DEVID_MODEL_63X],strlen(ModelString[DEVID_MODEL_63X])) == 0)
        || (strncmp(model,ModelString[DEVID_MODEL_64X],strlen(ModelString[DEVID_MODEL_64X])) == 0)
         )
        && (pSS->IOMode.bUSB)
       )
    {
        DBG1("This printer has limited USB status, but we did get DeviceIDString\n");
        pSS->IOMode.bStatus = FALSE;
    }

    if (device == UNSUPPORTED) return UNSUPPORTED_PRINTER;
    else return NO_ERROR;
} //SelectDevice


DRIVER_ERROR DeviceRegistry::InstantiatePrinter(Printer*& p, SystemServices* pSS)
// Instantiate a printer object and return a pointer p based on the previously
// set 'device' variable
{
    p=NULL;

DBG1("DR::InstantiateGeneral: device = ");

    switch(device) {

#ifdef APDK_DJ350
        case (eDJ350):
            DBG1("DJ350\n");
            p = new DJ350 (pSS);
            break;
#endif

#ifdef APDK_DJ400
        case(eDJ400):
                     DBG1("DJ400\n");
                     p=new DJ400(pSS);
                     break;
#endif

#ifdef APDK_DJ540
        case(eDJ540):
                     DBG1("DJ540\n");
                     p=new DJ540(pSS);
                     break;
#endif

#ifdef APDK_DJ600
        case(eDJ600):
                     DBG1("DJ600\n");
                     p=new DJ600(pSS);
                     break;
#endif

#ifdef APDK_DJ6xx
        case(eDJ6xx):
                     DBG1("DJ6xx\n");
                     p=new DJ660(pSS);
                     break;
#endif

#ifdef APDK_DJ6xxPhoto
        case(eDJ6xxPhoto):
                     DBG1("DJ6xxPhoto\n");
                     p=new DJ6xxPhoto(pSS);
                     break;
#endif

#ifdef APDK_DJ630
            case(eDJ630):
                     DBG1("DJ630\n");
                     p=new DJ630(pSS);
                     break;
#endif

#ifdef APDK_DJ8xx
        case(eDJ8xx):
                     DBG1("DJ8xx\n");
                     p=new DJ8xx(pSS);
                     break;
#endif

#ifdef APDK_DJ8x5
        case(eDJ8x5):
                     DBG1("DJ8x5\n");
                     p = new DJ8x5 (pSS);
                     break;
#endif

#ifdef APDK_DJ9xx
            case(eDJ9xx):
                     DBG1("DJ9xx\n");
                     p=new DJ9xx(pSS);
                     break;
#endif

#ifdef APDK_DJ9xxVIP
            case(eDJ9xxVIP):
                     DBG1("DJ9xxVIP\n");
                     p=new DJ9xxVIP(pSS);
                     break;
            case (eDJGenericVIP):
                     DBG1("DJGenericVIP\n");
                     p = new DJGenericVIP (pSS);
                     break;
#endif

#ifdef APDK_PSP100
            case (ePSP100):
                DBG1("PSP100\n");
                p = new PSP100 (pSS);
                break;
#endif

#ifdef APDK_APOLLO2XXX
            case(eAP2xxx):
                     DBG1("Apollo2xxx\n");
                     p=new Apollo2xxx(pSS);
                     break;
#endif
#ifdef APDK_APOLLO21XX
            case(eAP21xx):
                     DBG1("Apollo21xx\n");
                     p=new Apollo21xx(pSS);
                     break;
#endif
#ifdef APDK_APOLLO2560
            case(eAP2560):
                     DBG1("Apollo2560\n");
                     p=new Apollo2560(pSS);
                     break;
#endif
        default:
                     DBG1("UNSUPPORTED_PRINTER\n");
                     return UNSUPPORTED_PRINTER;
                     break;
    }

    NEWCHECK(p);

    return p->constructor_error;
} //InstantiatePrinter



DRIVER_ERROR DeviceRegistry::GetPrinterModel(char* strModel, int *pVIPVersion, char* strPens, SystemServices* pSS)
{
    DRIVER_ERROR err;
    BYTE DevIDBuffer[DevIDBuffSize];

    err = pSS->GetDeviceID(DevIDBuffer, DevIDBuffSize, TRUE);
    ERRCHECK;   // should be either NO_ERROR or BAD_DEVICE_ID

    return ParseDevIDString((const char*)DevIDBuffer, strModel, pVIPVersion, strPens);

} //GetPrinterModel

#define HEXTOINT(x, p) if (x >= '0' && x <= '9')      *p |= x - '0'; \
                       else if (x >= 'A' && x <= 'F') *p |= 0xA + x - 'A'; \
                       else if (x >= 'a' && x <= 'f') *p |= 0xA + x - 'a'


//ParseDevIDString
//! Parse a device id string
/*!
Enter a full description of the method here. This will be the API doc.

******************************************************************************/
DRIVER_ERROR DeviceRegistry::ParseDevIDString(const char* sDevID, char* strModel, int *pVIPVersion, char* strPens)
{
    int i;  // simple counter
    char* pStr = NULL;  // string pointer used in parsing DevID

    // get the model name
    // - note: I'm setting pStr to the return of strstr
    //   so I need to increment past my search string
    if ( (pStr = strstr(sDevID+2,"MODEL:")) )
        pStr+=6;
    else
        if ( (pStr=strstr(sDevID+2,"MDL:")) )
            pStr+=4;
        else return BAD_DEVICE_ID;

    // my own version of strtok to pull out the model string here
    i = 0;
    while ( (pStr[i] != ';') && (pStr[i] != '\0') && (i < DevIDBuffSize))
        strModel[i] = pStr[i++];
    strModel[i] = '\0';


    // see if this printer support VIP or not
    if ( (pStr=strstr(sDevID+2,";S:00")) )   // binary encoded device ID status (version 0)
    {
        pStr += 15;     // get to the VIP support field (version of 0 == doesn't support VIP)
        if ((*pStr >= '0') && (*pStr <= '9'))
        {
            *pVIPVersion = *pStr - '0';
        }
        else if ((*pStr >= 'A') && (*pStr <= 'F'))
        {
            *pVIPVersion = 10 + (*pStr - 'A');
        }
        else
        {
            *pVIPVersion = 0;
        }
    }

/*
 *  DevID string has changed starting with Jupiter.
 *  Following ";S:", two nibbles for Version Number
 *  12 nibbles for Status Information, the last nibble
 *  is reserved for future use, the second from last
 *  indicates whether the printer has VIP support.
 *
 *  Actually, four nibbles were added.
 *  The first fourteen nibbles contain feature state info.
 *  So, starting with version 02 of device id, following ":S:" there are
 *   2 nibbles for version number
 *  14 nibbles for feature state  (the 15th nibble from ';' is the vip flag
 *   2 nibbles for printer status
 */

    else if ((pStr = strstr (sDevID+2, ";S:")))
    {
        *pVIPVersion = 0;
        HEXTOINT (*(pStr+3), pVIPVersion);
        *pVIPVersion = *pVIPVersion << 4;
        HEXTOINT (*(pStr+4), pVIPVersion);
        if (*(pStr + 15) == '1')
        {
            (*pVIPVersion)++;
        }
        else
        {
            *pVIPVersion = 0;
        }
    }

    else
    {
        *pVIPVersion = 0;
    }

    // now get the pen info
    if( (pStr=strstr(sDevID+2,"VSTATUS:")) )
    {
        pStr+=8;
        i=0;
        while ( (pStr[i] != ',') && (pStr[i] != ';') && (pStr[i] != '\0') )
            strPens[i] = pStr[i++];
        strPens[i] = '\0';
    }
    else if ( (pStr = strstr(sDevID + 2, ";S:00")) ||   // binary encoded device ID status (version 0)
             (pStr = strstr (sDevID + 2, ";S:")))  // Jupiter and later style
    {

        int     iVersion = 0;
        HEXTOINT (*(pStr+3), &iVersion);
        iVersion = iVersion << 4;
        HEXTOINT (*(pStr+4), &iVersion);
        if (iVersion < 2)
        {
            pStr += 19;     // get to the number of pens field
        }
        else
        {
            pStr += 21;
        }

        // each supported pen has a block of 8 bytes of info so copy the number of pens byte
        // plus 8 bytes for each supported ped
        if ((*pStr >= '0') && (*pStr <= '9'))
        {
            i = 1 + ((*pStr-'0')*8);
        }
        else if ((*pStr >= 'A') && (*pStr <= 'F'))
        {
            i = 1 + ((10 + (*pStr-'A')) * 8);
        }
        else
        {   // bogus number of pens field
            i = 1;
        }
        memcpy(strPens, pStr, i);
        strPens[i] = '\0';
    }
    else   // no VSTATUS for 400 and sleek printers
        strPens[0] = '\0';

    return NO_ERROR;
} //ParseDevIDString


PRINTER_TYPE DeviceRegistry::EnumDevices(unsigned int& currIdx) const
// Returns next model; UNSUPPORTED when finished
{
    PRINTER_TYPE pt;

    if (currIdx >=  BuiltInSize)
        return UNSUPPORTED;

    pt = BuiltIn[currIdx];

    currIdx++;

    return pt;
} //EnumDevices

APDK_END_NAMESPACE

