/*
 *   Copyright (c) International Business Machines  Corp., 2000
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   COMPONENT_NAME: jfs utilities
 *
 *   FUNCTIONS:
 *               fsck_send_msg
 *               fsck_record_msg
 *               fsck_send_debug_msg
 *
 */

/*
 * defines and includes common among the xfsck modules
 */
#include "xfsckint.h"

/*
 * local includes
 */
#include "xchkdsk.h"
#include "fsckwsp.h"
#include "fsckmsgc.h"

/* + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + +
 *
 * fsck aggregate info structure pointer
 *
 *      defined in xchkdsk.c
 */
extern struct fsck_agg_record *agg_recptr;

/* + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + +
 *
 * For message processing
 */
#define STDOUT_HANDLE  1
#define STDERR_HANDLE  2

/*
 *      defined in xchkdsk.c
 */
extern int16_t  MsgProtocol[][2];

extern char *Vol_Label;

extern char *verbose_msg_ptr;

extern char *MsgText[];

extern char *msgprms[];
extern int16_t  msgprmidx[];

/* + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + +
 *
 * For query response processing
 *
 *      defined in xchkdsk.c
 */
extern char  *msgs_txt_ptr;
extern unsigned long  msgs_txt_maxlen;

/* VVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVV
 *
 * The following are internal to this file
 *
 */
void fsck_record_msg ( int, int );

void fsck_send_debug_msg ( int, int );

/* VVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVV */

/*****************************************************************************
 * NAME: fsck_record_msg
 *
 * FUNCTION:
 *
 * PARAMETERS:
 *      ?                 - input -
 *      ?                 - returned -
 *
 * NOTES:
 *
 * RETURNS:
 *	nothing
 */
void fsck_record_msg( int  mess_num, int  num_parms )
{
  int                prmidx;
  char               log_entry[512];
  int                entry_length = sizeof(fscklog_entry_hdr_t);
  logent_hdptr       hdptr;
  logent_prmtype     prmtype;
  logent_prmval      prmval;
  logent_prmstr_ptr  prmlit;
  int32_t            buffer_bytes_left;
  char               *buf_entry_ptr;

  if ( (!agg_recptr->fscklog_full) &&
       (agg_recptr->fscklog_buf_allocated) &&
       (!agg_recptr->fscklog_buf_alloc_err)   ) {  /* logging is active */

    hdptr = (logent_hdptr) log_entry;
    hdptr->msg_num = mess_num;
    hdptr->num_parms = num_parms;

    if ( num_parms != 0 ) {
      prmtype = (int8_t *) &(log_entry[entry_length]);
      entry_length += num_parms * sizeof(int8_t);
      prmval = (int16_t *) &(prmtype[num_parms]);
      entry_length += num_parms * sizeof(int16_t);
      prmlit = (char *) &(log_entry[entry_length]);
      prmidx = 0;
      while ( prmidx < num_parms ) {
        if ( msgprmidx[prmidx] != 0 ) {  /* variable text insert */
          prmtype[prmidx] = fscklog_var_text;
          prmval[prmidx] = msgprmidx[prmidx];
        } else {  /* string literal insert */
          prmtype[prmidx] = fscklog_literal;
          prmval[prmidx] = strlen(msgprms[prmidx]);
          entry_length += prmval[prmidx];
          /*
           * If the string is longer than the space allocated for 
           * log_entry, simply return.  The string in the log file 
           * will be truncated, but not the string printed to stdout.
           */ 
          if (entry_length >= 511) return;
          strncpy( prmlit, msgprms[prmidx], prmval[prmidx] );
          /*
           * add a null terminator to the literal
           */
          log_entry[entry_length] = '\0';
          prmval[prmidx] += 1;
          entry_length += 1;
          /*
           * set up for the next literal
           */
          prmlit = (char *) &(log_entry[entry_length]);
        }  /* end string literal insert */

        prmidx += 1;
      }
    }

    entry_length = ((entry_length + 3) / 4 ) * 4;  /*
                          * pad to the next 4 byte boundary
                          */
    hdptr->entry_length = entry_length;
    buffer_bytes_left = agg_recptr->fscklog_buf_length -
                        agg_recptr->fscklog_buf_data_len;
    if ( buffer_bytes_left < entry_length ) {
      agg_recptr->fscklog_last_msghdr->entry_length += buffer_bytes_left;
      fscklog_put_buffer();
      memset( (void *) (agg_recptr->fscklog_buf_ptr),
              0,
              agg_recptr->fscklog_buf_length);  /*
                        * clear the buffer
                        */
    }

    if ( ! agg_recptr->fscklog_full ) {
      buf_entry_ptr = (char *) ( (char *) agg_recptr->fscklog_buf_ptr +
                                          agg_recptr->fscklog_buf_data_len );

      /* swap if on big endian machine */
      ujfs_swap_fscklog_entry_hdr_t( hdptr );

      memcpy( (void *) buf_entry_ptr,
              (void *) hdptr,
              entry_length );

      agg_recptr->fscklog_last_msghdr =
               (struct fscklog_entry_hdr *) buf_entry_ptr;
      agg_recptr->fscklog_buf_data_len += entry_length;
    }
  }  /* end logging is active */

  return;
}  /* end fsck_record_msg() */


/*****************************************************************************
 * NAME: fsck_send_debug_msg
 *
 * FUNCTION: Issue an fsck message to stdout in English.
 *
 * PARAMETERS:
 *      ?                 - input -
 *      ?                 - returned -
 *
 * NOTES:    none
 *
 * RETURNS:  none
 */
void fsck_send_debug_msg( int  msg_num, int  num_inserts )
{
  int  prmidx;

  /*
   * if the messaging level is debug, issue the message in English
   */
  memset( (void *) verbose_msg_ptr, '\0', msgs_txt_maxlen );
  memset( (void *) msgs_txt_ptr, '\0', msgs_txt_maxlen );
  prmidx = 0;
  while ( prmidx < num_inserts ) {
    if ( msgprmidx[prmidx] != 0 ) {  /* variable text insert */
      msgprms[prmidx] = MsgText[ msgprmidx[prmidx] ];
    }  /* end variable text insert */
    prmidx += 1;
  }
  sprintf( msgs_txt_ptr, MsgText[ msg_num ],
           msgprms[0], msgprms[1], msgprms[2], msgprms[3], msgprms[4],
           msgprms[5], msgprms[6], msgprms[7], msgprms[8], msgprms[9]  );
  sprintf( verbose_msg_ptr, "%s%s", "(chklog) ", msgs_txt_ptr );

  printf("%s\n", verbose_msg_ptr );

  return;
}  /* end fsck_send_debug_msg() */


/*****************************************************************************
 * NAME: fsck_send_msg
 *
 * FUNCTION: Issue an fsck message, depending on the message's protocol
 *           according to the fsck message arrays.
 *
 * PARAMETERS:
 *      ?                 - input -
 *      ?                 - returned -
 *
 * NOTES:  none
 *
 * RETURNS:
 *	nothing
 */
void fsck_send_msg( int  msg_num,
                    int  num_inserts )
{
  int    prmidx;
  unsigned long  mess_number = 0;
  int    msgfile_idx = OSO_MSG;
  int    displayed_flag = 0;

  if ( MsgProtocol[msg_num][MP_MSGFILE] == jfs_msgfile ) {
    msgfile_idx = JFS_MSG;
    mess_number = msg_num + fsck_msgid_offset;
  }

  /*
   *     - effective messaging level is >= the level of the given message
   * or  - this is an autocheck message and processing in autocheck mode
   */
  if ( agg_recptr->effective_msg_level >= MsgProtocol[msg_num][MP_MSGLVL] ) {
    /*
     * if the message has a local language translation,
     * assemble it and issue it in the local language.
     */
    if ( MsgProtocol[msg_num][MP_MSGFILE] != no_msgfile ) {  /* translate */
      memset( (void *) msgs_txt_ptr, '\0', msgs_txt_maxlen );
      prmidx = 0;
      /*
       * translated insert text ALWAYS comes from the JFS message file
       */
      while ( prmidx < num_inserts ) {
        if ( msgprmidx[prmidx] != 0 ) {  /* variable text insert */

          strcpy(msgprms[prmidx],MsgText[msgprmidx[prmidx]]);
          msgprms[prmidx][strlen(msgprms[prmidx]) - 2] = '\0';
        }  /* end variable text insert */
        prmidx += 1;
      }

      message_user( mess_number, msgprms, num_inserts, msgfile_idx );
      displayed_flag = -1;

    }  /* end translate */
  }

  /*
   * Log the message
   */
  fsck_record_msg( msg_num, num_inserts );

  /*
   * Issue the message if appropriate
   */

  if ( ( agg_recptr->effective_msg_level == fsck_debug ) && (!displayed_flag) ) {  /* messaging debug */
    fsck_send_debug_msg( msg_num, num_inserts );
  }  /* end messaging debug */

  return;
}  /* end fsck_send_msg() */
