/*
 * xlog - GTK+ logging program for amateur radio operators
 * Copyright (C) 2001-2002 Joop Stakenborg <pa4tu@amsat.org>
 *
 * This program is free oftware; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * remote.c - remote data support
 */

/*
 * Messages send to xlog should consist of fields, separated by the '\1' character.
 * Fields are started with a descriptor, followed by a colon (':'). Possible fields are:
 *
 ****************************************************************************
 * program:<name of program which sends the QSO information>
 * version:<version string for the message queue, must be '1' for now>
 * date:<date of QSO, preferable 'dd mmm yyyy'>
 * time:<start time of QSO, preferably in GMT >
 * call:<callsign of your contact (will be converted to uppercase)>
 * mhz:<frequency in MHz>
 * mode:<any one of CW,SSB,RTTY,PSK31,etc (will be converted to uppercase)>
 * tx:<report (RST) which you have send>
 * rx:<report (RST) which you have received>
 * name:<name of the operator you have contacted>
 * qth:<town of the operator you have contacted>
 * notes:<additional notes>
 ****************************************************************************
 *
 * See sendtoxlog.c, included with the sources for examples.
 */

#include <stdlib.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <errno.h>
#include <sys/msg.h>
#include <gtk/gtk.h>
#include "remote.h"
#include "types.h"
#include "log.h"
#include "utils.h"
#include "support.h"

msgtype msgbuf;
remotetype remote;
extern statetype state;
extern preferencestype preferences;
extern glong msgid;
extern GtkWidget *mainwindow;
extern GtkWidget *mainnotebook;
extern gchar **qso;
extern gchar **bandsplit;
extern gchar **modesplit;
extern GList *logwindowlist;
 
/* find colon in a string and return the right part */
static gchar *getargument(gchar *remotestring) {
	gchar **split, *found = NULL;

	split = g_strsplit(remotestring, ":", 2);
	if (split[1]) found = g_strdup(split[1]);
	g_strfreev(split);
	return(g_strdup(found));
}

static void addtolog(gint type, gchar *entry) {
	GtkWidget *menu, *bandoptionmenu, *modeoptionmenu, *active_item,
		*bandentry, *modeentry;
	gchar *temp, **remoteinfo, *argument = NULL, *remarks;
	gint i, j = 0, newrow, bandindex, modeindex;
	logtype *logwindow;

	/* add QSO to the log */
	if (type == 88)
	{
		logwindow =  g_list_nth_data(logwindowlist, 
			gtk_notebook_get_current_page(GTK_NOTEBOOK(mainnotebook)));
		/* empty qso array */
		for (i = 0; i < 11; i++) qso[i] = g_strdup("");
		remarks = g_strdup("");
		bandentry = lookup_widget(mainwindow, "bandentry");
		bandoptionmenu = lookup_widget(mainwindow, "bandoptionmenu");
		modeentry = lookup_widget(mainwindow, "modeentry");
		modeoptionmenu = lookup_widget(mainwindow, "modeoptionmenu");
		remoteinfo = g_strsplit(entry, "\1", 0);
		for (;;)
		{
			if (remoteinfo[j] == NULL) break;
			if (g_strncasecmp(remoteinfo[j], "version:", 8) == 0)
			{
				if ((argument = getargument(remoteinfo[j])))
					remote.version = atoi(argument);
			}
			else if (g_strncasecmp(remoteinfo[j], "program:", 8) == 0)
			{
				if ((argument = getargument(remoteinfo[j])))
					remote.program = g_strdup(argument);
			}
			else if (g_strncasecmp(remoteinfo[j], "date:", 5) == 0)
			{
				if ((argument = getargument(remoteinfo[j])))
					qso[DATE] = g_strndup(argument, 15);
				else qso[DATE] = getdate();
			}
			else if (g_strncasecmp(remoteinfo[j], "time:", 5) == 0)
			{
				if ((argument = getargument(remoteinfo[j])))
					qso[GMT] = g_strndup(argument, 15);
				else qso[GMT] = gettime();
			}
			else if (g_strncasecmp(remoteinfo[j], "call:", 5) == 0)
			{
				if ((argument = getargument(remoteinfo[j]))) {
					g_strup(argument);
					qso[CALL] = g_strndup(argument, 15);
				}
			}
			else if (g_strncasecmp(remoteinfo[j], "mhz:", 4) == 0)
			{
				if ((argument = getargument(remoteinfo[j])))
					qso[BAND] = g_strndup(argument, 15);
			}
			else if (g_strncasecmp(remoteinfo[j], "mode:", 5) == 0)
			{
				if ((argument = getargument(remoteinfo[j]))) {
					g_strup(argument);
					qso[MODE] = g_strndup(argument, 15);
				}
			}
			else if (g_strncasecmp(remoteinfo[j], "tx:", 3) == 0)
			{
				if ((argument = getargument(remoteinfo[j])))
					qso[RST] = g_strndup(argument, 15);
			}
			else if (g_strncasecmp(remoteinfo[j], "rx:", 3) == 0)
			{
				if ((argument = getargument(remoteinfo[j])))
					qso[MYRST] = g_strndup(argument, 15);
			}
			/* endtime, name, QTH, notes, power and locator will all go onto the remarks field, */
			/* seperated by a ',' */
			else if (g_strncasecmp(remoteinfo[j], "endtime:", 8) == 0)
			{
				if ((argument = getargument(remoteinfo[j])))
				{
					if (g_strcasecmp(remarks, "") == 0)
						remarks = g_strdup(argument);
					else
						remarks = g_strconcat(remarks, ", ", argument, 0);
				}
			}
			else if (g_strncasecmp(remoteinfo[j], "name:", 5) == 0)
			{
				if ((argument = getargument(remoteinfo[j])))
				{
					if (g_strcasecmp(remarks, "") == 0)
						remarks = g_strdup(argument);
					else
						remarks = g_strconcat(remarks, ", ", argument, 0);
				}
			}
			else if (g_strncasecmp(remoteinfo[j], "qth:", 4) == 0)
			{
				if ((argument = getargument(remoteinfo[j])))
				{
					if (g_strcasecmp(remarks, "") == 0)
						remarks = g_strdup(argument);
					else
						remarks = g_strconcat(remarks, ", ", argument, 0);
				}
			}
			else if (g_strncasecmp(remoteinfo[j], "notes:", 6) == 0)
			{
				if ((argument = getargument(remoteinfo[j])))
				{
					if (g_strcasecmp(remarks, "") == 0)
						remarks = g_strdup(argument);
					else
						remarks = g_strconcat(remarks, ", ", argument, 0);
				}
			}
			else if (g_strncasecmp(remoteinfo[j], "power:", 6) == 0)
			{
				if ((argument = getargument(remoteinfo[j])))
				{
					if (g_strcasecmp(remarks, "") == 0)
						remarks = g_strdup(argument);
					else
						remarks = g_strconcat(remarks, ", ", argument, 0);
				}
			}
			else if (g_strncasecmp(remoteinfo[j], "locator:", 8) == 0)
			{
				if ((argument = getargument(remoteinfo[j])))
				{
					if (g_strcasecmp(remarks, "") == 0)
						remarks = g_strdup(argument);
					else
						remarks = g_strconcat(remarks, ", ", argument, 0);
				}
			}
			j++;
		}
		g_strfreev(remoteinfo);
		g_free(argument);

		/* Total length of the remarks field can be 80 characters */
		if (g_strcasecmp(remarks, "") != 0)
			qso[REMARKS] = g_strndup(remarks, 80);
		g_free(remarks);
		
		/* some checks: if date, gmt, band and mode is unknown we get it from xlog */
		if (g_strcasecmp(qso[DATE], "") == 0) qso[DATE] = getdate();
		if (g_strcasecmp(qso[GMT], "") == 0) qso[GMT] = gettime();
		if (g_strcasecmp(qso[BAND], "") == 0)
		{
			if (preferences.bandseditbox == 0)
			{
				menu = GTK_OPTION_MENU(bandoptionmenu)->menu;
				active_item = gtk_menu_get_active(GTK_MENU(menu));
				bandindex = g_list_index(GTK_MENU_SHELL(menu)->children, active_item);
				qso[BAND] = lookup_band(bandsplit[bandindex]);
			}
			else qso[BAND] = gtk_editable_get_chars(GTK_EDITABLE(bandentry), 0, -1);
		}
		if (g_strcasecmp(qso[MODE], "") == 0)
		{
			if (preferences.modeseditbox == 0)
			{
				menu = GTK_OPTION_MENU(modeoptionmenu)->menu;
				active_item = gtk_menu_get_active(GTK_MENU(menu));
				modeindex = g_list_index(GTK_MENU_SHELL(menu)->children, active_item);
				qso[MODE] = lookup_mode(modesplit[modeindex]);
			}
			else qso[MODE] = gtk_editable_get_chars(GTK_EDITABLE(modeentry), 0, -1);
		}

		/* add the new QSO to the log */
		qso[NR] = g_strdup_printf("%d", ++logwindow->qsos);
		newrow = gtk_clist_prepend(GTK_CLIST(logwindow->clist), qso);
		if (preferences.logcolor == 1) setcallstyle(logwindow->clist, newrow, preferences.themecolor);

		gtk_label_set_text(GTK_LABEL(logwindow->label), g_strconcat(logwindow->logname, " *", NULL));

		/* get new QSO number and update the statusbar */
		temp = g_strdup_printf(_("Remote data received from %s (#%d), QSO %s added"), 
			remote.program, ++remote.nr, qso[0]);
		update_statusbar(temp);
		g_free(temp);

		/* set some variables */
		remote.program = g_strdup("unknown");
		remote.version = 0;
		logwindow->logchanged = TRUE;
	}
}

gint remote_entry(void) {
	glong msgtyp = 0;
	ssize_t status = -1;

	status = msgrcv(msgid, (void *)&msgbuf, 1024, msgtyp, MSG_NOERROR|IPC_NOWAIT);
	if (status != -1) addtolog(msgbuf.mtype, msgbuf.mtext);
	return 1;
}
