/**********************************************************************
zyGrib: meteorological GRIB file viewer
Copyright (C) 2008 - Jacques Zaninetti - http://www.zygrib.org

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
***********************************************************************/

#include <iostream>
#include <cassert>

#include <QApplication>
#include <QMouseEvent>
#include <QPaintEvent>
#include <QPainter>

#include "MapDrawer.h"
#include "GribPlot.h"
#include "IacPlot.h"
#include "LonLatGrid.h"

//===========================================================
MapDrawer::MapDrawer(GshhsReader *gshhsReader)
	: QObject()
{    
    imgEarth = NULL;
    imgAll   = NULL;
    
    gisReader = new GisReader();
    gisReaderIsNew = true;
    
    assert(gisReader);
	this->gshhsReader = gshhsReader;
	initGraphicsParameters();
}
//---------------------------------------------------------------------
MapDrawer::MapDrawer(const MapDrawer &model)
	: QObject()
{
    imgEarth = NULL;
    imgAll   = NULL;
	gisReader = model.gisReader;
    gisReaderIsNew = false;
	this->gshhsReader = new GshhsReader(*model.gshhsReader);

	initGraphicsParameters();
	updateGraphicsParameters();
}
//---------------------------------------------------------------------
MapDrawer::~MapDrawer()
{
	if (gisReaderIsNew) {
		delete gisReader;
	}
}	
    
//===========================================================
void MapDrawer::initGraphicsParameters()
{    
    showCountriesBorders  = Util::getSetting("showCountriesBorders", true).toBool();
    showRivers   = Util::getSetting("showRivers", false).toBool();
    showCitiesNamesLevel = Util::getSetting("showCitiesNamesLevel", 0).toInt();
    showCountriesNames = Util::getSetting("showCountriesNames", false).toBool();
    showLonLatGrid   = Util::getSetting("showLonLatGrid", true).toBool();
    
    showIsobars  = Util::getSetting("showIsobars", true).toBool();
    showIsobarsLabels = Util::getSetting("showIsobarsLabels", false).toBool();
    isobarsStep = Util::getSetting("isobarsStep", 2).toDouble();
    showPressureMinMax = Util::getSetting("showPressureMinMax", false).toBool();
    
    showIsotherms0  = Util::getSetting("showIsotherms0", false).toBool();
    showIsotherms0Labels  = Util::getSetting("showIsotherms0Labels", false).toBool();
    isotherms0Step = Util::getSetting("isotherms0Step", 50).toDouble();
    
    colorMapMode = Util::getSetting("colorMapMode", MapDrawer::drawWind).toInt();
    
    colorMapSmooth = Util::getSetting("colorMapSmooth", true).toBool();
    showWindArrows  = Util::getSetting("showWindArrows", true).toBool();
    showBarbules = Util::getSetting("showBarbules", true).toBool();
    
    showTemperatureLabels = Util::getSetting("showTemperatureLabels", false).toBool();
    showGribGrid = Util::getSetting("showGribGrid", false).toBool();

}

//-------------------------------------------
void MapDrawer::updateGraphicsParameters()
{            
    backgroundColor  = Util::getSetting("backgroundColor", QColor(0,0,45)).value<QColor>();
    seaColor  = Util::getSetting("seaColor", QColor(50,50,150)).value<QColor>();
    landColor = Util::getSetting("landColor", QColor(200,200,120)).value<QColor>();
    
    seaBordersPen.setColor(Util::getSetting("seaBordersLineColor", QColor(40,45,30)).value<QColor>());
    seaBordersPen.setWidthF(Util::getSetting("seaBordersLineWidth", 1.8).toDouble());
    
    boundariesPen.setColor(Util::getSetting("boundariesLineColor", QColor(40,40,40)).value<QColor>());
    boundariesPen.setWidthF(Util::getSetting("boundariesLineWidth", 1.4).toDouble());
    
    riversPen.setColor(Util::getSetting("riversLineColor", QColor(50,50,150)).value<QColor>());
    riversPen.setWidthF(Util::getSetting("riversLineWidth", 1.0).toDouble());
    
    isobarsPen.setColor(Util::getSetting("isobarsLineColor", QColor(80,80,80)).value<QColor>());
    isobarsPen.setWidthF(Util::getSetting("isobarsLineWidth", 2.0).toDouble());
    
    isotherms0Pen.setColor(Util::getSetting("isotherms0LineColor", QColor(200,120,100)).value<QColor>());
    isotherms0Pen.setWidthF(Util::getSetting("isotherms0LineWidth", 1.6).toDouble());
}

//---------------------------------------------------------------------
void MapDrawer::draw_Map_Background(bool isEarthMapValid, Projection *proj)
{
	if (imgAll != NULL) {
		delete imgAll;
		imgAll = NULL;
	}
	imgAll = new QPixmap(proj->getW(), proj->getH());
	assert(imgAll);
	
	QPainter pnt(imgAll);
	pnt.setRenderHint(QPainter::Antialiasing, true);

	if (!isEarthMapValid)
	{
		if (imgEarth != NULL) {
			delete imgEarth;
			imgEarth = NULL;
		}

		imgEarth = new QPixmap(proj->getW(), proj->getH());
		assert(imgEarth);

		if (gshhsReader != NULL)
		{
			QPainter pnt1(imgEarth);
			pnt1.setRenderHint(QPainter::Antialiasing, false);
			gshhsReader->drawBackground(pnt1, proj, seaColor, backgroundColor);
			gshhsReader->drawContinents(pnt1, proj, seaColor, landColor);
		}
	}
	pnt.drawPixmap(0,0, *imgEarth);
}
//----------------------------------------------------------------------
void MapDrawer::draw_Map_Foreground(QPainter &pnt, Projection *proj)
{		
	if (gshhsReader != NULL)
	{
		pnt.setPen(seaBordersPen);
		gshhsReader->drawSeaBorders(pnt, proj);

		if (showCountriesBorders) {
			pnt.setPen(boundariesPen);
			gshhsReader->drawBoundaries(pnt, proj);
		}
		if (showRivers) {
			pnt.setPen(riversPen);
			gshhsReader->drawRivers(pnt, proj);
		}
	}
	if (showLonLatGrid) {
		LonLatGrid gr;
		gr.drawLonLatGrid(pnt, proj);
	}
	if (showCountriesNames) {
		gisReader->drawCountriesNames(pnt, proj);
	}
	if (showCitiesNamesLevel > 0) {
		gisReader->drawCitiesNames(pnt, proj, showCitiesNamesLevel);
	}
}		

//=======================================================================
// GSHHS Map
//=======================================================================
void MapDrawer::draw_GSHHS (
			QPainter &pntGlobal,
			bool mustRedraw, bool isEarthMapValid,
			Projection *proj
	)
{
    if (mustRedraw  ||  !isEarthMapValid)
    {
		//===================================================
		// Dessin du fond de carte
		//===================================================
		draw_Map_Background(isEarthMapValid, proj);
		QPainter pnt(imgAll);
		pnt.setRenderHint(QPainter::Antialiasing, true);
		//===================================================
		// Dessin des bordures et frontières
		//===================================================
		draw_Map_Foreground(pnt, proj);
    }
    // Recopie l'image complète
    pntGlobal.drawPixmap(0,0, *imgAll);
}

//=======================================================================
// IAC Files
//=======================================================================
void MapDrawer::draw_GSHHSandIAC (
			QPainter &pntGlobal,
			bool mustRedraw, bool isEarthMapValid,
			Projection *proj,
			IacPlot    *iacPlot
	)
{
    if (mustRedraw  ||  !isEarthMapValid)
    {
		//===================================================
		// Dessin du fond de carte
		//===================================================
		draw_Map_Background(isEarthMapValid, proj);
		QPainter pnt(imgAll);
		pnt.setRenderHint(QPainter::Antialiasing, true);
		//===================================================
		// Dessin des bordures et frontières
		//===================================================
		draw_Map_Foreground(pnt, proj);
		//===================================================
		// Dessin des données Meteo
		//===================================================
		draw_MeteoData_IAC (pnt, proj, iacPlot);
		
		//===================================================
		// Cartouche
		//===================================================
		draw_Cartouche_IAC(pnt, proj, iacPlot);
    }
    // Recopie l'image complète
    pntGlobal.drawPixmap(0,0, *imgAll);
}
//=======================================================================
// GRIB1 Files
//=======================================================================
void MapDrawer::draw_GSHHSandGRIB1 (
			QPainter &pntGlobal,
			bool mustRedraw, bool isEarthMapValid,
			Projection *proj,
			GribPlot   *gribPlot
	)
{
    if (mustRedraw  ||  !isEarthMapValid)
    {
		//===================================================
		// Dessin du fond de carte
		//===================================================
		draw_Map_Background(isEarthMapValid, proj);
		
		//===================================================
		// Dessin des données Meteo
		//===================================================
		QPainter pnt(imgAll);
		pnt.setRenderHint(QPainter::Antialiasing, true);
		draw_MeteoData_GRIB1 (pnt, proj, gribPlot);
		
		//===================================================
		// Dessin des bordures et frontières
		//===================================================
		draw_Map_Foreground(pnt, proj);

		//===================================================
		// Cartouche
		//===================================================
		draw_Cartouche_GRIB1(pnt, proj, gribPlot);
    }
    // Recopie l'image complète
    pntGlobal.drawPixmap(0,0, *imgAll);
}

//===================================================================
// Draw IAC data
//===================================================================
void MapDrawer::draw_MeteoData_IAC(QPainter &pnt, Projection *proj, IacPlot *iacPlot)
{
bool iac_showPressureMinMax = true;
bool iac_showPressureTroughLine = true;
bool iac_showIsobars = true;
bool iac_showIsobarsLabels = true;
bool iac_showFronts = true;
    

	iacPlot->show_CoverZone(pnt, proj);

	if (iac_showFronts)
	{
		iacPlot->draw_FRONT_all (pnt, proj);
	}
	if (iac_showPressureTroughLine)
	{
    	QPen pen;
		pen.setColor(Util::getSetting("IAC_TroughLinesColor", QColor(60,120,60)).value<QColor>());
		pen.setWidthF(Util::getSetting("IAC_TroughLinesWidth", 2.0).toDouble());
		pen.setStyle(Qt::DashLine);
		pnt.setPen(pen);
		iacPlot->draw_PRESSURE_TroughLines (pnt, proj);
	}
	if (iac_showIsobars)
	{
    	QPen pen;
		pen.setColor(Util::getSetting("IAC_IsobarsColor", QColor(50,20,20)).value<QColor>());
		pen.setWidthF(Util::getSetting("IAC_IsobarsWidth", 2.0).toDouble());
		pnt.setPen(pen);
		iacPlot->draw_PRESSURE_Isobars (pnt, proj);
	}
	if (iac_showIsobarsLabels)
	{
		iacPlot->draw_PRESSURE_IsobarsLabels (pnt, proj);
	}
	if (iac_showPressureMinMax) {
		iacPlot->draw_PRESSURE_MinMax (pnt, proj);
	}
}

//===================================================================
// Draw GRIB data
//===================================================================
void MapDrawer::draw_MeteoData_GRIB1(QPainter &pnt, Projection *proj, GribPlot *gribPlot)
{
	gribPlot->show_CoverZone(pnt, proj);
	
	//QTime t1 = QTime::currentTime();
	switch (colorMapMode)
	{
		case MapDrawer::drawWind :
			windArrowsColor.setRgb(255, 255, 255);
			gribPlot->draw_WIND_Color(pnt, proj, colorMapSmooth);
			break;
		case MapDrawer::drawRain :
			windArrowsColor.setRgb(140, 120, 100);
			gribPlot->draw_RAIN_Color(pnt, proj, colorMapSmooth);
			break;
		case MapDrawer::drawCloud :
			windArrowsColor.setRgb(180, 180, 80);
			gribPlot->draw_CLOUD_Color(pnt, proj, colorMapSmooth);
			break;
		case MapDrawer::drawHumid :
			windArrowsColor.setRgb(180, 180, 80);
			gribPlot->draw_HUMID_Color(pnt, proj, colorMapSmooth);
			break;
		case MapDrawer::drawTemp :
			windArrowsColor.setRgb(255, 255, 255);
			gribPlot->draw_Temp_Color(pnt, proj, colorMapSmooth);
			break;
		case MapDrawer::drawTempPot :
			windArrowsColor.setRgb(255, 255, 255);
			gribPlot->draw_TempPot_Color(pnt, proj, colorMapSmooth);
			break;
		case MapDrawer::drawDewpoint :
			windArrowsColor.setRgb(255, 255, 255);
			gribPlot->draw_Dewpoint_Color(pnt, proj, colorMapSmooth);
			break;
		case MapDrawer::drawDeltaDewpoint :
			windArrowsColor.setRgb(180, 180, 80);
			gribPlot->draw_DeltaDewpoint_Color(pnt, proj, colorMapSmooth);
			break;
		case MapDrawer::drawSnowDepth :
			windArrowsColor.setRgb(140, 120, 100);
			gribPlot->draw_SNOW_DEPTH_Color(pnt, proj, colorMapSmooth);
			break;
		case MapDrawer::drawSnowCateg :
			windArrowsColor.setRgb(140, 120, 100);
			gribPlot->draw_SNOW_CATEG_Color(pnt, proj, colorMapSmooth);
			break;
		case MapDrawer::drawFrzRainCateg :
			windArrowsColor.setRgb(140, 120, 100);
			gribPlot->draw_FRZRAIN_CATEG_Color(pnt, proj, colorMapSmooth);
			break;
		case MapDrawer::drawCAPEsfc :
			windArrowsColor.setRgb(100, 80, 80);
			gribPlot->draw_CAPEsfc(pnt, proj, colorMapSmooth);
			break;
	}
	//printf("time show ColorMap = %d ms\n", t1.elapsed());

	if (showIsobars) {
		pnt.setPen(isobarsPen);
		gribPlot->draw_Isobars(pnt, proj);
	}

	if (showIsotherms0) {
		pnt.setPen(isotherms0Pen);
		gribPlot->draw_Isotherms0(pnt, proj);
	}

	if (showWindArrows) {
		gribPlot->draw_WIND_Arrows(pnt, proj, showBarbules, windArrowsColor);
	}

	if (showIsobarsLabels) {
		gribPlot->draw_IsobarsLabels(pnt, proj);
	}
	if (showIsotherms0Labels) {
		gribPlot->draw_Isotherms0Labels(pnt, proj);
	}
	if (showPressureMinMax) {
		gribPlot->draw_PRESSURE_MinMax (pnt, proj);
	}
	if (showTemperatureLabels) {
		gribPlot->draw_TEMPERATURE_Labels (pnt, proj);
	}

	//===================================================
	// Grille GRIB
	//===================================================
	if (showGribGrid) {
		gribPlot->draw_GribGrid(pnt, proj);
	}
}
//-------------------------------------------------------------
// Cartouche : dates de la prévision courante + infos générales
void MapDrawer::draw_Cartouche_IAC(QPainter &pnt, const Projection *proj, IacPlot *iacPlot)
{
	IacReader *iacReader = iacPlot->getIacReader();
	if (iacReader!=NULL && iacReader->isOk())
	{
        int dy, w, h, x, y;
		dy = 11;
        QFont fontsmall("TypeWriter", dy-1, QFont::Bold, false);
			fontsmall.setStyleHint(QFont::TypeWriter);
			fontsmall.setStretch(QFont::Condensed);
        QFont fontsign("TypeWriter", dy-1, QFont::Normal, false);
			fontsign.setStyleHint(QFont::TypeWriter);
			fontsign.setStretch(QFont::Condensed);
        QFont fontbig("TypeWriter", dy+1, QFont::Bold, false);
			fontbig.setStyleHint(QFont::TypeWriter);
			fontbig.setStretch(QFont::Condensed);
		QColor   transpcolor(255,255,255,120);
		QColor   textcolor(20,20,20,255);
        pnt.setBrush(transpcolor);
		//---------------------------------------------------------
		// forecast validity date (top left corner)
		int n = 2;
		QString lig1= "";
        QString lig2= "";
        IacReader::IacFileType type = iacReader->getIacFileType();
        if (type == IacReader::IAC_ANALYSE)
        {
			lig1 = tr("IAC fleetcode : Analyse");
			lig2 = lig2.sprintf("Ref: %02d-%02dh%02d UTC",
						iacReader->getIacFileDay(),
						iacReader->getIacFileHour(),
						iacReader->getIacFileMinute() );
        }
        else if (type == IacReader::IAC_FORECAST)
        {
			lig1 = tr("IAC fleetcode : Prévision");
			lig2 = lig2.sprintf("Ref: %02d-%02dh%02d UTC - Val:%+dh",
						iacReader->getIacFileDay(),
						iacReader->getIacFileHour(),
						iacReader->getIacFileMinute(),
						iacReader->getIacFileValidHour() );
        }
        pnt.setFont(fontbig);
        w = 210;
        h = n*dy+8;
        x = 0;
        y = 0;
        pnt.setPen(transpcolor);
        pnt.drawRect(x,y,w,h);
        pnt.setPen(textcolor);
        pnt.drawText(x+3, y+dy+2, lig1);// forecast validity date
		if (lig2 != "") {
        	pnt.drawText(x+3, y+2*dy+7, lig2);	// type of color map
		}

		//--------------------------------------------------
        // Signature zyGrib (coin bas droite)
        pnt.setFont(fontsign);
        w = 44;
        h = dy+2;
        x = proj->getW() - w;
        y = proj->getH() - h;
        pnt.setPen(transpcolor);
        pnt.drawRect(x,y,w,h);
        pnt.setPen(textcolor);
        pnt.drawText(x+3, y+dy, tr("zyGrib"));
	}
}
//-------------------------------------------------------------
// Cartouche : dates de la prévision courante + infos générales
void MapDrawer::draw_Cartouche_GRIB1(QPainter &pnt, const Projection *proj, GribPlot *gribPlot)
{
	GribReader *gribReader = gribPlot->getGribReader();
    if (gribReader == NULL) {
        return;
    }
    
    if (gribReader->getNumberOfDates() > 0)
    {
        QString  tref = "Ref: "+Util::formatDateTimeLong(gribReader->getRefDate());
        QString  tval = "Val: "+Util::formatDateTimeLong(gribPlot->getCurrentDate());
        int n = 1;
        QString data="";
		switch (colorMapMode) {
			case MapDrawer::drawWind:
				data = tr("Vent à 10 m");
				break;
			case MapDrawer::drawCloud :
				data = tr("Couverture nuageuse");
				break;
			case MapDrawer::drawRain :
				data = tr("Précipitations");
				break;
			case MapDrawer::drawHumid :
				data = tr("Humidité relative");
				break;
			case MapDrawer::drawTemp :
				data = tr("Température à 2 m");
				break;
			case MapDrawer::drawDewpoint :
				data = tr("Point de rosée à 2 m");
				break;
			case MapDrawer::drawDeltaDewpoint :
				data = tr("Ecart temp-pt de rosée");
				break;
		 
			case MapDrawer::drawTempMin :
				data = tr("Temp. Min à 2 m");
				break;
			case MapDrawer::drawTempMax :
				data = tr("Temp. Max à 2 m");
				break;
			case MapDrawer::drawTempPot :
				data = tr("Temp. Potentielle");
				break;
		 
			case MapDrawer::drawSnowDepth :
				data = tr("Neige (épaisseur)");
				break;
			case MapDrawer::drawSnowCateg :
				data = tr("Neige (chute possible)");
				break;
			case MapDrawer::drawCAPEsfc :
				data = tr("CAPE (surface)");
				break;
		}
		if (data != "") {
			n ++;
		}
        int dy, w, h, x, y;
		
		dy = 11;
        QFont fontsmall("TypeWriter", dy-1, QFont::Bold, false);
			fontsmall.setStyleHint(QFont::TypeWriter);
			fontsmall.setStretch(QFont::Condensed);
        QFont fontsign("TypeWriter", dy-1, QFont::Normal, false);
			fontsign.setStyleHint(QFont::TypeWriter);
			fontsign.setStretch(QFont::Condensed);
        QFont fontbig("TypeWriter", dy+1, QFont::Bold, false);
			fontbig.setStyleHint(QFont::TypeWriter);
			fontbig.setStretch(QFont::Condensed);
		QColor   transpcolor(255,255,255,120);
		QColor   textcolor(20,20,20,255);
        pnt.setBrush(transpcolor);
		//---------------------------------------------------------
		// forecast validity date (top left corner) + data type
        QString  tval2 = Util::formatDateTimeLong(gribPlot->getCurrentDate());
        pnt.setFont(fontbig);
        w = 190;
        h = n*dy+8;
        x = 0;
        y = 0;
        pnt.setPen(transpcolor);
        pnt.drawRect(x,y,w,h);
        pnt.setPen(textcolor);
        pnt.drawText(x+3, y+dy+2, tval2);// forecast validity date
		if (data != "") {
        	pnt.drawText(x+3, y+2*dy+7, data);	// type of color map
		}
		//--------------------------------------------------
        // Date de référence (bas gauche)
        pnt.setFont(fontsmall);
        w = 190;
        h = dy+2;
        x = 0;
        y = proj->getH() - h;
        pnt.setPen(transpcolor);
        pnt.drawRect(x,y,w,h);
        pnt.setPen(textcolor);
        pnt.drawText(x+3, y+dy, tref);
		//--------------------------------------------------
        // Signature zyGrib (coin bas droite)
        pnt.setFont(fontsign);
        w = 44;
        h = dy+2;
        x = proj->getW() - w;
        y = proj->getH() - h;
        pnt.setPen(transpcolor);
        pnt.drawRect(x,y,w,h);
        pnt.setPen(textcolor);
        pnt.drawText(x+3, y+dy, tr("zyGrib"));
	}
}




