############################################################################
##
## Copyright (c) 2000, 2001, 2002, 2003, 2004, 2005 BalaBit IT Ltd, Budapest, Hungary
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; if not, write to the Free Software
## Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
##
##
## $Id: Session.py,v 1.58 2003/12/15 15:47:07 bazsi Exp $
##
## Author  : Bazsi
## Auditor : kisza
## Last audoted version: 1.9
## Notes:
##
############################################################################

"""Module defining session related classes and functions. 

This module defines the abstract session interface in a class named
'AbstractSession', and two descendants 'MasterSession' and 'StackedSession'.

Sessions are hierarchically stacked into each other just like proxies.
Each session has an owner session (except for the master session which
is on the top), and variables are "inherited" from owner sessions. 
(implemented using a simple getattr wrapper) This way stacked sessions
can inherit data from encapsulating proxies. (an HTTP proxy may define an
URL and a mime-type, and stacked CVP module may inspect those values)
"""

import Zorp
from Zorp import *
from Zone import root_zone
from Cache import ShiftCache

inbound_cache = ShiftCache('inbound_cache', 1000)
outbound_cache = ShiftCache('outbound_cache', 1000)

class AbstractSession:
	"""Abstract base class for different session types (master, or stacked)

	Abstract base class for different session types (master, or stacked), 
	both MasterSession and StackedSession are derived from this class.
	
	Attributes
	
	  none
	
	"""

	def destroy(self):
		"""Destroys the session. 
		
		We close filedescriptors here, in case no proxy
		module could be started (because of policy violations,
		or because the module cannot be found).
		
		Arguments
		
		  self -- this instance
		"""
		if self.client_stream:
			self.client_stream.close()
		if self.server_stream:
			self.server_stream.close()

class MasterSession(AbstractSession):
	"""Class encapsulating a master session.
	
	This class encapsulates a master session, which means
	that it's on the top of the session hierarchy.
	
	Referencing attributes exported by parent proxies
	
	  When a stacked proxy needs some information exported by its parent
	  it can simply be done by using the by-name references in the
	  session structure. For example a proxy named 'pssl' will export an
	  attribute named 'pssl' in its session which is inherited in the
	  session hierarchy, so a stacked proxy can refer to any of the
	  attributes of pssl through this reference:
	  
	  Example
	  
	    class MyPsslProxy(PsslProxy):
	    	class EmbeddedHttpProxy(HttpProxy):
		    def config(self):
		        HttpProxy.config(self)
		        
		        peer = self.session.pssl.server_peer_certificate.name
		
		def config(self):
		    PsslProxy.config(self)
		    self.stack_proxy = self.EmbeddedHttpProxy
	
	Attributes
	
	  client_stream  -- client stream
	  
	  client_address -- SockAddr instance containing client address
	  
	  client_local   -- local address (on the firewall)
	  
	  client_zone    -- zone of the client
	  
	  client_tos     -- Type of Service value for the client connection 
	                    as initiated by the client

	  server_stream  -- server stream
	  
	  server_address -- SockAddr instance containing server address

	  server_address_inband -- destination address is determined inband the channel
	  
	  server_local   -- local address (on the firewall)

	  server_local_loose -- allow loosely allocated source ports (e.g. 
	 		        it is not absoletely necessary to allocate
	 		        the same port as specified in server_local,
	 		        it is enough if it matches its category)
	  
	  server_zone    -- zone of the server
	  
	  server_tos     -- Type of Service value for the server connection

	  service        -- service instance this session runs
	  
	  session_id     -- unique identifier for this session in the
	                    format: "(firewall/service:instance id/proxy)"
	  
	  instance_id    -- the instance identifier of the service
	                    (sequence number)
	                    
	  started        -- indicates that the instance has been started

	  auth_policy    -- authentication/authorization policy

	  auth_user	 --

	  auth_groups    --

	  protocol	 -- the client side protocol

	  protocol_name  -- stringified name of @protocol

	"""
	
	def __init__(self, base_session_id=None):
		"""Constructor to initialize a MasterSession instance.

		This constructor initializes a new MasterSession instance
		based on its arguments.

		Arguments
		
		  self -- this instance
		
		"""
		self.base_session_id = base_session_id

		if not base_session_id:
			base_session_id = Zorp.firewall_name
		self.session_id = base_session_id
		
		self.client_stream = None
		self.client_address = None
		self.client_local = None
		self.client_zone = None
		self.client_tos = -1
		
		self.server_stream = None
		self.server_address = None
		self.server_address_inband = 0
		self.server_local = None
		self.server_local_loose = TRUE
		self.server_zone = None
		self.server_tos = -1
		
	        self.auth_policy = None
      	        self.auth_user = ""
		self.auth_groups = ()
	
		self.started = 0
		self.service = None
		self.instance_id = 0

		self.setProtocol(0)
		self.proxy = None

	def __del__(self):
		"""Function called when the master session is freed.
		
		This function is called when the master session is freed,
		thus the session ended. We inform our spawner service
		about this event.
		
		Arguments
		
		  self -- this instance
		"""
		if self.service:
			self.service.stopInstance(self)


	def setProtocol(self, protocol):
	    """Sets the protocol and its stringified name also.

	    Arguments
	    self -- this instance
	    protocol -- the protocol
	    """
	    self.protocol = protocol
	    try:
	   	    self.protocol_name = ZD_PROTO_NAME[protocol]
	    except KeyError:
	        self.protocol_name = "Unknown(%d)" % (self.protocol)



	def setService(self, service):
		"""Sets the service belonging to this session.

		Stores the service reference, and recalculates the session_id.
		This is called by the Listener after the service is determined.
		
		Arguments

		  self -- this instance
		  
		  service  -- Service instance
		"""
		self.service = service
		self.session_id = "%s/%s" % (self.base_session_id, service.name)
		## LOG ##
		# This message reports that the given service is started, because of a new connection.
		##
		log(self.session_id, CORE_SESSION, 5, "Starting service; name='%s'", service.name)

	def setClient(self, stream, addr):
		"""Set client address and perform access control.
		
		Sets the client address of the given session, and performs
		access control checks.
                   
		Arguments

		  self   -- this instance
		
		  stream -- stream of the client
		  
		  addr   -- sockaddr of the client
		"""
		self.client_stream = stream
		self.client_address = addr
		self.client_zone = root_zone.findZone(addr)

	def isClientPermitted(self):
		"""Function to actually check access control.
		
		This function is called when a connection is established to
		perform access control checks whether the client is
		permitted to use the requested service. Its return value
		specifies the result of the check.
		
		Arguments
		
		  self -- this instance
		  
		Returns
		
		  Z_ACCEPT for success, and Z_REJECT for failure.
		"""
		global outbound_cache
		try:
			zone_name = self.client_zone.getName()
			cached = outbound_cache.lookup((zone_name, self.service.name))
			if cached == Z_REJECT:
				## LOG ##
				# This message indicates that because of a cached decision this service is not permitted as an outbound service from that zone.
				# It means that the client from that zone tried to use this service and it is not permitted to do so.
				# Check that the service is included in the outbound_services set of the Zone.
				# @see: Zone
				##
				log(self.session_id, CORE_POLICY, 1, "Outbound service not permitted (cached); service='%s', client_zone='%s', client='%s', server_zone='%s', server='%s'", (self.service, self.client_zone, self.client_address, self.server_zone, self.server_address))
			return cached
		except KeyError:
			pass
		if self.client_zone.isOutboundServicePermitted(self) != Z_ACCEPT:
			outbound_cache.store((zone_name, self.service.name), Z_REJECT)
			## LOG ##
			# This message indicates that a service going out from the given
			# zone was denied by the policy. Check that the service is included in
			# the outbound_services set of the Zone.
			##
			log(self.session_id, CORE_POLICY, 1, "Outbound service not permitted; service='%s', client_zone='%s', client='%s', server_zone='%s', server='%s'", (self.service, self.client_zone, self.client_address, self.server_zone, self.server_address))
			return Z_REJECT
		outbound_cache.store((zone_name, self.service.name), Z_ACCEPT)
		return Z_ACCEPT

	def setServer(self, addr):
		"""Set the server address and perform access control checks.
		
		Stores the server address of the given connection, looks
		up server zone and performs access control and raises an
		exception upon failure.

		Arguments

		  self -- this instance
		  
		  addr -- Server address

		"""
		self.server_address = addr
		try:
			addr = addr[0]
		except TypeError:
			pass
		self.server_zone = root_zone.findZone(addr)
		
	def isServerPermitted(self):
		"""Function to actually check access control.
		
		This function is called when a connection is to be
		established with the server. It performs access control
		checks whether the connection to the server is permitted by
		the policy.  Its return value specifies the result of the
		check.
		
		Arguments
		
		  self -- this instance
		  
		Returns
		
		  Z_ACCEPT for success, and Z_REJECT for failure.
		"""
		global inbound_cache
		try:
			zone_name = self.server_zone.getName()
			cached = inbound_cache.lookup((zone_name, self.service.name))
			if cached == Z_REJECT:
				## LOG ##
				# This message indicates that because of a cached decision this service is not permitted as an inbound service to that zone.
				# It means that this service tried to connect to a  server in that zone and it is not permitted to do so.
				# Check that the service is included in the inbound_services set of the Zone.
				# @see: Zone
				##
				log(self.session_id, CORE_POLICY, 1, "Inbound service not permitted (cached); service='%s', client_zone='%s', client='%s', server_zone='%s', server='%s'", (self.service, self.client_zone, self.client_address, self.server_zone, self.server_address))
			return cached
		except KeyError:
			pass

		if self.server_zone.isInboundServicePermitted(self) != Z_ACCEPT:
			inbound_cache.store((zone_name, self.service.name), Z_REJECT)
			## LOG ##
			# This message indicates that a service trying to enter to the given
			# zone was denied by the policy. Check that the service is included in
			# the inbound_services set of the Zone.
			##
			log(self.session_id, CORE_POLICY, 1, "Inbound service not permitted; service='%s', client_zone='%s', client='%s', server_zone='%s', server='%s'", (self.service, self.client_zone, self.client_address, self.server_zone, self.server_address))
			return Z_REJECT
		inbound_cache.store((zone_name, self.service.name), Z_ACCEPT)
		return Z_ACCEPT
	
	def setServiceInstance(self, instance_id):
		"""Set service instance number and recalculate session id.
		
		Sets service instance number, and makes up a unique
		identifier for this session.

		Arguments

		  self -- this instance
		  
		  instance_id -- unique identifier of the service instance

		"""
		self.instance_id = instance_id
		self.session_id = "%s/%s:%d" % (self.base_session_id, self.name, self.instance_id)

class StackedSession(AbstractSession):
	"""Session class for subsessions.
	
	A StackedSession is a subsession, inheriting attributes from
	its parent.
	
	Attributes
	
	  owner     -- Parent session
	  
	  chainer   -- Chainer used to chain up to parent. If none
	               simply server_stream is used.
	"""

	def __init__(self, owner, chainer = None):
                """Constructor to initialize a StackedSession instance.

		This constructor initializes a new StackedSession instance
		based on parameters.
		
		Arguments

		  self        --    this instance
		  
		  owner       --    Parent session
		  
		  chainer     --    Chainer used to chain up to parent.
                """
		self.owner = owner
		self.chainer = chainer
		
	def __getattr__(self, name):
		"""Function to perform attribute inheritance.
		
		This function is called by the Python core when an attribute
		is referenced. It returns variables from the parent session, if
		not overriden here.

		Arguments

		  self         --   this instance
		  
		  name         --   Name of the attribute to get.

		Returns
		
		  The value of the given attribute.

		"""
		try:
			if name != '__dict__':
				return self.__dict__[name]
			else:
				raise KeyError
		except KeyError:
			return getattr(self.owner, name)

	def setProxy(self, proxy):
		"""Set the proxy name used in this subsession.

		Stores a reference to the proxy class, and modifies
		the session_id to include the proxy name. This is
		called by the Listener after the proxy module to 
		use is determined.
		
		Arguments

		  self       --     this instance
		  
		  proxy      --     Proxy class, derived from Proxy

                """
		self.session_id = "%s/%s:%d/%s" % (self.base_session_id, self.name, self.instance_id, proxy)

