""" A form action adapter that saves form submissions for download """

__author__  = 'Steve McMahon <steve@dcn.org>'
__docformat__ = 'plaintext'

from AccessControl import ClassSecurityInfo

from Products.Archetypes.public import *
from Products.Archetypes.utils import contentDispositionHeader
from Products.ATContentTypes.content.schemata import finalizeATCTSchema
from Products.ATContentTypes.content.base import registerATCT
from Products.CMFCore.permissions import View, ModifyPortalContent

from Products.PloneFormGen.config import *
from Products.PloneFormGen.content.actionAdapter import \
    FormActionAdapter, FormAdapterSchema


from DateTime import DateTime
import csv
from StringIO import StringIO

class FormSaveDataAdapter(FormActionAdapter):
    """A form action adapter that will save form input data in tab-delimited format."""

    schema = FormAdapterSchema.copy() + Schema((
        LinesField('ExtraData',
            widget=MultiSelectionWidget(
                label='Extra Data',
                description="""
                    Pick any extra data you'd like saved with the form input.
                    """,
                format='checkbox',
                i18n_domain = "ploneformgen",
                label_msgid = "label_savedataextra_text",
                description_msgid = "help_savedataextra_text",
                ),
            vocabulary = 'vocabExtraDataDL',
            ),
        StringField('DownloadFormat',
            searchable=0,
            required=1,
            default='tsv',
            vocabulary = 'vocabFormatDL',
            widget=SelectionWidget(
                label='Download Format',
                i18n_domain = "ploneformgen",
                label_msgid = "label_downloadformat_text",
                ),
            ),
        BooleanField("UseColumnNames",
            required=False,
            searchable=False,
            widget=BooleanWidget(
                label = "Include Column Names",
                description = "Do you wish to have column names on the first line of downloaded input?",
                i18n_domain = "ploneformgen",
                label_msgid = "label_usecolumnnames_text",
                description_msgid = "help_usecolumnnames_text",
                ),
            ),
        LinesField('SavedFormInput',
            edit_accessor='getSavedFormInputForEdit',
            searchable=0,
            required=0,
            primary=1,
            read_permission=DOWNLOAD_SAVED_PERMISSION,
            widget=TextAreaWidget(
                label="Saved Form Input",
                i18n_domain = "ploneformgen",
                label_msgid = "label_savedatainput_text",
                description_msgid = "help_savedatainput_text",
                ),
            ),
    ))
    finalizeATCTSchema(schema, folderish=True, moveDiscussion=False)

    meta_type      = 'FormSaveDataAdapter'
    portal_type    = 'FormSaveDataAdapter'
    archetype_name = 'Save Data Adapter'

    immediate_view = 'fg_savedata_view'
    default_view   = 'fg_savedata_view'

    security       = ClassSecurityInfo()


    def initializeArchetype(self, **kwargs):
        """ the trick of using a textarea widget with a lines field
            is causing a bad interpretation of the default. (It's
            getting represented as a string.)
        """

        FormActionAdapter.initializeArchetype(self, **kwargs)
        
        self.SavedFormInput = []


    def _isStringLike(self, obj):
        try:
            obj + ''
        except:
            return False
        return True


    security.declareProtected(ModifyPortalContent, 'getSavedFormInputForEdit')
    def getSavedFormInputForEdit(self, **kwargs):
        """ returns saved as CSV text """

        sbuf = StringIO()
        writer = csv.writer(sbuf)
        for row in self.SavedFormInput:
            if self._isStringLike(row):
                # legacy from version < 0.1.1
                row = row.split('\t')
            writer.writerow( row )
        res = sbuf.getvalue()
        sbuf.close()

        return res


    security.declareProtected(ModifyPortalContent, 'setSavedFormInput')
    def setSavedFormInput(self, value, **kwargs):
        """ expects value as csv text string, stores as list of lists """

        sbuf = StringIO( value )
        reader = csv.reader(sbuf)
        res = [row for row in reader]
        sbuf.close()

        self.SavedFormInput = res


    security.declareProtected(ModifyPortalContent, 'clearSavedFormInput')
    def clearSavedFormInput(self, **kwargs):
        """ convenience method to clear input buffer """

        self.SavedFormInput = []


    def _addDataRow(self, value):

        res = list(self.SavedFormInput)
        res.append(value)
        self.SavedFormInput = res


    def onSuccess(self, fields, REQUEST=None):
        """
        saves data.
        """

        from ZPublisher.HTTPRequest import FileUpload
        from OFS.content_types import guess_content_type

        data = []
        for f in fields:
            if f.meta_type == 'FormFileField':
                file = REQUEST.form.get('%s_file' % f.fgField.getName())
                if isinstance(file, FileUpload) and file.filename != '':
                    file.seek(0)
                    fdata = file.read()
                    filename = file.filename
                    mimetype, enc = guess_content_type(filename, fdata, None)
                    if mimetype.find('text/') >= 0:
                        # convert to native eols
                        fdata = fdata.replace('\x0d\x0a', '\n').replace('\x0a', '\n').replace('\x0d', '\n')
                        data.append( '%s:%s:%s:%s' %  (filename, mimetype, enc, fdata) )
                    else:
                        data.append( '%s:%s:%s:Binary upload discarded' %  (filename, mimetype, enc) )
                else:
                    data.append( 'NO UPLOAD' )
            elif f.meta_type not in ('FormLabelField','FormRichLabelField'):
                data.append( REQUEST.form.get(f.fgField.getName(),'') )

        if self.ExtraData:
            for f in self.ExtraData:
                if f == 'dt':
                    data.append( str(DateTime()) )
                else:
                    data.append( getattr(REQUEST, f, '') )


        self._addDataRow( data )


    def _cleanInput(self, value):
        """ make data safe to store in tab-delimited format """

        return  str(value).replace('\x0d\x0a', r'\n').replace('\x0a', r'\n').replace('\x0d', r'\n').replace('\t', r'\t')


    security.declareProtected(DOWNLOAD_SAVED_PERMISSION, 'getColumnNames')
    def getColumnNames(self):
        """Returns a list of column names"""
        
        names = [field.getName() for field in self.fgFields(displayOnly=True)]
        for f in self.ExtraData:
            names.append(f)
        
        return names
        

    security.declareProtected(DOWNLOAD_SAVED_PERMISSION, 'download_tsv')
    def download_tsv(self, REQUEST=None, RESPONSE=None):
        """Download the saved data
        """

        filename = self.id
        if filename.find('.') < 0:
            filename = '%s.tsv' % filename
        header_value = contentDispositionHeader('attachment', self.getCharset(), filename=filename)
        RESPONSE.setHeader("Content-disposition", header_value)
        RESPONSE.setHeader("Content-Type", 'text/tab-separated-values')

        if getattr(self, 'UseColumnNames', False):
            res = "%s\n" % '\t'.join( self.getColumnNames() )
        else:
            res = ''

        for row in self.SavedFormInput:
            if self._isStringLike(row):
                res = '%s%s\n' % (res, row)
            else:
                res = '%s%s\n' % (res, '\t'.join( [self._cleanInput(col) for col in row] ))

        return res


    security.declareProtected(DOWNLOAD_SAVED_PERMISSION, 'download_csv')
    def download_csv(self, REQUEST=None, RESPONSE=None):
        """Download the saved data
        """

        filename = self.id
        if filename.find('.') < 0:
            filename = '%s.csv' % filename
        header_value = contentDispositionHeader('attachment', self.getCharset(), filename=filename)
        RESPONSE.setHeader("Content-disposition", header_value)
        RESPONSE.setHeader("Content-Type", 'text/comma-separated-values')

        if getattr(self, 'UseColumnNames', False):
            res = "%s\n" % ','.join( self.getColumnNames() )
        else:
            res = ''

        return '%s%s' % (res, self.getSavedFormInputForEdit())


    security.declareProtected(DOWNLOAD_SAVED_PERMISSION, 'download')
    def download(self, REQUEST=None, RESPONSE=None):
        """Download the saved data
        """

        format = getattr(self, 'DownloadFormat', 'tsv')
        if format == 'tsv':
            return self.download_tsv(REQUEST, RESPONSE)
        else:
            assert format == 'csv', 'Unknown download format'
            return self.download_csv(REQUEST, RESPONSE)


    security.declareProtected(DOWNLOAD_SAVED_PERMISSION, 'formatMIME')
    def formatMIME(self):
        """MIME format selected for download
        """

        format = getattr(self, 'DownloadFormat', 'tsv')
        if format == 'tsv':
            return 'text/tab-separated-values'
        else:
            assert format == 'csv', 'Unknown download format'
            return 'text/comma-separated-values'


    security.declareProtected(DOWNLOAD_SAVED_PERMISSION, 'itemsSaved')
    def itemsSaved(self):
        """Download the saved data
        """

        return len(self.SavedFormInput)


    def vocabExtraDataDL(self):
        """ returns vocabulary for extra data """

        return DisplayList( (
                ('dt',
                    self.translate( msgid='vocabulary_postingdt_text',
                    domain='ploneformgen',
                    default='Posting Date/Time')
                    ),
                ('HTTP_X_FORWARDED_FOR','HTTP_X_FORWARDED_FOR',),
                ('REMOTE_ADDR','REMOTE_ADDR',),
                ('HTTP_USER_AGENT','HTTP_USER_AGENT',),
                ) )


    def vocabFormatDL(self):
        """ returns vocabulary for format """

        return DisplayList( (
                ('tsv',
                    self.translate( msgid='vocabulary_tsv_text',
                    domain='ploneformgen',
                    default='Tab-Separated Values')
                    ),
                ('csv',
                    self.translate( msgid='vocabulary_csv_text',
                    domain='ploneformgen',
                    default='Comma-Separated Values')
                    ),
            ) )



registerATCT(FormSaveDataAdapter, PROJECTNAME)
