""" Form fields, implemented via Archetypes Fields, Validators and Widgets"""

__author__  = 'Steve McMahon <steve@dcn.org>'
__docformat__ = 'plaintext'

from Products.Archetypes.public import *
from Products.Archetypes.utils import shasattr

from Products.ATContentTypes.content.base import registerATCT
from Products.ATContentTypes.content.base import ATCTContent
from Products.ATContentTypes.content.schemata import ATContentTypeSchema
from Products.ATContentTypes.content.schemata import finalizeATCTSchema
from Products.ATContentTypes.configuration import zconf

from Products.TALESField import TALESString, TALESLines

from Products.PloneFormGen.config import PROJECTNAME, EDIT_TALES_PERMISSION

from Products.CMFCore.permissions import View, ModifyPortalContent

from AccessControl import ClassSecurityInfo

from Products.PloneFormGen import PloneFormGenMessageFactory as _
from Products.PloneFormGen import HAS_PLONE25
from Products.PloneFormGen.widgets import RichLabelWidget

from Products.PloneFormGen.content.fieldsBase import *
# \
#    BaseFieldSchema, BareFieldSchema, BaseFieldSchemaLinesDefault, BaseFieldSchemaTextDefault, \
#    BaseFormField, BaseFieldSchemaStringDefault, \
#    maxlengthField, sizeField, vocabularyField, vocabularyOverrideField

class FGStringField(BaseFormField):
    """ A string entry field """

    security  = ClassSecurityInfo()

    schema = BaseFieldSchemaStringDefault + Schema((
        maxlengthField,
        sizeField,
        StringField('fgStringValidator',
            vocabulary='stringValidatorsDL',
            enforceVocabulary=1,
            widget=SelectionWidget(label='Validator',
                description="""Tests input against simple string patterns.""",
                i18n_domain = "ploneformgen",
                label_msgid = "label_fgstringvalidator_text",
                description_msgid = "help_fgstringvalidator_text",
                ),
        ),
    ))

    # hide references & discussion
    finalizeATCTSchema(schema, folderish=True, moveDiscussion=False)

    # Standard content type setup
    portal_type = meta_type = 'FormStringField'
    archetype_name = 'String Field'
    content_icon = 'StringField.gif'
    typeDescription= 'A string field'

    def __init__(self, oid, **kwargs):
        """ initialize class """

        BaseFormField.__init__(self, oid, **kwargs)

        # set a preconfigured field as an instance attribute
        self.fgField = StringField('fg_string_field',
            searchable=0,
            required=0,
            write_permission = View,
            validators=('isNotTooLong',),
            )

    def stringValidatorsDL(self):
        """ return a display list of string validators.
            this is a hack for 118n
        """

        # Check for Plone 2.5
        if HAS_PLONE25:
          return DisplayList( (
                ('',
                    _(u'vocabulary_none_text', u'None')
                    ),
                ('isEmail',
                    _(u'vocabulary_isemailaddress_text', u'Is an E-Mail Address')
                    ),
                ('isPrintable',
                    _(u'vocabulary_onlyprintable_text', u'Contains only printable characters')
                    ),
                ('isURL',
                    _(u'vocabulary_isurl_text', u'Is a well-formed URL')
                    ),
                ('isUSPhoneNumber',
                    _(u'vocabulary_isusphone_text', u'Is a valid US phone number')
                    ),
                ('isInternationalPhoneNumber',
                    _(u'vocabulary_isintphone_text', u'Is a valid international phone number')
                    ),
                ('isZipCode',
                    _(u'vocabulary_iszipcode_text', u'Is a valid zip code')
                    ),
                ) )
        else:
            return DisplayList( (
                ('',
                    self.translate( msgid='vocabulary_none_text',
                    domain='ploneformgen',
                    default='None')
                    ),
                ('isEmail',
                    self.translate( msgid='vocabulary_isemailaddress_text',
                    domain='ploneformgen',
                    default='Is an E-Mail Address')
                    ),
                ('isPrintable',
                    self.translate( msgid='vocabulary_onlyprintable_text',
                    domain='ploneformgen',
                    default='Contains only printable characters')
                    ),
                ('isURL',
                    self.translate( msgid='vocabulary_isurl_text',
                    domain='ploneformgen',
                    default='Is a well-formed URL')
                    ),
                ('isUSPhoneNumber',
                    self.translate( msgid='vocabulary_isusphone_text',
                    domain='ploneformgen',
                    default='Is a valid US phone number')
                    ),
                ('isInternationalPhoneNumber',
                    self.translate( msgid='vocabulary_isintphone_text',
                    domain='ploneformgen',
                    default='Is a valid international phone number')
                    ),
                ('isZipCode',
                    self.translate( msgid='vocabulary_iszipcode_text',
                    domain='ploneformgen',
                    default='Is a valid zip code')
                    ),
                ) )

registerATCT(FGStringField, PROJECTNAME)


class FGPasswordField(FGStringField):
    """ Password entry field (input is masked) """

    schema = BaseFieldSchemaStringDefault + Schema((
        maxlengthField,
        sizeField,
    ))

    # hide references & discussion
    finalizeATCTSchema(schema, folderish=True, moveDiscussion=False)
    # Standard content type setup
    portal_type = meta_type = 'FormPasswordField'
    archetype_name = 'Password Field'
    content_icon = 'PasswordField.gif'
    typeDescription= 'A password field (input masked)'

    def __init__(self, oid, **kwargs):
        """ initialize class """

        BaseFormField.__init__(self, oid, **kwargs)

        # set a preconfigured field as an instance attribute
        self.fgField = StringField('fg_string_field',
            searchable=0,
            required=0,
            write_permission = View,
            validators=('isNotTooLong',),
            widget=PasswordWidget(),
            )

registerATCT(FGPasswordField, PROJECTNAME)


class FGIntegerField(BaseFormField):
    """ Integer entry field """

    security  = ClassSecurityInfo()

    schema = BaseFieldSchemaStringDefault + Schema((
        IntegerField('minval',
            searchable=0,
            required=1,
            default='0',
            widget=IntegerWidget(
                label="Minimum Acceptable Value",
                description="""
                    The form will not accept values less than the number you enter here.
                """,
                i18n_domain = "ploneformgen",
                label_msgid = "label_minval_text",
                description_msgid = "help_minval_text",
                ),
            ),
        IntegerField('maxval',
            searchable=0,
            required=1,
            default='10000',
            widget=IntegerWidget(
                label="Maximum Acceptable Value",
                description="""
                    The form will not accept values greater than the number you enter here.
                """,
                i18n_domain = "ploneformgen",
                label_msgid = "label_maxval_text",
                description_msgid = "help_maxval_text",
                ),
            ),
        maxlengthField,
        sizeField,
    ))
    # hide references & discussion
    finalizeATCTSchema(schema, folderish=True, moveDiscussion=False)

    # Standard content type setup
    portal_type = meta_type = 'FormIntegerField'
    archetype_name = 'Integer Field'
    content_icon = 'IntegerField.gif'
    typeDescription= 'A integer field'

    def __init__(self, oid, **kwargs):
        """ initialize class """

        BaseFormField.__init__(self, oid, **kwargs)

        # set a preconfigured field as an instance attribute
        self.fgField = IntegerField('fg_integer_field',
            searchable=0,
            required=0,
            minval=0,
            maxval=10000,
            validators=('isNotTooLong', 'inExNumericRange',),
            write_permission = View,
            )

registerATCT(FGIntegerField, PROJECTNAME)


class FGFixedPointField(BaseFormField):
    """ Fixed-Point (float) entry field """

    security  = ClassSecurityInfo()

    schema = BaseFieldSchemaStringDefault + Schema((
        FixedPointField('minval',
            searchable=0,
            required=1,
            default='0.0',
            widget=DecimalWidget(
                label="Minimum Acceptable Value",
                description="""
                    The form will not accept values less than the number you enter here.
                """,
                size=8,
                i18n_domain = "ploneformgen",
                label_msgid = "label_minval_text",
                description_msgid = "help_minval_text",
                ),
            ),
        FixedPointField('maxval',
            searchable=0,
            required=1,
            default='10000.0',
            widget=DecimalWidget(
                label="Maximum Acceptable Value",
                description="""
                    The form will not accept values greater than the number you enter here.
                """,
                size=8,
                i18n_domain = "ploneformgen",
                label_msgid = "label_maxval_text",
                description_msgid = "help_maxval_text",
                ),
            ),
        maxlengthField,
        sizeField,
    ))
    # hide references & discussion
    finalizeATCTSchema(schema, folderish=True, moveDiscussion=False)

    # Standard content type setup
    portal_type = meta_type = 'FormFixedPointField'
    archetype_name = 'Fixed-Point Field'
    content_icon = 'FloatField.gif'
    typeDescription= 'A fixed-point (float) field'

    def __init__(self, oid, **kwargs):
        """ initialize class """

        BaseFormField.__init__(self, oid, **kwargs)

        # set a preconfigured field as an instance attribute
        self.fgField = FixedPointField('fg_date_field',
            searchable=0,
            required=0,
            write_permission = View,
            validators=('isNotTooLong', 'inExNumericRange',),
            )

#    security.declareProtected(ModifyPortalContent, 'setThousands_commas')
#    def setThousands_commas(self, value, **kw):
#        """ set widget's thousands_commas """
#
#        self.fgField.widget.thousands_commas = value == '1'
#
#
#    security.declareProtected(ModifyPortalContent, 'getThousands_commas')
#    def getThousands_commas(self, **kw):
#        """ get widget's thousands_commas """
#
#        return self.fgField.widget.thousands_commas

registerATCT(FGFixedPointField, PROJECTNAME)


class FGBooleanField(BaseFormField):
    """ Boolean (checkbox) field """

    security  = ClassSecurityInfo()

    schema = BaseFieldSchema.copy() + Schema((
        StringField('fgDefault',
            searchable=0,
            required=0,
            widget=BooleanWidget(
                label='Default',
                i18n_domain = "ploneformgen",
                label_msgid = "label_fgdefault_text",
                description_msgid = "help_fgdefault_text",
                ),
        ),
        StringField('fgBooleanValidator',
            vocabulary='boolVocabDL',
            enforceVocabulary=1,
            widget=SelectionWidget(label='Validator',
                description="""Choose a validator to require a particular response.""",
                i18n_domain = "ploneformgen",
                label_msgid = "label_fgbooleanvalidator_text",
                description_msgid = "help_fgbooleanvalidator_text",
                ),
        ),
    ))
    schema['required'].widget.description = \
        """NOTE: For a checkbox field, the required flag doesn't do anything beyond
           putting a 'required' marker next to the label. If you wish to require a
           particular input, choose a validator below.
        """
    schema['required'].widget.description_msgid = "help_boolrequired_text"
    # hide references & discussion
    finalizeATCTSchema(schema, folderish=True, moveDiscussion=False)

    # Standard content type setup
    portal_type = meta_type = 'FormBooleanField'
    archetype_name = 'Boolean Field'
    content_icon = 'CheckBoxField.gif'
    typeDescription= 'A CheckBox (Boolean) field'

    def __init__(self, oid, **kwargs):
        """ initialize class """

        BaseFormField.__init__(self, oid, **kwargs)

        # set a preconfigured field as an instance attribute
        self.fgField = BooleanField('fg_boolean_field',
            searchable=0,
            required=0,
            write_permission = View,
            )

    security.declareProtected(ModifyPortalContent, 'setFgBooleanValidator')
    def setFgBooleanValidator(self, value, **kw):
        """ set boolean validator """

        if value:
            self.fgField.validators = (value,)
        else:
            self.fgField.validators = ()
        self.fgField._validationLayer()

        self.fgBooleanValidator = value

    def boolVocabDL(self):
        """ returns DisplayList of vocabulary for fgBooleanValidator """

        # Check for Plone 2.5
        if HAS_PLONE25:
          return DisplayList( (
                ('',
                    _(u'vocabulary_none_text', u'None')
                    ),
                ('isChecked',
                    _(u'vocabulary_ischecked_text', u'Is checked')
                    ),
                ('isUnchecked',
                    _(u'vocabulary_isnotchecked_text', u'Is not checked')
                    ),
                ) )
        else:
            return DisplayList( (
                ('',
                    self.translate( msgid='vocabulary_none_text',
                    domain='ploneformgen',
                    default='None')
                    ),
                ('isChecked',
                    self.translate( msgid='vocabulary_ischecked_text',
                    domain='ploneformgen',
                    default='Is checked')
                    ),
                ('isUnchecked',
                    self.translate( msgid='vocabulary_isnotchecked_text',
                    domain='ploneformgen',
                    default='Is not checked')
                    ),
                ) )


registerATCT(FGBooleanField, PROJECTNAME)


class FGDateField(BaseFormField):
    """ Date/Time Entry Field """

    security  = ClassSecurityInfo()

    schema = BaseFieldSchemaStringDefault + Schema((
        BooleanField('fgShowHM',
            searchable=0,
            required=0,
            default=1,
            widget=BooleanWidget(
                label='Show Time Selection Options',
                i18n_domain = "ploneformgen",
                label_msgid = "label_fgshowhm_text",
                description_msgid = "help_fgshowhm_text",
                ),
        ),
    ))
    # hide references & discussion
    finalizeATCTSchema(schema, folderish=True, moveDiscussion=False)

    # Standard content type setup
    portal_type = meta_type = 'FormDateField'
    archetype_name = 'Date/Time Field'
    content_icon = 'DateTimeField.gif'
    typeDescription= 'A date/time field. Time component is optional.'

    def __init__(self, oid, **kwargs):
        """ initialize class """

        BaseFormField.__init__(self, oid, **kwargs)

        # set a preconfigured field as an instance attribute
        self.fgField = DateTimeField('fg_date_field',
            searchable=0,
            required=0,
            write_permission = View,
            widget=CalendarWidget(),
            )

    security.declareProtected(ModifyPortalContent, 'setFgShowHM')
    def setFgShowHM(self, value, **kw):
        """ set show_hm """

        self.fgField.widget.show_hm = value == '1'
        self.fgShowHM = value

registerATCT(FGDateField, PROJECTNAME)


class FGLabelField(BaseFormField):
    """ Label-Only field (no input component) """

    security  = ClassSecurityInfo()

    schema = BareFieldSchema
    # hide references & discussion
    finalizeATCTSchema(schema, folderish=True, moveDiscussion=False)

    # Standard content type setup
    portal_type = meta_type = 'FormLabelField'
    archetype_name = 'Label Field'
    content_icon = 'LabelField.gif'
    typeDescription= 'A Label (No Input) field'

    def __init__(self, oid, **kwargs):
        """ initialize class """

        BaseFormField.__init__(self, oid, **kwargs)

        # set a preconfigured field as an instance attribute
        self.fgField = StringField('fg_label_field',
            searchable=0,
            required=0,
            write_permission = View,
            widget=LabelWidget(),
            )


registerATCT(FGLabelField, PROJECTNAME)


class FGLinesField(BaseFormField):
    """ A line entry field. This appears as a textarea without wordwrap. """

    security  = ClassSecurityInfo()

    schema = BaseFieldSchemaLinesDefault

    # Standard content type setup
    portal_type = meta_type = 'FormLinesField'
    archetype_name = 'Lines Field'
    content_icon = 'LinesField.gif'
    typeDescription= 'A lines field. This appears as a textarea without wordwrap.'

    def __init__(self, oid, **kwargs):
        """ initialize class """

        BaseFormField.__init__(self, oid, **kwargs)

        # set a preconfigured field as an instance attribute
        self.fgField = LinesField('fg_lines_field',
            searchable=0,
            required=0,
            write_permission = View,
            )


registerATCT(FGLinesField, PROJECTNAME)


class FGSelectionField(BaseFormField):
    """ Selection Field (radio buttons or select) """

    security  = ClassSecurityInfo()

    schema = BaseFieldSchemaStringDefault + Schema((
        vocabularyField,
        vocabularyOverrideField,
        StringField('fgFormat',
            searchable=0,
            required=0,
            default='flex',
            enforceVocabulary=1,
            vocabulary='formatVocabDL',
            widget=SelectionWidget(
                label='Presentation Widget',
                i18n_domain = "ploneformgen",
                label_msgid = "label_fgformat_text",
                description_msgid = "help_fgformat_text",
                ),
        ),
    ))
    # hide references & discussion
    finalizeATCTSchema(schema, folderish=True, moveDiscussion=False)

    # Standard content type setup
    portal_type = meta_type = 'FormSelectionField'
    archetype_name = 'Selection Field'
    content_icon = 'ListField.gif'
    typeDescription= 'A selection field'

    def __init__(self, oid, **kwargs):
        """ initialize class """

        BaseFormField.__init__(self, oid, **kwargs)

        # set a preconfigured field as an instance attribute
        self.fgField = StringVocabularyField('fg_selection_field',
            searchable=0,
            required=0,
            widget=SelectionWidget(),
            vocabulary = '_get_selection_vocab',
            enforceVocabulary=1,
            write_permission = View,
            )


    security.declareProtected(ModifyPortalContent, 'setFgFormat')
    def setFgFormat(self, value, **kw):
        """ set selection format """

        self.fgField.widget.format = value
        self.fgFormat = value


    def formatVocabDL(self):
        """ returns vocabulary for fgFormat """

        # Check for Plone 2.5
        if HAS_PLONE25:
          return DisplayList( (
                ('flex',
                        _(u'vocabulary_flex_text', u'Flexible (radio for short, select for longer)')
                    ),
                ('select',
                        _(u'vocabulary_selection_text', u'Selection list')
                    ),
                ('radio',
                        _(u'vocabulary_radio_text', u'Radio buttons')
                    ),
            ) )
        else:
            return DisplayList( (
                ('flex',
                    self.translate( msgid='vocabulary_flex_text',
                    domain='ploneformgen',
                    default='Flexible (radio for short, select for longer)')
                    ),
                ('select',
                    self.translate( msgid='vocabulary_selection_text',
                    domain='ploneformgen',
                    default='Selection list')
                    ),
                ('radio',
                    self.translate( msgid='vocabulary_radio_text',
                    domain='ploneformgen',
                    default='Radio buttons')
                    ),
            ) )



registerATCT(FGSelectionField, PROJECTNAME)


class FGMultiSelectField(BaseFormField):
    """ Multiple selection field (select with multiple or check boxes) """

    security  = ClassSecurityInfo()

    schema = BaseFieldSchemaLinesDefault + Schema((
        vocabularyField,
        vocabularyOverrideField,
        StringField('fgFormat',
            searchable=0,
            required=0,
            default='select',
            enforceVocabulary=1,
            vocabulary='formatVocabDL',
            widget=SelectionWidget(
                label='Presentation Widget',
                i18n_domain = "ploneformgen",
                label_msgid = "label_fgmsformat_text",
                description_msgid = "help_fgmsformat_text",
                ),
        ),
    ))
    # hide references & discussion
    finalizeATCTSchema(schema, folderish=True, moveDiscussion=False)

    # Standard content type setup
    portal_type = meta_type = 'FormMultiSelectionField'
    archetype_name = 'Multi-Select Field'
    content_icon = 'MultipleListField.gif'
    typeDescription= 'A multiple-selection field'

    def __init__(self, oid, **kwargs):
        """ initialize class """

        BaseFormField.__init__(self, oid, **kwargs)

        # set a preconfigured field as an instance attribute
        self.fgField = LinesVocabularyField('fg_mselection_field',
            searchable=0,
            required=0,
            widget=MultiSelectionWidget(),
            vocabulary = '_get_selection_vocab',
            enforceVocabulary=1,
            write_permission = View,
            )


    security.declareProtected(ModifyPortalContent, 'setFgFormat')
    def setFgFormat(self, value, **kw):
        """ set selection format """

        self.fgField.widget.format = value
        self.fgFormat = value


    security.declareProtected(ModifyPortalContent, 'setFgRows')
    def setFgRows(self, value, **kw):
        """ sets widget rows """

        self.fgField.widget.size = value


    security.declareProtected(View, 'getFgRows')
    def getFgRows(self, **kw):
        """ gets widget rows """

        return self.fgField.widget.size


    def formatVocabDL(self):
        """ returns vocabulary for fgFormat """

        # Check for Plone 2.5
        if HAS_PLONE25:
          return DisplayList( (
            ('select',
                    _(u'vocabulary_selection_text', u'Selection list')
                ),
            ('checkbox',
                    _(u'vocabulary_checkbox_text', u'Checkbox list')
                )
            ) )
        else:
            return DisplayList( (
                ('select',
                    self.translate( msgid='vocabulary_selection_text',
                    domain='ploneformgen',
                    default='Selection list')
                    ),
                ('checkbox',
                    self.translate( msgid='vocabulary_checkbox_text',
                    domain='ploneformgen',
                    default='Checkbox list')
                    )
                ) )


registerATCT(FGMultiSelectField, PROJECTNAME)


class PlainTextField(TextField):
    """
    A text field forced text/plain.
    Without this hack, the textarea widget will (by acquisition) call
    the form folder's getContentType method.
    """

    security  = ClassSecurityInfo()

    security.declarePublic('getContentType')
    def getContentType(self, instance, fromBaseUnit=True):
        return 'text/plain'


class FGTextField(BaseFormField):
    """ Text (textarea) field """

    security  = ClassSecurityInfo()

    schema = BaseFieldSchemaTextDefault

    # Standard content type setup
    portal_type = meta_type = 'FormTextField'
    archetype_name = 'Text Field'
    content_icon = 'TextAreaField.gif'
    typeDescription= 'A text area field'

    def __init__(self, oid, **kwargs):
        """ initialize class """

        BaseFormField.__init__(self, oid, **kwargs)

        # set a preconfigured field as an instance attribute
        self.fgField = PlainTextField('fg_text_field',
            searchable=0,
            required=0,
            write_permission = View,
            validators=('isNotTooLong',),
            widget = TextAreaWidget(
                maxlength=0,
                ),
            )

    security.declareProtected(View, 'isBinary')
    def isBinary(self, key):
        return False;

    security.declareProtected(View, 'getContentType')
    def getContentType(self, key=None):
        return 'text/plain'


registerATCT(FGTextField, PROJECTNAME)


class HtmlTextField(TextField):
    """
    A text field forced text/html.
    Without this hack, the rich widget will (by acquisition) call
    the form folder's getContentType method.
    """

    security  = ClassSecurityInfo()

    security.declarePublic('getContentType')
    def getContentType(self, instance, fromBaseUnit=True):
        return 'text/html'


class FGRichTextField(BaseFormField):
    """ Rich-text (visual editor) field """

    from Products.ATContentTypes.config import HAS_MX_TIDY    
    
    security  = ClassSecurityInfo()

    schema = BaseFieldSchemaRichTextDefault.copy()

    if HAS_MX_TIDY:
        schema = schema + Schema((
            StringField('fgStringValidator',
                vocabulary='htmlValidatorsDL',
                enforceVocabulary=1,
                default='isTidyHtmlWithCleanup',
                widget=SelectionWidget(label='Validator',
                    description="""Input tests using HTMLTidy (if installed).""",
                    i18n_domain = "ploneformgen",
                    label_msgid = "label_fgstringvalidator_text",
                    description_msgid = "help_fgrtvalidator_text",
                    ),
                ),
        ))

    # Standard content type setup
    portal_type = meta_type = 'FormRichTextField'
    archetype_name = 'RichText Field'
    content_icon = 'RichTextField.gif'
    typeDescription= 'A rich text area field'

    def __init__(self, oid, **kwargs):
        """ initialize class """

        BaseFormField.__init__(self, oid, **kwargs)

        # set a preconfigured field as an instance attribute
        self.fgField = HtmlTextField('fg_text_field',
            searchable=0,
            required=0,
            write_permission = View,
            validators = ('isNotTooLong', 'isTidyHtmlWithCleanup',),
            default_content_type = 'text/html',
            default_output_type = 'text/x-html-safe',
            allowable_content_types = zconf.ATDocument.allowed_content_types,
            widget = RichWidget(
                    allow_file_upload = False,
                    ),
            )


    security.declareProtected(View, 'isBinary')
    def isBinary(self, key):
        return False;

    security.declareProtected(View, 'getContentType')
    def getContentType(self, key=None):
        return 'text/html'

    def htmlValidatorsDL(self):
        """ return a display list of string validators.
            this is a hack for 118n
        """

        # Check for Plone 2.5
        if HAS_PLONE25:
            return DisplayList( (
                ('',
                    _(u'vocabulary_none_text', u'None')
                    ),
                ('isTidyHtml',
                    _(u'vocabulary_istidyhtml_text', u'Is Tidy HTML (fails on errors and warnings)')
                    ),
                ('isTidyHtmlWithCleanup',
                    _(u'vocabulary_istidyhtmlwithcleanup_text', u'Tidy HTML With Cleanup (fails on errors, cleans up rest)')
                    ),
                ) )
        else:
            return DisplayList( (
                ('',
                    self.translate( msgid='vocabulary_none_text',
                    domain='ploneformgen',
                    default='None')
                    ),
                ('isTidyHtml',
                    self.translate( msgid='vocabulary_istidyhtml_text',
                    domain='ploneformgen',
                    default='Is Tidy HTML (fails on errors and warnings)')
                    ),
                ('isTidyHtmlWithCleanup',
                    self.translate( msgid='vocabulary_istidyhtmlwithcleanup_text',
                    domain='ploneformgen',
                    default='Tidy HTML With Cleanup (fails on errors, cleans up rest)')
                    ),
                ) )

registerATCT(FGRichTextField, PROJECTNAME)


class FGRichLabelField(BaseFormField):
    """ Rich-text label field """

    security  = ClassSecurityInfo()

    schema = BareFieldSchema.copy() + Schema((
        TextField('fgDefault',
            searchable=0,
            required=0,
            validators = ('isTidyHtmlWithCleanup',),
            default_content_type = 'text/html',
            default_output_type = 'text/x-html-safe',
            allowable_content_types = zconf.ATDocument.allowed_content_types,
            widget=RichWidget(label='Label body',
                description="""
                    The text to display in the form.
                """,
                i18n_domain = "ploneformgen",
                label_msgid = "label_fglabelbody_text",
                description_msgid = "help_fglabelbody_text",
                allow_file_upload = False,
                ),
            ),
        ))

    schema['title'].widget.label = "Title"
    schema['title'].widget.label_msgid='label_title'
    schema['title'].widget.description = "Not displayed on form."
    schema['title'].widget.description_msgid = 'help_notdisplayed_text'
    schema['description'].schemata = 'metadata'

    # Standard content type setup
    portal_type = meta_type = 'FormRichLabelField'
    archetype_name = 'Rich Label Field'
    content_icon = 'RichLabelField.gif'
    typeDescription= 'A rich-text label'

    def __init__(self, oid, **kwargs):
        """ initialize class """

        BaseFormField.__init__(self, oid, **kwargs)

        # set a preconfigured field as an instance attribute
        self.fgField = HtmlTextField('fg_text_field',
            searchable=0,
            required=0,
            write_permission = View,
            default_content_type = 'text/html',
            default_output_type = 'text/x-html-safe',
            allowable_content_types = zconf.ATDocument.allowed_content_types,
            widget = RichLabelWidget(),
            )

    security.declareProtected(ModifyPortalContent, 'setFgDefault')
    def setFgDefault(self, value, **kw):
        """ set default for field """

        self.fgField.default = value

    def getRawFgDefault(self, **kw):
        """ get default for field """

        return self.fgField.default

    security.declareProtected(View, 'isBinary')
    def isBinary(self, key):
        return False;

    security.declareProtected(View, 'getContentType')
    def getContentType(self, key=None):
        return 'text/html'

registerATCT(FGRichLabelField, PROJECTNAME)


class FGFileField(BaseFormField):
    """ File (upload) field """

    security  = ClassSecurityInfo()

    schema = BaseFieldSchema.copy() + Schema((
        StringField('fgMaxMB',
            searchable=0,
            required=0,
            default=0,
            widget=IntegerWidget(
                label='Maximum Upload Size (Megabytes)',
                description="""Set to 0 for no limit.""",
                label_msgid = "label_filemaxmb_text",
                description_msgid = "help_filemaxmb_text",
                i18n_domain = "ploneformgen",
                ),
        ),

    ))
    finalizeATCTSchema(schema, folderish=True, moveDiscussion=False)

    # Standard content type setup
    portal_type = meta_type = 'FormFileField'
    archetype_name = 'File Field'
    content_icon = 'FileField.gif'
    typeDescription= 'A file field'

    def __init__(self, oid, **kwargs):
        """ initialize class """

        BaseFormField.__init__(self, oid, **kwargs)

        # set a preconfigured field as an instance attribute
        self.fgField = FileField('fg_file_field',
            searchable=0,
            required=0,
            write_permission = View,
            accessor = 'nullAccessor',
            maxsize=0,
            validators=('isMaxSize',),
            )

    security.declareProtected(ModifyPortalContent, 'setFgMaxMB')
    def setFgMaxMB(self, value, **kw):
        """ set the maxmb """

        self.fgField.maxsize = int(value)

    security.declareProtected(View, 'getFgMaxMB')
    def getFgMaxMB(self, **kw):
        """ get the maxmb """

        return self.fgField.maxsize


registerATCT(FGFileField, PROJECTNAME)
