/**
 * \file libyasm/errwarn.h
 * \brief YASM error and warning reporting interface.
 *
 * \rcs
 * $Id: errwarn.h 1591 2006-07-13 03:25:38Z peter $
 * \endrcs
 *
 * \license
 *  Copyright (C) 2001  Peter Johnson
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *  - Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *  - Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND OTHER CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR OTHER CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * \endlicense
 */
#ifndef YASM_ERRWARN_H
#define YASM_ERRWARN_H

/** Warning classes (that may be enabled/disabled). */
typedef enum {
    YASM_WARN_GENERAL = 0,  /**< Non-specific warnings */
    YASM_WARN_UNREC_CHAR,   /**< Unrecognized characters (while tokenizing) */
    YASM_WARN_PREPROC,	    /**< Preprocessor warnings */
    YASM_WARN_ORPHAN_LABEL, /**< Label alone on a line without a colon */
    YASM_WARN_UNINIT_CONTENTS	/**< Uninitialized space in code/data section */
} yasm_warn_class;

/** Initialize any internal data structures. */
void yasm_errwarn_initialize(void);

/** Clean up any memory allocated by yasm_errwarn_initialize() or other
 * functions.
 */
void yasm_errwarn_cleanup(void);

/** Reporting point of internal errors.  These are usually due to sanity
 * check failures in the code.
 * \warning This function must NOT return to calling code; exit or longjmp
 *          instead.
 * \param file      source file (ala __FILE__)
 * \param line      source line (ala __LINE__)
 * \param message   internal error message
 */
extern /*@exits@*/ void (*yasm_internal_error_)
    (const char *file, unsigned int line, const char *message);

/** Easily-callable version of yasm_internal_error_().  Automatically uses
 * __FILE__ and __LINE__ as the file and line.
 * \param message   internal error message
 */
#define yasm_internal_error(message) \
    yasm_internal_error_(__FILE__, __LINE__, message)

/** Reporting point of fatal errors.
 * \warning This function must NOT return to calling code; exit or longjmp
 *          instead.
 * \param message   fatal error message
 * \param va	    va_list argument list for message
 */
extern /*@exits@*/ void (*yasm_fatal) (const char *message, va_list va);

/** Reporting point of fatal errors, with variable arguments (internal only).
 * \warning This function calls #yasm_fatal, and thus does not return to the
 *          calling code.
 * \param message   fatal error message
 * \param ...	    argument list for message
 */
/*@exits@*/ void yasm__fatal(const char *message, ...);

/** Log an error at a given line, displaying a different line.  va_list version
 * of yasm__error_at().
 * \internal
 * \param line      virtual line
 * \param displine  displayed virtual line
 * \param message   printf-like-format message
 * \param va	    argument list for message
 */
void yasm__error_va_at(unsigned long line, unsigned long displine,
		       const char *message, va_list va);

/** Log an error.  va_list version of yasm__error().
 * \internal
 * \param line      virtual line
 * \param message   printf-like-format message
 * \param va	    argument list for message
 */
#define yasm__error_va(line, message, va) \
    yasm__error_va_at(line, line, message, va)

/** Log a warning at a given line, displaying a different line.  va_list
 * version of yasm__warning_at().
 * \internal
 * \param wclass    warning class
 * \param line      virtual line
 * \param displine  displayed virtual line
 * \param message   printf-like-format message
 * \param va	    argument list for message
 */
void yasm__warning_va_at(yasm_warn_class wclass, unsigned long line,
			 unsigned long displine, const char *message,
			 va_list va);

/** Log a warning.  va_list version of yasm__warning().
 * \internal
 * \param wclass    warning class
 * \param line      virtual line
 * \param message   printf-like-format message
 * \param va	    argument list for message
 */
#define yasm__warning_va(wclass, line, message, va) \
    yasm__warning_va_at(wclass, line, line, message, va)

/** Log an error.  Does not print it out immediately; yasm_errwarn_output_all()
 * outputs errors and warnings.
 * \internal
 * \param line      virtual line
 * \param message   printf-like-format message
 * \param ...	    argument list for message
 */
void yasm__error(unsigned long line, const char *message, ...)
    /*@printflike@*/;

/** Log an error at a given line, displaying a different line.  Does not print
 * it out immediately; yasm_errwarn_output_all() outputs errors and warnings.
 * \internal
 * \param line      virtual line
 * \param displine  displayed virtual line
 * \param message   printf-like-format message
 * \param ...	    argument list for message
 */
void yasm__error_at(unsigned long line, unsigned long displine,
		    const char *message, ...) /*@printflike@*/;

/** Log a warning.  Does not print it out immediately;
 * yasm_errwarn_output_all() outputs errors and warnings.
 * \internal
 * \param wclass    warning class
 * \param line      virtual line
 * \param message   printf-like-format message
 * \param ...	    argument list for message
 */
void yasm__warning(yasm_warn_class wclass, unsigned long line,
		   const char *message, ...) /*@printflike@*/;

/** Log a warning at a given line, displaying a different line.  Does not print
 * it out immediately; yasm_errwarn_output_all() outputs errors and warnings.
 * \internal
 * \param wclass    warning class
 * \param line      virtual line
 * \param displine  displayed virtual line
 * \param message   printf-like-format message
 * \param ...	    argument list for message
 */
void yasm__warning_at(yasm_warn_class wclass, unsigned long line,
		      unsigned long displine, const char *message, ...)
    /*@printflike@*/;

/** Log a parser error.  Parser errors can be overwritten by non-parser errors
 * on the same line.
 * \internal
 * \param line      virtual line
 * \param message   parser error message
 */
void yasm__parser_error(unsigned long line, const char *message);

/** Enable a class of warnings.
 * \param wclass    warning class
 */
void yasm_warn_enable(yasm_warn_class wclass);

/** Disable a class of warnings.
 * \param wclass    warning class
 */
void yasm_warn_disable(yasm_warn_class wclass);

/** Disable all classes of warnings. */
void yasm_warn_disable_all(void);

/** Get total number of errors logged.
 * \param warning_as_error  if nonzero, warnings are treated as errors.
 * \return Number of errors.
 */
unsigned int yasm_get_num_errors(int warning_as_error);

/** Print out an error.
 * \param fn	filename of source file
 * \param line	line number
 * \param msg	error message
 */
typedef void (*yasm_print_error_func)
    (const char *fn, unsigned long line, const char *msg);

/** Print out a warning.
 * \param fn	filename of source file
 * \param line	line number
 * \param msg	warning message
 */
typedef void (*yasm_print_warning_func)
    (const char *fn, unsigned long line, const char *msg);

/** Outputs all errors and warnings.
 * \param lm	line map (to convert virtual lines into filename/line pairs)
 * \param warning_as_error  if nonzero, treat warnings as errors
 * \param print_error	    function called to print out errors
 * \param print_warning	    function called to print out warnings
 */
void yasm_errwarn_output_all
    (yasm_linemap *lm, int warning_as_error, yasm_print_error_func print_error,
     yasm_print_warning_func print_warning);

/** Convert a possibly unprintable character into a printable string.
 * \internal
 * \param ch	possibly unprintable character
 * \return Printable string representation (static buffer).
 */
char *yasm__conv_unprint(int ch);

/** Hook for library users to map to gettext() if GNU gettext is being used.
 * \param msgid	    message catalog identifier
 * \return Translated message.
 */
extern const char * (*yasm_gettext_hook) (const char *msgid);

#endif
