/*******************************************************************
 * Linux wireless extensions interface.
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * File: cardif_linux_wext.c
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: cardif_linux_wext.c,v 1.58 2006/02/23 22:26:53 chessing Exp $
 * $Date: 2006/02/23 22:26:53 $
 * $Log: cardif_linux_wext.c,v $
 * Revision 1.58  2006/02/23 22:26:53  chessing
 * Fix for bug id #1415020.  'Building Xsupplicant 1.2.3 Fails on FC4'.
 *
 * Revision 1.57  2006/01/24 04:42:26  chessing
 * A few more fixes to WPA code, along with a fix to the scan reaping code.
 *
 * Revision 1.56  2006/01/23 05:28:37  chessing
 * Fixed a few settings that were causing errors with IOCTLs on some cards.  Updated WPA2 code to properly process group key packets. We now record quality, signal level, and noise level from scan results, so that we can start to make better decisions on which AP to associate to.
 *
 * Revision 1.55  2006/01/19 05:37:04  chessing
 * WPA2 is working correctly.  Added the ability to query the card to gather encryption/authentication capabilities.  1.2.3 is now ready to go.
 *
 * Revision 1.54  2006/01/14 06:22:10  chessing
 * Fixed WPA functionality for cards that don't use IWAUTH to set up the WPA IE.
 *
 * Revision 1.53  2006/01/12 17:54:03  chessing
 * WPA almost works again on cards that use the GENIE setting.  Added a small fix to allow Xsupplicant to build on Open SuSE 10.
 *
 * Revision 1.52  2005/12/25 04:54:11  chessing
 * Fixup to code that added wext support needed for using ndiswrapper.
 *
 * Revision 1.51  2005/12/24 04:51:51  chessing
 * Removed an unneeded file.  Updated scanning code to parse IWEVGENIE events correctly, and make use of them.
 *
 * Revision 1.50  2005/12/24 02:39:48  chessing
 * Fixes to autoconf script to correctly identify the number of arguments that iw_extract_event_stream() takes, along with checking iwlib.h compiles correctly.
 *
 * Revision 1.49  2005/12/18 07:44:12  chessing
 * Added the ability to associate using IWAUTH options instead of setting a full IE.  This allows NDISwrapper to work with vanilla wireless extensions.  For some reason, we can't parse IWEVASSOCREQIE/IWEVASSOCRESPIE since we see the length as 0, even though iwevent sees the correct IE information. :-/  Need to figure out why.
 *
 * Revision 1.48  2005/12/03 22:18:51  chessing
 * Added an include to fix some problems when compiling with WE19.
 *
 * Revision 1.47  2005/11/14 17:27:39  chessing
 * Removed the driver specific code for hostap, ipw, and ndiswrapper, since they all support WE18 now.
 *
 * Revision 1.46  2005/11/10 04:56:54  chessing
 * Added patch from Ben Gardner to add support for setting a specific WEP key prior to attempting to associte.  (With a few slight modifications by me to make it fit in the current CVS code, and get it supported in config-parse.)  Added patch from Pekka Savola to fix some header ordering issues, and a potential buffer overflow.
 *
 * Revision 1.45  2005/10/17 03:56:54  chessing
 * Updates to the libxsupconfig library.  It no longer relies on other source from the main tree, so it can be used safely in other code with problems.
 *
 * Revision 1.44  2005/10/14 02:26:18  shaftoe
 * - cleanup gcc 4 warnings
 * - (re)add support for a pid in the form of /var/run/xsupplicant.<iface>.pid
 *
 * -- Eric Evans <eevans@sym-link.com>
 *
 * Revision 1.43  2005/10/13 18:46:47  chessing
 * Fixed to the Madwifi driver to allow it to do dynamic WEP again.  Fixed up the wext driver to behave correctly again. ;)
 *
 * Revision 1.42  2005/09/19 21:45:46  chessing
 * Fix for the madwifi driver when it connects to certain APs to do WEP.  (Currently the only known one with this problem is the Trapeze APs when they are running MSS 4.x+.)
 *
 * Revision 1.41  2005/09/17 18:38:17  chessing
 * Some updates to make the madwifi driver work with dynamic WEP again.
 *
 * Revision 1.40  2005/09/14 02:50:44  chessing
 * Major updates.  Auto association now works.  Started to rewrite the rtnetlink pieces using iwlib from the wireless tools to avoid compatibility issues.  As a result, getting the WPA and RSN IEs via the IWEVGENIE event no longer works for some reason, but the old style using IWEVCUSTOM events should still work like a champ.
 *
 * Revision 1.39  2005/09/08 16:27:02  chessing
 * Some small updates to the new state machine code.  First attempt at an auto association mode.  (It mostly works. ;)
 *
 * Revision 1.38  2005/09/05 01:00:37  chessing
 * Major overhaul to most of the state machines in Xsupplicant.  Also added additional error messages to the TLS functions to try to debug the one of the problems reported on the list.  Basic testing shows this new code to be more stable than previous code, but it needs more testing.
 *
 * Revision 1.37  2005/08/20 19:06:54  chessing
 * Patch from Carsten Grohmann to fix a few things in xsup_get_state.c.  Also added the ability to define an empty network clause, that will set the card in to encryption disabled mode.  From there, anything short of changing the SSID will be ignored by Xsupplicant.
 *
 * Revision 1.36  2005/08/09 01:39:15  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

#ifdef LINUX_FRAMER

// Use kernel headers.
#define HEADERS_KERNEL

#include <string.h>
#include <strings.h>
#include <errno.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <netinet/in.h>
#include <linux/types.h>
#include <linux/socket.h>
#include <linux/compiler.h>
#include <linux/wireless.h>
#include <iwlib.h>
#include <linux/if_packet.h>
#include <linux/netlink.h>

#include "profile.h"
#include "xsupconfig.h"
#include "config_ssid.h"
#include "xsup_debug.h"
#include "xsup_err.h"
#include "wpa.h"
#include "wpa2.h"
#include "wpa_common.h"
#include "cardif/cardif.h"
#include "cardif/linux/cardif_linux.h"
#include "cardif/linux/cardif_linux_wext.h"
#include "wireless_sm.h"
#include "cardif/linux/cardif_linux_rtnetlink.h"

// Old versions of wireless.h may not have this defined.
#ifndef IW_ENCODE_TEMP
#define IW_ENCODE_TEMP            0x0400
#endif

/**************************************************************
 *
 * Tell the wireless card to start scanning for wireless networks.
 *
 **************************************************************/
int cardif_linux_wext_scan(struct interface_data *thisint, char passive)
{
  struct lin_sock_data *sockData;
  struct iwreq iwr;
#if WIRELESS_EXT > 17
  struct iw_scan_req iwsr;
#endif

  if (!TEST_FLAG(thisint->flags, IS_WIRELESS))
    {
      debug_printf(DEBUG_INT, "%s is not a wireless interface!\n", 
		   thisint->intName);
      return XENOWIRELESS;
    }

  sockData = thisint->sockData;

  if (sockData->sockInt <= 0)
    return XENOSOCK;

  cardif_linux_wext_wpa_state(thisint, 1);

  debug_printf(DEBUG_INT, "Issuing %s scan request for interface %s!\n",
	       passive ? "passive":"active", thisint->intName);

#if WIRELESS_EXT > 17  
  // Build our extended scan structure.
  bzero(&iwsr, sizeof(iwsr));

  if (passive)
    {
      iwsr.scan_type = IW_SCAN_TYPE_PASSIVE;
      //      iwsr.scan_type = IW_SCAN_TYPE_ACTIVE;

      // If we are doing a passive scan, then we only care about other APs
      // that are on this SSID.  Otherwise, we might end up picking an SSID
      // later that isn't in the same layer2/3 space.
      //      iwr.u.data.flags = IW_SCAN_THIS_ESSID | IW_SCAN_ALL_FREQ | 
      //	IW_SCAN_THIS_MODE | IW_SCAN_ALL_RATE;

      //      iwr.u.data.flags = IW_SCAN_DEFAULT;
      iwr.u.data.flags = IW_SCAN_THIS_ESSID;
    }
  else
    {
      iwsr.scan_type = IW_SCAN_TYPE_ACTIVE;
      iwr.u.data.flags = IW_SCAN_DEFAULT;
    }

  // We aren't looking for a specific BSSID.
  memset(iwsr.bssid.sa_data, 0xff, 6);

  iwr.u.data.length = sizeof(iwsr);
  iwr.u.data.pointer = (caddr_t) &iwsr;

#else
  iwr.u.data.flags = IW_SCAN_DEFAULT;
#endif
  
  strcpy((char *)&iwr.ifr_name, thisint->intName);

  if (ioctl(sockData->sockInt, SIOCSIWSCAN, &iwr) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Error with SCAN ioctl!  (Perhaps your card "
		   "doesn't support scanning, or isn't up?)\n");

      return -1;
    }

  SET_FLAG(thisint->flags, SCANNING);

  return XENONE;
}

/*************************************************************
 *
 * Set all of the keys to 0s.
 *
 *************************************************************/
void cardif_linux_wext_set_zero_keys(struct interface_data *thisint)
{
  char zerokey[13];
  char keylen = 13;

  debug_printf(DEBUG_INT, "Setting keys to zeros!\n");

  bzero(zerokey, 13);

  // We set the key index to 0x80, to force key 0 to be set to all 0s,
  // and to have key 0 be set as the default transmit key.
  cardif_set_wep_key(thisint, (u_char *)&zerokey, keylen, 0x80);
  cardif_set_wep_key(thisint, (u_char *)&zerokey, keylen, 0x01);
  cardif_set_wep_key(thisint, (u_char *)&zerokey, keylen, 0x02);
  cardif_set_wep_key(thisint, (u_char *)&zerokey, keylen, 0x03);
}

/**************************************************************
 *
 * If we have detected, or forced this interface to reset keys, then
 * we need to reset them.  Otherwise, we will just ignore the fact that
 * we changed APs, and return.
 *
 **************************************************************/
void cardif_linux_wext_zero_keys(struct interface_data *thisint)
{
  struct config_network *network_data;

  network_data = config_get_network_config();

  if (network_data == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid network configuration structure! "
		   "(%s:%d)\n", __FUNCTION__, __LINE__);
      return;
    }

  if (network_data->wireless_ctrl == CTL_NO) 
    {
      debug_printf(DEBUG_INT, "Config file has instructed us not to reset the key!  Roaming may not work!!!\n");
      return;
    }

  if (thisint->flags & ROAMED)
    {
      return;
    }

  SET_FLAG(thisint->flags, ROAMED);

  cardif_linux_wext_set_zero_keys(thisint);
  cardif_linux_wext_enc_open(thisint);
}

/**************************************************************
 *
 * Disable encryption on the wireless card.  This is used in cases
 * where we roam to a different AP and the card needs to have WEP
 * disabled.
 *
 **************************************************************/
int cardif_linux_wext_enc_disable(struct interface_data *thisint)
{
  int rc = 0;
  int skfd;
  struct iwreq wrq;
  struct config_network *network_data;

  network_data = config_get_network_config();
  
  if (network_data == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid network configuration structure! "
		   "(%s:%d)\n", __FUNCTION__, __LINE__);
      return XEBADCONFIG;
    }

  if (network_data->wireless_ctrl == CTL_NO) 
    {
      debug_printf(DEBUG_INT, "Config file has instructed us not to reset the"
		   " key!  Roaming may not work!!!\n");
      return -1;
    }

  bzero((struct iwreq *)&wrq, sizeof(struct iwreq));

  skfd = socket(AF_INET, SOCK_DGRAM, 0);
  if (skfd < 0)
    return -1;

  strncpy(wrq.ifr_name, thisint->intName, IFNAMSIZ);

  // We got some data, so see if we have encryption or not.
  wrq.u.encoding.flags = IW_ENCODE_DISABLED;
  wrq.u.encoding.length = 0;
  wrq.u.encoding.pointer = (caddr_t)NULL;

  rc = ioctl(skfd, SIOCSIWENCODE, &wrq);
  if (rc < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't disable encryption!\n");
    } else {
      debug_printf(DEBUG_INT, "Encryption disabled!\n");
    }
 
  close(skfd);
  return rc;
}

int cardif_linux_wext_get_wpa2_ie(struct interface_data *thisint, 
				  char *iedata, int *ielen)
{
#if WIRELESS_EXT > 17

  // Should we use capabilities here?
  wpa2_gen_ie(thisint, iedata, ielen);

  debug_printf(DEBUG_INT, "Setting WPA2 IE : ");
  debug_hex_printf(DEBUG_INT, (u_char *)iedata, *ielen);
  debug_printf(DEBUG_INT, "\n");
#else
  debug_printf(DEBUG_NORMAL, "WPA2 isn't implemented in this version of the "
	       "wireless extensions!  Please upgrade to the latest version "
	       "of wireless extensions, or specify the driver to use with the"
	       " -D option!\n");

  iedata = NULL;
  *ielen = 0;
#endif
  return XENONE;
}

int cardif_linux_wext_get_wpa_ie(struct interface_data *thisint, 
				 char *iedata, int *ielen)
{
#if WIRELESS_EXT > 17

  // Should we use capabilities here?
  wpa_gen_ie(thisint, iedata);
  *ielen = 24;

  debug_printf(DEBUG_INT, "Setting WPA IE : ");
  debug_hex_printf(DEBUG_INT, (u_char *)iedata, *ielen);
  debug_printf(DEBUG_INT, "\n");
#else
  debug_printf(DEBUG_NORMAL, "WPA isn't implemented in this version of the "
	       "wireless extensions!  Please upgrade to the latest version "
	       "of wireless extensions, or specify the driver to use with the"
	       " -D option!\n");

  iedata = NULL;
  *ielen = 0;
#endif
  return XENONE;
}

/**************************************************************
 *
 * Set encryption on the wireless card to open.
 *
 **************************************************************/
int cardif_linux_wext_enc_open(struct interface_data *thisint)
{
  int rc = 0;
  int skfd;
  struct iwreq wrq;
  struct config_network *network_data;

  network_data = config_get_network_config();

  if (network_data == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid network configuration structure! "
		   "(%s:%d)\n", __FUNCTION__, __LINE__);
      return XEBADCONFIG;
    }

  if (network_data->wireless_ctrl == CTL_NO) 
    {
      debug_printf(DEBUG_INT, "Config file has instructed us not to reset the "
		   "key!  Roaming may not work!!!\n");
      return -1;
    }

  bzero((struct iwreq *)&wrq, sizeof(struct iwreq));

  skfd = socket(AF_INET, SOCK_DGRAM, 0);
  if (skfd < 0)
    return -1;

  strncpy(wrq.ifr_name, thisint->intName, IFNAMSIZ);

  // We got some data, so see if we have encryption or not.
  wrq.u.encoding.flags = IW_ENCODE_OPEN;
  wrq.u.encoding.length = 0;
  wrq.u.encoding.pointer = (caddr_t)NULL;

  rc = ioctl(skfd, SIOCSIWENCODE, &wrq);
  if (rc < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't disable encryption!\n");
    } else {
      debug_printf(DEBUG_INT, "Encryption set to Open!\n");
    }
 
  close(skfd);
  return rc;
}


/******************************************
 *
 * Set a WEP key.  Also, based on the index, we may change the transmit
 * key.
 *
 ******************************************/
int cardif_linux_wext_set_WEP_key(struct interface_data *thisint, u_char *key, 
				  int keylen, int index)
{
  int rc = 0;
  int skfd;
  struct iwreq wrq;
  struct config_network *network_data;

  network_data = config_get_network_config();

  if (network_data == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid network configuration structure! "
		   "(%s:%d)\n", __FUNCTION__, __LINE__);
      return XEBADCONFIG;
    }

  if (!(thisint->flags & IS_WIRELESS))
    {
      if ((cardif_int_is_wireless(thisint->intName) != TRUE) ||
	  (network_data->type == WIRED) ||
	  (network_data->wireless_ctrl == CTL_NO))
	{
	  debug_printf(DEBUG_NORMAL, "Interface isn't wireless, but an attempt"
		       " to set a key was made!\n");
	  return XENOWIRELESS;
	} else {
	  thisint->flags |= IS_WIRELESS;
	}
    }

  skfd = socket(AF_INET, SOCK_DGRAM, 0);
  if (skfd < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't allocate socket!\n");
      return -1;
    }

  strncpy(wrq.ifr_name, thisint->intName, IFNAMSIZ);

  wrq.u.data.flags = ((index & 0x7f)+1);

  if (thisint->flags & DONT_USE_TEMP)
    wrq.u.data.flags |= IW_ENCODE_OPEN;
  else
    wrq.u.data.flags |= IW_ENCODE_OPEN | IW_ENCODE_TEMP;

  wrq.u.data.length = keylen;
  wrq.u.data.pointer = (caddr_t)key;

  if ((rc = ioctl(skfd, SIOCSIWENCODE, &wrq)) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Failed to set WEP key [%d], error %d : %s\n",
		   (index & 0x7f) + 1, errno, strerror(errno));

      rc = XENOKEYSUPPORT;
    } else {
      debug_printf(DEBUG_INT, "Successfully set WEP key [%d]\n",
		   (index & 0x7f)+1);

      if (index & 0x80)
	{
	  // This is a unicast key, use it for transmissions.
	  strncpy(wrq.ifr_name, thisint->intName, IFNAMSIZ);

	  wrq.u.data.flags = (((index & 0x7f) + 1) & IW_ENCODE_INDEX) | IW_ENCODE_NOKEY;
	  if (thisint->flags & DONT_USE_TEMP)
	    wrq.u.data.flags |= IW_ENCODE_OPEN;
	  else
	    wrq.u.data.flags |= IW_ENCODE_OPEN | IW_ENCODE_TEMP;

	  wrq.u.data.length = 0;
	  wrq.u.data.pointer = (caddr_t)NULL;

	  if (ioctl(skfd, SIOCSIWENCODE, &wrq) < 0)
	    {
	      debug_printf(DEBUG_NORMAL, "Failed to set the WEP transmit key ID [%d]\n", (index & 0x7f)+1);
	      rc = XENOKEYSUPPORT;
	    } else {
	      debug_printf(DEBUG_INT, "Successfully set the WEP transmit key [%d]\n", (index & 0x7f)+1);
	    }
	}  
    }
 
  close(skfd);
  return rc;
}

/**********************************************************
 *
 * Set the SSID of the wireless card.
 *
 **********************************************************/
int cardif_linux_wext_set_ssid(struct interface_data *thisint, char *ssid_name)
{
  struct iwreq iwr;
  struct lin_sock_data *sockData;
  char newssid[100];

  if (thisint == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface struct passed in to %s!\n",
		   __FUNCTION__);
      return XEGENERROR;
    }

  if (ssid_name == NULL)
    {
      debug_printf(DEBUG_NORMAL, "A NULL ssid was passed in to %s! "
		   "Authentication may not work!\n", __FUNCTION__);
      return -1;
    }

  sockData = thisint->sockData;
 
  if (!(thisint->flags & IS_WIRELESS))
    {
      // We want to verify that the interface is in fact, not wireless, and
      // not that we are in a situation where the interface has just been 
      // down.
      if (!(thisint->flags & WAS_DOWN))
	{
	  return XENOWIRELESS;
	}
    } 

  // If we get here, and isWireless == FALSE, then we need to double
  // check that our interface is really not wireless.
  if (!(thisint->flags & IS_WIRELESS))
    {
      if (cardif_int_is_wireless(thisint->intName) == TRUE)
	{
	  thisint->flags |= IS_WIRELESS;
	} else {
	  thisint->flags &= (~IS_WIRELESS);
	}

      if (!(thisint->flags & IS_WIRELESS))
	{
	  thisint->flags &= (~WAS_DOWN);
	}
    }

  // Specify the interface name we are asking about.
  strncpy(iwr.ifr_name, thisint->intName, sizeof(iwr.ifr_name));

  bzero(newssid, 100);
  strcpy(newssid, ssid_name);

  iwr.u.essid.pointer = (caddr_t) newssid;
  iwr.u.essid.length = strlen(newssid)+1;
  iwr.u.essid.flags = 1;

  if (ioctl(sockData->sockInt, SIOCSIWESSID, &iwr) < 0) return XENOWIRELESS;

  debug_printf(DEBUG_INT, "Requested SSID be set to '%s'\n", newssid);

  thisint->flags &= (~WAS_DOWN);

  return XENONE;
}

/******************************************
 *
 * Set the Broadcast SSID (MAC address) of the AP we are connected to.
 *
 ******************************************/
int cardif_linux_wext_set_bssid(struct interface_data *intdata, u_char *bssid)
{
  struct iwreq wrq;
  struct lin_sock_data *sockData;

  if ((intdata == NULL) || (bssid == NULL))
    {
      debug_printf(DEBUG_NORMAL, "Invalid data passed to %s!\n", __FUNCTION__);
      return -1;
    }

  sockData = intdata->sockData;

  if (sockData == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid socket data!! (%s:%d)\n", 
		   __FUNCTION__, __LINE__);
      return XENOSOCK;
    }

  bzero(&wrq, sizeof(wrq));

  strncpy((char *)&wrq.ifr_name, intdata->intName, IFNAMSIZ);
  memcpy(&wrq.u.ap_addr.sa_data, bssid, 6);
  wrq.u.ap_addr.sa_family = ARPHRD_ETHER;

  if (ioctl(sockData->sockInt, SIOCSIWAP, &wrq) < 0)
    {
      // If we couldn't set the BSSID, it isn't the end of the world.  The
      // driver just may not need it.
      debug_printf(DEBUG_NORMAL, "Error setting BSSID!  We may not associate/"
      		   "authenticate correctly!\n");
      return XESOCKOP;
    }

  return XENONE;
}


/******************************************
 *
 * Get the Broadcast SSID (MAC address) of the Access Point we are connected 
 * to.  If this is not a wireless card, or the information is not available,
 * we should return an error.
 *
 ******************************************/
int cardif_linux_wext_get_bssid(struct interface_data *thisint, 
				char *bssid_dest)
{
  struct iwreq iwr;
  struct lin_sock_data *sockData;

  if ((thisint == NULL) || (bssid_dest == NULL))
    {
      debug_printf(DEBUG_NORMAL, "Invalid data passed to %s!\n", __FUNCTION__);
      return -1;
    }

  // If we are a wired interface, don't bother.
  if (!TEST_FLAG(thisint->flags, IS_WIRELESS)) return XENONE;

  sockData = thisint->sockData;

  if (sockData == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid sockData in %s:%d\n", __FUNCTION__,
		   __LINE__);
      debug_printf(DEBUG_NORMAL, "FATAL ERROR!\n");
      exit(1);
    }

  // Specify the interface name we are asking about.
  strncpy(iwr.ifr_name, thisint->intName, sizeof(iwr.ifr_name));

  if (ioctl(sockData->sockInt, SIOCGIWAP, &iwr) < 0) 
    {
      debug_printf(DEBUG_NORMAL, "Couldn't get MAC address for AP!\n");
      return XENOWIRELESS;
    }

  memcpy(bssid_dest, iwr.u.ap_addr.sa_data, 6);
  return XENONE;
}

/******************************************
 *
 * Ask the wireless card for the ESSID that we are currently connected to.  If
 * this is not a wireless card, or the information is not available, we should
 * return an error.
 *
 ******************************************/
int cardif_linux_wext_get_ssid(struct interface_data *thisint, char *ssid_name)
{
  struct iwreq iwr;
  struct lin_sock_data *sockData;
  char newssid[100];

  if (thisint == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface structure!  (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      return -1;
    }

  if (ssid_name == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid bucket for ssid_name!  (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      return -1;
    }

  sockData = thisint->sockData;
 
  if (!TEST_FLAG(thisint->flags, IS_WIRELESS))
    {
      // We want to verify that the interface is in fact, not wireless, and
      // not that we are in a situation where the interface has just been 
      // down.
      debug_printf(DEBUG_NORMAL, "This interface isn't wireless!\n");
      return XENOWIRELESS;
    } 

  // If we get here, and isWireless == FALSE, then we need to double
  // check that our interface is really not wireless.
  if (!(thisint->flags & IS_WIRELESS))
    {
      if (cardif_int_is_wireless(thisint->intName) == TRUE)
	{
	  thisint->flags |= IS_WIRELESS;
	} else {
	  thisint->flags &= (~IS_WIRELESS);
	}

      if (!(thisint->flags & IS_WIRELESS))
	{
	  thisint->flags &= (~WAS_DOWN);
	}
    }

  // Specify the interface name we are asking about.
  strncpy(iwr.ifr_name, thisint->intName, sizeof(iwr.ifr_name));

  bzero(newssid, 100);
  iwr.u.essid.pointer = (caddr_t) newssid;
  iwr.u.essid.length = 100;
  iwr.u.essid.flags = 0;

  if (ioctl(sockData->sockInt, SIOCGIWESSID, &iwr) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't get ESSID!\n");
      debug_printf(DEBUG_NORMAL, "Error (%d) : %s\n", errno, strerror(errno));
      return XENOWIRELESS;
    }

  thisint->flags &= (~WAS_DOWN);

  strcpy(ssid_name, newssid);

  return XENONE;
}

/***********************************************************************
 *
 *  This function is called when we roam, or disassociate.  It should
 *  reset the card to a state where it can associate with a new AP.
 *
 ***********************************************************************/
int cardif_linux_wext_wep_associate(struct interface_data *intdata, 
				    int zero_keys)
{
  char *bssid;
  struct config_globals *globals;

  cardif_linux_wext_wpa_state(intdata, 0);

  cardif_linux_wext_set_ssid(intdata, intdata->cur_essid);

  if (zero_keys == 0)
    {
      debug_printf(DEBUG_INT, "WEP: turning encryption off.\n");
      return cardif_linux_wext_enc_disable(intdata);
    } else if (zero_keys == 1)
      {
      debug_printf(DEBUG_INT, "WEP: zero keys.\n");
	cardif_linux_wext_zero_keys(intdata);
	return XENONE;
      } else
	{
	  debug_printf(DEBUG_NORMAL, "Invalid association value.\n");
	}

  bssid = (char *) config_ssid_get_mac();
  if (bssid != NULL)
    {
      debug_printf(DEBUG_INT, "Dest. BSSID : ");
      debug_hex_printf(DEBUG_INT, (u_char *) bssid, 6);
    }

  globals = config_get_globals();

  if ((!globals) || (!TEST_FLAG(globals->flags, CONFIG_GLOBALS_FIRMWARE_ROAM)))
    {
      cardif_linux_wext_set_bssid(intdata, bssid);
    }

  return XENOTHING_TO_DO;
}

#if WIRELESS_EXT > 17
int cardif_linux_wext_mlme(struct interface_data *thisint, uint16_t mlme_type,
			   uint16_t mlme_reason)
{
  struct iwreq iwr;
  struct lin_sock_data *sockData;
  struct iw_mlme iwm;

  sockData = thisint->sockData;

  // If we get here, and isWireless == FALSE, then we need to double
  // check that our interface is really not wireless.
  if (!(thisint->flags & IS_WIRELESS))
    {
      if (cardif_int_is_wireless(thisint->intName) == TRUE)
	{
	  thisint->flags |= IS_WIRELESS;
	} else {
	  thisint->flags &= (~IS_WIRELESS);
	}

      if (!(thisint->flags & IS_WIRELESS))
	{
	  thisint->flags &= (~WAS_DOWN);
	}
    }  

  memset(&iwr, 0, sizeof(iwr));
  // Specify the interface name we are asking about.
  strncpy(iwr.ifr_name, thisint->intName, sizeof(iwr.ifr_name));

  memset(&iwm, 0, sizeof(iwm));
  
  // Set up our MLME struct.
  iwm.cmd = mlme_type;
  iwm.reason_code = mlme_reason;
  iwm.addr.sa_family = ARPHRD_ETHER;

  // Need to specify the MAC address that we want to do this MLME for.
  memcpy(iwm.addr.sa_data, thisint->source_mac, 6);
  iwr.u.data.pointer = (caddr_t)&iwm;
  iwr.u.data.length = sizeof(iwm);

  if (ioctl(sockData->sockInt, SIOCSIWMLME, &iwr))
    {
      debug_printf(DEBUG_NORMAL, "Couldn't issue MLME request!\n");
    }
  
  return XENONE;
}
#endif

int cardif_linux_wext_disassociate(struct interface_data *intdata, int reason)
{
#if WIRELESS_EXT > 17
  cardif_linux_wext_mlme(intdata, IW_MLME_DISASSOC, reason);
#endif
  return XENONE;
}

int cardif_linux_wext_set_key_ext(struct interface_data *intdata, int alg, 
				   unsigned char *addr, int keyidx, int settx, 
				   char *seq,  int seqlen, char *key, 
				   int keylen)
{
#if WIRELESS_EXT > 17
  struct iwreq wrq;
  struct lin_sock_data *sockData;
  struct iw_encode_ext *iwee;

  sockData = intdata->sockData;

  bzero(&wrq, sizeof(wrq));
  strncpy((char *)&wrq.ifr_name, intdata->intName, IFNAMSIZ);

  // Allocate enough memory to hold our iw_encode_ext struct, and the
  // key itself.
  iwee = (struct iw_encode_ext *)malloc(sizeof(struct iw_encode_ext) + keylen);

  bzero(iwee, sizeof(struct iw_encode_ext));

  iwee->alg = alg;
  iwee->ext_flags = keyidx+1;

  if ((seq != NULL) && (seqlen > 0))
    {
      iwee->ext_flags |= IW_ENCODE_EXT_RX_SEQ_VALID;
      memcpy(iwee->rx_seq, seq, seqlen);
      debug_printf(DEBUG_INT, "SEQ (%d) : ", seqlen);
      debug_hex_printf(DEBUG_INT, seq, seqlen);
    }
  
  if (settx) 
    {
      iwee->ext_flags |= IW_ENCODE_EXT_SET_TX_KEY;
      if ((addr != NULL) && 
	  (memcmp(addr, "\0xff\0xff\0xff\0xff\0xff\0xff", 6) != 0))
	{
	  memcpy(iwee->addr.sa_data, addr, 6);
	} else {
	  memcpy(iwee->addr.sa_data, intdata->dest_mac, 6);
	}
    } else {
      iwee->ext_flags |= IW_ENCODE_EXT_GROUP_KEY;
      memset(iwee->addr.sa_data, 0xff, 6);
    }
  iwee->addr.sa_family = ARPHRD_ETHER;

  iwee->key_len = keylen;
  memcpy(iwee->key, key, keylen);

  if (key)
    {
      debug_printf(DEBUG_INT, "Setting key : ");
      debug_hex_printf(DEBUG_INT, iwee->key, keylen);
    }

  wrq.u.encoding.pointer = (caddr_t)iwee;
  wrq.u.encoding.flags = (keyidx + 1);
  wrq.u.encoding.length = sizeof(struct iw_encode_ext) + keylen;

  if (ioctl(sockData->sockInt, SIOCSIWENCODEEXT, &wrq) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Error setting key!! (IOCTL "
		   "failure.)\n");
      debug_printf(DEBUG_NORMAL, "Error %d : %s\n", errno, strerror(errno));
    }
#else
  debug_printf(DEBUG_NORMAL, "%s : Not supported by WE(%d)!\n", __FUNCTION__,
	       WIRELESS_EXT);
#endif
  return XENONE;
}


int cardif_linux_wext_set_tkip_key(struct interface_data *intdata, 
				   unsigned char *addr, int keyidx, int settx, 
				   char *seq,  int seqlen, char *key, 
				   int keylen)
{
#if WIRELESS_EXT > 17
  return cardif_linux_wext_set_key_ext(intdata, IW_ENCODE_ALG_TKIP, addr,
				       keyidx, settx, seq, seqlen, key, 
				       keylen);
#else
  debug_printf(DEBUG_NORMAL, "%s : Not supported by WE(%d)!\n", __FUNCTION__,
	       WIRELESS_EXT);
#endif
  return XENONE;
}

int cardif_linux_wext_set_ccmp_key(struct interface_data *intdata,
				   unsigned char *addr, int keyidx, int settx,
				   char *seq, int seqlen, char *key,
				   int keylen)
{
#if WIRELESS_EXT > 17
  return cardif_linux_wext_set_key_ext(intdata, IW_ENCODE_ALG_CCMP, addr,
				       keyidx, settx, seq, seqlen, key,
				       keylen);
#else
  debug_printf(DEBUG_NORMAL, "%s : Not supported by WE(%d)!\n", __FUNCTION__,
	       WIRELESS_EXT);
#endif
  return XENONE;
}

int cardif_linux_wext_wpa(struct interface_data *intdata, char state)
{
#if WIRELESS_EXT > 17
  struct iwreq wrq;
  struct lin_sock_data *sockData;

  sockData = intdata->sockData;

  bzero(&wrq, sizeof(wrq));
  strncpy((char *)&wrq.ifr_name, intdata->intName, IFNAMSIZ);

  wrq.u.param.flags = IW_AUTH_WPA_ENABLED & IW_AUTH_INDEX;
  wrq.u.param.value = state;

  if (ioctl(sockData->sockInt, SIOCSIWAUTH, &wrq) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Error setting WPA state enable/disable!\n");
      return -1;
    }
#endif
  return XENONE;
}

int cardif_linux_wext_set_wpa_ie(struct interface_data *intdata, unsigned char *wpaie,
				 unsigned int wpalen)
{
#if WIRELESS_EXT > 17
  struct iwreq wrq;
  struct lin_sock_data *sockData;

  sockData = intdata->sockData;

  bzero (&wrq, sizeof(wrq));

  strncpy((char *)&wrq.ifr_name, intdata->intName, IFNAMSIZ);

  wrq.u.data.pointer = (caddr_t) wpaie;
  wrq.u.data.length = wpalen;
  wrq.u.data.flags = 0;

  if (ioctl(sockData->sockInt, SIOCSIWGENIE, &wrq) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Error setting WPA IE!\n");
    } 
#endif
  return XENONE;
}


int cardif_linux_wext_wpa_state(struct interface_data *intdata, char state)
{

  // If we have wireless extensions 18 or higher, we can support WPA/WPA2
  // with standard ioctls.

#if WIRELESS_EXT > 17
  char wpaie[24];

  if (state)
    {
      // Enable WPA if the interface doesn't already have it.
      cardif_linux_wext_wpa(intdata, TRUE);
    } else {
      cardif_linux_wext_wpa(intdata, FALSE);

      // Otherwise, make sure we don't have an IE set.
      memset(wpaie, 0, sizeof(wpaie));
      if (cardif_linux_wext_set_wpa_ie(intdata, (unsigned char *)wpaie, 0) < 0)
	{
	  debug_printf(DEBUG_NORMAL, "Couldn't clear WPA IE on device %s!\n",
		       intdata->intName);
	}
    }
#endif

  return XENONE;
}

int cardif_linux_wext_unencrypted_eapol(struct interface_data *intdata,
					int state)
{
#if WIRELESS_EXT > 17
  struct iwreq wrq;
  struct lin_sock_data *sockData;

  sockData = intdata->sockData;

  bzero(&wrq, sizeof(wrq));
  strncpy((char *)&wrq.ifr_name, intdata->intName, IFNAMSIZ);

  wrq.u.param.flags = IW_AUTH_RX_UNENCRYPTED_EAPOL & IW_AUTH_INDEX;
  wrq.u.param.value = state;

  if (ioctl(sockData->sockInt, SIOCSIWAUTH, &wrq) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Error setting allow unencrypted EAPOL! "
		   "It is possible that your driver is not yet compatible "
		   "with wireless extensions 17 or higher.\n");
      return -1;
    } else {
      debug_printf(DEBUG_INT, "Set allowing of unencrypted EAPOL!\n");
    }
#endif
  return XENONE;
}

/*************************************************************************
 *
 * Set values for IWAUTH.
 *
 *************************************************************************/
int cardif_linux_wext_set_iwauth(struct interface_data *intdata,
				 int setting, uint32_t value, 
				 char *setting_name)
{
#if WIRELESS_EXT > 17
  struct iwreq wrq;
  struct lin_sock_data *sockData;

  sockData = intdata->sockData;

  bzero(&wrq, sizeof(wrq));
  strncpy((char *)&wrq.ifr_name, intdata->intName, IFNAMSIZ);

  wrq.u.param.flags = setting & IW_AUTH_INDEX;
  wrq.u.param.value = value;

  if (ioctl(sockData->sockInt, SIOCSIWAUTH, &wrq) < 0)
    {
      if (errno != ENOTSUP)
	{
	  debug_printf(DEBUG_NORMAL, "Error changing setting for '%s'! "
		       "It is possible that your driver does not support the "
		       "needed functionality.\n", setting_name);
	  return -1;
	}
    }
#endif
  return XENONE;
}

/***********************************************************************
 *
 * Convert our cipher designator to something that will be understood by
 * the linux wireless extensions.
 *
 ***********************************************************************/
int cardif_linux_wext_iw_cipher(int cipher)
{
#if WIRELESS_EXT > 17
  switch (cipher)
    {
    case CIPHER_NONE:
      return 0;
      break;

    case CIPHER_WEP40:
      return IW_AUTH_CIPHER_WEP40;
      break;

    case CIPHER_TKIP:
      return IW_AUTH_CIPHER_TKIP;
      break;

    case CIPHER_WRAP:
      debug_printf(DEBUG_NORMAL, "WRAP is not supported!\n");
      return -1;
      break;

    case CIPHER_CCMP:
      return IW_AUTH_CIPHER_CCMP;
      break;
      
    case CIPHER_WEP104:
      return IW_AUTH_CIPHER_WEP104;
      break;

    default:
      debug_printf(DEBUG_NORMAL, "Unknown cipher value of %d!\n", cipher);
      return -1;
      break;
    }
#else
  return -1;
#endif
}

/*********************************************************************
 *
 * Set all of the card settings that are needed in order to complete an
 * association, so that we can begin the authentication.
 *
 *********************************************************************/
void cardif_linux_wext_associate(struct interface_data *intdata, char *newssid)
{
  char *bssid;
#if WIRELESS_EXT > 17
  int len, akm = 0;
  uint32_t cipher;
  u_char wpaie[255];
#endif
  struct config_network *network_data;
  struct config_globals *globals;

  if (!intdata)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface data passed to %s!\n",
		   __FUNCTION__);
      return;
    }

  network_data = config_get_network_config();
  
  if (!network_data)
    {
      debug_printf(DEBUG_NORMAL, "Invalid network data in %s():%d\n",
		   __FUNCTION__, __LINE__);
      return;
    }

  if (config_ssid_get_ssid_abilities() & RSN_IE)
    {
#if WIRELESS_EXT > 17
      cardif_linux_wext_get_wpa2_ie(intdata, (char *) wpaie, &len);
      if (cardif_linux_wext_set_iwauth(intdata, IW_AUTH_WPA_VERSION,
				       IW_AUTH_WPA_VERSION_WPA2,
				       "WPA version") < 0)
	{
	  debug_printf(DEBUG_NORMAL, "Couldn't set WPA version!\n");
	}
#endif
    } else if (config_ssid_get_ssid_abilities() & WPA_IE)
      {
#if WIRELESS_EXT > 17
	cardif_linux_wext_get_wpa_ie(intdata, (char *) wpaie, &len);
	if (cardif_linux_wext_set_iwauth(intdata, IW_AUTH_WPA_VERSION,
					 IW_AUTH_WPA_VERSION_WPA,
					 "WPA2 version") < 0)
	  {
	    debug_printf(DEBUG_NORMAL, "Couldn't set WPA2 version!\n");
	  }
#endif
      }

#if WIRELESS_EXT > 17
  // If the length of the WPA/WPA2 IE is > 0.
  if (len > 0)
    {
      // Turn off countermeasures, if they were running.
      if (cardif_linux_wext_set_iwauth(intdata, IW_AUTH_TKIP_COUNTERMEASURES,
				       FALSE, "tkip countermeasures") < 0)
	{
	  debug_printf(DEBUG_NORMAL, "Couldn't disable TKIP countermeasures!\n");
	}

      if (cardif_linux_wext_set_iwauth(intdata, IW_AUTH_DROP_UNENCRYPTED,
				       TRUE, "drop unencrypted data") < 0)
	{
	  debug_printf(DEBUG_NORMAL, "Couldn't enable dropping of unencrypted"
		       " data!\n");
	}

      if (cardif_linux_wext_set_iwauth(intdata, IW_AUTH_80211_AUTH_ALG,
				       IW_AUTH_ALG_OPEN_SYSTEM, "802.11 auth."
				       " alg to open") < 0)
	{
	  debug_printf(DEBUG_NORMAL, "Couldn't set 802.11 auth. alg to open.\n");
	}

      if (cardif_linux_wext_set_wpa_ie(intdata, wpaie, len) < 0)
	{
	  debug_printf(DEBUG_NORMAL, "Couldn't set WPA IE on device %s!\n",
		       intdata->intName);
	}

      // For drivers that require something other than just setting a
      // WPA IE, we will set the components for the IE instead.
      cipher = cardif_linux_wext_iw_cipher(network_data->wpa_pairwise_crypt);
      if (cardif_linux_wext_set_iwauth(intdata, IW_AUTH_CIPHER_PAIRWISE,
       				       cipher, "pairwise cipher") < 0)
	{
	  debug_printf(DEBUG_NORMAL, "Couldn't set pairwise cipher.\n");
	}

      cipher = cardif_linux_wext_iw_cipher(network_data->wpa_group_crypt);
      if (cardif_linux_wext_set_iwauth(intdata, IW_AUTH_CIPHER_GROUP,
      				       cipher, "group cipher") < 0)
	{
	  debug_printf(DEBUG_NORMAL, "Couldn't set group cipher.\n");
	}

      if (network_data->methods->method_num != WPA_PSK)
	{
	  akm = IW_AUTH_KEY_MGMT_802_1X;
	} else {
	  akm = IW_AUTH_KEY_MGMT_PSK;
	}

      if (cardif_linux_wext_set_iwauth(intdata, IW_AUTH_KEY_MGMT, akm,
				       "Authenticated Key Management Suite") 
	  < 0)
	{
	  debug_printf(DEBUG_NORMAL, "Couldn't set Authenticated Key "
		       "Management Suite.\n");
	}
    }
#endif

  cardif_linux_wext_set_ssid(intdata, newssid);

  bssid = (char *) config_ssid_get_mac();
  if (bssid != NULL)
    {
      debug_printf(DEBUG_INT, "Dest. BSSID : ");
      debug_hex_printf(DEBUG_INT, (u_char *) bssid, 6);
    }

  globals = config_get_globals();

  if ((!globals) || (!TEST_FLAG(globals->flags, CONFIG_GLOBALS_FIRMWARE_ROAM)))
    {
      cardif_linux_wext_set_bssid(intdata, bssid);
    }
  
  if ((intdata->wpa_ie == NULL) && (intdata->rsn_ie == NULL))
    {
      // We need to set up the card to allow unencrypted EAPoL frames.
      cardif_linux_wext_unencrypted_eapol(intdata, TRUE);
    }
  
  return;
}

int cardif_linux_wext_countermeasures(struct interface_data *intdata,
				      char endis)
{
#if WIRELESS_EXT > 17
  struct iwreq wrq;
  struct lin_sock_data *sockData;

  sockData = intdata->sockData;

  bzero(&wrq, sizeof(wrq));
  strncpy((char *)&wrq.ifr_name, intdata->intName, IFNAMSIZ);

  wrq.u.param.flags = IW_AUTH_TKIP_COUNTERMEASURES & IW_AUTH_INDEX;
  wrq.u.param.value = endis;

  if (ioctl(sockData->sockInt, SIOCSIWAUTH, &wrq) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Error setting WPA countermeasures enable/disable!\n");
      return -1;
    }
#endif
  return XENONE;
}

int cardif_linux_wext_drop_unencrypted(struct interface_data *intdata,
				       char endis)
{
#if WIRELESS_EXT > 17
  struct iwreq wrq;
  struct lin_sock_data *sockData;

  sockData = intdata->sockData;

  bzero(&wrq, sizeof(wrq));
  strncpy((char *)&wrq.ifr_name, intdata->intName, IFNAMSIZ);

  wrq.u.param.flags = IW_AUTH_DROP_UNENCRYPTED & IW_AUTH_INDEX;
  wrq.u.param.value = endis;

  if (ioctl(sockData->sockInt, SIOCSIWAUTH, &wrq) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Error setting WPA countermeasures enable/disable!\n");
      return -1;
    }
#endif
  return XENONE;
}

void cardif_linux_wext_enc_capabilities(struct interface_data *intdata)
{
#if WIRELESS_EXT > 17
  struct iwreq wrq;
  struct lin_sock_data *sockData;
  struct iw_range *range;
  char buffer[sizeof(struct iw_range) * 2];
  int i;

  sockData = intdata->sockData;

  intdata->enc_capa = 0;

  bzero(&wrq, sizeof(wrq));
  strncpy((char *)&wrq.ifr_name, intdata->intName, IFNAMSIZ);

  wrq.u.data.pointer = (caddr_t) buffer;
  wrq.u.data.length = sizeof(buffer);
  wrq.u.data.flags = 0;

  if (ioctl(sockData->sockInt, SIOCGIWRANGE, &wrq) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't get encryption capabilites!\n");
      return;
    }

  // Otherwise, determine what we have.
  range = (struct iw_range *)buffer;

  for (i=0; i<range->num_encoding_sizes; i++)
    {
      if (range->encoding_size[i] == 5) intdata->enc_capa |= DOES_WEP40;
      if (range->encoding_size[i] == 13) intdata->enc_capa |= DOES_WEP104;
    }

  if (range->enc_capa & IW_ENC_CAPA_WPA) intdata->enc_capa |= DOES_WPA;
  if (range->enc_capa & IW_ENC_CAPA_WPA2) intdata->enc_capa |= DOES_WPA2;
  if (range->enc_capa & IW_ENC_CAPA_CIPHER_TKIP)
    intdata->enc_capa |= DOES_TKIP;
  if (range->enc_capa & IW_ENC_CAPA_CIPHER_CCMP)
    intdata->enc_capa |= DOES_CCMP;
#else
  intdata->enc_capa = 0;
#endif
}

int cardif_linux_wext_delete_key(struct interface_data *intdata, int key_idx,
				 int set_tx)
{
#if WIRELESS_EXT > 17
  return cardif_linux_wext_set_key_ext(intdata, IW_ENCODE_ALG_NONE, NULL,
				       key_idx, set_tx, NULL, 0, NULL, 0);
#else
  debug_printf(DEBUG_NORMAL, "%s : Not supported by WE(%d)!\n", __FUNCTION__,
	       WIRELESS_EXT);
#endif
  return XENONE;
}


struct cardif_funcs cardif_linux_wext_driver = {
  .scan = cardif_linux_wext_scan,
  .disassociate = cardif_linux_wext_disassociate,
  .set_wep_key = cardif_linux_wext_set_WEP_key,
  .set_tkip_key = cardif_linux_wext_set_tkip_key,
  .set_ccmp_key = cardif_linux_wext_set_ccmp_key,
  .delete_key = cardif_linux_wext_delete_key,
  .associate = cardif_linux_wext_associate,
  .get_ssid = cardif_linux_wext_get_ssid,
  .get_bssid = cardif_linux_wext_get_bssid,
  .wpa_state = cardif_linux_wext_wpa_state,
  .wpa = cardif_linux_wext_wpa,
  .wep_associate = cardif_linux_wext_wep_associate,
  .countermeasures = cardif_linux_wext_countermeasures,
  .drop_unencrypted = cardif_linux_wext_drop_unencrypted,
  .get_wpa_ie = cardif_linux_wext_get_wpa_ie,
  .get_wpa2_ie = cardif_linux_wext_get_wpa2_ie,
  .enc_disable = cardif_linux_wext_enc_disable,
  .enc_capabilities = cardif_linux_wext_enc_capabilities,
  .setbssid = cardif_linux_wext_set_bssid,
};


#endif
