/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef XSID_WB_DEBUG
#include <iomanip>
#endif
#include <cctype>
#include <fstream>
#include <iostream>
#include <cstring>
using namespace std;

#ifdef HAVE_STRNCASECMP
#define MYSTRNCMP strncasecmp
#elif HAVE_STRNICMP
#define MYSTRNCMP strnicmp
#else
#error Neither strncasecmp nor strnicmp available.
#endif

#include "TextFile.h"

TextFile::TextFile(const int newMaxLineLen) : maxLineLen(newMaxLineLen)
{
    lineBuf = new char[maxLineLen+1];
    parseBuf = new char[maxLineLen+1];
    status = (lineBuf!=0 && parseBuf!=0);
    inFile = 0;
    open(0);
}

TextFile::~TextFile()
{
    close();
    if (lineBuf != 0)
        delete[] lineBuf;
    if (parseBuf != 0)
        delete[] parseBuf;
}

bool TextFile::open(const char* fileName)
{
    lineBuf[maxLineLen] = 0;
    lineLen = 0;
    lineNum = 0;
    
    close();  // always re-open
    
    if (fileName == 0)
        return false;
    
    // Create input file stream.
    // Make it binary because of compatibility to text files from any
    // known operating system.
#ifdef XSID_HAVE_IOS_BIN
    inFile = new ifstream(fileName,ios::in|ios::bin);
#else
    inFile = new ifstream(fileName,ios::in|ios::binary);
#endif
#ifdef XSID_DONT_HAVE_IS_OPEN
    isGood = ( !inFile.fail() && inFile!=0 && !inFile->bad() );
#else
    isGood = ( inFile->is_open() && inFile!=0 && !inFile->bad() );
#endif
    return isGood;
}

void TextFile::close()
{
    if ( inFile!=0 )
    {
        inFile->close();
        delete inFile;
    }
    inFile = 0;
    isGood = false;
}

const char* TextFile::getLineBuf() const
{ 
    return lineBuf;
}

const char* TextFile::getCurLineBuf() const
{ 
    return lineBuf+keyLen;
}

int TextFile::getLineNum() const
{
    return lineNum;
}

int TextFile::getLineLen() const
{
    return lineLen;
}

bool TextFile::isEOF() const
{
    if ( inFile!=0 && inFile->eof() )
        return true;
    else
        return false;
}

bool TextFile::getLine()
{
    // Reset line settings for new and empty lines.
    lineLen = 0;
    lineBuf[0] = 0;
    haveParseCopy = false;

    if ( !status || !isGood )  // input file stream & object status?
        return false;

    bool haveLine = false;
    int n = 0;
    char c;
    while ( inFile->get(c) && n<maxLineLen )
    {
        haveLine = true;
        // Unix: LF = 0x0A
        // MS-Windows, MS-DOS: CR,LF = 0x0D,0x0A
        // MacOS: CR = 0x0D
        if ( c!=0x0a && c!=0x0d )
        {
            lineBuf[n++] = c;
            lineBuf[n] = 0;
        }
        else if ( c==0x0a )
        {
            break;
        }
        else if ( c==0x0d )
        {
            if ( !inFile->get(c) )
                break;
            if ( c==0x0a )
                break;
            else
                inFile->putback(c);
        }
    };
    lineLen = n;
    if ( haveLine )
        lineNum++;
    return haveLine;
}

bool TextFile::isComment()
{
    if ( !haveParseCopy )
    {
        createParseCopy();
    }
    return ((parseBuf[0]==';')||(parseBuf[0]=='#'));
}

bool TextFile::isBlank()
{
    if ( !haveParseCopy )
    {
        createParseCopy();
    }
    return (parseBuf[0]==0);
}

bool TextFile::isKey(const char* keyword, bool advance)
{
    if ( !haveParseCopy )
    {
        createParseCopy();
    }
    int keywordLen = strlen(keyword);
    keyLen = 0;
    bool matches = (MYSTRNCMP(curParseBuf,keyword,keywordLen)==0);
    matches &= *(curParseBuf+keywordLen)=='=';
    if (matches && advance)
    {
        keyLen = keywordLen+1;
        curParseBuf += keywordLen+1;
        if (curParseBuf > (parseBuf+maxLineLen))
        {
            curParseBuf = parseBuf+maxLineLen;
        }
    }
    return matches;
}

bool TextFile::isValue(const char* value, bool advance)
{
    if ( !haveParseCopy )
    {
        createParseCopy();
    }
    int valueLen = strlen(value);
    bool matches = (MYSTRNCMP(curParseBuf,value,valueLen)==0);
    if (matches && advance)
    {
        curParseBuf += valueLen;
        if (curParseBuf > (parseBuf+maxLineLen))
        {
            curParseBuf = parseBuf+maxLineLen;
        }
    }
    return matches;
}

ofstream& TextFile::writeKey(ofstream& toFile, const char* key)
{
    toFile << key << '=';
    return toFile;
}

bool TextFile::createParseCopy()
{
    int di = 0;
    for ( int i = 0; i < lineLen; i++ )
    {
        char c = lineBuf[i];
        parseBuf[di] = c;
        if ( !isspace(c) )
        {
            di++;
        }
    }
    parseBuf[di] = 0;
    curParseBuf = parseBuf;
    return (haveParseCopy=true);
}

const char* TextFile::getParseBuf()
{ 
    if ( !haveParseCopy )
    {
        createParseCopy();
    }
    return parseBuf;
}

const char* TextFile::getCurParseBuf()
{ 
    if ( !haveParseCopy )
    {
        createParseCopy();
    }
    return curParseBuf;
}
