/*
 * Copyright 2006-2007 Luc Verhaegen.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "via_driver.h"

#ifndef _XF86_ANSIC_H
#include <string.h>
#endif

#include "via_output.h"

/*
 *
 */
enum SiI16xDevices {
    SII164,
    SII178,
    TI410,
    TI510,
    VT1632
};

/*
 * Output->Private
 */
struct SiI16xOutputPrivate {
    int Device;

    int DotclockMin;
    int DotclockMax;

    CARD8 Reg08;
    CARD8 Reg09;
};

/*
 *
 */
static void
SiI16xPrivateDestroy(struct ViaOutput *Output)
{
    xfree(Output->Private);

    Output->PrivateDestroy = NULL;
}

/*
 *
 */
static struct SiI16xOutputPrivate *
SiI16xPrivateCreate(struct ViaOutput *Output)
{
    struct SiI16xOutputPrivate *Private;
    CARD8 buf;

    VIAFUNC(Output->scrnIndex);

    Output->PrivSize = sizeof(struct SiI16xOutputPrivate);
    Output->Private =  xnfcalloc(1, Output->PrivSize);
    Private = Output->Private;

    /* Get the dotclock ranges while we are at it. */
    xf86I2CReadByte(Output->I2CDev, 0x06, &buf);
    Private->DotclockMin = buf;

    xf86I2CReadByte(Output->I2CDev, 0x07, &buf);
    Private->DotclockMax = buf + 65;

    Output->PrivateDestroy = SiI16xPrivateDestroy;

    return Output->Private;
}

/*
 *
 */
static void
SiI16xPrintRegs(struct ViaOutput *Output, const char *function)
{
    int i;
    CARD8 buf;

    ViaDebug(Output->scrnIndex, "%s: Printing registers for %s\n",
             function, Output->I2CDev->DevName);

    for (i = 0; i < 0x0E; i++) {
	xf86I2CReadByte(Output->I2CDev, i, &buf);
	ViaDebug(Output->scrnIndex, "%02X: 0x%02X\n", i, buf);
    }

    ViaDebug(Output->scrnIndex, "End of %s registers.\n",
             Output->I2CDev->DevName);
}

/*
 *
 */
static void
SiI16xSave(struct ViaOutput *Output)
{
    struct SiI16xOutputPrivate *Private = Output->Private;

    VIAFUNC(Output->scrnIndex);

    xf86I2CReadByte(Output->I2CDev, 0x08, &Private->Reg08);
}

/*
 *
 */
static void
SiI16xRestore(struct ViaOutput *Output)
{
    struct SiI16xOutputPrivate *Private = Output->Private;

    VIAFUNC(Output->scrnIndex);

    xf86I2CWriteByte(Output->I2CDev, 0x08, Private->Reg08);
}

/*
 * Unable to test this currently.
 */
static Bool
SiI16xDACSense(struct ViaOutput *Output)
{
    return FALSE;
}

/*
 *
 */
static ModeStatus
SiI16xModeValid(struct ViaOutput *Output, DisplayModePtr mode)
{
    struct SiI16xOutputPrivate *Private = Output->Private;

    VIAFUNC(Output->scrnIndex);

    if (mode->Clock < Private->DotclockMin)
        return MODE_CLOCK_LOW;

    if (mode->Clock > Private->DotclockMax)
        return MODE_CLOCK_HIGH;

    return MODE_OK;
}

/*
 * Empty.
 */
static void
SiI16xMode(struct ViaOutput *Output, DisplayModePtr mode)
{
    VIAFUNC(Output->scrnIndex);
}

/*
 *
 */
static void
SiI16xPower(struct ViaOutput *Output, Bool On)
{
    CARD8 buf;

    VIAFUNC(Output->scrnIndex);

    xf86I2CReadByte(Output->I2CDev, 0x08, &buf);

    if (On)
	xf86I2CWriteByte(Output->I2CDev, 0x08, buf | 0x01);
    else
	xf86I2CWriteByte(Output->I2CDev, 0x08, buf & 0xFE);
}

/*
 * This detects the following TMDS encoders, which are more or less SiI164
 * compatible (there are probably countless more devices):
 *
 * Silicon Image SiI164
 * Silicon Image SiI178 (dual link?)
 * Texas Instruments TFP410 (remember to disable DE for full compatibility)
 * Texas Instruments TFP510/513
 * VIAs VT1632(A) (actually tested)
 *
 */
struct ViaOutput *
ViaSiI16xInit(ScrnInfoPtr pScrn, I2CDevPtr pDev)
{
    struct ViaOutput *Output;
    CARD8 buf = 0, Read[5];
    CARD32 ID;

    VIAFUNC(pScrn->scrnIndex);

    /* Read the first 5 bytes */
    if (!xf86I2CWriteRead(pDev, &buf, 1, Read, 5)) {
        xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
                   "%s: Unable to read from %s Slave %d.\n",
                   __func__,  pDev->pI2CBus->BusName, pDev->SlaveAddr);
        return NULL;
    }

    ID = (Read[1] << 24) | (Read[0] << 16) | (Read[3] << 8) | Read[2];

    switch (ID) {
    case 0x00010006: /* Silicon Image SiI164 */
        xf86DrvMsg(pScrn->scrnIndex, X_PROBED, "Detected Silicon Image SiI164"
                   " (Rev. %d) TMDS Transmitter.\n", Read[4]);
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "SiI164 is not supported yet."
                   " Please contact unichrome-devel@lists.sf.net\n");
        return NULL;

    case 0x00010008: /* Silicon Image SiI178 */
        xf86DrvMsg(pScrn->scrnIndex, X_PROBED, "Detected Silicon Image SiI178"
                   " (Rev. %d) TMDS Transmitter.\n", Read[4]);
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "SiI178 is not supported yet."
                   " Please contact unichrome-devel@lists.sf.net\n");
        return NULL;

    case 0x014C0410: /* Texas Instruments TFP410 */
        xf86DrvMsg(pScrn->scrnIndex, X_PROBED, "Detected Texas Instruments "
                   "TFP410 (Rev. %d) TMDS Transmitter.\n", Read[4]);
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "TFP410 is not supported yet."
                   " Please contact unichrome-devel@lists.sf.net\n");
        return NULL;

    case 0x014C0510: /* Texas Instruments TFP510/513 */
        xf86DrvMsg(pScrn->scrnIndex, X_PROBED, "Detected Texas Instruments "
                   "TFP510/513 (Rev. %d) TMDS Transmitter.\n", Read[4]);
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "TFP510 is not supported yet."
                   " Please contact unichrome-devel@lists.sf.net\n");
        return NULL;

    case 0x11063192: /* VIA VT1632 */
        xf86DrvMsg(pScrn->scrnIndex, X_PROBED, "Detected Via Technologies "
                   "VT1632(A) (Rev. %d) TMDS Transmitter.\n", Read[4]);
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "VT1632 support is not"
                   " complete yet.\n");

        Output = xnfcalloc(1, sizeof(struct ViaOutput));

        Output->Prev = NULL;
        Output->Next = NULL;
        Output->scrnIndex = pScrn->scrnIndex;
        Output->I2CDev = pDev;
        Output->Type = OUTPUT_TMDS;

        pDev->DevName = "VT1632";
        Output->Name = "VT1632";

        Output->ClockMaster = FALSE;

        SiI16xPrivateCreate(Output);

        Output->Save = SiI16xSave;
        Output->Restore = SiI16xRestore;
        Output->Sense = SiI16xDACSense;
        Output->ModeValid = SiI16xModeValid;
        Output->Mode = SiI16xMode;
        Output->Power = SiI16xPower;
        Output->PrintRegs = SiI16xPrintRegs;

        return Output;

    default:
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "%s: Unknown SiI16x compatible"
                   " device detected on %s:0x%02X: 0x%08lX.\n", __func__,
                   pDev->pI2CBus->BusName, pDev->SlaveAddr, ID);
        return NULL;
    }

    return NULL;
}
