/*
 * Copyright 2004-2007 Luc Verhaegen.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

/*
 * Crappy panel code i haven't been able to clean up yet.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "via_driver.h"

#ifndef _XF86_ANSIC_H
#include <string.h>
#include <unistd.h>
#include <stdio.h>
#endif

#include "via_vgahw.h"
#include "via_id.h"
#include "via_output.h"
#include "via_mode.h"

/*
 * Output->Private
 */
struct ViaPanelOutputPrivate {
    /* private use */
    Bool Present; /* only used until init is over */

    int  X;
    int  Y;

    /* To enforce Native resolution */
    int  HTotalMax;
    int  HTotalMin;
    int  HSyncMax;
    int  HSyncMin;

    int  VTotalMax;
    int  VTotalMin;
    int  VSyncMax;
    int  VSyncMin;

    /* private options */
    Bool  Center;

#define VIA_DI_12BIT  0x00
#define VIA_DI_24BIT  0x01
    CARD8  BusWidth; /* Digital Output Bus Width */
};

/*
 *
 */
static void
ViaPanelPrivateDestroy(struct ViaOutput *Output)
{
    xfree(Output->Private);

    /* So we won't try to destroy this twice */
    Output->PrivateDestroy = NULL;
}

/*
 *
 */
static struct ViaPanelOutputPrivate *
ViaPanelPrivateCreate(struct ViaOutput *Output)
{
    VIAFUNC(Output->scrnIndex);

    Output->PrivSize = sizeof(struct ViaPanelOutputPrivate);
    Output->Private =  xnfcalloc(1, Output->PrivSize);
    memset(Output->Private, 0, Output->PrivSize);

    Output->PrivateDestroy = ViaPanelPrivateDestroy;

    return Output->Private;
}

/*
 *
 * Option handling.
 *
 */
enum ViaPanelOpts {
    OPTION_BUSWIDTH,
    OPTION_CENTER,
    OPTION_FORCEPANEL,
    OPTION_PANELSIZE
};

static OptionInfoRec ViaPanelOptions[] =
{
#ifdef UNUSED
    {OPTION_BUSWIDTH,	 "BusWidth",  	OPTV_ANYSTR,  {0}, FALSE},
#endif
    {OPTION_CENTER,      "Center",      OPTV_BOOLEAN, {0}, FALSE},
    {OPTION_FORCEPANEL,  "ForcePanel",  OPTV_BOOLEAN, {0}, FALSE}, /* last resort - don't doc */
    {OPTION_PANELSIZE,   "PanelSize",   OPTV_ANYSTR,  {0}, FALSE},
    {-1,                 NULL,          OPTV_NONE,    {0}, FALSE}
};

/*
 * Stop the user from telling us lies.
 */
/* Scratch register contents. */
#define VIA_PANEL6X4         0
#define VIA_PANEL8X6         1
#define VIA_PANEL10X7        2
#define VIA_PANEL12X7        3
#define VIA_PANEL12X10       4
#define VIA_PANEL14X10       5
#define VIA_PANEL16X12       6
#define VIA_PANEL_INVALID  255

struct {
    int   X;
    int   Y;
    CARD8 Scratch;
} ViaNativePanelSizes[] = {
    { 640,  480, VIA_PANEL6X4  },
    { 800,  600, VIA_PANEL8X6  },
    {1024,  768, VIA_PANEL10X7 },
    {1280,  768, VIA_PANEL12X7 },
    {1280, 1024, VIA_PANEL12X10},
    {1400, 1050, VIA_PANEL14X10},
    {1600, 1200, VIA_PANEL16X12},
    {0, 0, VIA_PANEL_INVALID}
};


/*
 *
 */
static OptionInfoPtr
ViaPanelGetOptions(ScrnInfoPtr pScrn, struct ViaPanelOutputPrivate *Private)
{
    OptionInfoPtr  Options;
    char  *s = NULL;

    Options = xnfalloc(sizeof(ViaPanelOptions));
    memcpy(Options, ViaPanelOptions, sizeof(ViaPanelOptions));

    xf86ProcessOptions(pScrn->scrnIndex, pScrn->options, Options);

#ifdef UNUSED
    /* Digital Output Bus Width Option */
    Private->BusWidth = VIA_DI_12BIT;
    if ((s = xf86GetOptValString(Options, OPTION_BUSWIDTH))) {
        if (!xf86NameCmp(s, "12BIT")) {
            Private->BusWidth = VIA_DI_12BIT;
            xf86DrvMsg(pScrn->scrnIndex, X_CONFIG,
                       "Digital Output Bus Width is 12BIT\n");
        } else if (!xf86NameCmp(s, "24BIT")) {
            Private->BusWidth = VIA_DI_24BIT;
            xf86DrvMsg(pScrn->scrnIndex, X_CONFIG,
                       "Digital Output Bus Width is 24BIT\n");
        }
    }
#endif

    if (1) {
        /* primary can't do scaling on its own */
        Private->Center = TRUE;
    } else { /* pretty useless */
        /* LCD Center/Expend Option */
        if (xf86ReturnOptValBool(Options, OPTION_CENTER, FALSE)) {
            Private->Center = TRUE;
            xf86DrvMsg(pScrn->scrnIndex, X_CONFIG, "DVI Center is On\n");
        } else
            Private->Center = FALSE;
    }

    /* Force the use of the Panel? */
    if (xf86ReturnOptValBool(Options, OPTION_FORCEPANEL, FALSE)) {
        xf86DrvMsg(pScrn->scrnIndex, X_INFO, "Enabling panel from config.\n");
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING,
                   "This is a last resort option only.\n");
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING,
                   "Report PCI IDs to http://unichrome.sf.net/ ASAP.\n");
        Private->Present = TRUE;
    } else
        Private->Present = FALSE;

    /* Panel Size Option */
    if ((s = xf86GetOptValString(Options, OPTION_PANELSIZE))) {
        int X, Y;
        if (sscanf(s, "%4dx%4d", &X, &Y) == 2) {
            int i;

            for (i = 0; ViaNativePanelSizes[i].X; i++) {
                if ((X == ViaNativePanelSizes[i].X) &&
                    (Y == ViaNativePanelSizes[i].Y)) {
                    Private->X = X;
                    Private->Y = Y;
                    break;
                }
            }

            if (!ViaNativePanelSizes[i].X) {
                xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "Unknown or Unhandled"
                           " Native Panel Resolution: %dx%d\n", X, Y);
                xf86DrvMsg(pScrn->scrnIndex, X_INFO,
                           "If this is the actual resolution of the panel, "
                           "contact the driver author.\n");
            }
        } else {
            xf86DrvMsg(pScrn->scrnIndex, X_WARNING,
                       "Failed to parse PanelSize Option: %s\n", s);
        }
    }

    return Options;
}


/*
 * Only run this during PreInit.
 */
static Bool
ViaPanelNativeResolutionFromConsole(ScrnInfoPtr pScrn, CARD16 *HDisplay,
                                 CARD16 *VDisplay)
{
    VIAPtr pVia = VIAPTR(pScrn);
    vgaHWPtr hwp = VGAHWPTR(pScrn);
    CARD8 CR6A = hwp->readCrtc(hwp, 0x6A);

    VIAFUNC(pScrn->scrnIndex);

    /* Primary VGA and Panel enabled? */
    if (xf86IsPrimaryPci(pVia->PciInfo) && (CR6A & 0x08)) {
        CARD8 CR97 = hwp->readCrtc(hwp, 0x97);
        CARD8 CR99 = hwp->readCrtc(hwp, 0x99);

        /* Secondary enabled? DFP source is secondary? */
        if (((CR6A & 0xC0) == 0x40) && (CR97 & 0x10) && (CR99 & 0x10)) {
            *HDisplay = (hwp->readCrtc(hwp, 0x51) |
                         ((hwp->readCrtc(hwp, 0x55) & 0x70) << 4)) + 1;

            *VDisplay = (hwp->readCrtc(hwp, 0x59) |
                         ((hwp->readCrtc(hwp, 0x5D) & 0x38) << 5)) + 1;

            return TRUE;
        } else if (!(CR97 & 0x10) && !(CR99 & 0x10)) {
            /* Go for primary timing */
            *HDisplay = (hwp->readCrtc(hwp, 0x01) + 1) * 8;

            *VDisplay = (hwp->readCrtc(hwp, 0x12) |
                         ((hwp->readCrtc(hwp, 0x07) & 0x02) << 7) |
                         ((hwp->readCrtc(hwp, 0x07) & 0x40) << 3) |
                         ((hwp->readCrtc(hwp, 0x35) & 0x04) << 8)) + 1;

            return TRUE;
        } else
            xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "%s: Unhandled combination:"
                       " CR6A 0x%02X; CR97 0x%02X; CR99 0x%02X\n", __func__,
                       CR6A, CR97, CR99);
    }
    return FALSE;
}

/*
 *
 */
static void
ViaPanelNativeResolution(ScrnInfoPtr pScrn,
                         struct ViaPanelOutputPrivate *Private)
{
    VIAPtr pVia = VIAPTR(pScrn);

    VIAFUNC(pScrn->scrnIndex);

    /* This could have been provided by config */
    if (!Private->X || !Private->Y) {
        CARD16 HDisplay = 0, VDisplay = 0;
        int i;

        for (i = 0; ViaNativePanelSizes[i].X; i++) {
            if (ViaNativePanelSizes[i].Scratch == pVia->Scratch->PanelSize) {
                Private->X = ViaNativePanelSizes[i].X;
                Private->Y = ViaNativePanelSizes[i].Y;
                break;
            }
        }

        if (!ViaNativePanelSizes[i].X) {
            xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "%s: Unhandled PanelSize"
                       "(%d): Contact the driver author ASAP.\n",
                       __func__, pVia->Scratch->PanelSize);
            Private->X = 1024;
            Private->Y = 768;
        }

        if (ViaPanelNativeResolutionFromConsole(pScrn, &HDisplay, &VDisplay)) {
            if ((Private->X != HDisplay) || (Private->Y != VDisplay)) {
                if (!pVia->Scratch->PanelSize) {
                    xf86DrvMsg(pScrn->scrnIndex, X_INFO, "%s: Scratch register"
                               " panel sizing was probably wrong.\n", __func__);
                    xf86DrvMsg(pScrn->scrnIndex, X_INFO, "%s: Using resolution"
                               " of text-mode panel instead.\n", __func__);
                    Private->X = HDisplay;
                    Private->Y = VDisplay;
                } else
                    xf86DrvMsg(pScrn->scrnIndex, X_INFO, "%s: Scratch based"
                               "sizing (%dx%d) doesn't match text-mode "
                               "(%dx%d)\n", __func__, Private->X, Private->Y,
                               HDisplay, VDisplay);
            }
        }
    }

    xf86DrvMsg(pScrn->scrnIndex, X_INFO, "Panel Native Resolution is %dx%d.\n",
               Private->X, Private->Y);
}

/*
 * Here we assume that panels are capable of at least 57.5 and 65Hz CVT timing
 * and CVT reduced blanking timing. We use those timings to impose our limits.
 *
 */
static void
ViaPanelCenteredTiming(struct ViaOutput *Output)
{
    struct ViaPanelOutputPrivate *Private = Output->Private;
    DisplayModePtr Mode, RMode;

    /* Faster normal mode = upper limit for timing*/
    Mode = ViaCVTMode(Private->X, Private->Y, 65.0, FALSE, FALSE);

    /* Reduced blanking mode = lower limit for timing */
    RMode = ViaCVTMode(Private->X, Private->Y, 60.0, TRUE, FALSE);

    Private->HTotalMax = Mode->HTotal;
    Private->HTotalMin = RMode->HTotal;
    Private->HSyncMax = Mode->HSyncEnd - Mode->HSyncStart;
    Private->HSyncMin = 32; /* CVT -r default */

    Private->VTotalMax = Mode->VTotal;
    Private->VTotalMin = RMode->VTotal;
    Private->VSyncMax = Mode->VSyncEnd - Mode->VSyncStart;
    Private->VSyncMin = 3; /* CVT timing only goes down to 4 */

    ViaModesDestroy(RMode);

    /* Safe bets */
    Output->numVRefresh = 1;
    Output->VRefresh[0].lo = 57.5;
    Output->VRefresh[0].hi = 65.0;

    /* now get slow timing; for HSync */
    RMode = ViaCVTMode(Private->X, Private->Y, 57.5, FALSE, FALSE);

    Output->numHSync = 1;
    Output->HSync[0].hi = (float) Mode->Clock / Mode->HTotal;
    Output->HSync[0].lo = (float) RMode->Clock / RMode->HTotal;

    ViaModesDestroy(Mode);
    ViaModesDestroy(RMode);
}

/*
 *
 * Primary only Panel implementation.
 *
 */
static struct {
    int X;
    int Y;
    char Name[10];
} ViaPanelModes[] = {
    {1280, 768, "1280x768"},
    {1024, 768, "1024x768"},
    { 800, 600, "800x600"},
    { 640, 480, "640x480"},
    {   0,   0, ""},
};

/*
 * Add centered modes.
 *
 * We use Mode destructively here.
 *
 */
static void
ViaPanelCenteredAdd(struct ViaOutput *Output,  DisplayModePtr Mode)
{
    struct ViaPanelOutputPrivate *Private = Output->Private;
    int i;

    for (i = 0; ViaPanelModes[i].X; i++) {
        if ((ViaPanelModes[i].X > Private->X) ||
            (ViaPanelModes[i].Y > Private->Y))
            continue;

        /* We already added native resolution */
        if ((ViaPanelModes[i].X == Private->X) &&
            (ViaPanelModes[i].Y == Private->Y))
            continue;

        Mode->HDisplay = ViaPanelModes[i].X;
        Mode->VDisplay = ViaPanelModes[i].Y;
        xfree(Mode->name);
        Mode->name = xnfstrdup(ViaPanelModes[i].Name);

        Mode->type = M_T_DRIVER;

        ViaOutputModesCopyAdd(Output, Mode);
    }
}

/*
 *
 */
static void
ViaPanelTimingSet(struct ViaOutput *Output)
{
    struct ViaPanelOutputPrivate *Private = Output->Private;
    DisplayModePtr Mode;

    VIAFUNC(Output->scrnIndex);

    /* We need to add both CVT as normal modes when we can. */
    Output->ReducedAllowed = TRUE;

    /* Native mode */
    Mode = ViaCVTMode(Private->X, Private->Y, 60.0, TRUE, FALSE);
    Mode->type = M_T_DRIVER | M_T_PREFERRED;
    ViaOutputModesCopyAdd(Output, Mode);

    /* Add centered Modes */
    ViaPanelCenteredAdd(Output, Mode);

    ViaModesDestroy(Mode);

    /* Native mode */
    Mode = ViaCVTMode(Private->X, Private->Y, 60.0, FALSE, FALSE);
    Mode->type = M_T_DRIVER | M_T_PREFERRED;
    ViaOutputModesCopyAdd(Output, Mode);

    /* Add centered Modes */
    ViaPanelCenteredAdd(Output, Mode);

    ViaModesDestroy(Mode);

    Output->MonitorName = xnfstrdup("TTL Panel");

    ViaPanelCenteredTiming(Output);

    Output->ModesExclusive = FALSE;
}

/*
 *
 */
static ModeStatus
ViaPanelModeValid(struct ViaOutput *Output, DisplayModePtr mode)
{
    struct ViaPanelOutputPrivate *Private = Output->Private;
    int Sync;

    VIAFUNC(Output->scrnIndex);

    if ((mode->HDisplay > Private->X) || (mode->VDisplay > Private->Y))
        return MODE_PANEL;

    if ((mode->CrtcHTotal > Private->HTotalMax) ||
        (mode->CrtcHTotal < Private->HTotalMin))
        return MODE_H_ILLEGAL;

    if ((mode->CrtcVTotal > Private->VTotalMax) ||
        (mode->CrtcVTotal < Private->VTotalMin))
        return MODE_V_ILLEGAL;

    Sync = mode->CrtcHSyncEnd - mode->CrtcHSyncStart;
    if (Sync > Private->HSyncMax)
        return MODE_HSYNC_WIDE;
    if (Sync < Private->HSyncMin)
        return MODE_HSYNC_NARROW;

    Sync = mode->CrtcVSyncEnd - mode->CrtcVSyncStart;
    if (Sync > Private->VSyncMax)
        return MODE_VSYNC_WIDE;
    if (Sync < Private->VSyncMin)
        return MODE_VSYNC_NARROW;

    if ((mode->HDisplay == Private->X) || (mode->VDisplay == Private->Y))
        return MODE_OK;

    if (Private->Center) {
        int Shift;

        /* The timing is that of native resolution but with H/VDisplay smaller
         * than native resolution.
         *
         * We shift blanking away from H/VDisplay as panels are not sync
         * triggered but data-enable triggered.
         *
         * Sync is shifted too so that an attached CRT will be centered as well.
         */

        Shift = ((Private->X - mode->HDisplay) / 2) & ~0x07;
        if (Shift && ((mode->CrtcHBlankEnd != (mode->CrtcHTotal - Shift)) ||
                      (mode->CrtcHBlankStart != mode->CrtcHDisplay + Shift))) {
            mode->CrtcHBlankEnd = mode->CrtcHTotal - Shift;
            mode->CrtcHBlankStart = mode->CrtcHDisplay + Shift;

            /* calculate new sync posisition */
            Shift = mode->CrtcHSyncEnd - mode->CrtcHSyncStart;
            mode->CrtcHSyncEnd = ((mode->CrtcHTotal + mode->CrtcHDisplay) & ~0x0F) >> 1;
            mode->CrtcHSyncStart = mode->CrtcHSyncEnd - Shift;

            mode->CrtcHAdjusted = TRUE;
        }

        Shift = (Private->Y - mode->VDisplay) / 2;
        if (Shift && ((mode->CrtcVBlankEnd != (mode->CrtcVTotal - Shift)) ||
                      (mode->CrtcVBlankStart != mode->CrtcVDisplay + Shift))) {
            mode->CrtcVBlankEnd = mode->CrtcVTotal - Shift;
            mode->CrtcVBlankStart = mode->CrtcVDisplay + Shift;

             /* calculate new sync posisition */
            Shift = mode->CrtcVSyncEnd - mode->CrtcVSyncStart;
            mode->CrtcVSyncEnd = (mode->CrtcVTotal + mode->CrtcVDisplay) >> 1;
            mode->CrtcVSyncStart = mode->CrtcVSyncEnd - Shift;

            mode->CrtcVAdjusted = TRUE;
        }
    }

    return MODE_OK;
}

/*
 *
 */
static void
ViaPanelMode(struct ViaOutput *Output, DisplayModePtr mode)
{
    ScrnInfoPtr pScrn = xf86Screens[Output->scrnIndex];
    vgaHWPtr hwp = VGAHWPTR(pScrn);
    struct ViaPanelOutputPrivate *Private = Output->Private;

    VIAFUNC(Output->scrnIndex);

    if (Private->Center ||
        ((mode->HDisplay == Private->X) && (mode->VDisplay == Private->Y))) {
        ViaCrtcMask(hwp, 0x79, 0x00, 0x01); /* Disable Scaling */
    } else {
        /* Scaling is not possible from the primary only */
        CARD32 HScale = (mode->HDisplay - 1) * 4096 / (Private->X - 1);
        CARD32 VScale = (mode->VDisplay - 1) * 2048 / (Private->Y - 1);

        ViaCrtcMask(hwp, 0x9F, HScale, 0x03);
        hwp->writeCrtc(hwp, 0x77, HScale >> 2);
        ViaCrtcMask(hwp, 0x79, HScale >> 6, 0x30);

        ViaCrtcMask(hwp, 0x79, VScale << 3, 0x08);
        hwp->writeCrtc(hwp, 0x78, VScale >> 1);
        ViaCrtcMask(hwp, 0x79, VScale >> 3, 0xC0);

        /* Enable Scaling and H/V interpolation. */
        ViaCrtcMask(hwp, 0x79, 0x07, 0x07);
    }
}

/*
 * Reduction of old VIA code, untested.
 */
static void
VT3122PanelPower(struct ViaOutput *Output, Bool On)
{
    vgaHWPtr hwp = VGAHWPTR(xf86Screens[Output->scrnIndex]);

    ViaDebug(Output->scrnIndex, "%s: %s.\n", __func__, On ? "On" : "Off");

    /* Enable LCD */
    if (On)
	ViaCrtcMask(hwp, 0x6A, 0x08, 0x08);
    else
	ViaCrtcMask(hwp, 0x6A, 0x00, 0x08);

    usleep(1);

    if (On) {
        ViaCrtcMask(hwp, 0x91, 0x10, 0x10);
        usleep(0x19);
        ViaCrtcMask(hwp, 0x91, 0x08, 0x08);
        usleep(0x1FE);
        ViaCrtcMask(hwp, 0x91, 0x06, 0x06);
        usleep(0x01);
    } else {
        ViaCrtcMask(hwp, 0x91, 0x00, 0x06);
        usleep(0xD2);
        ViaCrtcMask(hwp, 0x91, 0x00, 0x08);
        usleep(0x19);
        ViaCrtcMask(hwp, 0x91, 0x00, 0x10);
        usleep(0x01);
    }
}

/*
 * Reduction of old VIA code, untested.
 */
static void
VT7205PanelPower(struct ViaOutput *Output, Bool On)
{
    vgaHWPtr hwp = VGAHWPTR(xf86Screens[Output->scrnIndex]);

    ViaDebug(Output->scrnIndex, "%s: %s.\n", __func__, On ? "On" : "Off");

    /* Enable LCD */
    if (On)
	ViaCrtcMask(hwp, 0x6A, 0x08, 0x08);
    else
	ViaCrtcMask(hwp, 0x6A, 0x00, 0x08);

    usleep(1);

    if (On) {
        ViaCrtcMask(hwp, 0x91, 0x10, 0x10);
        usleep(0x19);
        ViaCrtcMask(hwp, 0x91, 0x08, 0x08);
        usleep(0x1FE);
        ViaSeqMask(hwp, 0x3D, 0x20, 0x20);
        usleep(0x01);
    } else {
        ViaSeqMask(hwp, 0x3D, 0x00, 0x20);
        usleep(0xD2);
        ViaCrtcMask(hwp, 0x91, 0x00, 0x08);
        usleep(0x19);
        ViaCrtcMask(hwp, 0x91, 0x00, 0x10);
        usleep(0x01);
    }

    usleep(1);
}

/*
 * This implements only superficial power down. Good enough for DPMS.
 */
static void
VT3108PanelPower(struct ViaOutput *Output, Bool On)
{
    ScrnInfoPtr pScrn = xf86Screens[Output->scrnIndex];
    vgaHWPtr hwp = VGAHWPTR(pScrn);

    ViaDebug(Output->scrnIndex, "%s: %s.\n", __func__, On ? "On" : "Off");

    ViaCrtcMask(hwp, 0x91, 0x00, 0x01); /* Disable Software Control */
    usleep(1);

    if (On) /* Direct Panel Display and Backlight control */
        ViaCrtcMask(hwp, 0x91, 0x00, 0xC0);
    else
        ViaCrtcMask(hwp, 0x91, 0xC0, 0xC0);
}

/*
 * future: I2CDevPtr for LVDS encoders.
 */
struct ViaOutput *
ViaPanelInit(ScrnInfoPtr pScrn, I2CDevPtr pDev)
{
    VIAPtr pVia = VIAPTR(pScrn);
    struct ViaOutput *Output;
    struct ViaPanelOutputPrivate *Private;

    VIAFUNC(pScrn->scrnIndex);

    if (pDev)
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "%s: Ignoring I2C Device"
                   " passed.\n", __func__);

    Output = xnfcalloc(1, sizeof(struct ViaOutput));

    Output->Prev = NULL;
    Output->Next = NULL;
    Output->scrnIndex = pScrn->scrnIndex;
    Output->I2CDev = pDev;
    Output->Type = OUTPUT_PANEL;

    Private = ViaPanelPrivateCreate(Output);

    /* Parse options here ourselves. */
    Output->Options = ViaPanelGetOptions(pScrn, Private);

    /* Is there even a panel present? */
    if (!Private->Present) {
        if (pVia->Id && pVia->Id->HasPanel) {
            xf86DrvMsg(pScrn->scrnIndex, X_INFO, "Enabling panel from"
                       " PCI-Subsystem Id information.\n");
            Private->Present = TRUE;
        } else {
            Output->PrivateDestroy(Output);

            xfree(Output->Options);
            xfree(Output);
            return NULL;
        }
    }

    ViaPanelNativeResolution(pScrn, Private);

    Output->Name = "Panel";

    Output->ClockMaster = FALSE;

    ViaPanelTimingSet(Output);

    Output->Save = NULL;
    Output->Restore = NULL;
    Output->Sense = NULL;
    Output->ModeValid = ViaPanelModeValid;
    Output->Mode = ViaPanelMode;
    Output->PrintRegs = NULL;

    switch (pVia->Chipset) {
    case VT3122:
        Output->Power = VT3122PanelPower;
        break;
    case VT7205:
        Output->Power = VT7205PanelPower;
        break;
    case VT3108:
        Output->Power = VT3108PanelPower;
        break;
    default:
	xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "Panel on %s is currently "
                   "not supported.\n", pScrn->chipset);
        Output->PrivateDestroy(Output);
        xfree(Output->Options);
        xfree(Output);
        return NULL;
    }

    return Output;
}
