/*
 * Copyright 2004-2007 Luc Verhaegen.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

/*
 * via_output.c
 *
 * Output Handling.
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "via_driver.h"

#ifndef _XF86_ANSIC_H
#include <string.h>
#endif

#include "via_vgahw.h"
#include "via_id.h"
#include "via_crtc.h"
#include "via_output.h"
#include "via_mode.h"

/*
 *
 */
static struct ViaOutput *
ViaOutputDestroy(struct ViaOutput *Output)
{
    struct ViaOutput *Next;

    Next = Output->Next;

    if (Output->PrivateDestroy)
        Output->PrivateDestroy(Output);

    ViaModesDestroy(Output->Modes);

    if (Output->I2CDev)
        xf86DestroyI2CDevRec(Output->I2CDev, TRUE);

    if (Output->Options)
        xfree(Output->Options);

    if (Output->MonitorName)
        xfree(Output->MonitorName);

    xfree(Output);

    return Next;
}

/*
 *
 */
void
ViaOutputsDestroy(ScrnInfoPtr pScrn)
{
    struct ViaOutput *Output = VIAPTR(pScrn)->Outputs;

    while (Output)
        Output = ViaOutputDestroy(Output);
}

/*
 *
 */
static Bool
ViaOutputAdd(ScrnInfoPtr pScrn, struct ViaOutput *Output)
{
    struct ViaOutput *Last = NULL;

    /* Is this Output sound? */
    if (!Output->Name || !Output->ModeValid || !Output->Mode ||
        !Output->Power) {

        if (!Output->Name)
            xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
                       "%s: Nameless output device.\n", __func__);

        if (!Output->ModeValid)
            xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "%s: %s: ModeValid "
                       "callback missing\n", __func__, Output->Name);

        if (!Output->Mode)
            xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "%s: %s: Mode "
                       "callback missing\n", __func__, Output->Name);

        if (!Output->Power)
            xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "%s: %s: Power "
                       "callback missing\n", __func__, Output->Name);

        return FALSE;
    }

    /* add Output to list */
    Last = VIAPTR(pScrn)->Outputs;
    if (Last) {
        while (Last->Next)
            Last = Last->Next;

        Last->Next = Output;
        Output->Prev = Last;
    } else {
        VIAPTR(pScrn)->Outputs = Output;
        Output->Prev = NULL;
    }

    return TRUE;
}

#ifdef UNUSED
/*
 *
 */
static void
ViaOutputRemove(ScrnInfoPtr pScrn, struct ViaOutput *Output)
{
    if (!Output)
        return;

    /* remove from list */
    if (Output->Prev)
        Output->Prev->Next = Output->Next;
    else
        VIAPTR(pScrn)->Outputs = Output->Next;

    if (Output->Next)
        Output->Next->Prev = Output->Prev;

    ViaOutputDestroy(Output);
}
#endif

/*
 *
 * Whine about TODOs.
 *
 */

/*
 * VIAs VT1631 LVDS encoder. (aka VT3191)
 */
static struct ViaOutput *
ViaVT1631Init(ScrnInfoPtr pScrn, I2CDevPtr pDev)
{
    CARD8 buf = 0, Read[4];
    CARD32 ID;

    VIAFUNC(pScrn->scrnIndex);

    if (!xf86I2CWriteRead(pDev, &buf, 1, Read, 4)) {
        xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
                   "%s: Unable to read from %s Slave %d.\n",
                   __func__,  pDev->pI2CBus->BusName, pDev->SlaveAddr);
        return NULL;
    }

    ID = (Read[1] << 24) | (Read[0] << 16) | (Read[3] << 8) | Read[2];

    switch (ID) {
    case 0x11063191: /* VIA VT1631 */
        xf86DrvMsg(pScrn->scrnIndex, X_PROBED,
                   "Detected Via Technologies VT1631 LVDS Transmitter.\n");
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "VT1631 is not supported yet."
                   " Please contact unichrome-devel@lists.sf.net\n");
        return NULL;
    case 0x11063192: /* VIA VT1632 - Not handled here => Sil164 compatible. */
        return NULL;
    default:
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "%s: Unknown VIA device "
                   "detected on %s:0x%02X: 0x%08lX.\n", __func__,
                   pDev->pI2CBus->BusName, pDev->SlaveAddr, ID);
        return NULL;
    }

    return NULL;
}

/*
 * Philips TV encoders.
 *
 * SAA7108
 * SAA7109
 */
static struct ViaOutput *
ViaSAA710xInit(ScrnInfoPtr pScrn, I2CDevPtr pDev)
{
    CARD8 buf;

    VIAFUNC(pScrn->scrnIndex);

    if (!xf86I2CReadByte(pDev, 0x1C, &buf)) {
        xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
                   "%s: Unable to read from %s Slave %d.\n",
                   __func__,  pDev->pI2CBus->BusName, pDev->SlaveAddr);
        return NULL;
    }

    switch (buf) {
    case 0x02:
        xf86DrvMsg(pScrn->scrnIndex, X_PROBED,
                   "Detected Philips Semiconductors SAA7108E TV Encoder.\n");
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "SAA7108E is not supported yet."
                   " Please contact unichrome-devel@lists.sf.net\n");
        return NULL;
    case 0x03:
        xf86DrvMsg(pScrn->scrnIndex, X_PROBED,
                   "Detected Philips Semiconductors SAA7109E TV Encoder.\n");
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "SAA7109E is not supported yet."
                   " Please contact unichrome-devel@lists.sf.net\n");
        return NULL;
    default:
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING,
                   "%s: Unknown device detected: 0x%02X.\n", __func__, buf);
        return NULL;
    };

    return NULL;
}

/*
 * Focus enhancements tv encoders.
 */
static struct ViaOutput *
ViaFS45xInit(ScrnInfoPtr pScrn, I2CDevPtr pDev)
{
    CARD8 buf;

    VIAFUNC(pScrn->scrnIndex);

    if (!xf86I2CReadByte(pDev, 0x7F, &buf)) {
        xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
                   "%s: Unable to read from %s Slave %d.\n",
                   __func__,  pDev->pI2CBus->BusName, pDev->SlaveAddr);
        return NULL;
    }

    switch (buf) {
    case 0x20:
        xf86DrvMsg(pScrn->scrnIndex, X_PROBED,
                   "Detected Focus Enhancements 454 TV Encoder.\n");
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "FS454 is not supported yet."
                   " Please contact unichrome-devel@lists.sf.net\n");
        return NULL;
    default:
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING,
                   "%s: Unknown device detected: 0x%02X.\n", __func__, buf);
        return NULL;
    };

    return NULL;
}

/*
 * Aitech AIT2139
 */
static struct ViaOutput *
ViaAIT213xInit(ScrnInfoPtr pScrn, I2CDevPtr pDev)
{
    CARD8 buf;

    VIAFUNC(pScrn->scrnIndex);

    if (!xf86I2CReadByte(pDev, 0x3D, &buf)) {
        xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
                   "%s: Unable to read from %s Slave %d.\n",
                   __func__,  pDev->pI2CBus->BusName, pDev->SlaveAddr);
        return NULL;
    }

    switch (buf) {
    case 0x11:
        xf86DrvMsg(pScrn->scrnIndex, X_PROBED,
                   "Detected AITech AIT2139 TV Encoder.\n");
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "AIT2139 is not supported yet."
                   " Please contact unichrome-devel@lists.sf.net\n");
        return NULL;
    default:
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING,
                   "%s: Unknown device detected: 0x%02X.\n", __func__, buf);
        return NULL;

    }
}

static struct {
    CARD8 Address;
    char * Name;
    struct ViaOutput * (*Init) (ScrnInfoPtr pScrn, I2CDevPtr pDev);
} ViaI2CDevices[] = {
    {0x10, "VT1631",  ViaVT1631Init},
    {0x10, "SiI16x",  ViaSiI16xInit},
    {0x40, "VT162x",  ViaVT162xInit},
    {0x42, "VT162x",  ViaVT162xInit},
    {0x70, "SiI16x",  ViaSiI16xInit},
    {0x72, "SiI16x",  ViaSiI16xInit},
    {0x88, "SAA710x", ViaSAA710xInit},
    {0x88, "AIT213x", ViaAIT213xInit},
    {0x8A, "AIT213x", ViaAIT213xInit},
    {0xD4, "FS45x",   ViaFS45xInit},
    {0xEA, "CH7xxx",  ViaCH7xxxInit},
    {0xEC, "CH7xxx",  ViaCH7xxxInit},
    {0x00, NULL, NULL}
};

/*
 *
 */
static void
ViaScanBus(ScrnInfoPtr pScrn, I2CBusPtr pI2CBus)
{
    struct ViaOutput *Output;
    CARD8 SkipAddress = 0x00;
    int i;

    VIAFUNC(pScrn->scrnIndex);

    for (i = 0; ViaI2CDevices[i].Init; i++) {
        I2CDevPtr pDev = NULL;

        /* Make sure we don't try to Init a successfully inited device again */
        if (ViaI2CDevices[i].Address == SkipAddress)
            continue;

        if (!xf86I2CProbeAddress(pI2CBus, ViaI2CDevices[i].Address))
            continue;

        pDev = xf86CreateI2CDevRec();

        pDev->DevName = ViaI2CDevices[i].Name;
        pDev->SlaveAddr = ViaI2CDevices[i].Address;
        pDev->pI2CBus = pI2CBus;

        if (!xf86I2CDevInit(pDev)) {
            xf86DestroyI2CDevRec(pDev, TRUE);
            continue;
        }

        Output = ViaI2CDevices[i].Init(pScrn, pDev);

        /* did we actually find anything? */
        if (!Output) {
            xf86DestroyI2CDevRec(pDev,TRUE);
        } else {
            SkipAddress = ViaI2CDevices[i].Address;

            if (!ViaOutputAdd(pScrn, Output)) {
                while (Output) /* could be a whole list */
                    Output = ViaOutputDestroy(Output);
            }
        }
    }
}

/*
 *
 */
static char *
ViaOutputBusName(int Position)
{
    switch(Position) {
    case OUTPUT_BUS_CRT:
        return "CRT";
    case OUTPUT_BUS_DVP0:
        return "DVP0";
    case OUTPUT_BUS_DVP1:
        return "DVP1";
    case OUTPUT_BUS_DFP:
        return "DFP";
    case OUTPUT_BUS_DFPHIGH:
        return "DFPHigh";
    case OUTPUT_BUS_DFPLOW:
        return "DFPLow";
    default:
        return "Unknown";
    }
}

/*
 *
 */
void
ViaOutputsDetect(ScrnInfoPtr pScrn)
{
    vgaHWPtr hwp = VGAHWPTR(pScrn);
    VIAPtr pVia = VIAPTR(pScrn);
    struct ViaOutput *Output;
    CARD8 SR12 = hwp->readSeq(hwp, 0x12);

    VIAFUNC(pScrn->scrnIndex);

    /* CRT */
    Output = ViaCRTInit(pScrn, NULL);
    if (Output && !ViaOutputAdd(pScrn, Output))
        while (Output) /* could be a whole list */
            Output = ViaOutputDestroy(Output);

    /* Panel */
    Output = ViaPanelInit(pScrn, NULL);
    if (Output && !ViaOutputAdd(pScrn, Output))
        while (Output) /* could be a whole list */
            Output = ViaOutputDestroy(Output);

    if (pVia->pI2CBus2)
        ViaScanBus(pScrn, pVia->pI2CBus2);

    if (pVia->pI2CBus3)
        ViaScanBus(pScrn, pVia->pI2CBus3);

    /* Match outputs to Bus */
    Output = pVia->Outputs;
    while (Output) {
        /* check bus position */
        switch (Output->Type) {
        case OUTPUT_CRT:
            Output->Position = OUTPUT_BUS_CRT; /* Duh! */
            break;
        case OUTPUT_TV:
            if (SR12 & 0x20)
                Output->Position = OUTPUT_BUS_DVP0;
            else
                Output->Position = OUTPUT_BUS_DVP1;
            break;
        case OUTPUT_TMDS:
            /* VT3122 EPIAs probably sit off DFPHigh, can't test this yet */
            Output->Position = OUTPUT_BUS_DFPHIGH;
            /* !!!! Check wether this is allowed !!!! */
            break;
        case OUTPUT_PANEL:
            /* split me! */
            Output->Position = OUTPUT_BUS_DFP;
            break;
        default:
            xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "%s: Unhandled output device"
                       " type.\n", __func__);
            break;
        }

        /* Dump registers as early as possible */
        if (pVia->PrintTVRegs && Output->PrintRegs)
            Output->PrintRegs(Output, __func__);

        Output = Output->Next;
    }
}

/*
 * Save all output devices.
 */
void
ViaOutputsSave(ScrnInfoPtr pScrn)
{
    struct ViaOutput *Output = VIAPTR(pScrn)->Outputs;

    while (Output) {
        if (Output->Save)
            Output->Save(Output);

        Output = Output->Next;
    }
}

/*
 * Restore all output devices.
 */
void
ViaOutputsRestore(ScrnInfoPtr pScrn)
{
    struct ViaOutput *Output = VIAPTR(pScrn)->Outputs;

    while (Output) {
        if (Output->Restore)
            Output->Restore(Output);

        Output = Output->Next;
    }
}

/*
 * Check which output devices are active.
 */
static Bool
ViaOutputsSense(ScrnInfoPtr pScrn)
{
    struct ViaOutput *Output = VIAPTR(pScrn)->Outputs;
    Bool Active = FALSE;

    while (Output) {
        if (Output->Sense) {
            Output->Active = Output->Sense(Output);
            if (Output->Active)
                Active = TRUE;
        } else {
            Output->Active = TRUE;
            Active = TRUE;
        }

        Output = Output->Next;
    }

    return Active;
}

/*
 * Copy the modes of a table with final entry name == NULL into a list.
 * Attach list to a MonRec.
 *
 * Stolen from xf86Config.c's addDefaultModes
 */
void
ViaOutputAddModetable(struct ViaOutput *Output, DisplayModePtr Modes)
{
    DisplayModePtr mode;
    DisplayModePtr last;
    int i;

    for (last = Output->Modes; last && last->next; last = last->next)
        ;

    for (i = 0; Modes[i].name; i++) {

        mode = xnfalloc(sizeof(DisplayModeRec));
        memcpy(mode, &Modes[i], sizeof(DisplayModeRec));

        mode->name = xnfstrdup(Modes[i].name);

        if (last) {
            mode->prev = last;
            last->next = mode;
        } else { /* this is the first mode */
            Output->Modes = mode;
            mode->prev = NULL;
        }
        last = mode;
    }
}

/*
 * Does the common checks, and then passes on the Mode to the Outputs
 * own validation.
 */
int
ViaModeOutputValid(struct ViaOutput *Output, DisplayModePtr Mode)
{
    int i;

    for (i = 0; i < Output->numHSync; i++)
        if ((Mode->HSync >= (Output->HSync[i].lo * (1.0 - SYNC_TOLERANCE))) &&
            (Mode->HSync <= (Output->HSync[i].hi * (1.0 + SYNC_TOLERANCE))))
            break;
    if (Output->numHSync && (i == Output->numHSync))
        return MODE_HSYNC;

    for (i = 0; i < Output->numVRefresh; i++)
        if ((Mode->VRefresh >= (Output->VRefresh[i].lo * (1.0 - SYNC_TOLERANCE))) &&
            (Mode->VRefresh <= (Output->VRefresh[i].hi * (1.0 + SYNC_TOLERANCE))))
            break;
    if (Output->numVRefresh && (i == Output->numVRefresh))
        return MODE_VSYNC;

    if (Output->MaxClock && (Mode->SynthClock > Output->MaxClock))
        return MODE_CLOCK_HIGH;

    /* Is the horizontal blanking a bit lowish? */
    if (((Mode->CrtcHDisplay * 5 / 4) & ~0x07) > Mode->CrtcHTotal) {
        /* is this a cvt -r Mode, and only a cvt -r Mode? */
        if (((Mode->CrtcHTotal - Mode->CrtcHDisplay) == 160) &&
            ((Mode->CrtcHSyncEnd - Mode->CrtcHDisplay) == 80) &&
            ((Mode->CrtcHSyncEnd - Mode->CrtcHSyncStart) == 32) &&
            ((Mode->CrtcVSyncStart - Mode->CrtcVDisplay) == 3)) {
            if (!Output->ReducedAllowed)
                return MODE_NO_REDUCED;
        } else if ((Mode->CrtcHDisplay * 1.10) > Mode->CrtcHTotal)
            return MODE_HSYNC_NARROW;
    }

    if (Output->ModeValid)
        return Output->ModeValid(Output, Mode);
    else
        return MODE_OK;
}

/*
 *
 */
void
ViaOutputsPrintRegs(ScrnInfoPtr pScrn, const char *function)
{
    struct ViaOutput *Output = VIAPTR(pScrn)->Outputs;

    while (Output) {
        if (Output->PrintRegs)
            Output->PrintRegs(Output, function);

        Output = Output->Next;
    }
}

#ifdef UNUSED
/*
 * Is the CRTC in control of the dotclock, or is an output device?
 */
static Bool
ViaOutputsFindDotclockMaster(ScrnInfoPtr pScrn)
{
    struct ViaOutput  *Output;

    for (Output = VIAPTR(pScrn)->Outputs; Output; Output = Output->Next)
        if (Output->Active && Output->ClockMaster)
            return TRUE;
    return FALSE;
}
#endif

/*
 * Print a list of active output devices.
 */
static void
ViaOutputsPrint(ScrnInfoPtr pScrn)
{
    struct ViaOutput *Output = VIAPTR(pScrn)->Outputs;

    xf86DrvMsg(pScrn->scrnIndex, X_INFO, "Listing Outputs:\n");

    while (Output) {
        xf86DrvMsg(pScrn->scrnIndex, X_INFO, "Bus %s: Output %s: %s\n",
                   ViaOutputBusName(Output->Position), Output->Name,
                   Output->Active ? "Active" : "Disabled");

        Output = Output->Next;
    }
}

/*
 *
 */
Bool
ViaOutputsSelect(ScrnInfoPtr pScrn)
{
    VIAPtr pVia = VIAPTR(pScrn);
    struct ViaOutput *Output;
    Bool Active = FALSE, Found;

    if (pVia->IsSecondary) { /* we just abort for now */
	xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "%s: Not handling secondary.\n",
                   __func__);
	return FALSE;
    }

    VIAFUNC(pScrn->scrnIndex);
    ViaDebug(pScrn->scrnIndex, "X Configuration: 0x%02x\n", pVia->ActiveDevice);

    if (!pVia->ActiveDevice)
        Active = ViaOutputsSense(pScrn);
    else {
        if (pVia->ActiveDevice & VIA_DEVICE_LCD) {
            Found = FALSE;

            Output = pVia->Outputs;

            while (Output) {
                if (Output->Type & OUTPUT_PANEL) {
                    Found = TRUE;
                    Output->Active = TRUE;
                    break;
                }

                Output = Output->Next;
            }

            if (!Found)
		xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "Unable to activate"
			   " panel: no panel is present.\n");
            else
                Active = TRUE;
	}

        /* way too simplistic */
	if (pVia->ActiveDevice & VIA_DEVICE_TV) {
            Found = FALSE;

            Output = pVia->Outputs;
            while (Output) {
                if (Output->Type & OUTPUT_TV) {
                    if (Output->Sense) {
                        if (Output->Sense(Output))
                            Output->Active = TRUE;
                        else {
                            xf86DrvMsg(pScrn->scrnIndex, X_WARNING,
                                       "Unable to activate TV encoder: "
                                       "no cable attached.\n");

                            Output->Active = FALSE;
                        }
                    } else
                        Output->Active = TRUE;
                    Found = TRUE;
                }

                Output = Output->Next;
            }

            if (!Found)
                xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "Unable to activate"
                           " TV encoder: no TV encoder present.\n");
            else
                Active = TRUE;
        }

        if (pVia->ActiveDevice & VIA_DEVICE_CRT) {
            Active = TRUE;

            Output = pVia->Outputs;
            while (Output) {
                if (Output->Type & OUTPUT_CRT)
                    Output->Active = TRUE;

                Output = Output->Next;
            }
        }
    }

    if (!Active)
        return FALSE;

    /* Check our Outputs in case 2 devices demand control over the dotclock. */
    Found = FALSE;
    for (Output = pVia->Outputs; Output; Output = Output->Next)
        if (Output->Active && Output->ClockMaster) {
            if (!Found)
                Found = TRUE;
            else
                Output->Active = FALSE;
        }

    /* Check our Outputs in case we are trying to active both TV encoders
     * and the panel. This needs to get a proper solution soon */
    for (Output = pVia->Outputs; Output; Output = Output->Next)
        if (Output->Active && (Output->Type & OUTPUT_PANEL))
            break;

    if (Output) {
        for (Output = pVia->Outputs; Output; Output = Output->Next)
            if (Output->Active && (Output->Type & OUTPUT_TV))
                Output->Active = FALSE;
    }

#ifdef USE_SECONDARY
    /* set up outputs for CRTC */
    for (Output = pVia->Outputs; Output; Output = Output->Next)
        if (Output->Active)
            Output->Owner = VIA_CRTC_SECONDARY;
#endif

    ViaOutputsPrint(pScrn);

    return TRUE;
}

/*
 *
 */
void
ViaOutputModesCopyAdd(struct ViaOutput *Output, DisplayModePtr Additions)
{
    DisplayModePtr Mode, Last;

    if (!Additions)
        return;

    Last = Output->Modes;
    Mode = Additions;

    if (Last) {
        while (Last->next)
            Last = Last->next;
    } else {
        Output->Modes = ViaModeCopy(Mode);
        Last = Output->Modes;
        Mode = Mode->next;
    }

    while (Mode) {
        Last->next = ViaModeCopy(Mode);
        Last->next->prev = Last;

        Last = Last->next;
        Mode = Mode->next;
    }
}

/*
 *
 */
void
ViaOutputTimingSetFromConfig(ScrnInfoPtr pScrn)
{
    VIAPtr pVia = VIAPTR(pScrn);
    MonPtr Config = pScrn->confScreen->monitor;
    struct ViaOutput *Output;
    int i;

    /* find the first CRT */
    for (Output = pVia->Outputs; Output; Output = Output->Next)
        if (Output->Type & OUTPUT_CRT) /* LVDS and TV _always_ know better */
            break;

    if (!Output) /* Our work is done. */
        return;

    if (!Output->MonitorName)
        Output->MonitorName = xnfstrdup(Config->id);

    if (Config->nHsync) {
        xf86DrvMsg(pScrn->scrnIndex, X_INFO,
                   "\"%s - %s\": Imposing HSync values from config monitor \"%s\".\n",
                   Output->Name, Output->MonitorName, Config->id);
        Output->numHSync = Config->nHsync;
        for (i = 0; i < Config->nHsync; i++) {
            Output->HSync[i].lo = Config->hsync[i].lo;
            Output->HSync[i].hi = Config->hsync[i].hi;
        }
    } else if (!Output->numHSync) {
        Output->numHSync = 3;
        Output->HSync[0].lo = 31.5;
        Output->HSync[0].hi = 31.5;
        Output->HSync[1].lo = 35.15;
        Output->HSync[1].hi = 35.15;
        Output->HSync[2].lo = 35.5;
        Output->HSync[2].hi = 35.5;
    }

    if (Config->nVrefresh) {
        xf86DrvMsg(pScrn->scrnIndex, X_INFO,
                   "\"%s - %s\": Imposing VRefresh values from config monitor \"%s\".\n",
                   Output->Name, Output->MonitorName, Config->id);
        Output->numVRefresh = Config->nVrefresh;
        for (i = 0; i < Config->nVrefresh; i++) {
            Output->VRefresh[i].lo = Config->vrefresh[i].lo;
            Output->VRefresh[i].hi = Config->vrefresh[i].hi;
        }
    } else if (!Output->numVRefresh) {
        Output->numVRefresh = 1;
        Output->VRefresh[0].lo = 50;
        Output->VRefresh[0].hi = 61;
    }

#ifdef MONREC_HAS_REDUCED
    if (Config->reducedblanking)
        Output->ReducedAllowed = TRUE;
#endif

    ViaOutputModesCopyAdd(Output, Config->Modes);

    /* other stuff will not be needed for us here */
}

/*
 *
 * Output Bus handling.
 *
 */
/*
 *
 */
void
ViaOutputBusPower(struct ViaCrtc *Crtc, int Bus, Bool On)
{
    ScrnInfoPtr pScrn = xf86Screens[Crtc->scrnIndex];
    vgaHWPtr hwp = VGAHWPTR(pScrn);
    VIAPtr pVia = VIAPTR(pScrn);

    ViaDebug(Crtc->scrnIndex, "%s: Bus %s %s.\n", __func__,
             ViaOutputBusName(Bus), On ? "On" : "Off");

    switch (Bus) {
    case OUTPUT_BUS_CRT:
        ;
        break;
    case OUTPUT_BUS_DVP0:
        if (On)
            ViaSeqMask(hwp, 0x1E, 0xC0, 0xC0);
        else
            ViaSeqMask(hwp, 0x1E, 0x00, 0xC0);
        break;
    case OUTPUT_BUS_DVP1:
        if (On)
            ViaSeqMask(hwp, 0x1E, 0x30, 0x30);
        else
            ViaSeqMask(hwp, 0x1E, 0x00, 0x30);
        break;
    case OUTPUT_BUS_DFP:
        if (pVia->Chipset == VT3122) {
            if (On) {
                ViaSeqMask(hwp, 0x1A, 0x30, 0x30); /* Power on DI1 on */
                ViaCrtcMask(hwp, 0x93, 0x01, 0x01); /* Enable DI1 */
            } else {
                ViaSeqMask(hwp, 0x1A, 0x00, 0x30);
                ViaCrtcMask(hwp, 0x93, 0x00, 0x01);
            }
        } else {
            if (On)
                ViaSeqMask(hwp, 0x2A, 0x0F, 0x0F);
            else
                ViaSeqMask(hwp, 0x2A, 0x00, 0x0F);
        }
        break;
    case OUTPUT_BUS_DFPHIGH:
        if (On)
            ViaSeqMask(hwp, 0x2A, 0x0C, 0x0C);
        else
            ViaSeqMask(hwp, 0x2A, 0x00, 0x0C);
        break;
    case OUTPUT_BUS_DFPLOW:
        if (On)
            ViaSeqMask(hwp, 0x2A, 0x03, 0x03);
        else
            ViaSeqMask(hwp, 0x2A, 0x00, 0x03);
            break;
    default:
        xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "%s: unhandled bus: %d\n",
                   __func__, Bus);
        break;
    }
}

/*
 * Prepare the bus for this output.
 *
 */
void
ViaOutputBusSet(struct ViaCrtc *Crtc, int Bus)
{
    vgaHWPtr hwp = VGAHWPTR(xf86Screens[Crtc->scrnIndex]);
    VIAPtr pVia = VIAPTR(xf86Screens[Crtc->scrnIndex]);

    VIAFUNC(Crtc->scrnIndex);

    switch (Bus) {
    case OUTPUT_BUS_CRT:
        if (Crtc->ID == VIA_CRTC_SECONDARY)
            ViaSeqMask(hwp, 0x16, 0x40, 0x40);
        else
            ViaSeqMask(hwp, 0x16, 0x00, 0x40);

        break;
    case OUTPUT_BUS_DVP0:
        if (pVia->Chipset == VT3122) {
            Crtc->PLLFlags |= PLL_FLAG_EXTERNAL;

            if ((pVia->Chipset == VT3122) && VT3122_REV_IS_AX(pVia->HostRev)) {
                Crtc->PLLFlags |= PLL_FLAG_HALVE;
                if (Crtc->ID == VIA_CRTC_SECONDARY) /* polarity maybe?  */
                    ViaCrtcMask(hwp, 0x6C, 0x10, 0x10);

                if (pVia->HostRev != 0x02)
                    ViaCrtcMask(hwp, 0x6C, 0x00, 0x0E);
            }

            if (Crtc->ID == VIA_CRTC_SECONDARY) /* TV CLK source */
                ViaCrtcMask(hwp, 0x6C, 0x80, 0x80);
            else
                ViaCrtcMask(hwp, 0x6C, 0x00, 0x80);

            ViaCrtcMask(hwp, 0x6C, 0x01, 0x01); /* TV CLK input enable */
        } else {
            if (Crtc->ID == VIA_CRTC_SECONDARY)
                ViaCrtcMask(hwp, 0x96, 0x10, 0x10);
            else
                ViaCrtcMask(hwp, 0x96, 0x00, 0x10);
        }

        break;
    case OUTPUT_BUS_DVP1:
        if (Crtc->ID == VIA_CRTC_SECONDARY)
            ViaCrtcMask(hwp, 0x9B, 0x10, 0x10);
        else
            ViaCrtcMask(hwp, 0x9B, 0x00, 0x10);

        break;
    case OUTPUT_BUS_DFP:
        if (pVia->Chipset == VT3122) {
            if (Crtc->ID == VIA_CRTC_SECONDARY)
                ViaCrtcMask(hwp, 0x93, 0x80, 0x80); /* Set Source */
            else
                ViaCrtcMask(hwp, 0x93, 0x00, 0x80);
        } else {
            if (Crtc->ID == VIA_CRTC_SECONDARY) { /* Set DFP source */
                ViaCrtcMask(hwp, 0x97, 0x10, 0x10);
                ViaCrtcMask(hwp, 0x99, 0x10, 0x10);
            } else {
                ViaCrtcMask(hwp, 0x97, 0x00, 0x10);
                ViaCrtcMask(hwp, 0x99, 0x00, 0x10);
            }
        }

        ViaCrtcMask(hwp, 0x6A, 0x08, 0x08);

        break;
    default:
        xf86DrvMsg(Crtc->scrnIndex, X_ERROR, "%s: unhandled bus: %d\n",
                   __func__, Bus);
        return;
    }
}

/*
 * Power according to On all active devices, power off when inactive.
 *
 */
void
ViaOutputsPower(struct ViaCrtc *Crtc, Bool On)
{
    struct ViaOutput *Output = VIAPTR(xf86Screens[Crtc->scrnIndex])->Outputs;

    ViaDebug(Crtc->scrnIndex, "%s: %s.\n", __func__, On ? "On" : "Off");

    for (; Output; Output = Output->Next) {
        if (Output->Owner != Crtc->ID)
            continue;

        if (Output->Active && On) {
            ViaOutputBusPower(Crtc, Output->Position, TRUE);
        
            if (Output->Power)
                Output->Power(Output, TRUE);
        } else {
            if (Output->Power)
                Output->Power(Output, FALSE);

            ViaOutputBusPower(Crtc, Output->Position, FALSE);
        }
    }
}

/*
 * Handle Grammar Correction. ;)
 */
void
ViaOutputsGamma(ScrnInfoPtr pScrn, Bool On)
{
    vgaHWPtr hwp = VGAHWPTR(pScrn);
    VIAPtr pVia = VIAPTR(pScrn);
    struct ViaOutput *Output = pVia->Outputs;

    if (pVia->Chipset < VT3108) {
        if (On)
            ViaSeqMask(hwp, 0x16, 0x80, 0x80);
        else
            ViaSeqMask(hwp, 0x16, 0x00, 0x80);
    } else {
        if (On)
            ViaCrtcMask(hwp, 0x33, 0x80, 0x80);
        else
            ViaCrtcMask(hwp, 0x33, 0x00, 0x80);
    }

    while (Output) {
        if (Output->Active) {
            switch (Output->Position) {
            case OUTPUT_BUS_CRT:
                break;
            case OUTPUT_BUS_DVP0:
            case OUTPUT_BUS_DVP1:
                if (On)
                    ViaCrtcMask(hwp, 0x32, 0x04, 0x04);
                else
                    ViaCrtcMask(hwp, 0x32, 0x00, 0x04);
                break;
            case OUTPUT_BUS_DFP:
                if (On)
                    ViaCrtcMask(hwp, 0x6A, 0x02, 0x02);
                else
                    ViaCrtcMask(hwp, 0x6A, 0x00, 0x02);
                break;
            default:
                xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "%s: unhandled bus: %d\n",
                           __func__, Output->Position);
                break;
            }
        }
        Output = Output->Next;
    }
}
