/*
 * Copyright 2004-2007 Luc Verhaegen.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

/*
 * via_mode.c
 *
 * Everything to do with setting and changing modes.
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "via_driver.h"

#ifndef _XF86_ANSIC_H
#include <string.h>
#include <stdio.h>
#include <unistd.h>
#endif

#include "via_vgahw.h"
#include "via_id.h"
#include "via_crtc.h"
#include "via_output.h"
#include "via_mode.h"

/*
 * Some VGA register debugging.
 */
void
ViaVgaPrintRegs(ScrnInfoPtr pScrn, const char *function)
{
    vgaHWPtr hwp = VGAHWPTR(pScrn);
    int i;

    ViaDebug(pScrn->scrnIndex, "%s: Printing VGA registers:\n", function);
    ViaDebug(pScrn->scrnIndex, "Printing VGA Sequence registers:\n");
    for (i = 0x00; i < 0x80; i++)
	ViaDebug(pScrn->scrnIndex, "SR%02X: 0x%02X\n", i, hwp->readSeq(hwp, i));

    ViaDebug(pScrn->scrnIndex, "Printing VGA CRTM/C registers:\n");
    for (i = 0x00; i < 0x19; i++)
        ViaDebug(pScrn->scrnIndex, "CR%02X: 0x%02X\n", i, hwp->readCrtc(hwp, i));
    for (i = 0x33; i < 0xA3; i++)
	ViaDebug(pScrn->scrnIndex, "CR%02X: 0x%02X\n", i, hwp->readCrtc(hwp, i));

    ViaDebug(pScrn->scrnIndex, "Printing VGA Graphics registers:\n");
    for (i = 0x00; i < 0x08; i++)
	ViaDebug(pScrn->scrnIndex, "GR%02X: 0x%02X\n", i, hwp->readGr(hwp, i));

    ViaDebug(pScrn->scrnIndex, "Printing VGA Attribute registers:\n");
    for (i = 0x00; i < 0x14; i++)
	ViaDebug(pScrn->scrnIndex, "AR%02X: 0x%02X\n", i, hwp->readAttr(hwp, i));

    ViaDebug(pScrn->scrnIndex, "Printing VGA Miscellaneous register:\n");
    ViaDebug(pScrn->scrnIndex, "Misc: 0x%02X\n", hwp->readMiscOut(hwp));

    ViaDebug(pScrn->scrnIndex, "End of VGA Registers.\n");
}

/*
 * Temporary.
 */
static void
add(char **p, char *new)
{
    *p = xnfrealloc(*p, strlen(*p) + strlen(new) + 2);
    strcat(*p, " ");
    strcat(*p, new);
}

void
ViaPrintModeline(int scrnIndex, DisplayModePtr mode)
{
    char tmp[256];
    char *flags = xnfcalloc(1, 1);

    if (mode->HSkew) {
	snprintf(tmp, 256, "hskew %i", mode->HSkew);
	add(&flags, tmp);
    }
    if (mode->VScan) {
	snprintf(tmp, 256, "vscan %i", mode->VScan);
	add(&flags, tmp);
    }
    if (mode->Flags & V_INTERLACE) add(&flags, "interlace");
    if (mode->Flags & V_CSYNC) add(&flags, "composite");
    if (mode->Flags & V_DBLSCAN) add(&flags, "doublescan");
    if (mode->Flags & V_BCAST) add(&flags, "bcast");
    if (mode->Flags & V_PHSYNC) add(&flags, "+hsync");
    if (mode->Flags & V_NHSYNC) add(&flags, "-hsync");
    if (mode->Flags & V_PVSYNC) add(&flags, "+vsync");
    if (mode->Flags & V_NVSYNC) add(&flags, "-vsync");
    if (mode->Flags & V_PCSYNC) add(&flags, "+csync");
    if (mode->Flags & V_NCSYNC) add(&flags, "-csync");
#if 0
    if (mode->Flags & V_CLKDIV2) add(&flags, "vclk/2");
#endif
    xf86DrvMsgVerb(scrnIndex, X_INFO, 7,
		   "Modeline \"%s\"  %6.2f  %i %i %i %i  %i %i %i %i%s\n",
		   mode->name, mode->Clock/1000., mode->HDisplay,
		   mode->HSyncStart, mode->HSyncEnd, mode->HTotal,
		   mode->VDisplay, mode->VSyncStart, mode->VSyncEnd,
		   mode->VTotal, flags);
    xfree(flags);
}

#if 0
/*
 *
 */
static void
ViaMonitorDebug(int scrnIndex, MonPtr Monitor)
{
    int i;
    DisplayModePtr Mode;

    ViaDebug(scrnIndex, "Monitor: %s (%s - %s)\n", Monitor->id,
             Monitor->vendor, Monitor->model);

    for (i = 0; i < Monitor->nHsync; i++)
        ViaDebug(scrnIndex, "Horizontal Range %d: %f - %f\n", i,
                 Monitor->hsync[i].lo, Monitor->hsync[i].hi);

    for (i = 0; i < Monitor->numVRefresh; i++)
        ViaDebug(scrnIndex, "Vertical Range %d: %f - %f\n", i,
                 Monitor->VRefresh[i].lo, Monitor->VRefresh[i].hi);

    Mode = Monitor->Modes;
    while (Mode) {
        ViaPrintModeline(scrnIndex, Mode);
        Mode = Mode->next;
    }
    ViaDebug(scrnIndex, "Physical dimensions: %dmm x %dmm\n",
             Monitor->widthmm, Monitor->heightmm);

    ViaDebug(scrnIndex, "Gamma %f, %f, %f\n", Monitor->gamma.red,
             Monitor->gamma.green, Monitor->gamma.blue);

    if (Monitor->DDC)
        ViaDebug(scrnIndex, "Monitor has DDC\n");

#ifdef MONREC_HAS_REDUCED
    if (Monitor->reducedblanking)
        ViaDebug(scrnIndex, "Monitor allows for reduced blanking\n");
#endif
}
#endif

#if 0
/*
 *
 */
static void
ViaModeListPrint(DisplayModePtr Modes)
{
    DisplayModePtr Mode;

    xf86Msg(X_INFO, "%s:\n", __func__);
    for (Mode = Modes; Mode && (Mode != Modes); Mode = Mode->next)
        xf86Msg(X_INFO, "%p: \"%s\" (%dx%d:%3.1fMhz): (%p <- x -> %p)\n",
                (void *) Mode, Mode->name, Mode->HDisplay, Mode->VDisplay,
                Mode->Clock / 1000.0, (void *) Mode->prev, (void *) Mode->next);
}
#endif

/*
 * Generate a CVT standard mode from HDisplay, VDisplay and VRefresh.
 *
 * These calculations are stolen from the CVT calculation spreadsheet written
 * by Graham Loveridge. He seems to be claiming no copyright and there seems to
 * be no license attached to this. He apparently just wants to see his name
 * mentioned.
 *
 * This file can be found at http://www.vesa.org/Public/CVT/CVTd6r1.xls
 *
 * Comments and structure corresponds to the comments and structure of the xls.
 * This should ease importing of future changes to the standard (not very
 * likely though).
 *
 * About margins; i'm sure that they are to be the bit between HDisplay and
 * HBlankStart, HBlankEnd and HTotal, VDisplay and VBlankStart, VBlankEnd and
 * VTotal, where the overscan colour is shown. FB seems to call _all_ blanking
 * outside sync "margin" for some reason. Since we prefer seeing proper
 * blanking instead of the overscan colour, and since the Crtc* values will
 * probably get altered after us, we will disable margins altogether. With
 * these calculations, Margins will plainly expand H/VDisplay, and we don't
 * want that. -- libv
 *
 */
DisplayModePtr
ViaCVTMode(int HDisplay, int VDisplay, float VRefresh, Bool Reduced,
           Bool Interlaced)
{
    DisplayModeRec  *Mode = xnfalloc(sizeof(DisplayModeRec));

    /* 1) top/bottom margin size (% of height) - default: 1.8 */
#define CVT_MARGIN_PERCENTAGE 1.8

    /* 2) character cell horizontal granularity (pixels) - default 8 */
#define CVT_H_GRANULARITY 8

    /* 4) Minimum vertical porch (lines) - default 3 */
#define CVT_MIN_V_PORCH 3

    /* 4) Minimum number of vertical back porch lines - default 6 */
#define CVT_MIN_V_BPORCH 6

    /* Pixel Clock step (kHz) */
#define CVT_CLOCK_STEP 250

    Bool Margins = FALSE;
    float  VFieldRate, HPeriod;
    int  HDisplayRnd, HMargin;
    int  VDisplayRnd, VMargin, VSync;
    float  Interlace; /* Please rename this */

    memset(Mode, 0, sizeof(DisplayModeRec));

    /* CVT default is 60.0Hz */
    if (!VRefresh)
        VRefresh = 60.0;

    /* 1. Required field rate */
    if (Interlaced)
        VFieldRate = VRefresh * 2;
    else
        VFieldRate = VRefresh;

    /* 2. Horizontal pixels */
    HDisplayRnd = HDisplay - (HDisplay % CVT_H_GRANULARITY);

    /* 3. Determine left and right borders */
    if (Margins) {
        /* right margin is actually exactly the same as left */
        HMargin = (((float) HDisplayRnd) * CVT_MARGIN_PERCENTAGE / 100.0);
        HMargin -= HMargin % CVT_H_GRANULARITY;
    } else
        HMargin = 0;

    /* 4. Find total active pixels */
    Mode->HDisplay = HDisplayRnd + 2*HMargin;

    /* 5. Find number of lines per field */
    if (Interlaced)
        VDisplayRnd = VDisplay / 2;
    else
        VDisplayRnd = VDisplay;

    /* 6. Find top and bottom margins */
    /* nope. */
    if (Margins)
        /* top and bottom margins are equal again. */
        VMargin = (((float) VDisplayRnd) * CVT_MARGIN_PERCENTAGE / 100.0);
    else
        VMargin = 0;

    Mode->VDisplay = VDisplay + 2*VMargin;

    /* 7. Interlace */
    if (Interlaced)
        Interlace = 0.5;
    else
        Interlace = 0.0;

    /* Determine VSync Width from aspect ratio */
    if (!(VDisplay % 3) && ((VDisplay * 4 / 3) == HDisplay))
        VSync = 4;
    else if (!(VDisplay % 9) && ((VDisplay * 16 / 9) == HDisplay))
        VSync = 5;
    else if (!(VDisplay % 10) && ((VDisplay * 16 / 10) == HDisplay))
        VSync = 6;
    else if (!(VDisplay % 4) && ((VDisplay * 5 / 4) == HDisplay))
        VSync = 7;
    else if (!(VDisplay % 9) && ((VDisplay * 15 / 9) == HDisplay))
        VSync = 7;
    else /* Custom */
        VSync = 10;

    if (!Reduced) { /* simplified GTF calculation */

        /* 4) Minimum time of vertical sync + back porch interval (s)
         * default 550.0 */
#define CVT_MIN_VSYNC_BP 550.0

        /* 3) Nominal HSync width (% of line period) - default 8 */
#define CVT_HSYNC_PERCENTAGE 8

        float  HBlankPercentage;
        int  VSyncAndBackPorch, VBackPorch;
        int  HBlank;

        /* 8. Estimated Horizontal period */
        HPeriod = ((float) (1000000.0 / VFieldRate - CVT_MIN_VSYNC_BP)) /
            (VDisplayRnd + 2 * VMargin + CVT_MIN_V_PORCH + Interlace);

        /* 9. Find number of lines in sync + backporch */
        if (((int)(CVT_MIN_VSYNC_BP / HPeriod) + 1) < (VSync + CVT_MIN_V_PORCH))
            VSyncAndBackPorch = VSync + CVT_MIN_V_PORCH;
        else
            VSyncAndBackPorch = (int)(CVT_MIN_VSYNC_BP / HPeriod) + 1;

        /* 10. Find number of lines in back porch */
        VBackPorch = VSyncAndBackPorch - VSync;

        /* 11. Find total number of lines in vertical field */
        Mode->VTotal = VDisplayRnd + 2 * VMargin + VSyncAndBackPorch + Interlace
            + CVT_MIN_V_PORCH;

        /* 5) Definition of Horizontal blanking time limitation */
        /* Gradient (%/kHz) - default 600 */
#define CVT_M_FACTOR 600

        /* Offset (%) - default 40 */
#define CVT_C_FACTOR 40

        /* Blanking time scaling factor - default 128 */
#define CVT_K_FACTOR 128

        /* Scaling factor weighting - default 20 */
#define CVT_J_FACTOR 20

#define CVT_M_PRIME CVT_M_FACTOR * CVT_K_FACTOR / 256
#define CVT_C_PRIME (CVT_C_FACTOR - CVT_J_FACTOR) * CVT_K_FACTOR / 256 + \
        CVT_J_FACTOR

        /* 12. Find ideal blanking duty cycle from formula */
        HBlankPercentage = CVT_C_PRIME - CVT_M_PRIME * HPeriod/1000.0;

        /* 13. Blanking time */
        if (HBlankPercentage < 20)
            HBlankPercentage = 20;

        HBlank = Mode->HDisplay * HBlankPercentage/(100.0 - HBlankPercentage);
        HBlank -= HBlank % (2*CVT_H_GRANULARITY);

        /* 14. Find total number of pixels in a line. */
        Mode->HTotal = Mode->HDisplay + HBlank;

        /* Fill in HSync values */
        Mode->HSyncEnd = Mode->HDisplay + HBlank / 2;

        Mode->HSyncStart = Mode->HSyncEnd -
            (Mode->HTotal * CVT_HSYNC_PERCENTAGE) / 100;
        Mode->HSyncStart += CVT_H_GRANULARITY -
            Mode->HSyncStart % CVT_H_GRANULARITY;

        /* Fill in VSync values */
        Mode->VSyncStart = Mode->VDisplay + CVT_MIN_V_PORCH;
        Mode->VSyncEnd = Mode->VSyncStart + VSync;

    } else { /* Reduced blanking */
        /* Minimum vertical blanking interval time (s) - default 460 */
#define CVT_RB_MIN_VBLANK 460.0

        /* Fixed number of clocks for horizontal sync */
#define CVT_RB_H_SYNC 32.0

        /* Fixed number of clocks for horizontal blanking */
#define CVT_RB_H_BLANK 160.0

        /* Fixed number of lines for vertical front porch - default 3 */
#define CVT_RB_VFPORCH 3

        int  VBILines;

        /* 8. Estimate Horizontal period. */
        HPeriod = ((float) (1000000.0 / VFieldRate - CVT_RB_MIN_VBLANK)) /
            (VDisplayRnd + 2*VMargin);

        /* 9. Find number of lines in vertical blanking */
        VBILines = ((float) CVT_RB_MIN_VBLANK) / HPeriod + 1;

        /* 10. Check if vertical blanking is sufficient */
        if (VBILines < (CVT_RB_VFPORCH + VSync + CVT_MIN_V_BPORCH))
            VBILines = CVT_RB_VFPORCH + VSync + CVT_MIN_V_BPORCH;

        /* 11. Find total number of lines in vertical field */
        Mode->VTotal = VDisplayRnd + 2 * VMargin + Interlace + VBILines;

        /* 12. Find total number of pixels in a line */
        Mode->HTotal = Mode->HDisplay + CVT_RB_H_BLANK;

        /* Fill in HSync values */
        Mode->HSyncEnd = Mode->HDisplay + CVT_RB_H_BLANK / 2;
        Mode->HSyncStart = Mode->HSyncEnd - CVT_RB_H_SYNC;

        /* Fill in VSync values */
        Mode->VSyncStart = Mode->VDisplay + CVT_RB_VFPORCH;
        Mode->VSyncEnd = Mode->VSyncStart + VSync;
    }

    /* 15/13. Find pixel clock frequency (kHz for xf86) */
    Mode->Clock = Mode->HTotal * 1000.0 / HPeriod;
    Mode->Clock -= Mode->Clock % CVT_CLOCK_STEP;

    /* 16/14. Find actual Horizontal Frequency (kHz) */
    Mode->HSync = ((float) Mode->Clock) / ((float) Mode->HTotal);

    /* 17/15. Find actual Field rate */
    Mode->VRefresh = (1000.0 * ((float) Mode->Clock)) /
        ((float) (Mode->HTotal * Mode->VTotal));

    /* 18/16. Find actual vertical frame frequency */
    /* ignore - just set the mode flag for interlaced */
    if (Interlaced)
        Mode->VTotal *= 2;

    {
        char  Name[256];
        Name[0] = 0;

        snprintf(Name, 256, "%dx%d", HDisplay, VDisplay);
        Mode->name = xnfstrdup(Name);
    }

    if (Reduced)
        Mode->Flags |= V_PHSYNC | V_NVSYNC;
    else
        Mode->Flags |= V_NHSYNC | V_PVSYNC;

    if (Interlaced)
        Mode->Flags |= V_INTERLACE;

    return Mode;
}

/*
 * xf86Mode.c should have a some more DisplayModePtr list handling.
 */
DisplayModePtr
ViaModesAdd(DisplayModePtr Modes, DisplayModePtr Additions)
{
    if (!Modes) {
        if (Additions)
            return Additions;
        else
            return NULL;
    }

    if (Additions) {
        DisplayModePtr Mode = Modes;

        while (Mode->next)
            Mode = Mode->next;

        Mode->next = Additions;
        Additions->prev = Mode;
    }

    return Modes;
}

/*
 *
 */
DisplayModePtr
ViaModeCopy(DisplayModePtr Mode)
{
    DisplayModePtr New;

    if (!Mode)
        return NULL;

    New = xnfalloc(sizeof(DisplayModeRec));
    memcpy(New, Mode, sizeof(DisplayModeRec)); /* re-use private */
    New->name = xnfstrdup(Mode->name);
    New->prev = NULL;
    New->next = NULL;
    New->Private = Mode->Private;
    New->PrivSize = Mode->PrivSize;

    return New;
}

/*
 *
 */
void
ViaModesDestroy(DisplayModePtr Modes)
{
    DisplayModePtr mode = Modes, next;

    while (mode) {
        next = mode->next;
        xfree(mode->name);
        xfree(mode);
        mode = next;
    }
}

/*
 * Basic sanity checks.
 */
static int
ViaModeSanity(DisplayModePtr Mode)
{
    /* do we need to bother at all? */
    if (Mode->status != MODE_OK)
        return Mode->status;

    if (!Mode->name)
        return MODE_ERROR;

    if (Mode->Clock <= 0)
        return MODE_NOCLOCK;

    if ((Mode->HDisplay <= 0) || (Mode->HSyncStart <= 0) ||
        (Mode->HSyncEnd <= 0) || (Mode->HTotal <= 0))
        return MODE_H_ILLEGAL;

    /* Align to character width */
    if (Mode->HDisplay & 0x07)
        Mode->HDisplay = (Mode->HDisplay + 7) & ~7;
    if (Mode->HSyncStart & 0x07)
        Mode->HSyncStart = (Mode->HSyncStart + 7) & ~7;
    if (Mode->HSyncEnd & 0x07)
        Mode->HSyncEnd = (Mode->HSyncEnd + 7) & ~7;
    if (Mode->HTotal & 0x07)
        Mode->HTotal = (Mode->HTotal + 7) & ~7;

    if ((Mode->HTotal <= Mode->HSyncEnd) ||
        (Mode->HSyncEnd <= Mode->HSyncStart) ||
        (Mode->HSyncStart < Mode->HDisplay))
        return MODE_H_ILLEGAL;

    /* HSkew? */

    if ((Mode->VDisplay <= 0) || (Mode->VSyncStart <= 0) ||
        (Mode->VSyncEnd <= 0) || (Mode->VTotal <= 0))
        return MODE_V_ILLEGAL;

    if ((Mode->VTotal <= Mode->VSyncEnd) ||
        (Mode->VSyncEnd <= Mode->VSyncStart) ||
        (Mode->VSyncStart < Mode->VDisplay))
        return MODE_V_ILLEGAL;

    if ((Mode->VScan != 0) && (Mode->VScan != 1))
        return MODE_NO_VSCAN;

    if (Mode->Flags & V_INTERLACE)
        return MODE_NO_INTERLACE;

    if (Mode->Flags & V_DBLSCAN)
        return MODE_NO_DBLESCAN;

    /* Flags */
    return MODE_OK;
}

/*
 * After we passed the initial sanity check, we need to fill out the CRTC
 * values.
 */
static void
ViaModeFillOutCrtcValues(DisplayModePtr Mode)
{
    /* do we need to bother at all? */
    if (Mode->status != MODE_OK)
        return;

    Mode->ClockIndex = -1; /* Always! direct non-programmable support must die. */

    if (!Mode->SynthClock)
        Mode->SynthClock = Mode->Clock;

    if (!Mode->CrtcHDisplay)
        Mode->CrtcHDisplay = Mode->HDisplay;

    if (!Mode->CrtcHBlankStart)
        Mode->CrtcHBlankStart = Mode->HDisplay;

    if (!Mode->CrtcHSyncStart)
        Mode->CrtcHSyncStart = Mode->HSyncStart;

    if (!Mode->CrtcHSyncEnd)
        Mode->CrtcHSyncEnd = Mode->HSyncEnd;

    if (!Mode->CrtcHBlankEnd)
        Mode->CrtcHBlankEnd = Mode->HTotal;

    if (!Mode->CrtcHTotal)
        Mode->CrtcHTotal = Mode->HTotal;

    if (!Mode->CrtcHSkew)
        Mode->CrtcHSkew = Mode->HSkew;

    if (!Mode->CrtcVDisplay)
        Mode->CrtcVDisplay = Mode->VDisplay;

    if (!Mode->CrtcVBlankStart)
        Mode->CrtcVBlankStart = Mode->VDisplay;

    if (!Mode->CrtcVSyncStart)
        Mode->CrtcVSyncStart = Mode->VSyncStart;

    if (!Mode->CrtcVSyncEnd)
        Mode->CrtcVSyncEnd = Mode->VSyncEnd;

    if (!Mode->CrtcVBlankEnd)
        Mode->CrtcVBlankEnd = Mode->VTotal;

    if (!Mode->CrtcVTotal)
        Mode->CrtcVTotal = Mode->VTotal;

    /* Always change these */
    Mode->HSync = ((float) Mode->SynthClock) / Mode->CrtcHTotal;
    Mode->VRefresh = (Mode->SynthClock * 1000.0) /
        (Mode->CrtcHTotal * Mode->CrtcVTotal);

    /* We're usually first in the chain, right after ViaModeSanity. */
    Mode->CrtcHAdjusted = FALSE;
    Mode->CrtcVAdjusted = FALSE;

    /* Steer clear of PrivSize, Private and PrivFlags */
}

/*
 * Basic sanity checks.
 */
static int
ViaModeCrtcSanity(DisplayModePtr Mode)
{
    if (Mode->SynthClock <= 0)
        return MODE_NOCLOCK;

    if ((Mode->CrtcHDisplay <= 0) || (Mode->CrtcHBlankStart <= 0) ||
        (Mode->CrtcHSyncStart <= 0) || (Mode->CrtcHSyncEnd <= 0) ||
        (Mode->CrtcHBlankEnd <= 0) || (Mode->CrtcHTotal <= 0))
        return MODE_H_ILLEGAL;

    /* Align to character width */
    if (Mode->CrtcHDisplay & 0x07) {
        Mode->CrtcHDisplay = (Mode->CrtcHDisplay + 7) & ~7;
        Mode->CrtcHAdjusted = TRUE;
    }
    if (Mode->CrtcHBlankStart & 0x07) {
        Mode->CrtcHBlankStart = (Mode->CrtcHBlankStart + 7) & ~7;
        Mode->CrtcHAdjusted = TRUE;
    }
    if (Mode->CrtcHSyncStart & 0x07) {
        Mode->CrtcHSyncStart = (Mode->CrtcHSyncStart + 7) & ~7;
        Mode->CrtcHAdjusted = TRUE;
    }
    if (Mode->CrtcHSyncEnd & 0x07) {
        Mode->CrtcHSyncEnd = (Mode->CrtcHSyncEnd + 7) & ~7;
        Mode->CrtcHAdjusted = TRUE;
    }
    if (Mode->CrtcHBlankEnd & 0x07) {
        Mode->CrtcHBlankEnd = (Mode->CrtcHBlankEnd + 7) & ~7;
        Mode->CrtcHAdjusted = TRUE;
    }
    if (Mode->CrtcHTotal & 0x07) {
        Mode->CrtcHTotal = (Mode->CrtcHTotal + 7) & ~7;
        Mode->CrtcHAdjusted = TRUE;
    }

    if ((Mode->CrtcHTotal < Mode->CrtcHBlankEnd) ||
        (Mode->CrtcHBlankEnd <= Mode->CrtcHSyncEnd) ||
        (Mode->CrtcHSyncEnd <= Mode->CrtcHSyncStart) ||
        (Mode->CrtcHSyncStart < Mode->CrtcHBlankStart) ||
        (Mode->CrtcHBlankStart < Mode->CrtcHDisplay))
        return MODE_H_ILLEGAL;

    /* CrtcHSkew? */

    if ((Mode->CrtcVDisplay <= 0) || (Mode->CrtcVBlankStart <= 0) ||
        (Mode->CrtcVSyncStart <= 0) || (Mode->CrtcVSyncEnd <= 0) ||
        (Mode->CrtcVBlankEnd <= 0) || (Mode->CrtcVTotal <= 0))
        return MODE_V_ILLEGAL;

    if ((Mode->CrtcVTotal < Mode->CrtcVBlankEnd) ||
        (Mode->CrtcVBlankEnd <= Mode->CrtcVSyncEnd) ||
        (Mode->CrtcVSyncEnd <= Mode->CrtcVSyncStart) ||
        (Mode->CrtcVSyncStart < Mode->CrtcVBlankStart) ||
        (Mode->CrtcVBlankStart < Mode->CrtcVDisplay))
        return MODE_V_ILLEGAL;

    /* Always change these */
    Mode->HSync = ((float) Mode->SynthClock) / Mode->CrtcHTotal;
    Mode->VRefresh = (Mode->SynthClock * 1000.0) /
        (Mode->CrtcHTotal * Mode->CrtcVTotal);

    return MODE_OK;
}

/*
 *
 */
static int
ViaModeValidate(ScrnInfoPtr pScrn, DisplayModePtr Mode)
{
    VIAPtr pVia = VIAPTR(pScrn);
#ifndef USE_SECONDARY
    struct ViaCrtc *Crtc = pVia->Crtc[0];
#else
    struct ViaCrtc *Crtc = pVia->Crtc[1];
#endif
    int Status, i;

    Status = ViaModeSanity(Mode);
    if (Status != MODE_OK)
        return Status;

    ViaModeFillOutCrtcValues(Mode);

    /* We don't want to loop around this forever */
    for (i = 10; i; i--) {
        struct ViaOutput *Output;

        Mode->CrtcHAdjusted = FALSE;
        Mode->CrtcVAdjusted = FALSE;

        Status = ViaModeCrtcSanity(Mode);
        if (Status != MODE_OK)
            return Status;
        if (Mode->CrtcHAdjusted || Mode->CrtcVAdjusted)
            continue;

        /* Did we set up virtual resolution already? */
        if ((pScrn->virtualX > 0) && (pScrn->virtualY > 0)) {
            if (pScrn->virtualX < Mode->HDisplay)
                return MODE_VIRTUAL_X;
            if (pScrn->virtualY < Mode->VDisplay)
                return MODE_VIRTUAL_Y;
        } else { /* just check our videoRam for now */
            if (((((Mode->CrtcHDisplay >> 3) + 0x03) & ~0x03) * Mode->CrtcVDisplay
                 * pScrn->bitsPerPixel) > (pScrn->videoRam * 1024))
                return MODE_MEM;
        }

        Status = Crtc->ModeValidate(Crtc, Mode);
        if (Status != MODE_OK)
            return Status;
        if (Mode->CrtcHAdjusted || Mode->CrtcVAdjusted)
            continue;

        for (Output = pVia->Outputs; Output; Output = Output->Next)
            if (Output->Active) {
                Status = ViaModeOutputValid(Output, Mode);
                if (Status != MODE_OK)
                    return Status;
                if (Mode->CrtcHAdjusted || Mode->CrtcVAdjusted)
                    break; /* restart. */
            }

        if (!Output) /* We're done. This must be a good mode. */
            return MODE_OK;
    }

    /* Mode has been bouncing around for ages, on adjustments */
    xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "%s: Mode \"%s\" (%dx%d:%3.1fMhz) was"
               " thrown around for too long.\n", __func__, Mode->name,
               Mode->HDisplay, Mode->VDisplay, Mode->Clock/1000.0);
    return MODE_ERROR;
}

/*
 * Wrap the limited xf86 Mode statusses with our own message.
 */
struct {
    int Status;
    char *Message;
} ViaModeStatusMessages[] = {
    { MODE_NO_REDUCED,    "Reduced blanking is not supported."},
    { MODE_MEM_BW,        "Memory bandwidth exceeded."},
    { MODE_OUTPUT_UNDEF,  "Mode not defined by output device."},
    { MODE_NOT_PAL,       "This is not a PAL TV mode."},
    { MODE_NOT_NTSC,      "This is not an NTSC TV mode."},
    { MODE_HTOTAL_WIDE,   "Horizontal Total is out of range."},
    { MODE_HDISPLAY_WIDE, "Mode is too wide."},
    { MODE_HSYNC_RANGE,   "Horizontal Sync Start is out of range."},
    { MODE_HBLANK_RANGE,  "Horizontal Blanking Start is out of range."},
    { MODE_VTOTAL_WIDE,   "Vertical Total is out of range.\n"},
    { MODE_VDISPLAY_WIDE, "Mode is too high."},
    { MODE_VSYNC_RANGE,   "Vertical Sync Start is out of range.\n"},
    { MODE_VBLANK_RANGE,  "Vertical Blanking Start is out of range."},
    { MODE_PITCH,         "Scanout buffer Pitch too wide."},
    { MODE_OFFSET,        "Scanout buffer offset too high in FB."},
    { MODE_MINHEIGHT,     "Height too low."},
    { 0, NULL}
};

static const char *
ViaModeStatusToString(int Status)
{
    if ((Status & 0xFFF00) == VIA_MODE_STATUS) {
        int i;

        for (i = 0; ViaModeStatusMessages[i].Message; i++)
            if (ViaModeStatusMessages[i].Status == Status)
                return ViaModeStatusMessages[i].Message;
        ErrorF("%s: unhandled Status type: 0x%X\n", __func__, Status);
        return "Unknown status.";

    } else
        return xf86ModeStatusToString(Status);
}

/*
 * Create the list of all modes that are currently valid
 */
static DisplayModePtr
ViaCreateModesListAndValidate(ScrnInfoPtr pScrn, Bool Silent)
{
    VIAPtr pVia = VIAPTR(pScrn);
    DisplayModePtr Keepers = NULL, Mode, Check;
    struct ViaOutput *Output = pVia->Outputs;
    int Status;

    /* First Pass, X's own Modes. */
    if (!Silent)
        xf86DrvMsg(pScrn->scrnIndex, X_INFO, "Validating Modes from X (config &"
                   " built-in)\n");
    Check = pVia->XModes;
    while (1) {
        for (; Check; Check = Check->next) {
            Mode = ViaModeCopy(Check);

            Status = ViaModeValidate(pScrn, Mode);
            if (Status == MODE_OK)
                Keepers = ViaModesAdd(Keepers, Mode);
            else {
                if (!Silent)
                    xf86DrvMsg(pScrn->scrnIndex, X_INFO, "Rejected mode \"%s\" "
                               "(%dx%d:%3.1fMhz): %s\n", Mode->name,
                               Mode->HDisplay, Mode->VDisplay,
                               Mode->Clock / 1000.0, ViaModeStatusToString(Status));
                xfree(Mode->name);
                xfree(Mode);
            }
        }

        /* Move on to the Outputs, in turn. */
        while (Output && !Check) {
            if (Output->Active && Output->Modes) {
                if (!Silent)
                    xf86DrvMsg(pScrn->scrnIndex, X_INFO, "Validating Modes "
                               "from Output \"%s - %s\"\n", Output->Name,
                               Output->MonitorName);
                Check = Output->Modes;
            }
            Output = Output->Next;
        }
        if (!Check && !Output) /* No more outputs?, we're done! */
            break;
    }

    return Keepers;
}

/*
 *
 */
static DisplayModePtr
ViaModesGrabOnNameAll(DisplayModePtr *Modes, char *name)
{
    DisplayModePtr Mode, Matched = NULL, Temp;

    for (Mode = *Modes; Mode; ) {
        if (!strcmp(Mode->name, name)) {
            Temp = Mode;
            Mode = Mode->next;

            if (Temp->prev)
                Temp->prev->next = Mode;
            else
                *Modes = Mode;

            if (Mode)
                Mode->prev = Temp->prev;

            Temp->prev = NULL;
            Temp->next = Matched;
            if (Matched)
                Matched->prev = Temp;
            Matched = Temp;
        } else
            Mode = Mode->next;
    }

    return Matched;
}

/*
 *
 */
static DisplayModePtr
ViaModesGrabOnTypeAll(DisplayModePtr *Modes, int Type, int Mask)
{
    DisplayModePtr Mode, Matched = NULL, Temp;

    for (Mode = *Modes; Mode; ) {
        if ((Mode->type & Mask) == (Type & Mask)) {
            Temp = Mode;
            Mode = Mode->next;

            if (Temp->prev)
                Temp->prev->next = Mode;
            else
                *Modes = Mode;

            if (Mode)
                Mode->prev = Temp->prev;

            Temp->next = Matched;
            if (Matched)
                Matched->prev = Temp;
            Temp->prev = NULL;
            Matched = Temp;
        } else
            Mode = Mode->next;
    }

    return Matched;
}

/*
 *
 */
static DisplayModePtr
ViaModesGrabBestRefresh(DisplayModePtr *Modes)
{
    DisplayModePtr Mode, Best = NULL;

    if (!*Modes)
        return NULL;

    Best = *Modes;

    for (Mode = Best->next; Mode; Mode = Mode->next)
        if (Best->VRefresh < Mode->VRefresh)
            Best = Mode;
        else if (Best->VRefresh == Mode->VRefresh) {
            /* Same name != same resolution */
            if ((Best->HDisplay * Best->VDisplay) <
                (Mode->HDisplay * Mode->VDisplay))
                Best = Mode;
            else if ((Best->HDisplay * Best->VDisplay) ==
                     (Mode->HDisplay * Mode->VDisplay)) {
                /* Lower bandwidth == better! */
                if (Best->Clock > Mode->Clock)
                    Best = Mode;
            }
        }

    if (Best->next)
        Best->next->prev = Best->prev;
    if (Best->prev)
        Best->prev->next = Best->next;
    if (Best == *Modes)
        *Modes = (*Modes)->next;

    Best->next = NULL;
    Best->prev = NULL;

    return Best;
}

/*
 *
 */
static DisplayModePtr
ViaModesGrabOnHighestType(DisplayModePtr *Modes)
{
    DisplayModePtr Mode;

    /* User provided, but can also have another source. */
    Mode = ViaModesGrabOnTypeAll(Modes, M_T_USERDEF, 0xF0);
    if (Mode)
        return Mode;

    /* Often EDID provided, but can also have another source. */
    Mode = ViaModesGrabOnTypeAll(Modes, M_T_DRIVER, 0xF0);
    if (Mode)
        return Mode;

    /* No reason why we should treat built-in and vesa seperately */
    Mode = *Modes;
    *Modes = NULL;
    return Mode;
}

/*
 *
 */
static DisplayModePtr
ViaModesSortOnSize(DisplayModePtr Modes)
{
    DisplayModePtr Sorted, Mode, Temp, Next;

    if (!Modes)
        return NULL;

    Sorted = Modes;
    Modes = Modes->next;

    Sorted->next = NULL;
    Sorted->prev = NULL;

    for (Next = Modes; Next; ) {
        /* since we're taking modelines from in between */
        Mode = Next;
        Next = Next->next;

        for (Temp = Sorted; Temp; Temp = Temp->next) {
            /* nasty ! */
            if (((Temp->CrtcHDisplay * Temp->CrtcVDisplay) <
                (Mode->CrtcHDisplay * Mode->CrtcVDisplay)) ||
                (((Temp->CrtcHDisplay * Temp->CrtcVDisplay) ==
                  (Mode->CrtcHDisplay * Mode->CrtcVDisplay)) &&
                 ((Temp->VRefresh < Mode->VRefresh)  ||
                  ((Temp->VRefresh < Mode->VRefresh) &&
                   (Temp->SynthClock < Mode->SynthClock))))) {
                Mode->next = Temp;
                Mode->prev = Temp->prev;
                Temp->prev = Mode;
                if (Mode->prev)
                    Mode->prev->next = Mode;
                else
                    Sorted = Mode;
                break;
            }

            if (!Temp->next) {
                Temp->next = Mode;
                Mode->prev = Temp;
                Mode->next = NULL;
                break;
            }
        }
    }

    return Sorted;
}

/*
 * take a modename, try to parse it, if that works, generate the CVT modeline.
 */
static DisplayModePtr
ViaModeCreateFromName(ScrnInfoPtr pScrn, char *name, Bool Silent)
{
    VIAPtr pVia = VIAPTR(pScrn);
    DisplayModePtr Mode;
    int HDisplay = 0, VDisplay = 0, tmp;
    float VRefresh = 0;
    Bool Reduced;

    sscanf(name, "%dx%d@%f", &HDisplay, &VDisplay, &VRefresh);
    if (!HDisplay || !VDisplay) {
        if (!Silent)
            xf86DrvMsg(pScrn->scrnIndex, X_INFO, "%s: Unable to generate "
                       "Modeline for Mode \"%s\"\n", __func__, name);
        return NULL;
    }

    tmp = strlen(name) - 1;
    if ((name[tmp] == 'r') || (name[tmp] == 'R'))
        Reduced = TRUE;
    else
        Reduced = FALSE;

    Mode = ViaCVTMode(HDisplay, VDisplay, VRefresh, Reduced, FALSE);
    xfree(Mode->name);
    Mode->name = xnfstrdup(name);
    Mode->type = M_T_USERDEF;

    /* Do this before validation, it might become valid later on */
    pVia->XModes = ViaModesAdd(pVia->XModes, ViaModeCopy(Mode));

    tmp = ViaModeValidate(pScrn, Mode);
    if (tmp == MODE_OK)
        return Mode;
    return NULL;
}

/*
 *
 */
DisplayModePtr
ViaModesPoolCreate(ScrnInfoPtr pScrn, Bool Silent)
{
    DisplayModePtr Pool = NULL, List, TempList, Temp;
    char **ModeNames = pScrn->display->modes;
    int i;

    List = ViaCreateModesListAndValidate(pScrn, Silent);
    if (!List)
        return List;

    /* Reduce our list */
    if (ModeNames && ModeNames[0]) { /* Find the best matching mode for each name */
        for (i = 0; ModeNames[i]; i++) {
            TempList = ViaModesGrabOnNameAll(&List, ModeNames[i]);
            if (TempList) {
                Temp = ViaModesGrabOnHighestType(&TempList);
                ViaModesDestroy(TempList);

                TempList = Temp;
                Temp = ViaModesGrabOnTypeAll(&TempList, M_T_PREFERRED, M_T_PREFERRED);
                if (Temp) {
                    ViaModesDestroy(TempList);
                    TempList = Temp;
                }

                Temp = ViaModesGrabBestRefresh(&TempList);

                ViaModesDestroy(TempList);
            } else /* No matching modes found, generate */
                Temp = ViaModeCreateFromName(pScrn, ModeNames[i], Silent);

            if (Temp)
                Pool = ViaModesAdd(Pool, Temp);
        }

        ViaModesDestroy(List);
    } else { /* No names, just work the list directly */
        Temp = ViaModesGrabOnHighestType(&List);
        ViaModesDestroy(List);
        List = Temp;

        while (List) {
            TempList = ViaModesGrabOnNameAll(&List, List->name);

            Temp = ViaModesGrabOnTypeAll(&TempList, M_T_PREFERRED, M_T_PREFERRED);
            if (Temp) {
                ViaModesDestroy(TempList);
                TempList = Temp;
            }

            Temp = ViaModesGrabBestRefresh(&TempList);
            ViaModesDestroy(TempList);

            Pool = ViaModesAdd(Pool, Temp);
        }

        /* Sort our list */
        TempList = Pool;

        /* Sort higher priority modes seperately */
        Pool = ViaModesGrabOnTypeAll(&TempList, M_T_PREFERRED, M_T_PREFERRED);
        Pool = ViaModesSortOnSize(Pool);

        TempList = ViaModesSortOnSize(TempList);

        Pool = ViaModesAdd(Pool, TempList);
    }

    return Pool;
}

/*
 * Force 8 pixel alignment (VGA) for width here ourselves.
 * This while X needs 32 byte alignment for the framebuffer.
 */
void
ViaVirtualGetFromConfig(ScrnInfoPtr pScrn)
{
    VIAPtr pVia = VIAPTR(pScrn);
#ifndef USE_SECONDARY
    struct ViaCrtc *Crtc = pVia->Crtc[0];
#else
    struct ViaCrtc *Crtc = pVia->Crtc[1];
#endif
    int VirtualX = pScrn->display->virtualX;
    int VirtualY = pScrn->display->virtualY;
    int step = 32 * 8 / Crtc->bpp;
    int VideoRam = pScrn->videoRam * 1024;
    int tmpX, tmpY;
    int size;
    int MaxFrame;

    if (Crtc->MinPitch && (VirtualX < Crtc->MinPitch))
        VirtualX = Crtc->MinPitch;
    else if (Crtc->MaxPitch && (VirtualX > Crtc->MaxPitch))
        VirtualX = Crtc->MaxPitch;
    else
        VirtualX &= ~0x07;

    if (VirtualY < 128) /* Be reasonable */
        VirtualY = 128;

    MaxFrame = Crtc->MaxOffset - Crtc->Offset;
    if (MaxFrame > VideoRam)
        MaxFrame = VideoRam;

    /* 32byte alignment, plus, bitsPerPixel / 8 */
    size = (((VirtualX >> 3) + 0x03) & ~0x03) * VirtualY * Crtc->bpp;

    if (size >= MaxFrame) { /* scale down both VirtualX and VirtualY */
        tmpX = VirtualX & ~(step - 1); /* make sure we've already rounded down */
        tmpY = VirtualY;

        while (tmpX > Crtc->MinPitch) {
            tmpY = VirtualY * tmpX / VirtualX;
            size = (((tmpX >> 3) + 0x03) & ~0x03) * tmpY * Crtc->bpp;

            if (size < MaxFrame)
                break;

            tmpX -= step;
        }

        VirtualX = tmpX;
        VirtualY = tmpY;
    }

    pScrn->virtualX = VirtualX;
    pScrn->virtualY = VirtualY;
    pScrn->displayWidth = (VirtualX + step - 1) & ~(step - 1);
}

/*
 *
 */
void
ViaVirtualGetFromModes(ScrnInfoPtr pScrn)
{
    VIAPtr pVia = VIAPTR(pScrn);
#ifndef USE_SECONDARY
    struct ViaCrtc *Crtc = pVia->Crtc[0];
#else
    struct ViaCrtc *Crtc = pVia->Crtc[1];
#endif
    DisplayModePtr Mode;
    int VirtualX = 0;
    int VirtualY = 0;
    int step = 32 * 8 / Crtc->bpp;
    int TmpX, TmpY;
    int Offset;

    Mode = pScrn->modes;
    do {
        /* Before we accept the highest combined resolution, we need to check
           whether the offset is acceptable. Pitch was already handled. */
        if ((Mode->CrtcHDisplay > VirtualX) ||
            (Mode->CrtcVDisplay > VirtualY)) {

            TmpX = VirtualX;
            TmpY = VirtualY;

            if (Mode->CrtcHDisplay > VirtualX)
                TmpX = Mode->CrtcHDisplay;
            if (Mode->CrtcVDisplay > VirtualY)
                TmpY = Mode->CrtcVDisplay;

            Offset = (((TmpX >> 3) + 0x03) & ~0x03) * TmpY * Crtc->bpp;
            Offset += Crtc->Offset;
            if ((Offset <= Crtc->MaxOffset) && (Offset <= (pScrn->videoRam * 1024))) {
                VirtualX = TmpX;
                VirtualY = TmpY;
            }
        }
        Mode = Mode->next;
    } while (Mode != pScrn->modes);

    pScrn->virtualX = VirtualX;
    pScrn->virtualY = VirtualY;
    pScrn->displayWidth = (VirtualX + step - 1) & ~(step - 1);
}

/*
 * Does all the work necessary to add this list to our screen.
 */
void
ViaModesAttach(ScrnInfoPtr pScrn, DisplayModePtr Modes)
{
    DisplayModePtr Mode = Modes;

    pScrn->modes = Modes;
    pScrn->currentMode = Modes;

    while (Mode->next) {
        Mode->type = M_T_USERDEF; /* satisfy xf86ZoomViewport */
        Mode = Mode->next;
    }

    Mode->type = M_T_USERDEF;

    /* Make our list circular */
    Mode->next = pScrn->modes;
    pScrn->modes->prev = Mode;
}

/*
 *
 */
void
ViaModeSet(ScrnInfoPtr pScrn, DisplayModePtr mode)
{
    vgaHWPtr hwp = VGAHWPTR(pScrn);
    VIAPtr pVia = VIAPTR(pScrn);
#ifndef USE_SECONDARY
    struct ViaCrtc *Crtc = pVia->Crtc[0];
#else
    struct ViaCrtc *Crtc = pVia->Crtc[1];
#endif
    struct ViaOutput *Output;

    VIAFUNC(pScrn->scrnIndex);

    ViaDebug(pScrn->scrnIndex, "%s: Using CRTC %s (%d)\n", __func__,
             Crtc->Name, Crtc->ID);

    ViaCrtcMask(hwp, 0x17, 0x00, 0x80);
    Crtc->Reset(Crtc, TRUE);

    /* The VT3122A can't really cope on secondary with primary active */
#ifndef USE_SECONDARY
    pVia->Crtc[1]->Reset(pVia->Crtc[1], TRUE);
#else
    pVia->Crtc[0]->Reset(pVia->Crtc[0], TRUE);
#endif

    Crtc->Enable(Crtc, TRUE);

    /* Disable simultaneous */
    ViaCrtcMask(hwp, 0x6B, 0x00, 0x08);

    ViaCrtcInitForFB(pScrn, Crtc);
    Crtc->ModeSet(Crtc, mode);
    Crtc->FIFOSet(Crtc, mode);

    for (Output = pVia->Outputs; Output; Output = Output->Next) {
        if (Output->Owner != Crtc->ID)
            continue;

        if (Output->Active) {
            ViaOutputBusPower(Crtc, Output->Position, TRUE);
            ViaOutputBusSet(Crtc, Output->Position);
            if (Output->Mode)
                Output->Mode(Output, mode);
        } else {
            if (Output->Power)
                Output->Power(Output, FALSE);
            ViaOutputBusPower(Crtc, Output->Position, FALSE);
        }
    }

    Crtc->PLLSet(Crtc, mode->Clock, Crtc->PLLFlags);

    if (Crtc->bpp == 8) /* 8bit is palette, not gamma */
        ViaOutputsGamma(pScrn, FALSE);
    else
        ViaOutputsGamma(pScrn, TRUE);

    Crtc->Reset(Crtc, FALSE);

    ViaCrtcMask(hwp, 0x17, 0x80, 0x80);
}
