/*  This file is part of "xprintmon"
 *  Copyright (C) 2006 Bernhard R. Link
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02111-1301  USA
 */
#include <config.h>

#include <assert.h>
#include <stdio.h>
#include <stdbool.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <strings.h>

#include <X11/Intrinsic.h>

#include "global.h"

struct request {
	struct request *next;
	struct requests *parent;
	char *name;
	struct child *lpq;
	size_t processed;
	timecode timestart, timefinished;
	/* call the next one, once this is finished */
	bool iterative;
	char *username;
} requests;

struct requests {
	summaryCallback *callback;
	finishedCallback *callatend;
	void *frame;
	struct request *requests;
};

void requestsFree(struct requests *requests) {
	struct request *request;

	if( requests == NULL )
		return;
	request = requests->requests;
	free(requests);
	while( request != NULL ) {
		struct request *next = request->next;

		free(request->name);
		free(request->username);
		if( request->lpq != NULL )
			abortCollector(request->lpq);
		free(request);
		request = next;
	}
}

struct requests *requestsCreate(const char *lookfor, summaryCallback callback, finishedCallback callatend, void *frame) {
	const char *start;
	struct requests *result;
	struct request **next_p;

	result = malloc(sizeof(struct requests));
	if( result == NULL )
		return NULL;
	result->callback = callback;
	result->callatend = callatend;
	result->frame = frame;
	start = lookfor;
	result->requests = NULL;
	next_p = &result->requests;
	while( start != NULL && *start != '\0' ) {
		char *name;
		struct request *n;
		const char *end = index(start, '|');
		if( end == NULL )
			name = strdup(start);
		else {
			name = strndup(start,end-start);
			end++;
		}
		if( name == NULL )
			continue;
		start = end;
		n = calloc(1,sizeof(struct request));
		if( n == NULL ) {
			free(name);
			continue;
		}
		n->name = name;
		n->lpq = NULL;
		n->processed = 0;
		n->parent = result;
		n->next = NULL;
		assert( *next_p == NULL );
		*next_p = n;
		next_p = &n->next;
	}
	return result;
}

/* This parses a line of length len, but assumes it can read over len and will
 * either find \n or \0 there */
static void parsesummaryline(struct request *request, const char *line, size_t len, timecode timecode) {
#define STATUSLINE "XPRINTMON:"
#define IGNORELINE1 "Make sure the remote hos supports the LPD protocol"
#define IGNORELINE2 "and accepts connections from this host and from non-privileged (>1023) ports"
	const char *e,*p,*printername,*remark,*destination;
	size_t printerlength, remarklength, destinationlength;
	enum lpq_state printerstate, spoolstate;
	int jobs;
	assert( request->parent != NULL );

	e = line;
	while( (size_t)(e-line) < len && *e != ' ' )
		e++;
	// TODO: handle unparseable lines somehow
	if( (size_t)(e-line) >= len )
		return;
	if( len >= sizeof(STATUSLINE) &&
			strncmp(line, STATUSLINE, sizeof(STATUSLINE)-1)== 0 )  {
		// TODO: check for error message
		return;
	}
	if( len >= sizeof(IGNORELINE1)-1 &&
			strncmp(line, IGNORELINE1, sizeof(IGNORELINE1)-1)== 0 )
		return;
	if( len >= sizeof(IGNORELINE2)-1 &&
			strncmp(line, IGNORELINE2, sizeof(IGNORELINE2)-1)== 0 )
		return;
#undef STATUSLINE
#undef IGNORELINE1
#undef IGNORELINE2
	if( e-line == 7 && strncmp(line, "Printer", 7) == 0 ) {
		e++;
		if( *e == '\'' ) {
			e++;
			printername = e;
			while( (size_t)(e-line) < len && *e != '\'' )
				e++;
			if( (size_t)(e-line) >= len )
				return;
			if( e[1] == ' ' && e[2] == '-' && e[3] == ' ' ) {
				printerlength = e-printername;
				e += 4;
				request->parent->callback(
						request->parent->frame,
						timecode,
						printername, printerlength,
						-1,
						LPQ_UNKNOWN,
						LPQ_UNKNOWN,
						"", 0,
						e, len-(e-line));
			}
			return;
		}
		return;
	}
	if( e-line == 3 && strncmp(line, "bad", 3) == 0 )
		return;
	printername = line;
	printerlength = e-line;
	e++;
	p = e;
	// TODO: check for '(' here, too
	jobs = strtol(p,(char**)&e,10);
	if( e <= p || *(e++) != ' ' )
		return;
	if( *(e++) != 'j' || *(e++) != 'o' || *(e++) != 'b' )
		return;
	if( *e == 's' )
		e++;
	printerstate = LPQ_ENABLED;
	spoolstate = LPQ_ENABLED;
	destination = "";
	destinationlength = 0;
	remark = "";
	remarklength = 0;
	while( (size_t)(e-line) < len ) {
		while( *e == ' ' )
			e++;
		if( *e != '(' )
			break;
		do {
			e++;
			while( *e == ' ' )
				e++;
			if( strncmp(e, "printing disabled", 17) == 0 ) {
				e+=17;
				printerstate = LPQ_DISABLED;
			} else if( strncmp(e, "spooling disabled", 17) == 0 ) {
				e+=17;
				spoolstate = LPQ_DISABLED;
			} else if( strncmp(e, "dest ", 5) == 0 ) {
				e+=5;
				destination = e;
				while( (size_t)(e-line) < len &&
						*e != ',' && *e != ')')
					e++;
				destinationlength = e-destination;
			} else {
				// put those in remarks instead....
				while( (size_t)(e-line) < len &&
						*e != ',' && *e != ')')
					e++;
			}
		} while( *e == ',' );
		if( *e == ')' )
			e++;
		else
			break;
	}
	request->parent->callback(request->parent->frame, timecode,
			printername, printerlength,
			jobs, spoolstate, printerstate,
			destination, destinationlength,
			remark, remarklength);
}

static void lpqs(void *privdata, const char *data, size_t newsize, bool finished);

static void startRequest(struct request *request, timecode timecode, char *user) {
	request->username = user;
	if( request->name[0] == '-' ) {
		request->lpq = newCollectorVa(context, lpqs, request,
				"lpq", "-s", request->name, user, NULL);
	} else {
		request->lpq = newCollectorVa(context, lpqs, request,
				"lpq", "-s", "-P", request->name, user, NULL);
	}
	request->timestart = timecode;
}

static void lpqs(void *privdata, const char *data, size_t newsize, bool finished) {
	struct request *request = privdata;
	timecode timecode;
	size_t i;

	assert( request != NULL );

	timecode = request->timestart;
	for( i = request->processed ; i < newsize ; i++ ) {
		if( data[i] == '\n' ) {
			size_t j = i;
		while( j > request->processed && isspace(data[j-1]) )
				j--;
			if( j >= request->processed ) {
				parsesummaryline(request,
						data+request->processed,
						j-request->processed,
						timecode);
			}
			request->processed = i+1;
		}
	}

	if( finished ) {
		struct request *r;
		char *username = request->username;

		request->username = NULL;
		if( request->iterative ) {
			/* start the next one */
			struct request *next = request->next;
			while( next != NULL ) {
				if( next->lpq == NULL ) {
					startRequest(next, timecode, username);
					next->iterative = true;
					if( next->lpq != NULL ) {
						username = NULL;
						break;
					} else
						next->username = NULL;
				}
			}
		}
		free(username);
		/* also process unterminated lines: */
		if( newsize > request->processed ) {
			parsesummaryline(request,
					data+request->processed,
					newsize-request->processed,
					timecode);
			request->processed = newsize;
		}
		request->processed = 0;
		/* remove the callback: */
		abortCollector(request->lpq);
		request->lpq = NULL;

		for( r = request->parent->requests; r != NULL ; r = r->next ){
			if( r->timefinished < timecode && r != request )
				break;
		}
		if( r == NULL ) {
			request->parent->callatend(request->parent->frame,
					timecode);

		}
		/* it's quite possible most timecodes do not get through
		 * this way, if two jobs are interlaced in a applicaple
		 * way, but no false ones should get through this way */
		request->timefinished = timecode;
	}
}

void requestsTrigger(struct requests *r, bool iterative, timecode timecode, const char *username) {
	struct request *request;

	for( request = r->requests; request != NULL; request = request->next) {
		if( request->lpq == NULL ) {
			char *user = strdup(username);
			startRequest(request, timecode, user);
		}
		request->iterative = iterative;
		if( iterative && request->lpq != NULL )
			break;
		else
			continue;
	}
}
